﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "Constants.h"
#include <nn/nn_Macro.h>
#include <memory> // For unique_ptr<>

class TestlistGenerator;

namespace ParserHelpers
{
    size_t LineLen(const char* pStr) NN_NOEXCEPT;
    bool ContainsNonWhiteSpace(const char* pLine, size_t lineLen) NN_NOEXCEPT;
}

// Abstract class used to read a line from any source
class LineReader
{
public:
    virtual ~LineReader() {}
    virtual char* ReadLine(char* pOutBuffer, uint32_t bufLen) NN_NOEXCEPT = 0;
    virtual bool IsEndOfStream() NN_NOEXCEPT = 0;
};

// Used to read a line from any a file
class FileLineReader : public LineReader
{
private:
    FILE* m_pReadFile;

public:
    explicit FileLineReader(FILE* pReadFile) NN_NOEXCEPT
        : m_pReadFile(pReadFile)
    {}

    virtual char* ReadLine(char* pOutBuffer, uint32_t bufLen) NN_NOEXCEPT NN_OVERRIDE
    {
        return fgets(pOutBuffer, bufLen, m_pReadFile);
    }

    virtual bool IsEndOfStream() NN_NOEXCEPT NN_OVERRIDE
    {
        return feof(m_pReadFile) != 0;
    }
};

// Used to read a line from a buffer
class BufferLineReader : public LineReader
{
private:
    const char* const m_pReadBuffer;
    const uint32_t m_bufLen;
    uint32_t m_readPos;

public:
    BufferLineReader(const char* pReadBuffer, uint32_t bufLen) NN_NOEXCEPT
        : m_pReadBuffer(pReadBuffer),
          m_bufLen(bufLen),
          m_readPos(0)
    {}

    virtual char* ReadLine(char* pOutBuffer, uint32_t bufLen) NN_NOEXCEPT NN_OVERRIDE;
    virtual bool IsEndOfStream() NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_bufLen == 0) || (m_readPos >= m_bufLen - 1); // -1 for null-terminator
    }
};

// Used to store parameters inside a <LOOP> definition
struct LoopParam
{
    LoopParam* pNextParam;
    LoopParam* pNextIter;
    uint32_t startLine;
    uint32_t dataLen;
    char pData[MaxLoopParamBufLen];

    LoopParam() NN_NOEXCEPT
        : pNextParam(nullptr),
            pNextIter(nullptr),
            startLine(0),
            dataLen(0)
    {
        memset(pData, '\0', sizeof(pData));
    }

    ~LoopParam() NN_NOEXCEPT
    {
        // Recursively deletes list
        if(nullptr != pNextParam)
        {
            delete pNextParam;
        }

        // Recursively deletes list
        if(nullptr != pNextIter)
        {
            delete pNextIter;
        }
    }
};

enum class LoopReadState : uint8_t
{
    ReadingNothing = 0,
    ReadingParam   = 1,
    ReadingBody    = 2,
};

// Keeps track of state information while parsing a <LOOP> command
struct LoopState
{
    LoopState() NN_NOEXCEPT
        : pRootParam(nullptr),
          pCurIter(nullptr),
          pCurParam(nullptr),
          isAtEndOfLoop(false),
          doWriteLine(true),
          nestedLoopCounter(0),
          iterationCount(0),
          bodyLine(0),
          loopBodyLen(0),
          eLoopState(LoopReadState::ReadingNothing),
          isInIteration(false),
          isNewIteration(false)
    {}

    std::unique_ptr<char[]> pLoopBody;
    LoopParam* pRootParam;
    LoopParam* pCurIter;
    LoopParam* pCurParam;
    bool isAtEndOfLoop;
    bool doWriteLine;
    uint32_t nestedLoopCounter;
    uint32_t iterationCount;
    uint32_t bodyLine;
    uint32_t loopBodyLen;
    LoopReadState eLoopState;
    bool isInIteration;
    bool isNewIteration;
};

// Used to keep track of the parsing state.
struct ParseInfo
{
    ParseInfo(const char* _pLine, size_t _lineLen, const ParseInfo& baseValues) NN_NOEXCEPT
        : lineReader(baseValues.lineReader),
            pLine(_pLine),
            lineLen(_lineLen),
            pPrevIter(baseValues.pPrevIter),
            pCurIter(baseValues.pCurIter),
            lineNumber(baseValues.lineNumber),
            isIgnoreEmptyLinesEnabled(baseValues.isIgnoreEmptyLinesEnabled),
            isIgnoreBeginningWhiteSpaceEnabled(baseValues.isIgnoreBeginningWhiteSpaceEnabled)
    {
        strncpy(pFileName, baseValues.pFileName, sizeof(pFileName));
    }

    ParseInfo(LineReader& _lineReader, const char* _pLine, size_t _lineLen, LoopParam* _pPrevIter, LoopParam* _pCurIter, uint32_t& _lineNumber, bool _isIgnoreEmptyLinesEnabled, bool _isIgnoreBeginningWhiteSpaceEnabled, const char* _pFileName) NN_NOEXCEPT
        : lineReader(_lineReader),
            pLine(_pLine),
            lineLen(_lineLen),
            pPrevIter(_pPrevIter),
            pCurIter(_pCurIter),
            lineNumber(_lineNumber),
            isIgnoreEmptyLinesEnabled(_isIgnoreEmptyLinesEnabled),
            isIgnoreBeginningWhiteSpaceEnabled(_isIgnoreBeginningWhiteSpaceEnabled)
    {
        strncpy(pFileName, _pFileName, sizeof(pFileName));
    }

    bool CheckForCommand(const char* pCommand, size_t commandLen) NN_NOEXCEPT;

    LineReader& lineReader;
    const char* pLine;
    size_t lineLen;
    LoopParam* pPrevIter;
    LoopParam* pCurIter;
    uint32_t& lineNumber;
    bool isIgnoreEmptyLinesEnabled;
    bool isIgnoreBeginningWhiteSpaceEnabled;
    char pFileName[MaxFileNameBufLen];
};

// Used to store parameters for the <DEFINE> command
struct DefineParam
{
    DefineParam* pNextParam;
    uint32_t startLine;
    uint32_t dataLen;
    char pFileName[MaxFileNameBufLen];
    char pName[MaxDefineNameBufLen];
    char pData[MaxDefineBufLen];

    DefineParam() NN_NOEXCEPT
        : pNextParam(nullptr),
            startLine(0),
            dataLen(0)
    {
        memset(pFileName, '\0', sizeof(pFileName));
        memset(pData,     '\0', sizeof(pData));
        memset(pName,     '\0', sizeof(pName));
    }

    ~DefineParam() NN_NOEXCEPT
    {
        if(nullptr != pNextParam)
        {
            delete pNextParam; // Recursively deletes the rest of the list
        }
    }
};

// Used to create a list of member function ptrs that parse all the commands.
class CommandParser
{
public:

    typedef int32_t (TestlistGenerator::* ParserFn)(ParseInfo&) NN_NOEXCEPT;

    explicit CommandParser(ParserFn _pParserFn) NN_NOEXCEPT
        : pParserFn(_pParserFn),
          pNext(nullptr) {}

    virtual ~CommandParser() NN_NOEXCEPT
    {
        if(nullptr != pNext)
        {
            delete pNext; // Recursively deletes every node after this in the list.
        }
    }

    const ParserFn pParserFn;
    CommandParser* pNext;
};
