﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
 HTCS Host-side test example
 *---------------------------------------------------------------------------*/


#include <nnt/nntest.h>
#include <nn/os.h>
#include <nn/nn_Log.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <Utils/HTCSUtil.h>
#include <Utils/GTestUtil.h>


namespace {

// transport coordinates.
const char * const HtcsSessionServerHost = "localhost";
const char * const HtcsSessionServerPort = "8003";
const char * const HtcsSdevServerName    = "ServerInTarget";
const char * const HtcsTicsSessionName   = "TicsServerName";

}




namespace NATF {
namespace Tests {



class TestHTCSSession: public nnt::net::SimpleHTCSPlug
{


private: // constants
    static const int BufSize = 1024;

private: // data contents constants
    static const char * const TestString;

public: // constants
    static const int HtcsConnTimeoutMs = 30000;

public:
    explicit TestHTCSSession( tics::CrossBar *pCrossBar ):
             nnt::net::SimpleHTCSPlug( pCrossBar,
                                       HtcsSessionServerHost, HtcsSessionServerPort,
                                       HtcsSdevServerName,    HtcsTicsSessionName ),
        m_buffer( BufSize ),
        m_pCrossBar( pCrossBar ),
        m_bStringMatched( false )
    {
    }

    void OnSessionStarted (::tics::portability::stl::string type, ::tics::Endpoint* connectedEP) NN_OVERRIDE;
    int  OnSendComplete(::tics::Buffer* buffer, long offset, long len) NN_OVERRIDE;
    int  OnReceiveComplete(::tics::Buffer* buffer, long offset, long len) NN_OVERRIDE;
    void PvtDetach();

    int OnDetach() NN_OVERRIDE
    {
        PvtDetach();
        return 0;
    }

    int OnRemoteDetach( ::tics::DisconnectReason ) NN_OVERRIDE
    {
        PvtDetach();
        return 0;
    }

    bool IsStringMatched()
    {
        return m_bStringMatched;
    }

private:
    ::tics::RawMemBuffer m_buffer;
    tics::CrossBar *m_pCrossBar;
    bool m_bStringMatched;
};



/// --------- constants from the class
const char * const TestHTCSSession::TestString          = "hello htcs\n";




void TestHTCSSession::OnSessionStarted (::tics::portability::stl::string type, ::tics::Endpoint* connectedEP)
{
    tics::Plug::OnSessionStarted( type, connectedEP );
    strcpy( (char *)(m_buffer.GetPayLoad()), TestString );
    BeginSend(&m_buffer, 0,  (long)strlen((const char *)(m_buffer.GetPayLoad())), true);
}

int TestHTCSSession::OnSendComplete(::tics::Buffer* buffer, long offset, long len)
{
    // echo - so I expect to get a buffer the exact size as what I sent
    BeginReceive( len, len, &m_buffer, 0 );
    return 0;
}

int TestHTCSSession::OnReceiveComplete(::tics::Buffer* buffer, long offset, long len)
{
    long expectedLen = (long)strlen(TestString);

    NN_LOG("\nReceived %d bytes from SDEV\n", len);
    if( len == expectedLen )
    {
        if( memcmp( buffer->GetPayLoad(), TestString, len) == 0 )
        {
            m_bStringMatched = true;
        }
    }

    // disconnect regardless
    BeginDetach();
    return 0;
}

void TestHTCSSession::PvtDetach()
{
    NN_LOG("\nv3Detach event caught in session, will shutdown\n");
    m_pCrossBar->BeginShutdown(::tics::CrossBar::ExitDone);
}




} }; // NATF::Tests namespace






TEST(natf, HTCSLib_Host)
{
    try
    {

        tics::CrossBar *pCb;
        NATF::Tests::TestHTCSSession *pSession;
        bool bMatch = false;

        static char __tics_heap[ 16 * 1024 * 1024 ];

        // Initialization
        ::tics::Initialize( (uintptr_t)__tics_heap, sizeof(__tics_heap) );


        NN_LOG("\nStarting: HTCSLib_Host v6\n");


        pCb = new tics::CrossBar();

        pSession = new NATF::Tests::TestHTCSSession( pCb );

        NATF_EXPECT_TRUE( pSession->GetFactory().ParseStateOk(),
             "ERROR: Parser did not initialize successfully, bMatch=%d\n", (int)bMatch );

        pSession->GetFactory().SetAutoTeardown(); // disconnect when found
        pSession->GetFactory().SetTimeout( NATF::Tests::TestHTCSSession::HtcsConnTimeoutMs );

        // blocking - will run all HTCS communication (host to discovery session plus session itself)
        pCb->Run();

        bMatch = pSession->IsStringMatched();

        // order matters here
        delete pSession;

        NATF_EXPECT_TRUE( bMatch,
             "ERROR: Echo string did not match, bMatch=%d\n", (int)bMatch );

        NN_LOG("Ending: string did match\n\n");
    }
    catch( const ::tics::portability::stl::string &xcpt )
    {
        NN_LOG("\nError: Exception [%s]\n\n", xcpt.c_str() );
        FAIL();
    }
    catch(...)
    {
        NN_LOG("\nError: unknown exception thrown during test\n\n");
        FAIL();
    }

}

