﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <cstdarg>
#include <nn/socket.h>
#include <nn/nn_Log.h>
#include <nn/tsc.h>
#include <nn/tsc/tsc_DebugLibrary.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/init.h>
#include <nnt/nntest.h>
#include <nn/nn_Assert.h>
#include <nnt.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/bsdsocket/cfg/cfg_Types.h>
#include <nn/bsdsocket/cfg/cfg_ClientApi.h>
#include <nn/eth/eth_EthClient.h>

#include <cstring> // memset

#define EXPECT_ERROR(CALL, ERROR)                                                \
do                                                                               \
{                                                                                \
    nn::Result RESULT = (CALL);                                                  \
    if( RESULT.IsSuccess() )                                                     \
    {                                                                            \
        NN_LOG("\n*********************************************************\n"); \
        NN_LOG("* Unexpected call succeeded. Expected error: %s\n", #ERROR);     \
        NN_LOG("* Call: %s - Line: %d\n", #CALL, __LINE__);                      \
        NN_LOG("*  Module: %d, Description: %d\n", RESULT.GetModule(), RESULT.GetDescription()); \
        NN_LOG("*  Error String: %s\n", nn::tsc::debug::GetErrorReasonString(RESULT)); \
        NN_LOG("*********************************************************\n\n"); \
        isSuccess = false;                                                       \
        goto out;                                                                \
    }                                                                            \
    else if( (RESULT.GetModule()      != ERROR.GetModule())      ||              \
             (RESULT.GetDescription() != ERROR.GetDescription()) )               \
    {                                                                            \
        NN_LOG("\n*********************************************************\n"); \
        NN_LOG("* Got unexpected error: %d, Expected error: %d\n", RESULT.GetDescription(), ERROR.GetDescription());     \
        NN_LOG("* Call: %s - Line: %d\n", #CALL, __LINE__);                      \
        NN_LOG("*  Module: %d, Description: %d\n", RESULT.GetModule(), RESULT.GetDescription()); \
        NN_LOG("*  Error String: %s\n", nn::tsc::debug::GetErrorReasonString(RESULT)); \
        NN_LOG("*********************************************************\n\n"); \
        isSuccess = false;                                                       \
        goto out;                                                                \
    }                                                                            \
} while( NN_STATIC_CONDITION(false) )

#define EXPECT_SUCCESS(CALL)                                                     \
do                                                                               \
{                                                                                \
    nn::Result RESULT = (CALL);                                                  \
    if( RESULT.IsFailure() )                                                     \
    {                                                                            \
        NN_LOG("\n*********************************************************\n"); \
        NN_LOG("* Error Call: %s - Line: %d\n", #CALL, __LINE__);                \
        NN_LOG("*  Module: %d, Description: %d\n", RESULT.GetModule(), RESULT.GetDescription()); \
        NN_LOG("*  Error String: %s\n", nn::tsc::debug::GetErrorReasonString(RESULT)); \
        NN_LOG("*********************************************************\n\n"); \
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(2000) );             \
        isSuccess = false;                                                       \
        goto out;                                                                \
    }                                                                            \
} while( NN_STATIC_CONDITION(false) )

extern "C" void nninitStartup()
{    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize + nn::socket::DefaultSocketMemoryPoolSize );

    NN_ASSERT( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

namespace
{
    const uint32_t MaxIfLen = 128;
    const uint32_t MaxWaitTimeSec = 3; // Wait time for usb eth interface to be detected.
    const size_t ClientStackSize = (16 * 1024);
    const char* g_pInterfaceName = nullptr;
    const uint16_t ServerPort   = 8088;  // There is a TCP listening socket on this port
    const uint16_t NoServerPort = 8089;  // There is NOT a TCP socket on this port.
    const char* const ServerIp   = "192.168.11.101";

    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
}
class EthernetHandler
{
public:
    uint8_t                                    m_Stack[ClientStackSize] NN_ALIGNAS(nn::os::ThreadStackAlignment);
    nn::os::MultiWaitType                      m_MultiWait;
    nn::os::MultiWaitHolderType                m_InterfaceGroupHolder;
    nn::os::ThreadType                         m_ClientThread;
    nn::eth::client::InterfaceGroupHandler*    m_EthernetInterfaceGroup;
    bool                                       m_Running;
    nn::eth::InterfaceList                     m_CurrentInterfaceList;
    char                                       m_pStringStorage[16];

private:
    // Link group or event handler into multi wait
    template <typename T>
    void SetupWaitHolder(nn::os::MultiWaitHolderType* holder, T* impl)
    NN_NOEXCEPT
    {
        nn::os::InitializeMultiWaitHolder(holder,  reinterpret_cast<nn::os::SystemEventType*>(impl->GetSystemEventPointer()));
        nn::os::SetMultiWaitHolderUserData(holder, reinterpret_cast<uintptr_t>(impl));
        nn::os::LinkMultiWaitHolder(&m_MultiWait, holder);
    }

    // Handle adapter insertion/removal
    void HandleInterfaceGroupEvent()
    NN_NOEXCEPT
    {
        nn::Result result;
        nn::eth::InterfaceList removedList;
        nn::eth::InterfaceList addedList;
        nn::eth::InterfaceList newList;

        m_EthernetInterfaceGroup->ClearEvent();

        result = m_EthernetInterfaceGroup->GetInterfaceList(&newList);
        NN_ABORT_UNLESS(result.IsSuccess());

        // parse and update local interface list
        m_EthernetInterfaceGroup->ParseInterfaceList(&newList, &m_CurrentInterfaceList, &addedList, &removedList);
        m_CurrentInterfaceList = newList;

        // handle removals
        for (uint32_t i = 0; i < removedList.adapterCount; i++)
        {
            NN_LOG("Removed:\n"
                   "[%s]model:    %08x\n"
                   "[%s]revision: %08x\n"
                   "[%s]mac:      %02x:%02x:%02x:%02x:%02x:%02x\n",
                    removedList.adapterInfo[i].interfaceName,
                    removedList.adapterInfo[i].interfaceModel,
                    removedList.adapterInfo[i].interfaceName,
                    removedList.adapterInfo[i].interfaceRevision,
                    removedList.adapterInfo[i].interfaceName,
                    removedList.adapterInfo[i].interfaceMacAddress[0],
                    removedList.adapterInfo[i].interfaceMacAddress[1],
                    removedList.adapterInfo[i].interfaceMacAddress[2],
                    removedList.adapterInfo[i].interfaceMacAddress[3],
                    removedList.adapterInfo[i].interfaceMacAddress[4],
                    removedList.adapterInfo[i].interfaceMacAddress[5]
            );

            // deallocations will be done inside link state handler on
            // an event containing ResultInterfaceRemoved() result
        }

        // handle additions
        for (uint32_t i = 0; i < addedList.adapterCount; i++)
        {
            nn::eth::client::InterfaceHandler* ethernetInterface;
            nn::os::MultiWaitHolderType*          ethernetInterfaceHolder;

            strncpy(m_pStringStorage, addedList.adapterInfo[i].interfaceName, 16);
            g_pInterfaceName = m_pStringStorage;

            NN_LOG("Added:\n"
                   "[%s]model:    %08x\n"
                   "[%s]revision: %08x\n"
                   "[%s]mac:      %02x:%02x:%02x:%02x:%02x:%02x\n",
                    addedList.adapterInfo[i].interfaceName,
                    addedList.adapterInfo[i].interfaceModel,
                    addedList.adapterInfo[i].interfaceName,
                    addedList.adapterInfo[i].interfaceRevision,
                    addedList.adapterInfo[i].interfaceName,
                    addedList.adapterInfo[i].interfaceMacAddress[0],
                    addedList.adapterInfo[i].interfaceMacAddress[1],
                    addedList.adapterInfo[i].interfaceMacAddress[2],
                    addedList.adapterInfo[i].interfaceMacAddress[3],
                    addedList.adapterInfo[i].interfaceMacAddress[4],
                    addedList.adapterInfo[i].interfaceMacAddress[5]
            );

            // create interface handler for new interface,
            // add its event to multi wait

            ethernetInterface = new nn::eth::client::InterfaceHandler;
            NN_ABORT_UNLESS(ethernetInterface != nullptr);

            result = ethernetInterface->Initialize(m_EthernetInterfaceGroup, &addedList.adapterInfo[i], nn::os::EventClearMode_ManualClear);
            NN_ABORT_UNLESS(result.IsSuccess());

            ethernetInterfaceHolder = new nn::os::MultiWaitHolderType;
            NN_ABORT_UNLESS(ethernetInterfaceHolder != nullptr);

            SetupWaitHolder<nn::eth::client::InterfaceHandler>(ethernetInterfaceHolder, ethernetInterface);

            // request auto negotiation
            result = ethernetInterface->SetMediaType(nn::eth::MediaType_AUTO);
            NN_ABORT_UNLESS(result.IsSuccess());

        }
    }

    // Handle link state changes
    void HandleInterfaceMediaEvent(nn::os::MultiWaitHolderType* holder)
    NN_NOEXCEPT
    {
        nn::eth::MediaType current;
        nn::eth::MediaType requested;
        nn::Result result;
        uint32_t   eventCounter;

        nn::eth::client::InterfaceHandler* ethernetInterface =
                reinterpret_cast<nn::eth::client::InterfaceHandler*>
                                    (nn::os::GetMultiWaitHolderUserData(holder));

        // clear this before processing so that if
        // event happens while processing is in progress
        // we would not miss it.
        ethernetInterface->ClearEvent();

        if (ethernetInterface->GetResult() <= ResultInterfaceRemoved())
        {
            NN_LOG("[%s] Adapter removed\n", ethernetInterface->GetInterfaceName());
            nn::os::UnlinkMultiWaitHolder(holder);
            delete holder;
            delete ethernetInterface;
        }
        else
        {
            // obtain interface state
            result = ethernetInterface->GetMediaType(&requested, &current, &eventCounter);
            NN_ABORT_UNLESS(result.IsSuccess());
        }

    }

    void ClientThread()
    NN_NOEXCEPT
    {
        nn::Result result;

        m_CurrentInterfaceList.adapterCount = 0;

        m_EthernetInterfaceGroup = new nn::eth::client::InterfaceGroupHandler;
        NN_ABORT_UNLESS(m_EthernetInterfaceGroup != nullptr);

        // open session to the server
        result = m_EthernetInterfaceGroup->Initialize(nn::os::EventClearMode_ManualClear);
        NN_ABORT_UNLESS(result.IsSuccess());

        // add group interface handler to multi wait
        nn::os::InitializeMultiWait(&m_MultiWait);
        SetupWaitHolder<nn::eth::client::InterfaceGroupHandler>(&m_InterfaceGroupHolder, m_EthernetInterfaceGroup);

        while (m_Running)
        {
            // wait for group change or link state change
            nn::os::MultiWaitHolderType* pendingEventHolder = nn::os::TimedWaitAny(&m_MultiWait, nn::TimeSpan::FromMilliSeconds(100));

            if (pendingEventHolder == &m_InterfaceGroupHolder)
            {
                // interface was added or removed
                HandleInterfaceGroupEvent();
            }
            else if (pendingEventHolder != nullptr)
            {
                // link state has changed
                HandleInterfaceMediaEvent(pendingEventHolder);
            }
        }
    }

    static void ClientThreadEntry(void* arg) NN_NOEXCEPT
    {
        EthernetHandler* p = (EthernetHandler*)arg;
        p->ClientThread();
    }

public:

    void Initialize()
    NN_NOEXCEPT
    {
        g_pInterfaceName = nullptr;
        m_Running = true;
        nn::os::CreateThread(&m_ClientThread, ClientThreadEntry, this, m_Stack, sizeof(m_Stack), nn::os::DefaultThreadPriority);
        nn::os::StartThread(&m_ClientThread);
    }

    void Shutdown()
    NN_NOEXCEPT
    {
        m_Running = false;
        nn::os::WaitThread(&m_ClientThread);
        nn::os::DestroyThread(&m_ClientThread);
    }
};

TEST(Tsc, Negative)
{
    bool isSuccess = true;
    char pInterfaceName[MaxIfLen];
    nn::tsc::Ipv4AddrStorage ipAddr;
    uint32_t mtuValue = 0;
    nn::tsc::Ipv4Config ipConfig;

    nn::tsc::ActiveConfigContext activeInterface(nullptr);
    nn::tsc::Ipv4ConfigContext ifConfig(nullptr, nn::os::EventClearMode_ManualClear);

    EXPECT_ERROR(activeInterface.SetInterfaceName(""), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(activeInterface.GetMtu(&mtuValue), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(activeInterface.GetInterfaceAddress(&ipAddr), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(activeInterface.GetSubnetMask(&ipAddr), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(activeInterface.GetDefaultGateway(&ipAddr), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(activeInterface.GetPreferredDns(&ipAddr), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(activeInterface.GetAlternativeDns(&ipAddr), nn::tsc::ResultLibraryNotInitialized());

    EXPECT_ERROR(ifConfig.SetInterfaceName(g_pInterfaceName), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.GetInterfaceName(pInterfaceName, MaxIfLen), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.SetMtu(1500), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.GetMtu(&mtuValue), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.SetConfig(&ipConfig), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.ApplyConfig(nn::tsc::IpApplyModeMask_None), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.NotifyInterfaceDown(), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.GetApplyResult(), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.CheckDuplicateIp(10000), nn::tsc::ResultLibraryNotInitialized());
    EXPECT_ERROR(ifConfig.CancelApplyConfig(), nn::tsc::ResultLibraryNotInitialized());

    EXPECT_ERROR(nn::tsc::Finalize(), nn::tsc::ResultLibraryNotInitialized());

out:

    if (isSuccess)
    {
        NN_LOG("\n ******** PASS ********\n\n");
    }
    else
    {
        NN_LOG("\n ******** FAIL ********\n\n");
    }

    EXPECT_EQ(isSuccess, true);
}

namespace
{
    EthernetHandler g_ethernetHandler NN_ALIGNAS(nn::os::ThreadStackAlignment);
}

TEST( Tsc, Basic )
{
    uint32_t mtuValue = 0;
    bool isSuccess = true;
    char pInterfaceName[MaxIfLen];
    nn::tsc::Ipv4AddrStorage ipAddr;
    nn::tsc::Ipv4ConfigContext ifConfig(nullptr, nn::os::EventClearMode_ManualClear);
    nn::tsc::Ipv4Config ipConfig;
    uint32_t interfaceCount = 0;
    nn::eth::client::InterfaceGroupHandler iHandler;

    NN_LOG("Starting Test\n");

    nn::socket::Initialize(reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                           nn::socket::DefaultSocketMemoryPoolSize,
                           nn::socket::DefaultSocketAllocatorSize,
                           nn::socket::DefaultConcurrencyLimit);

    g_ethernetHandler.Initialize();

    uint32_t waitTimeSec = 0;
    while (g_pInterfaceName == nullptr)
    {
        if (waitTimeSec++ >= MaxWaitTimeSec)
        {
            NN_LOG("Failed to detect USB-Eth interface within %d seconds.\n\n", MaxWaitTimeSec);
            isSuccess = false;
            goto out;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    g_ethernetHandler.Shutdown();

    EXPECT_SUCCESS(nn::tsc::Initialize());

    EXPECT_SUCCESS(iHandler.Initialize(nn::os::EventClearMode_ManualClear));
    EXPECT_SUCCESS(iHandler.GetInterfaceCount(&interfaceCount));
    NN_LOG("interfaceCount: %d\n\n", (int)interfaceCount);

    {   // Need this scope because EXPECT_SUCCESS macro uses 'goto out' and
        // ActiveConfigContext needs g_pInterfaceName to be initialized
        nn::tsc::ActiveConfigContext activeInterface(g_pInterfaceName);

        EXPECT_SUCCESS(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen));
        NN_LOG("Interface Name: %s\n", pInterfaceName);
        EXPECT_SUCCESS(activeInterface.SetInterfaceName(pInterfaceName));

        EXPECT_SUCCESS(activeInterface.GetMtu(&mtuValue));
        NN_LOG("MTU: %u\n", mtuValue);

        memset(&ipConfig, 0, sizeof(ipConfig));
        ipConfig.method = nn::tsc::Ipv4ConfigMethod_Dhcp;

        EXPECT_SUCCESS(ifConfig.SetInterfaceName(pInterfaceName));
        EXPECT_SUCCESS(ifConfig.SetMtu(1500));
        EXPECT_SUCCESS(ifConfig.SetConfig(&ipConfig));

        NN_LOG("Applying config...\n");
        EXPECT_SUCCESS(ifConfig.ApplyConfig(nn::tsc::IpApplyModeMask_None, 30000));
        EXPECT_ERROR(ifConfig.ApplyConfig(nn::tsc::IpApplyModeMask_None, 3000), nn::tsc::ResultConfigurationInProgress());

        NN_LOG("Waiting on event...\n");
        ifConfig.GetTimerEventPointer()->Wait();

        NN_LOG("Getting result...\n");
        EXPECT_SUCCESS(ifConfig.GetApplyResult());

        EXPECT_SUCCESS(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen));
        NN_LOG("Interface Name: %s\n", pInterfaceName);

        EXPECT_SUCCESS(activeInterface.GetInterfaceAddress(&ipAddr));
        NN_LOG("Interface Addr: %s\n", nn::tsc::debug::GetStringFromNetworkAddress(ipAddr));

        EXPECT_SUCCESS(activeInterface.GetSubnetMask(&ipAddr));
        NN_LOG("Subnet Mask: %s\n", nn::tsc::debug::GetStringFromNetworkAddress(ipAddr));

        EXPECT_SUCCESS(activeInterface.GetDefaultGateway(&ipAddr));
        NN_LOG("Default Gateway: %s\n", nn::tsc::debug::GetStringFromNetworkAddress(ipAddr));

        EXPECT_SUCCESS(activeInterface.GetPreferredDns(&ipAddr));
        NN_LOG("Preferred DNS: %s\n", nn::tsc::debug::GetStringFromNetworkAddress(ipAddr));

        EXPECT_SUCCESS(activeInterface.GetAlternativeDns(&ipAddr));
        NN_LOG("Alternate DNS: %s\n", nn::tsc::debug::GetStringFromNetworkAddress(ipAddr));

        EXPECT_SUCCESS(activeInterface.GetMtu(&mtuValue));
        NN_LOG("MTU: %u\n", mtuValue);

        NN_LOG("Setting interface down...\n");
        EXPECT_SUCCESS(ifConfig.NotifyInterfaceDown());
    }

out:
    NN_LOG("Finalizing TSC library...\n");
    nn::tsc::Finalize();

    NN_LOG("Finalizing Socket library...\n");
    nn::socket::Finalize();

    if( isSuccess )
    {
        NN_LOG("\n ******** PASS ********\n\n");
    }
    else
    {
        NN_LOG("\n ******** FAIL ********\n\n");
    }

    EXPECT_EQ(isSuccess, true);
}

namespace {
    const uint32_t StackSize = 4096;
    NN_ALIGNAS(nn::os::StackRegionAlignment) char g_pThreadStack[StackSize];

    static void CancelThread(void* pParam)
    {
        bool isSuccess = true;
        nn::tsc::Ipv4ConfigContext* pConfigContext = (nn::tsc::Ipv4ConfigContext*)pParam;

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

        NN_LOG("Canceling Configuration...\n");
        EXPECT_SUCCESS(pConfigContext->CancelApplyConfig());
        NN_LOG("Done canceling Configuration.\n");

    out:
        return;
    }
}

TEST( Tsc, CancelConfigThread )
{
    nn::os::ThreadType thread;
    const uint32_t MaxIfLen = 128;
    uint32_t mtuValue = 0;
    bool isSuccess = true;
    char pInterfaceName[MaxIfLen];
    nn::tsc::ActiveConfigContext activeInterface(nullptr);
    nn::tsc::Ipv4ConfigContext ifConfig(nullptr, nn::os::EventClearMode_ManualClear);
    nn::tsc::Ipv4Config ipConfig;
    uint32_t interfaceCount = 0;
    nn::eth::client::InterfaceGroupHandler iHandler;

    NN_LOG("Starting Test\n");

    nn::socket::Initialize(reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                           nn::socket::DefaultSocketMemoryPoolSize,
                           nn::socket::DefaultSocketAllocatorSize,
                           nn::socket::DefaultConcurrencyLimit);
    EXPECT_SUCCESS(nn::tsc::Initialize());

    EXPECT_SUCCESS(iHandler.Initialize(nn::os::EventClearMode_ManualClear));
    EXPECT_SUCCESS(iHandler.GetInterfaceCount(&interfaceCount));
    NN_LOG("interfaceCount: %d\n\n", (int)interfaceCount);
    EXPECT_SUCCESS(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen));

    EXPECT_SUCCESS(nn::os::CreateThread(&thread, CancelThread, &ifConfig, g_pThreadStack, StackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread())));
    EXPECT_SUCCESS(activeInterface.SetInterfaceName(g_pInterfaceName));
    EXPECT_SUCCESS(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen));
    NN_LOG("Interface Name: %s\n", pInterfaceName);

    EXPECT_SUCCESS(activeInterface.GetMtu(&mtuValue));
    NN_LOG("MTU: %u\n", mtuValue);

    memset(&ipConfig, 0, sizeof(ipConfig));
    ipConfig.method = nn::tsc::Ipv4ConfigMethod_Dhcp;

    EXPECT_SUCCESS(ifConfig.SetInterfaceName(g_pInterfaceName));
    EXPECT_SUCCESS(ifConfig.SetMtu(1500));
    EXPECT_SUCCESS(ifConfig.SetConfig(&ipConfig));

    NN_LOG("Applying config...\n");
    EXPECT_SUCCESS(ifConfig.ApplyConfig(nn::tsc::IpApplyModeMask_None, 3000));

    NN_LOG("Starting cancel thread...\n");
    nn::os::StartThread(&thread);

    NN_LOG("Waiting on event...\n");
    ifConfig.GetTimerEventPointer()->Wait();

    NN_LOG("Getting result...\n");
    EXPECT_ERROR(ifConfig.GetApplyResult(), nn::tsc::ResultProcessAborted());

    EXPECT_SUCCESS(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen));
    NN_LOG("Interface Name: %s\n", pInterfaceName);

out:
    nn::os::DestroyThread(&thread);

    NN_LOG("Finalizing TSC library...\n");
    nn::tsc::Finalize();

    nn::socket::Finalize();

    if( isSuccess )
    {
        NN_LOG("\n ******** PASS ********\n\n");
    }
    else
    {
        NN_LOG("\n ******** FAIL ********\n\n");
    }

    EXPECT_EQ(isSuccess, true);
}

TEST(Tsc, CancelConfig)
{
    const uint32_t MaxIfLen = 128;
    uint32_t mtuValue = 0;
    bool isSuccess = true;
    char pInterfaceName[MaxIfLen];
    nn::tsc::ActiveConfigContext activeInterface(nullptr);
    nn::tsc::Ipv4ConfigContext ifConfig(nullptr, nn::os::EventClearMode_ManualClear);
    nn::tsc::Ipv4Config ipConfig;
    uint32_t interfaceCount = 0;
    nn::eth::client::InterfaceGroupHandler iHandler;

    NN_LOG("Starting Test\n");

    nn::socket::Initialize(reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                           nn::socket::DefaultSocketMemoryPoolSize,
                           nn::socket::DefaultSocketAllocatorSize,
                           nn::socket::DefaultConcurrencyLimit);
    EXPECT_SUCCESS(nn::tsc::Initialize());

    EXPECT_SUCCESS(iHandler.Initialize(nn::os::EventClearMode_ManualClear));
    EXPECT_SUCCESS(iHandler.GetInterfaceCount(&interfaceCount));
    NN_LOG("interfaceCount: %d\n\n", (int)interfaceCount);
    EXPECT_SUCCESS(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen));
    EXPECT_SUCCESS(activeInterface.SetInterfaceName(g_pInterfaceName));
    EXPECT_SUCCESS(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen));
    NN_LOG("Interface Name: %s\n", pInterfaceName);

    EXPECT_SUCCESS(activeInterface.GetMtu(&mtuValue));
    NN_LOG("MTU: %u\n", mtuValue);

    memset(&ipConfig, 0, sizeof(ipConfig));
    ipConfig.method = nn::tsc::Ipv4ConfigMethod_Dhcp;

    EXPECT_SUCCESS(ifConfig.SetInterfaceName(g_pInterfaceName));
    EXPECT_SUCCESS(ifConfig.SetMtu(1500));
    EXPECT_SUCCESS(ifConfig.SetConfig(&ipConfig));

    NN_LOG("Applying config...\n");
    EXPECT_SUCCESS(ifConfig.ApplyConfig(nn::tsc::IpApplyModeMask_None));

    NN_LOG("Canceling configuration...\n");
    EXPECT_SUCCESS(ifConfig.CancelApplyConfig());

    NN_LOG("Getting result...\n");
    EXPECT_ERROR(ifConfig.GetApplyResult(), nn::tsc::ResultProcessAborted());

    EXPECT_SUCCESS(activeInterface.GetInterfaceName(pInterfaceName, MaxIfLen));
    NN_LOG("Interface Name: %s\n", pInterfaceName);

out:

    NN_LOG("Finalizing TSC library...\n");
    nn::tsc::Finalize();

    nn::socket::Finalize();

    if (isSuccess)
    {
        NN_LOG("\n ******** PASS ********\n\n");
    }
    else
    {
        NN_LOG("\n ******** FAIL ********\n\n");
    }

    EXPECT_EQ(isSuccess, true);
}

namespace
{
    uint8_t ifDownStack[ClientStackSize] NN_ALIGNAS(nn::os::ThreadStackAlignment);
    void InterfaceDown(void* arg) NN_NOEXCEPT
    {
        bool isSuccess = true;
        nn::tsc::Ipv4ConfigContext* pConfig = (nn::tsc::Ipv4ConfigContext*)arg;

        NN_LOG("Sleeping before taking IF down...\n");
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(4000));
        NN_LOG("Taking IF down...\n");
        EXPECT_SUCCESS(pConfig->NotifyInterfaceDown());
        NN_LOG("IF down\n");

    out:
        return;
    }
}

namespace
{
    bool BringInterfaceUp(nn::tsc::Ipv4ConfigContext& ifConfig)
    {
        bool isSuccess = true;
        nn::tsc::Ipv4AddrStorage ipAddr;

        nn::tsc::ActiveConfigContext activeInterface(nullptr);
        nn::tsc::Ipv4Config ipConfig;

        EXPECT_SUCCESS(activeInterface.SetInterfaceName(g_pInterfaceName));

        memset(&ipConfig, 0, sizeof(ipConfig));
        ipConfig.method = nn::tsc::Ipv4ConfigMethod_Dhcp;

        EXPECT_SUCCESS(ifConfig.SetInterfaceName(g_pInterfaceName));
        EXPECT_SUCCESS(ifConfig.SetMtu(1500));
        EXPECT_SUCCESS(ifConfig.SetConfig(&ipConfig));

        NN_LOG("Applying config...\n");
        EXPECT_SUCCESS(ifConfig.ApplyConfig(nn::tsc::IpApplyModeMask_None, 30000));
        EXPECT_ERROR(ifConfig.ApplyConfig(nn::tsc::IpApplyModeMask_None, 3000), nn::tsc::ResultConfigurationInProgress());

        NN_LOG("Waiting on event...\n");
        ifConfig.GetTimerEventPointer()->Wait();

        NN_LOG("Getting result...\n");
        EXPECT_SUCCESS(ifConfig.GetApplyResult());

        EXPECT_SUCCESS(activeInterface.GetInterfaceAddress(&ipAddr));
        NN_LOG("Interface Addr: %s\n", nn::tsc::debug::GetStringFromNetworkAddress(ipAddr));

        EXPECT_SUCCESS(activeInterface.GetSubnetMask(&ipAddr));
        NN_LOG("Subnet Mask: %s\n", nn::tsc::debug::GetStringFromNetworkAddress(ipAddr));

        EXPECT_SUCCESS(activeInterface.GetDefaultGateway(&ipAddr));
        NN_LOG("Default Gateway: %s\n", nn::tsc::debug::GetStringFromNetworkAddress(ipAddr));

    out:

        return isSuccess;
    }
}

namespace
{
    bool TestConnect(int socket, nn::tsc::Ipv4ConfigContext& ifConfig, nn::os::ThreadType& ifDownThread)
    {
        bool isSuccess = true;

        nn::socket::SockAddrIn addr;
        memset(&addr, 0, sizeof(addr));
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(NoServerPort);

        if (!BringInterfaceUp(ifConfig))
        {
            return false;
        }

        nn::socket::InetAton(ServerIp, &addr.sin_addr);

        nn::os::StartThread(&ifDownThread);

        NN_LOG("Connecting to invalid server...\n");
        int ret = nn::socket::Connect(socket, (nn::socket::SockAddr*)&addr, sizeof(addr));
        NN_LOG("Connect: ret %d, errno: %d\n\n", ret, nn::socket::GetLastError());
        if (ret != 0)
        {
            if (nn::socket::GetLastError() != nn::socket::Errno::ENetUnreach)
            {
                NN_LOG("Failure!! Connect: ret %d, errno: %d\n\n", ret, nn::socket::GetLastError());
                isSuccess = false;
            }
        }

        nn::os::WaitThread(&ifDownThread);

        return isSuccess;
    }

    bool TestPoll(int socket, nn::tsc::Ipv4ConfigContext& ifConfig, nn::os::ThreadType& ifDownThread)
    {
        bool isSuccess = true;

        if (!BringInterfaceUp(ifConfig))
        {
            return false;
        }

        nn::socket::PollFd fdPoll;
        fdPoll.fd = socket;
        fdPoll.events = nn::socket::PollEvent::PollRdNorm;
        fdPoll.revents = nn::socket::PollEvent::PollNone;

        nn::os::StartThread(&ifDownThread);

        // Wait for socket to be readable
        NN_LOG("Polling...\n");
        int ret = nn::socket::Poll(&fdPoll, 1, -1);
        NN_LOG("Poll: %d, errno: %d\n", ret, nn::socket::GetLastError());
        if (ret < 0)
        {
            if (nn::socket::GetLastError() != nn::socket::Errno::ENetUnreach)
            {
                NN_LOG("Error: Poll: rval: %d errno: %d\n", ret, nn::socket::GetLastError());
                isSuccess = false;
            }
        }

        nn::os::WaitThread(&ifDownThread);

        return isSuccess;
    }

    bool TestSelect(int socket, nn::tsc::Ipv4ConfigContext& ifConfig, nn::os::ThreadType& ifDownThread)
    {
        bool isSuccess = true;

        if (!BringInterfaceUp(ifConfig))
        {
            return false;
        }

        nn::socket::FdSet readSet;
        nn::socket::FdSetZero(&readSet);
        nn::socket::FdSetSet(socket, &readSet);

        nn::os::StartThread(&ifDownThread);

        NN_LOG("Selecting...\n");
        int ret = nn::socket::Select(1, &readSet, nullptr, nullptr, nullptr);
        NN_LOG("Select: %d, errno: %d\n", ret, nn::socket::GetLastError());
        if (ret < 0)
        {
            if (nn::socket::GetLastError() != nn::socket::Errno::ENetUnreach)
            {
                NN_LOG("Error: Select: rval: %d errno: %d\n", ret, nn::socket::GetLastError());
                isSuccess = false;
            }
        }

        nn::os::WaitThread(&ifDownThread);

        return isSuccess;
    }

    bool TestRecv(int socket, nn::tsc::Ipv4ConfigContext& ifConfig, nn::os::ThreadType& ifDownThread)
    {
        bool isSuccess = true;
        char pRecvBuffer[1024];

        if (!BringInterfaceUp(ifConfig))
        {
            return false;
        }

        nn::os::StartThread(&ifDownThread);

        NN_LOG("Receiving...\n");
        int ret = nn::socket::Recv(socket, pRecvBuffer, sizeof(pRecvBuffer), nn::socket::MsgFlag::Msg_None);
        if (ret < 0)
        {
            if (nn::socket::GetLastError() != nn::socket::Errno::ENetUnreach)
            {
                NN_LOG("Recv: %d, Errno: %d\n\n", ret, nn::socket::GetLastError());
                isSuccess = false;
            }
        }

        nn::os::WaitThread(&ifDownThread);

        return isSuccess;
    }

    bool TestAccept(int socket, nn::tsc::Ipv4ConfigContext& ifConfig, nn::os::ThreadType& ifDownThread)
    {
        bool isSuccess = true;

        nn::socket::SockAddrIn addr;
        memset(&addr, 0, sizeof(addr));
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(ServerPort);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;

        if (!BringInterfaceUp(ifConfig))
        {
            return false;
        }

        int ret = nn::socket::Bind(socket, (nn::socket::SockAddr*)&addr, sizeof(addr));
        if (ret != 0)
        {
            NN_LOG("Bind failed. ret: %d, errno: %d\n\n", ret, nn::socket::GetLastError());
            return false;
        }

        ret = nn::socket::Listen(socket, 10);
        if (ret != 0)
        {
            NN_LOG("listen: %d, errno: %d\n\n", ret, nn::socket::GetLastError());
            return false;
        }

        nn::os::StartThread(&ifDownThread);

        nn::socket::SockLenT addrLen = sizeof(addr);

        NN_LOG("Accepting...\n");
        int newSocket = nn::socket::Accept(socket, (nn::socket::SockAddr*)&addr, &addrLen);

        NN_LOG("Accept: ret %d, errno: %d\n\n", newSocket, nn::socket::GetLastError());
        if (newSocket < 0)
        {
            if (nn::socket::GetLastError() != nn::socket::Errno::ENetUnreach)
            {
                isSuccess = false;
            }
        }

        nn::os::WaitThread(&ifDownThread);

        return isSuccess;
    }
}

TEST( Tsc, BlockingSockets )
{
    bool isSuccess = true;
    nn::socket::SockAddrIn addr;
    int socket = -1;
    int ret = -1;
    nn::tsc::Ipv4ConfigContext ifConfig(nullptr, nn::os::EventClearMode_ManualClear);

    NN_LOG("Starting Test\n");

    nn::socket::Initialize(reinterpret_cast<void*>(g_SocketMemoryPoolBuffer),
                           nn::socket::DefaultSocketMemoryPoolSize,
                           nn::socket::DefaultSocketAllocatorSize,
                           nn::socket::DefaultConcurrencyLimit);
    EXPECT_SUCCESS(nn::tsc::Initialize());

    if (!BringInterfaceUp(ifConfig))
    {
        isSuccess = false;
        goto out;
    }

    socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    if( socket < 0 )
    {
        NN_LOG("Socket() failed. ret: %d, errno: %d\n\n", socket, nn::socket::GetLastError());
        isSuccess = false;
        goto out;
    }

    nn::os::ThreadType ifDownThread;
    nn::os::CreateThread(&ifDownThread, InterfaceDown, &ifConfig, ifDownStack, sizeof(ifDownStack), nn::os::DefaultThreadPriority);

    if (!TestConnect(socket, ifConfig, ifDownThread))
    {
        isSuccess = false;
        goto out;
    }

    memset(&addr, 0, sizeof(addr));
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(ServerPort);
    nn::socket::InetAton(ServerIp, &addr.sin_addr);

    NN_LOG("Connecting to valid server...");
    ret = nn::socket::Connect(socket, (nn::socket::SockAddr*)&addr, sizeof(addr));
    NN_LOG("done.\n", ret, nn::socket::GetLastError());
    if (ret != 0)
    {
        NN_LOG(" * ERROR: Failed to connect to Server: ret %d, errno: %d\n\n", ret, nn::socket::GetLastError());
        isSuccess = false;
        goto out;
    }

    if (!TestPoll(socket, ifConfig, ifDownThread))
    {
        isSuccess = false;
        goto out;
    }

    if (!TestSelect(socket, ifConfig, ifDownThread))
    {
        isSuccess = false;
        goto out;
    }

    if (!TestRecv(socket, ifConfig, ifDownThread))
    {
        isSuccess = false;
        goto out;
    }

    nn::socket::Close(socket);

    socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    if (socket < 0)
    {
        NN_LOG("Socket() failed. ret: %d, errno: %d\n\n", socket, nn::socket::GetLastError());
        isSuccess = false;
        goto out;
    }

    if (!TestAccept(socket, ifConfig, ifDownThread))
    {
        isSuccess = false;
        goto out;
    }

out:
    nn::os::DestroyThread(&ifDownThread);

    NN_LOG("Finalizing TSC library...\n");
    nn::tsc::Finalize();

    NN_LOG("Finalizing Socket library...\n");
    nn::socket::Finalize();

    if( isSuccess )
    {
        NN_LOG("\n ******** PASS ********\n\n");
    }
    else
    {
        NN_LOG("\n ******** FAIL ********\n\n");
    }

    EXPECT_EQ(isSuccess, true);
}
