﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <cstdlib>

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/nntest.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <curl/curl.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>
#include <Common/testServerPki.h>
#include <Common/testClientPki.h>
#include <Common/testClientPkiNoPwd.h>

#if defined (NN_BUILD_CONFIG_OS_WIN)
#define MY_SNPRINTF _snprintf
#else
#define MY_SNPRINTF snprintf
#endif

namespace
{
SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
const uint32_t           HostNameLength = 128;
const uint32_t           ArgOptionMessageLength = 128;

enum ArgOptionType
{
    ArgOptionType_Command = 0x00,
    ArgOptionType_Message,
    ArgOptionType_Max
};

enum ArgOption
{
    ArgOption_Help = 0x0,
    ArgOption_Host,
    ArgOption_Port,
    ArgOption_DisableServerValidation,
    ArgOption_DisableNameValidation,
    ArgOption_DisableDateValidation,
    ArgOption_ServerCert,
    ArgOption_ClientCert,
    ArgOption_NetProfile,
    ArgOption_Max
};

typedef struct ConfigParams
{
    char        pHostName[HostNameLength];
    uint16_t    portNumber;
    bool        isPerformServerValidation;
    bool        isPerformNameValidation;
    bool        isPerformDateValidation;
    const char* pServerCert;
    const char* pClientPki;
    const char* pPassword;
    uint32_t    serverCertSize;
    uint32_t    clientPkiSize;
    uint32_t    passwordSize;
    nn::util::Uuid netProfile;
} ConfigParams;

static const char g_ArgOptions[ArgOption_Max + 1][ArgOptionType_Max][ArgOptionMessageLength] = {
    {"--help", "This command."},
    {"--host", "Specify host name (no need to pass https:// in string or dotted decimal IP."},
    {"--port", "Specify port number (443 will be used when not specified)."},
    {"--disable_server_validation", "Disable server validation (it's performed by default)."},
    {"--disable_name_validation", "Disable host name validation (it's performed by default)."},
    {"--disable_date_validation", "Disable date validation (it's performed by default)."},
    {"--server_cert", "Specify server cert for ntd-net-server1 (this option shold be followed by port number - e.g.: --server_cert 444)"},
    {"--client_cert", "Specify client cert for ntd-net-server1 (this option shold be followed by port number - e.g.: --client_cert 444)"},
    {"--net_profile", "Specify the Uuid of the network profile to use with NIFM."}
};

void PrintHelp()
{
    NN_LOG("Usage\n");
    for(int i=0; i < ArgOption_Max; i++)
    {
        NN_LOG(" %s - %s\n", g_ArgOptions[i][ArgOptionType_Command], g_ArgOptions[i][ArgOptionType_Message]);
    }
    NN_LOG("example: testNet_SslHttpsUtil.exe --url www.google.com --port 443 --disable_server_validation\n");
}

void InitializeConfigParameters(ConfigParams* pInConfig)
{
    pInConfig->portNumber                = 443;
    pInConfig->isPerformServerValidation = true;
    pInConfig->isPerformNameValidation   = true;
    pInConfig->isPerformDateValidation   = true;
    pInConfig->pServerCert               = nullptr;
    pInConfig->pClientPki                = nullptr;
    pInConfig->pPassword                 = nullptr;
    pInConfig->serverCertSize            = 0;
    pInConfig->clientPkiSize             = 0;
    pInConfig->passwordSize              = 0;
    pInConfig->netProfile                = nn::util::InvalidUuid;
    memset(pInConfig->pHostName, 0x00, HostNameLength);
}

bool ConfigureParameters(int argc, char** argv, ConfigParams* pInConfig)
{
    InitializeConfigParameters(pInConfig);

    for (int i = 1; i < argc; i++)
    {
        bool isFound = false;
        int  j;

        for(j=0; j < ArgOption_Max; j++)
        {
            if (!strcmp(argv[i], g_ArgOptions[j][ArgOptionType_Command]))
            {
                isFound = true;
                break;
            }
        }

        if (isFound == false)
        {
            NN_LOG("Unknown option was passed (%s).\n", argv[i]);
            PrintHelp();
            return false;
        }

        switch(j)
        {
        case ArgOption_Help:
            PrintHelp();
            return false;
        case ArgOption_Host:
            if (++i > argc)
            {
                NN_LOG("Actual URL name is not passed.\n");
                return false;
            }
            strncpy(pInConfig->pHostName, argv[i], HostNameLength);
            break;
        case ArgOption_Port:
            if (++i > argc)
            {
                NN_LOG("Actual port number is not passed.\n");
                return false;
            }
            pInConfig->portNumber = static_cast<uint16_t>(atoi(argv[i]));
            break;
        case ArgOption_DisableServerValidation:
            pInConfig->isPerformNameValidation = false;
            break;
        case ArgOption_DisableNameValidation:
            pInConfig->isPerformServerValidation = false;
            break;
        case ArgOption_DisableDateValidation:
            pInConfig->isPerformDateValidation = false;
            break;
        case ArgOption_ServerCert:
            {
                if (++i > argc)
                {
                    NN_LOG("Actual port number to select server cert is not passed.\n");
                    return false;
                }
                int tmpPortNumber = atoi(argv[i]);
                switch(tmpPortNumber)
                {
                case ServerPort_ExpiredCert:
                    pInConfig->pServerCert = g_pTestCaCertExpired;
                    pInConfig->serverCertSize = sizeof(g_pTestCaCertExpired);
                    break;
                case ServerPort_Normal:
                case ServerPort_ClientAuth:
                    pInConfig->pServerCert = g_pTestCaCert;
                    pInConfig->serverCertSize = sizeof(g_pTestCaCert);
                    break;
                default:
                    NN_LOG("There's no dedicated CA certificate for given port number.\n");
                    return false;
                }
            }
            break;
        case ArgOption_ClientCert:
            {
                if (++i > argc)
                {
                    NN_LOG("Actual port number to select client cert is not passed.\n");
                    return false;
                }
                int tmpPortNumber = atoi(argv[i]);
                switch(tmpPortNumber)
                {
                case ServerPort_ClientAuth:
                    pInConfig->pClientPki    = reinterpret_cast<const char*>(g_pTestClientPki);
                    pInConfig->clientPkiSize = g_pTestClientPkiSize;
                    pInConfig->pPassword     = g_pTestClientPkiPassword;
                    pInConfig->passwordSize  = g_TestClientPkiPasswordLength;
                    break;
                default:
                    NN_LOG("There's no dedicated client PKI for given port number.\n");
                    return false;
                }
            }
            break;
        case ArgOption_NetProfile:
        {
            char pBuffer[nn::util::Uuid::StringSize + 1];

            if (++i > argc)
            {
                NN_LOG("\n ERROR: Actual Uuid is not passed.\n\n");
                return false;
            }

            pInConfig->netProfile.FromString(argv[i]);
            if(strcmp(pInConfig->netProfile.ToString(pBuffer, sizeof(pBuffer)), argv[i]) != 0)
            {
                pInConfig->netProfile = nn::util::InvalidUuid;
                NN_LOG("\n ERROR: Bad Uuid format! Passed Arg: %s.\n\n", argv[i]);
                return false;
            }
        }
        break;
        default:
            NN_LOG("Internal parser error.\n");
            return false;
        }
    }

    return true;
} // NOLINT(impl/function_size)
} // Un-named namespace

extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup loaded %p\n", nninitStartup);
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    ASSERT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    ASSERT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

extern "C" void nnMain()
{
    ConfigParams config;

    if(ConfigureParameters(nn::os::GetHostArgc(), nn::os::GetHostArgv(), &config) != true)
    {
        return;
    }

    if (config.pHostName[0] == '\0')
    {
        NN_LOG("URL is not specified.\n");
        return;
    }

    nn::socket::InAddr ipAddr;
    bool    isResolverRequired = true;
    ipAddr.S_addr = 0;
    if(nn::socket::InetAton(config.pHostName, &ipAddr) == 0)
    {
        isResolverRequired = false;
    }

    NN_LOG(" Host name        : %s\n", config.pHostName);
    NN_LOG(" IP               : 0x%x\n", ipAddr.S_addr);
    NN_LOG(" Server validation: %s\n", (config.isPerformServerValidation)?("YES"):("NO"));
    NN_LOG(" Name validation  : %s\n", (config.isPerformNameValidation)?("YES"):("NO"));

    ASSERT_TRUE(g_CommonUtil.SetupNetwork(config.netProfile));
    ASSERT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());
    EXPECT_TRUE(nn::ssl::Initialize().IsSuccess());

    nn::ssl::Connection::VerifyOption verifyOption =
        nn::ssl::Connection::VerifyOption::VerifyOption_None;
    if(config.isPerformServerValidation == true)
    {
        verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_PeerCa;
    }
    if(config.isPerformNameValidation == true)
    {
        verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_HostName;
    }
    if(config.isPerformDateValidation == true)
    {
        verifyOption |= nn::ssl::Connection::VerifyOption::VerifyOption_DateCheck;
    }

    SimpleHttpsClient httpsClient(
        false, // Blocking
        config.pHostName,
        config.portNumber);

    if(isResolverRequired == false)
    {
        httpsClient.SetIpAddress(ipAddr.S_addr);
    }

    EXPECT_TRUE(httpsClient.Initialize(
        nn::ssl::Context::SslVersion::SslVersion_Auto,
        verifyOption));

    if(config.pServerCert)
    {
        EXPECT_TRUE(httpsClient.ImportServerPki(
            config.pServerCert,
            config.serverCertSize,
            nn::ssl::CertificateFormat_Pem));
    }

    if(config.pClientPki)
    {
        EXPECT_TRUE(httpsClient.ImportClientPki(
            config.pClientPki,
            config.pPassword,
            config.clientPkiSize,
            config.passwordSize));
    }

    EXPECT_TRUE(httpsClient.PerformSslHandshake(false));
    EXPECT_TRUE(httpsClient.SendHttpGetOverSsl());
    EXPECT_TRUE(httpsClient.ReceiveAllHttpData());

    httpsClient.PrintReceivedData();

    httpsClient.Finalize();
    EXPECT_TRUE(nn::ssl::Finalize().IsSuccess());
    nn::socket::Finalize();
    g_CommonUtil.FinalizeNetwork();
}
