﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/ssl.h>
#include <nn/ssl/ssl_Context.private.h>

#include <Common/testCommonUtil.h>
#include <Common/testInfraInfo.h>

#include <Utils/CommandLineParser.h>

// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
#define USE_NETWORK

// ------------------------------------------------------------------------------------------------
// Build flags for tests to run
// ------------------------------------------------------------------------------------------------
#define RUN_NOT_INITIALIZED       // Calls APIs when library is not initialized
#define RUN_LIMIT                 // Creates context/connection more than max
#define RUN_CONN_REMAIN           // Destroys context when there is still valid connections
#define RUN_CONTEXT_PRIVATE       // Test ContextPrivate class
#define RUN_BUILTIN_MANAGER       // Test BuiltInManager APIs
#if defined(NN_SDK_BUILD_RELEASE)
#define RUN_INVALID_ARG           // Can be run only on Release build due to NN_SDK_REQUIRES_NOT_NULL
#endif

#if defined(USE_NETWORK)
#define RUN_INVALID_ARG_CONDITION // Passes args in invlid condition
#define RUN_INSUFF_SRV_BUFF       // Passes too small buffer as server cert buffer
//#define RUN_FORCE_DETACH          // Finalizes library when there are still valid contexts/connections
#define RUN_POSITIVE              // Performs normal SSL
#define RUN_POSITIVE_ARG          // Passes valid arg and verifies with Get APIs
#endif

// ------------------------------------------------------------------------------------------------
// Utils
// ------------------------------------------------------------------------------------------------
#if defined (NN_BUILD_CONFIG_OS_WIN)
#define MY_SNPRINTF _snprintf
#else
#define MY_SNPRINTF snprintf
#endif

// ------------------------------------------------------------------------------------------------
// Params
// ------------------------------------------------------------------------------------------------
namespace
{
const int g_MsecPollTimeout = 5000;

SslTestCommonUtil        g_CommonUtil;
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
} // Un-named namespace

//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    NN_LOG("nninitStartup -> %p\n", (void *)nninitStartup);

    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    EXPECT_TRUE( result.IsSuccess() );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    EXPECT_TRUE( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// Tests
// ------------------------------------------------------------------------------------------------
TEST(InitTest, Success)
{
    NN_LOG("nninitStartup: loaded at %p\n", (void *)nninitStartup);

#if defined(USE_NETWORK)
    nn::util::Uuid netProfile = nn::util::InvalidUuid;
    NATF::Utils::ParserGroup parser;

    parser.AddParser(NATF::Utils::UuidParser ("--NetProfile", &nn::util::InvalidUuid, netProfile));

    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();

    if (!parser.Parse(argc, argv))
    {
        NN_LOG("\n * Failed to parse command line arguements!\n\n");
        FAIL();
        return;
    }

    ASSERT_TRUE(g_CommonUtil.SetupNetwork(netProfile));
#endif

    EXPECT_TRUE(nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::MinSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit).IsSuccess());
}

#if defined(RUN_NOT_INITIALIZED)
TEST(ShimNotInitialized, Success)
{
    nn::Result           result;
    nn::ssl::Context*    pSslContext    = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;

    // --------------------------------------------------------------------------------------------
    // Context
    // --------------------------------------------------------------------------------------------
    pSslContext = new nn::ssl::Context();
    EXPECT_TRUE(pSslContext != nullptr);

    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslContext->Destroy();
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslContext->SetOption(nn::ssl::Context::ContextOption_None, 1);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    int optionValue;
    result = pSslContext->GetOption(&optionValue, nn::ssl::Context::ContextOption_None);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    // --------------------------------------------------------------------------------------------
    // Connection
    // --------------------------------------------------------------------------------------------
    pSslConnection = new nn::ssl::Connection();
    EXPECT_TRUE(pSslConnection != nullptr);

    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslConnection->Destroy();
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_DoNotCloseSocket, true);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
    bool isTempOptionValue;
    result = pSslConnection->GetOption(&isTempOptionValue, nn::ssl::Connection::OptionType_DoNotCloseSocket);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    int socketFd = 0;
    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
    result = pSslConnection->GetSocketDescriptor(&socketFd);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    const char* hostName = "test";
    char hostName2[16];

    uint32_t hostNameLen    = sizeof(hostName2);
    uint32_t hostNameOutLen = 0;
    result = pSslConnection->SetHostName(hostName, strlen(hostName));
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
    result = pSslConnection->GetHostName(hostName2, &hostNameOutLen, hostNameLen);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    nn::ssl::Connection::VerifyOption verifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
    result = pSslConnection->SetVerifyOption(verifyOption);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
    result = pSslConnection->GetVerifyOption(&verifyOption);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    char buff[32];
    result = pSslConnection->SetServerCertBuffer(buff, sizeof(buff));
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

#if defined(RUN_SERVER_CERT_CHAIN)
    nn::ssl::Connection::ServerCertDetail certDetail;
    result = pSslConnection->GetServerCertDetail(&certDetail, buff, 0);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
#endif

    nn::ssl::Connection::IoMode ioMode = nn::ssl::Connection::IoMode_Blocking;
    result = pSslConnection->SetIoMode(ioMode);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslConnection->GetIoMode(&ioMode);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    nn::ssl::Connection::SessionCacheMode sessionCacheMode = nn::ssl::Connection::SessionCacheMode_None;
    result = pSslConnection->SetSessionCacheMode(sessionCacheMode);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslConnection->GetSessionCacheMode(&sessionCacheMode);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    nn::ssl::Connection::RenegotiationMode renegotiationMode = nn::ssl::Connection::RenegotiationMode_None;
    result = pSslConnection->SetRenegotiationMode(renegotiationMode);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslConnection->GetRenegotiationMode(&renegotiationMode);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    uint32_t certSize = 0;
    result = pSslConnection->DoHandshake(&certSize, nullptr);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    nn::ssl::Connection::CipherInfo cipherInfo;
    result = pSslConnection->GetCipherInfo(&cipherInfo);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    char ioBuff[32];
    int ioBytes = 0;
    ioBytes = pSslConnection->Read(ioBuff, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->Read(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    ioBytes = pSslConnection->Write(ioBuff, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->Write(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = pSslConnection->Peek(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    ioBytes = pSslConnection->Pending();
    EXPECT_TRUE(ioBytes == -1);

    nn::ssl::Connection::PollEvent pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
    nn::ssl::Connection::PollEvent pollOutEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
    result = pSslConnection->Poll(&pollOutEvent, &pollEvent, 1000);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    nn::Result lastError;
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    nn::Result verifyError;
    result = pSslConnection->GetVerifyCertError(&verifyError);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    static const uint32_t VerifyErrorArrayLen = 4;
    nn::Result pVerifyErrors[VerifyErrorArrayLen];
    uint32_t writtenErrors = 0;
    uint32_t totalErrors = 0;
    result = pSslConnection->GetVerifyCertErrors(pVerifyErrors, &writtenErrors, &totalErrors, VerifyErrorArrayLen);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
    // --------------------------------------------------------------------------------------------
    // BuiltInManager
    // --------------------------------------------------------------------------------------------
    nn::ssl::CaCertificateId ids[] = { nn::ssl::CaCertificateId_NintendoCAG3,
                                       nn::ssl::CaCertificateId_AmazonRootCA1 };
    uint32_t bufSize;
    uint32_t dummy;
    nn::ssl::BuiltInManager::BuiltInCertificateInfo *pCertInfo;
    result = nn::ssl::BuiltInManager::GetBuiltInCertificateBufSize(&bufSize,
                                                                   ids,
                                                                   2);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = nn::ssl::BuiltInManager::GetBuiltInCertificates(&pCertInfo,
                                                             reinterpret_cast<uint8_t *>(&dummy),
                                                             sizeof(dummy),
                                                             ids,
                                                             2);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    result = nn::ssl::Finalize();
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    if(pSslContext)
    {
        delete pSslContext;
    }

    if(pSslConnection)
    {
        delete pSslConnection;
    }
} // NOLINT(impl/function_size)
#endif

//  This test needs to be declared before the others which use SSL so that
//  SSL is initialized properly for the remaining tests.
TEST(ShimInit, Success)
{
    nn::Result result = nn::ssl::Initialize();
    EXPECT_TRUE(result.IsSuccess());
}

#if defined(RUN_CONTEXT_PRIVATE)
TEST(ShimContextPrivate, Success)
{
    nn::ssl::ContextPrivate* pSslContext = nullptr;
    nn::ssl::Connection*     pSslConnection  = nullptr;

    pSslContext = new nn::ssl::ContextPrivate();
    EXPECT_TRUE(pSslContext != nullptr);

    NN_LOG("Create SSL context...\n");
    nn::Result result =
        pSslContext->Create(nn::ssl::ContextPrivate::SslVersion_SslV3);
    EXPECT_TRUE(result.IsSuccess());

    pSslConnection = new nn::ssl::Connection();
    EXPECT_TRUE(pSslConnection != nullptr);

    NN_LOG("Create SSL connection...\n");
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Create TCP socket...\n");
    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    NN_LOG("Set SSL socket descr %d...\n", socketFd);
    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Destroy SSL connection...\n");
    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Destroy SSL context...\n");
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    if(pSslConnection != nullptr)
    {
        delete pSslConnection;
        pSslConnection = nullptr;
    }
    if(pSslContext != nullptr)
    {
        delete pSslContext;
        pSslContext = nullptr;
    }
}
#endif // RUN_CONTEXT_PRIVATE

#if defined(RUN_INVALID_ARG)
TEST(ShimInvalidArg, Success)
{
    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    nn::ssl::Context*    pSslContext    = nullptr;
    nn::ssl::Connection* pSslConnection = nullptr;

    // --------------------------------------------------------------------------------------------
    // Context
    // --------------------------------------------------------------------------------------------
    pSslContext = new nn::ssl::Context();
    EXPECT_TRUE(pSslContext != nullptr);

    nn::Result result = pSslContext->GetOption(nullptr, nn::ssl::Context::ContextOption_None);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    result = pSslContext->GetContextId(nullptr);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    // --------------------------------------------------------------------------------------------
    // Connection
    // --------------------------------------------------------------------------------------------
    pSslConnection = new nn::ssl::Connection();
    EXPECT_TRUE(pSslConnection != nullptr);

    result = pSslConnection->Create(nullptr);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    result = pSslConnection->GetOption(nullptr, nn::ssl::Connection::OptionType_DoNotCloseSocket);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    result = pSslConnection->GetSocketDescriptor(nullptr);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    result = pSslConnection->GetCipherInfo(nullptr);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    char     hostName[16];
    uint32_t hostNameLen    = sizeof(hostName);
    uint32_t hostNameOutLen = 0;
    result = pSslConnection->SetHostName(nullptr, 1);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));
    result = pSslConnection->GetHostName(hostName, nullptr, hostNameLen);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));
    result = pSslConnection->GetHostName(nullptr, &hostNameOutLen, hostNameLen);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));
    result = pSslConnection->GetHostName(nullptr, nullptr, hostNameLen);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    result = pSslConnection->GetVerifyOption(nullptr);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    result = pSslConnection->SetServerCertBuffer(nullptr, 1);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    result = pSslConnection->GetIoMode(nullptr);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    int ioBytes;
    ioBytes = pSslConnection->Read(nullptr, 1);
    EXPECT_TRUE(ioBytes == -1);

    nn::Result lastError;
    result = pSslConnection->Read(nullptr, &ioBytes, 1);
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(lastError));

    ioBytes = pSslConnection->Write(nullptr, 1);
    EXPECT_TRUE(ioBytes == -1);

    result = pSslConnection->Write(nullptr, &ioBytes, 1);
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(lastError));

    result = pSslConnection->Peek(nullptr, &ioBytes, 1);
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(lastError));

    result = pSslConnection->Poll(nullptr, nullptr, 1000);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    result = pSslConnection->GetLastError(nullptr);
    EXPECT_TRUE(nn::ssl::ResultInvalidPointer::Includes(result));

    if(pSslContext)
    {
        delete pSslContext;
    }

    if(pSslConnection)
    {
        delete pSslConnection;
    }

    g_CommonUtil.CloseTcpSocket(socketFd);
}
#endif

#if defined(RUN_INVALID_ARG_CONDITION)
TEST(ShimInvalidArgCondition, Success)
{
    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    nn::ssl::Context*    pSslContext    = new nn::ssl::Context();
    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();
    EXPECT_TRUE(pSslContext != nullptr);
    EXPECT_TRUE(pSslConnection != nullptr);

    // --------------------------------------------------------------------------------------------
    // Context
    // --------------------------------------------------------------------------------------------
    nn::Result result = pSslContext->Destroy();
    EXPECT_TRUE(nn::ssl::ResultInvalidContext::Includes(result));

#if 0 // TODO: These APIs are not implemented yet
    result = pSslContext->SetOption(nn::ssl::Context::ContextOption_None, 1);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));

    int optionValue;
    result = pSslContext->GetOption(nn::ssl::Context::ContextOption_None, &optionValue);
    EXPECT_TRUE(nn::ssl::ResultLibraryNotInitialized::Includes(result));
#endif

    // --------------------------------------------------------------------------------------------
    // Connection
    // --------------------------------------------------------------------------------------------
    const char* hostName = "test";
    char hostName2[16];
    uint32_t hostNameLen    = sizeof(hostName2);
    uint32_t hostNameOutLen = 0;
    int tmpSockFd = 0;
    nn::ssl::Connection::VerifyOption verifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
    char buff[32];
    char ioBuff[32];
    int ioBytes = 0;
    uint32_t certSize = 0;
    nn::ssl::Connection::PollEvent pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
    nn::ssl::Connection::PollEvent pollOutEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
    nn::Result lastError;
    bool isSocketImported = false;

    // ResultInvalidConnectionContext *************************************************************
    result = pSslConnection->Destroy();
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_DoNotCloseSocket, true);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));
    bool isTempOptionValue;
    result = pSslConnection->GetOption(&isTempOptionValue, nn::ssl::Connection::OptionType_DoNotCloseSocket);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));
    result = pSslConnection->GetSocketDescriptor(&tmpSockFd);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    result = pSslConnection->SetHostName(hostName, strlen(hostName));
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));
    result = pSslConnection->GetHostName(hostName2, &hostNameOutLen, hostNameLen);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    result = pSslConnection->SetVerifyOption(verifyOption);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));
    result = pSslConnection->GetVerifyOption(&verifyOption);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    result = pSslConnection->SetServerCertBuffer(buff, sizeof(buff));
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    nn::ssl::Connection::IoMode ioMode = nn::ssl::Connection::IoMode_Blocking;
    result = pSslConnection->SetIoMode(ioMode);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    result = pSslConnection->GetIoMode(&ioMode);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    result = pSslConnection->DoHandshake(&certSize, nullptr);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    nn::ssl::Connection::CipherInfo cipherInfo;
    result = pSslConnection->GetCipherInfo(&cipherInfo);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    ioBytes = pSslConnection->Read(ioBuff, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(lastError));

    result = pSslConnection->Read(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(lastError));

    ioBytes = pSslConnection->Write(ioBuff, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(lastError));

    result = pSslConnection->Write(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(lastError));

    result = pSslConnection->Peek(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(lastError));

    ioBytes = pSslConnection->Pending();
    EXPECT_TRUE(ioBytes == -1);

    result = pSslConnection->Poll(&pollOutEvent, &pollEvent, 1000);
    EXPECT_TRUE(nn::ssl::ResultInvalidConnectionContext::Includes(result));

    // Some other errors **************************************************************************
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(nn::ssl::ResultInvalidContext::Includes(result));

    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    // ResultSocketNotRegistered
    result = pSslConnection->GetCipherInfo(&cipherInfo);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));

    result = pSslConnection->SetIoMode(ioMode);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));

    nn::ssl::Connection::SessionCacheMode sessionCacheMode = nn::ssl::Connection::SessionCacheMode_None;
    result = pSslConnection->SetSessionCacheMode(sessionCacheMode);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));

    result = pSslConnection->GetSessionCacheMode(&sessionCacheMode);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));

    nn::ssl::Connection::RenegotiationMode renegotiationMode = nn::ssl::Connection::RenegotiationMode_None;
    result = pSslConnection->SetRenegotiationMode(renegotiationMode);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));

    result = pSslConnection->GetRenegotiationMode(&renegotiationMode);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));

    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));

    result = pSslConnection->DoHandshake(&certSize, nullptr);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));

    ioBytes = pSslConnection->Read(ioBuff, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(lastError));

    result = pSslConnection->Read(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(lastError));

    ioBytes = pSslConnection->Write(ioBuff, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(lastError));

    result = pSslConnection->Write(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(lastError));

    result = pSslConnection->Peek(ioBuff, &ioBytes, sizeof(ioBuff));
    EXPECT_TRUE(ioBytes == -1);
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(result));
    result = pSslConnection->GetLastError(&lastError);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(nn::ssl::ResultSocketNotRegistered::Includes(lastError));

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    isSocketImported = true;

    // ResultNoSslConnection
    result = pSslConnection->GetCipherInfo(&cipherInfo);
    EXPECT_TRUE(nn::ssl::ResultNoSslConnection::Includes(result));

    // ResultSocketAlreadyRegistered
    result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_DoNotCloseSocket, true);
    EXPECT_TRUE(nn::ssl::ResultSocketAlreadyRegistered::Includes(result));

    // ResultHostNameNotRegistered
    EXPECT_TRUE(pSslConnection->SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_HostName);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->DoHandshake();
    EXPECT_TRUE(nn::ssl::ResultHostNameNotRegistered::Includes(result));
    result = pSslConnection->DoHandshake(&certSize, nullptr);
    EXPECT_TRUE(nn::ssl::ResultHostNameNotRegistered::Includes(result));

    // ResultHostNameTooLong
    char invalidHostName[nn::ssl::MaxHostNameLength + 1] = {0};
    memset(invalidHostName, 0x41, nn::ssl::MaxHostNameLength + 1);
    result = pSslConnection->SetHostName(invalidHostName, sizeof(invalidHostName));
    EXPECT_TRUE(nn::ssl::ResultHostNameTooLong::Includes(result));

    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    // ResultBufferAlreadyRegistered
    result = pSslConnection->SetServerCertBuffer(buff, sizeof(buff));
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->SetServerCertBuffer(buff, sizeof(buff));
    EXPECT_TRUE(nn::ssl::ResultBufferAlreadyRegistered::Includes(result));

    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());

    if(pSslContext)
    {
        delete pSslContext;
    }

    if(pSslConnection)
    {
        delete pSslConnection;
    }

    g_CommonUtil.CloseTcpSocket(socketFd);
} // NOLINT(impl/function_size)
#endif

#if defined(RUN_CONN_REMAIN)
TEST(ShimConnectionRemains, Success)
{
    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    // ResultConnectionRemaining ******************************************************************
    int i=0;
    const int connectionCount = 4;
    nn::ssl::Context*    pSslContext = nullptr;
    nn::ssl::Connection* pSslConnection[connectionCount] = {nullptr};

    pSslContext = new nn::ssl::Context();
    EXPECT_TRUE(pSslContext != nullptr);
    nn::Result result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());

    NN_LOG("Created context %p\n", pSslContext);

    for(i =0; i < connectionCount; i++)
    {
        pSslConnection[i] = new nn::ssl::Connection();
        EXPECT_TRUE(pSslConnection[i] != nullptr);
        result = pSslConnection[i]->Create(pSslContext);
        EXPECT_TRUE(result.IsSuccess());

        NN_LOG("Created connection %p\n", pSslConnection[i]);
    }

    NN_LOG("destroying context %p\n", pSslContext);
    result = pSslContext->Destroy();
    EXPECT_TRUE(nn::ssl::ResultConnectionRemaining::Includes(result));

    for(i =0; i < connectionCount; i++)
    {
        if(pSslConnection[i] != nullptr)
        {
            result = pSslConnection[i]->Destroy();
            EXPECT_TRUE(result.IsSuccess());
            delete pSslConnection[i];
        }
        if(i < connectionCount - 1)
        {
            result = pSslContext->Destroy();
            EXPECT_TRUE(nn::ssl::ResultConnectionRemaining::Includes(result));
        }
    }

    if(pSslContext != nullptr)
    {
        result = pSslContext->Destroy();
        EXPECT_TRUE(result.IsSuccess());
        delete pSslContext;
    }

    NN_LOG("Closing socket...\n");
    g_CommonUtil.CloseTcpSocket(socketFd);
}
#endif

#if defined(RUN_LIMIT)
TEST(ShimLimit, Success)
{
    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    nn::Result result;
    int i=0;
    const int maxResourceCount = 8;
    nn::ssl::Context*    pSslContext[maxResourceCount + 1]    = {nullptr};
    nn::ssl::Connection* pSslConnection[maxResourceCount + 1] = {nullptr};

    // Flood Context ******************************************************************************
    for(i =0; i < maxResourceCount + 1; i++)
    {
        pSslContext[i] = new nn::ssl::Context();
        EXPECT_TRUE(pSslContext[i] != nullptr);
    }

    for(i =0; i < maxResourceCount + 1; i++)
    {
        result = pSslContext[i]->Create(nn::ssl::Context::SslVersion_Auto);
        if(i < maxResourceCount)
        {
            EXPECT_TRUE(result.IsSuccess());
        }
        else
        {
            EXPECT_TRUE(nn::ssl::ResultResourceMax::Includes(result));
        }
    }

    // Flood Connection ***************************************************************************
    for(i =0; i < maxResourceCount + 1; i++)
    {
        pSslConnection[i] = new nn::ssl::Connection();
        EXPECT_TRUE(pSslContext[i] != nullptr);
    }

    for(i =0; i < maxResourceCount + 1; i++)
    {
        result = pSslConnection[i]->Create((i >= maxResourceCount)?(pSslContext[i - 1]):(pSslContext[i]));
        if(i < maxResourceCount)
        {
            EXPECT_TRUE(result.IsSuccess());
        }
        else
        {
            EXPECT_TRUE(nn::ssl::ResultResourceMax::Includes(result));
        }
    }

    // Cleanup ************************************************************************************
    for(i =0; i < maxResourceCount + 1; i++)
    {
        if(pSslConnection[i] != nullptr)
        {
            if(i < maxResourceCount)
            {
                pSslConnection[i]->Destroy();
            }
            delete pSslConnection[i];
        }

        if(pSslContext[i] != nullptr)
        {
            if(i < maxResourceCount)
            {
                pSslContext[i]->Destroy();
            }
            delete pSslContext[i];
        }
    }

    g_CommonUtil.CloseTcpSocket(socketFd);
}
#endif

#if defined(RUN_INSUFF_SRV_BUFF)
TEST(ShimInsufficientServerBuffer, Success)
{
    bool isSocketImported = false;

    nn::Result result;
    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();;
    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    EXPECT_TRUE(pSslContext != nullptr);
    EXPECT_TRUE(pSslConnection != nullptr);

    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    isSocketImported = true;

    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    EXPECT_TRUE(pSslConnection->SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
    EXPECT_TRUE(result.IsSuccess());

    // Set buffer to get peer certificate
    char* serverCertBuff = new char[4];
    result = pSslConnection->SetServerCertBuffer(serverCertBuff, 4);
    EXPECT_TRUE(result.IsSuccess());

    uint32_t certSize = 0;
    result = pSslConnection->DoHandshake(&certSize, nullptr);
    EXPECT_TRUE(nn::ssl::ResultInsufficientServerCertBuffer::Includes(result));
    result = pSslConnection->GetNeededServerCertBufferSize(&certSize);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(certSize > 4);
    NN_LOG(" SSL Handshake completed (server cert size:%d).\n", certSize);

    char httpReqBuff[64] = {0};
    MY_SNPRINTF(httpReqBuff, sizeof(httpReqBuff), "GET / HTTP/1.0\r\nHost: %s\r\n\r\n", ServerName);
    uint32_t httpReqBuffLen = static_cast<uint32_t>(strlen(httpReqBuff));
    int sentBytes = pSslConnection->Write(httpReqBuff, httpReqBuffLen);
    EXPECT_TRUE(sentBytes > 0);
    NN_LOG(" Sent HTTP request over SSL (%d bytes).\n", sentBytes);

    int receivedTotalBytes    = 0;
    do
    {
        nn::ssl::Connection::PollEvent pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
        nn::ssl::Connection::PollEvent pollOutEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
        pollEvent |= nn::ssl::Connection::PollEvent::PollEvent_Read;

        result = pSslConnection->Poll(&pollOutEvent, &pollEvent, g_MsecPollTimeout);
        EXPECT_TRUE(result.IsSuccess());

        if((pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Read) == nn::ssl::Connection::PollEvent::PollEvent_Read)
        {
            char tmpBuff[1024] = {0};
            int receivedBytes = pSslConnection->Read(tmpBuff, sizeof(tmpBuff));
            EXPECT_TRUE(receivedBytes >= 0);
            if(receivedBytes < 0)
            {
                NN_LOG(" nn::ssl::Read failed!\n");
                break;
            }
            if(receivedBytes == 0)
            {
                NN_LOG(" Connection closed by the server.\n");
                break;
            }
            receivedTotalBytes += receivedBytes;
        }
    } while(NN_STATIC_CONDITION(false));
    NN_LOG(" Received %d bytes\n", receivedTotalBytes);

    // Cleanup
    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    delete pSslConnection;

    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    delete pSslContext;

    delete[] serverCertBuff;

    if(isSocketImported != true)
    {
        g_CommonUtil.CloseTcpSocket(socketFd);
    }
}


#endif

#if defined(RUN_POSITIVE_ARG)
TEST(ShimPositiveArg, Success)
{
    bool isSocketImported = false;
    nn::Result result;

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();;
    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    EXPECT_TRUE(pSslContext != nullptr);
    EXPECT_TRUE(pSslConnection != nullptr);

    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    // --------------------------------------------------------------------------------------------
    // Context
    // --------------------------------------------------------------------------------------------

    // Currently there's nothing to test

    // --------------------------------------------------------------------------------------------
    // Connnection
    // --------------------------------------------------------------------------------------------
    bool isDontCloseSocket = true;
    result = pSslConnection->GetOption(&isDontCloseSocket, nn::ssl::Connection::OptionType_DoNotCloseSocket);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(isDontCloseSocket == false);
    result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_DoNotCloseSocket, true);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->GetOption(&isDontCloseSocket, nn::ssl::Connection::OptionType_DoNotCloseSocket);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(isDontCloseSocket == true);
    result = pSslConnection->SetOption(nn::ssl::Connection::OptionType_DoNotCloseSocket, false);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    isSocketImported = true;
    int tmpSocketFd = 0;
    result = pSslConnection->GetSocketDescriptor(&tmpSocketFd);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(socketFd == tmpSocketFd);


    nn::ssl::Connection::SessionCacheMode sessionCacheMode = nn::ssl::Connection::SessionCacheMode_None;
    result = pSslConnection->GetSessionCacheMode(&sessionCacheMode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(sessionCacheMode == nn::ssl::Connection::SessionCacheMode_SessionId);

    sessionCacheMode = nn::ssl::Connection::SessionCacheMode_None;
    result = pSslConnection->SetSessionCacheMode(sessionCacheMode);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->GetSessionCacheMode(&sessionCacheMode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(sessionCacheMode == nn::ssl::Connection::SessionCacheMode_None);


    nn::ssl::Connection::RenegotiationMode renegotiationMode = nn::ssl::Connection::RenegotiationMode_None;
    result = pSslConnection->GetRenegotiationMode(&renegotiationMode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(renegotiationMode == nn::ssl::Connection::RenegotiationMode_Secure);

    renegotiationMode = nn::ssl::Connection::RenegotiationMode_None;
    result = pSslConnection->SetRenegotiationMode(renegotiationMode);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->GetRenegotiationMode(&renegotiationMode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(renegotiationMode == nn::ssl::Connection::RenegotiationMode_None);

    char     tmpHostName[64] = {0};
    uint32_t tmpHostNameLen    = sizeof(tmpHostName);
    uint32_t tmpHostNameOutLen = 0;
    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->GetHostName(tmpHostName, &tmpHostNameOutLen, tmpHostNameLen);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(strcmp(ServerName,tmpHostName) == 0);
    EXPECT_TRUE(strlen(ServerName) == tmpHostNameOutLen);

    nn::ssl::Connection::VerifyOption tmpVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
    result = pSslConnection->GetVerifyOption(&tmpVerifyOption);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(tmpVerifyOption == nn::ssl::Connection::VerifyOption::VerifyOption_Default);

    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_All);
    EXPECT_TRUE(result.IsSuccess());

    tmpVerifyOption = nn::ssl::Connection::VerifyOption::VerifyOption_None;
    result = pSslConnection->GetVerifyOption(&tmpVerifyOption);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(tmpVerifyOption == nn::ssl::Connection::VerifyOption::VerifyOption_All);

    nn::ssl::Connection::IoMode tmpIoMode = nn::ssl::Connection::IoMode_NonBlocking;
    result = pSslConnection->GetIoMode(&tmpIoMode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(tmpIoMode == nn::ssl::Connection::IoMode_Blocking);

    tmpIoMode = nn::ssl::Connection::IoMode_Blocking;
    result = pSslConnection->SetIoMode(nn::ssl::Connection::IoMode_NonBlocking);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->GetIoMode(&tmpIoMode);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(tmpIoMode == nn::ssl::Connection::IoMode_NonBlocking);

    // Cleanup
    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    delete pSslConnection;

    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    delete pSslContext;

    if(isSocketImported != true)
    {
        g_CommonUtil.CloseTcpSocket(socketFd);
    }
}
#endif

#if defined(RUN_POSITIVE)
TEST(ShimPositive, Success)
{
    bool isSocketImported = false;
    nn::Result result;

    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    nn::ssl::Context*    pSslContext = new nn::ssl::Context();;
    nn::ssl::Connection* pSslConnection = new nn::ssl::Connection();;
    EXPECT_TRUE(pSslContext != nullptr);
    EXPECT_TRUE(pSslConnection != nullptr);

    result = pSslContext->Create(nn::ssl::Context::SslVersion_Auto);
    EXPECT_TRUE(result.IsSuccess());
    result = pSslConnection->Create(pSslContext);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetSocketDescriptor(socketFd);
    EXPECT_TRUE(result.IsSuccess());
    isSocketImported = true;

    result = pSslConnection->SetHostName(ServerName, strlen(ServerName));
    EXPECT_TRUE(result.IsSuccess());

    EXPECT_TRUE(pSslConnection->SetOption(nn::ssl::Connection::OptionType_SkipDefaultVerify, true).IsSuccess());
    result = pSslConnection->SetVerifyOption(nn::ssl::Connection::VerifyOption::VerifyOption_None);
    EXPECT_TRUE(result.IsSuccess());

    result = pSslConnection->SetSessionCacheMode(nn::ssl::Connection::SessionCacheMode_None);
    EXPECT_TRUE(result.IsSuccess());

    // Set buffer to get peer certificate
    char* serverCertBuff = new char[1024 * 4];
    result = pSslConnection->SetServerCertBuffer(serverCertBuff, 1024 * 4);
    EXPECT_TRUE(result.IsSuccess());

    uint32_t certSize = 0;
    result = pSslConnection->DoHandshake(&certSize, nullptr);
    ASSERT_TRUE(result.IsSuccess());
    EXPECT_TRUE(certSize > 0);
    NN_LOG(" SSL Handshake completed (server cert size:%d).\n", certSize);

    nn::ssl::Connection::CipherInfo cipherInfo;
    result = pSslConnection->GetCipherInfo(&cipherInfo);
    EXPECT_TRUE(result.IsSuccess());
    EXPECT_TRUE(strlen(cipherInfo.cipherName) > 0);
    EXPECT_TRUE(strlen(cipherInfo.versionName) > 0);

    char httpReqBuff[64] = {0};
    MY_SNPRINTF(httpReqBuff, sizeof(httpReqBuff), "GET / HTTP/1.0\r\nHost: %s\r\n\r\n", ServerName);
    uint32_t httpReqBuffLen = static_cast<uint32_t>(strlen(httpReqBuff));
    int sentBytes = pSslConnection->Write(httpReqBuff, httpReqBuffLen);
    ASSERT_TRUE(sentBytes > 0);
    NN_LOG(" Sent HTTP request over SSL (%d bytes).\n", sentBytes);

    int receivedTotalBytes    = 0;
    do
    {
        nn::ssl::Connection::PollEvent pollEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
        nn::ssl::Connection::PollEvent pollOutEvent = nn::ssl::Connection::PollEvent::PollEvent_None;
        pollEvent |= nn::ssl::Connection::PollEvent::PollEvent_Read;

        NN_LOG(" Calling poll w/to %d\n", g_MsecPollTimeout);
        result = pSslConnection->Poll(&pollOutEvent, &pollEvent, g_MsecPollTimeout);
        EXPECT_TRUE(result.IsSuccess());

        if((pollOutEvent & nn::ssl::Connection::PollEvent::PollEvent_Read)
           == nn::ssl::Connection::PollEvent::PollEvent_Read)
        {
            char tmpBuff[1024] = {0};
            int receivedBytes = pSslConnection->Read(tmpBuff, sizeof(tmpBuff));
            EXPECT_TRUE(receivedBytes >= 0);
            if(receivedBytes < 0)
            {
                NN_LOG(" nn::ssl::Read failed!\n");
                break;
            }
            if(receivedBytes == 0)
            {
                NN_LOG(" Connection closed by the server.\n");
                break;
            }
            receivedTotalBytes += receivedBytes;
        }
    } while(NN_STATIC_CONDITION(false));
    NN_LOG(" Received %d bytes\n", receivedTotalBytes);

    // Cleanup
    result = pSslConnection->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    delete pSslConnection;

    result = pSslContext->Destroy();
    EXPECT_TRUE(result.IsSuccess());
    delete pSslContext;

    delete[] serverCertBuff;

    if(isSocketImported != true)
    {
        g_CommonUtil.CloseTcpSocket(socketFd);
    }
}
#endif


#if defined(RUN_FORCE_DETACH)
TEST(ShimForceDetach, Success)
{
    int socketFd = g_CommonUtil.CreateTcpSocket(
        true,
        ServerPort_Normal,
        ServerName,
        0
    );
    EXPECT_TRUE(socketFd >= 0);

    nn::Result result;
    int i=0;
    const int resourceCount = 8;
    nn::ssl::Context*    pSslContext[resourceCount]    = {nullptr};
    nn::ssl::Connection* pSslConnection[resourceCount] = {nullptr};

    for(i =0; i < resourceCount; i++)
    {
        pSslContext[i] = new nn::ssl::Context();
        EXPECT_TRUE(pSslContext[i] != nullptr);
        result = pSslContext[i]->Create(nn::ssl::Context::SslVersion_Auto);
        EXPECT_TRUE(result.IsSuccess());
    }

    for(i =0; i < resourceCount; i++)
    {
        pSslConnection[i] = new nn::ssl::Connection();
        EXPECT_TRUE(pSslContext[i] != nullptr);
        result = pSslConnection[i]->Create(pSslContext[i]);
        EXPECT_TRUE(result.IsSuccess());
    }

    g_CommonUtil.CloseTcpSocket(socketFd);

    for(i =0; i < resourceCount; i++)
    {
        if(pSslContext[i])    delete pSslContext[i];
        if(pSslConnection[i]) delete pSslConnection[i];
    }

}
#endif

#ifdef RUN_BUILTIN_MANAGER
TEST(ShimBuiltIn, Success)
{
    uint32_t                                        bufSize = 0;
    nn::ssl::CaCertificateId                        nintendoId[] = { nn::ssl::CaCertificateId_NintendoCAG3 };
    nn::ssl::CaCertificateId                        commercialId[] = { nn::ssl::CaCertificateId_AmazonRootCA1 };
    nn::Result                                      result;
    uint8_t                                         *pBuf = nullptr;
    nn::ssl::BuiltInManager::BuiltInCertificateInfo *pCertInfoArray = nullptr;
    char                                            *pDerData = nullptr;
    nn::ssl::CertStoreId                            certId;

    //  Grab the first Nintendo trusted CA first
    result =
        nn::ssl::BuiltInManager::GetBuiltInCertificateBufSize(&bufSize,
                                                              nintendoId,
                                                              1);
    NN_LOG("[GetBuiltInCa, Success] need %u bytes\n", bufSize);
    ASSERT_TRUE(result.IsSuccess());

    pBuf = new uint8_t[bufSize];
    ASSERT_TRUE(pBuf != nullptr);

    result =
        nn::ssl::BuiltInManager::GetBuiltInCertificates(&pCertInfoArray,
                                                        pBuf,
                                                        bufSize,
                                                        nintendoId,
                                                        1);
    ASSERT_TRUE(result.IsSuccess());

    //  Attempt to import the cert into a nn::ssl::Context (this will
    //  verify the DER data is all there and correct.
    nn::ssl::Context *pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);

    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    pDerData =
        reinterpret_cast<char *>(pCertInfoArray[0].data.ptr.pCertificateDerData);
    result = pCtx->ImportServerPki(&certId,
                                   pDerData,
                                   pCertInfoArray[0].certificateSize,
                                   nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE(result.IsSuccess());

    delete[] pBuf;
    pBuf = nullptr;
    pCtx->Destroy();
    delete pCtx;
    pCtx = nullptr;

    //  Now repeat with the first commercial CA
    result =
        nn::ssl::BuiltInManager::GetBuiltInCertificateBufSize(&bufSize,
                                                              commercialId,
                                                              1);
    ASSERT_TRUE(result.IsSuccess());

    pBuf = new uint8_t[bufSize];
    ASSERT_TRUE(pBuf != nullptr);

    result =
        nn::ssl::BuiltInManager::GetBuiltInCertificates(&pCertInfoArray,
                                                        pBuf,
                                                        bufSize,
                                                        commercialId,
                                                        1);
    ASSERT_TRUE(result.IsSuccess());

    //  Attempt to import the cert into a nn::ssl::Context (this will
    //  verify the DER data is all there and correct.
    pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);

    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    pDerData =
        reinterpret_cast<char *>(pCertInfoArray[0].data.ptr.pCertificateDerData);
    result = pCtx->ImportServerPki(&certId,
                                   pDerData,
                                   pCertInfoArray[0].certificateSize,
                                   nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE(result.IsSuccess());

    delete[] pBuf;
    pBuf = nullptr;
    pCtx->Destroy();
    delete pCtx;
    pCtx = nullptr;
}

TEST(ShimMultiBuiltIn, Success)
{
    uint32_t                                        bufSize;
    nn::ssl::CaCertificateId                        twoIds[] = { nn::ssl::CaCertificateId_NintendoCAG3,
                                                                 nn::ssl::CaCertificateId_AmazonRootCA1 };
    nn::Result                                      result;
    uint8_t                                         *pBuf = nullptr;
    nn::ssl::BuiltInManager::BuiltInCertificateInfo *pCertInfoArray = nullptr;
    char                                            *pDerData = nullptr;
    nn::ssl::CertStoreId                            certId;

    //  Grab the first Nintendo trusted CA first
    result =
        nn::ssl::BuiltInManager::GetBuiltInCertificateBufSize(&bufSize,
                                                              twoIds,
                                                              2);
    ASSERT_TRUE(result.IsSuccess());

    pBuf = new uint8_t[bufSize];
    ASSERT_TRUE(pBuf != nullptr);

    result =
        nn::ssl::BuiltInManager::GetBuiltInCertificates(&pCertInfoArray,
                                                        pBuf,
                                                        bufSize,
                                                        twoIds,
                                                        2);
    ASSERT_TRUE(result.IsSuccess());

    //  Attempt to import the cert into a nn::ssl::Context (this will
    //  verify the DER data is all there and correct.
    nn::ssl::Context *pCtx = new nn::ssl::Context();
    ASSERT_TRUE(pCtx != nullptr);

    result = pCtx->Create(nn::ssl::Context::SslVersion_Auto);
    ASSERT_TRUE(result.IsSuccess());

    pDerData =
        reinterpret_cast<char *>(pCertInfoArray[0].data.ptr.pCertificateDerData);
    result = pCtx->ImportServerPki(&certId,
                                   pDerData,
                                   pCertInfoArray[0].certificateSize,
                                   nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE(result.IsSuccess());

    pDerData =
        reinterpret_cast<char *>(pCertInfoArray[1].data.ptr.pCertificateDerData);
    result = pCtx->ImportServerPki(&certId,
                                   pDerData,
                                   pCertInfoArray[1].certificateSize,
                                   nn::ssl::CertificateFormat_Der);
    EXPECT_TRUE(result.IsSuccess());

    delete[] pBuf;
    pBuf = nullptr;
    pCtx->Destroy();
    delete pCtx;
    pCtx = nullptr;
}
#endif    //  RUN_BUILTIN_MANAGER


// This test MUST be the last test to ensure SSL is finalized properly
TEST(ShimFinalize, Success)
{
    nn::Result result = nn::ssl::Finalize();
    EXPECT_TRUE(result.IsSuccess());
    nn::socket::Finalize();
#ifdef USE_NETWORK
    g_CommonUtil.FinalizeNetwork();
#endif
}
