﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNet_P2P.h"
#include "CommandLineParser.h"

#include <cmath>

namespace nnt { namespace net { namespace p2p {

// GetPeerByHandle
NodeInfo& PacketRateTest::GetPeerByHandle(uint32_t peerHandle) NN_NOEXCEPT
{
    if( peerHandle >= m_PeerCount || g_pPeerIndeces[peerHandle] >= PacketRateTest::GetParams().nodeCount)
    {
        NN_NETTEST_LOG("\n *** CRITICAL ERROR! Index out of range! Index: %d\n\n", peerHandle);
        NETTEST_ABORT();
#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wnull-dereference"
#endif
        return *(NodeInfo*)NULL;
#if defined (NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic pop
#endif

    }

    return g_pNodes[g_pPeerIndeces[peerHandle]];
}

// BufferedLogger
BufferedLogger::BufferedLogger(char* pBuffer, uint32_t bufferLen) NN_NOEXCEPT
    : m_pBuffer(pBuffer),
      m_BufferLen(bufferLen),
      m_PrintPos(0),
      m_FlushPos(0),
      m_IsOutOfSpace(false)
{
    if( m_pBuffer == NULL )
    {
        NN_NETTEST_LOG("\n *** CRITICAL ERROR! BufferedLogger: pBuffer passed to constructor is NULL!\n\n");
        NETTEST_ABORT();
        return;
    }
}

// Log
void BufferedLogger::Log(const char* format, ...) NN_NOEXCEPT
{
    va_list args;
    uint32_t flushPos = m_FlushPos;
    int remainingBufferLen;

    if( flushPos > m_PrintPos )
    {
        remainingBufferLen = flushPos - m_PrintPos;
    }
    else
    {
        remainingBufferLen = (int)m_BufferLen - (int)m_PrintPos;
    }

    if( remainingBufferLen <= 0 )
    {
        if( m_IsOutOfSpace )
        {
            NN_LOG(".");
        }
        else
        {
            NN_NETTEST_LOG(" ** WARNING! Print buffer ran out of space\n");
            m_IsOutOfSpace = true;
        }

        return;
    }

    va_start(args, format);
    int rval = vsnprintf(&m_pBuffer[m_PrintPos], remainingBufferLen, format, args);
    va_end(args);

    if( rval < remainingBufferLen )
    {
        m_PrintPos += rval;
    }
    else if( rval >= 0 )
    {
        if( flushPos == 0 || flushPos > m_PrintPos )
        {
            if( m_IsOutOfSpace )
            {
                NN_LOG(".");
            }
            else
            {
                NN_NETTEST_LOG(" ** WARNING! Print buffer ran out of space!\n");
                m_IsOutOfSpace = true;
            }

            return;
        }
        else
        {
            m_PrintPos = 0;
        }
    }

    m_IsOutOfSpace = false;
}

// Flush
void BufferedLogger::Flush(uint32_t charCount) NN_NOEXCEPT
{
    uint32_t printPos = m_PrintPos;
    uint32_t printLen = 0;
    bool doLoop = false;
    int32_t secondPrint = 0;

    // Pass zero to flush everythin!
    if( charCount == 0 )
    {
        if( printPos >= m_FlushPos )
        {
            printLen = printPos - m_FlushPos;
        }
        else
        {
            printLen = m_BufferLen - m_FlushPos;
            secondPrint = printPos;
            doLoop = true;
        }
    }
    else
    {
        printLen = printPos - m_FlushPos < charCount ? printPos - m_FlushPos : charCount;
        if( printPos >= m_FlushPos )
        {
            printLen = printPos - m_FlushPos < charCount ? printPos - m_FlushPos : charCount;
            if( printLen > (m_BufferLen - printPos) )
            {
                printLen = (m_BufferLen - printPos);
                doLoop = true;
            }
        }
        else
        {
            if( charCount < m_BufferLen - m_FlushPos )
            {
                printLen = charCount;
            }
            else if( charCount == m_BufferLen - m_FlushPos )
            {
                printLen = charCount;
                doLoop = true;
            }
            else
            {
                printLen = m_BufferLen - m_FlushPos;
                secondPrint = charCount - printLen;
                doLoop = true;
            }
        }
    }

    NN_LOG("%.*s", printLen, &m_pBuffer[m_FlushPos]);

    if( doLoop )
    {
        m_FlushPos = 0;
    }
    else
    {
        m_FlushPos += printLen;
    }

    if( secondPrint )
    {
        NN_LOG("%.*s", secondPrint, &m_pBuffer[m_FlushPos]);
        m_FlushPos += secondPrint;
    }

    if( m_FlushPos >= m_BufferLen )
    {
        m_FlushPos = 0;
    }
}

// SetHeader
void NetMessage::SetHeader(uint8_t nodeIndex, uint32_t sequenceNum, NetHeaderFlags flags, uint32_t packetSize) NN_NOEXCEPT
{
    m_Header.headerLen   = sizeof(m_Header);
    m_Header.nodeIndex   = nodeIndex;
    m_Header.checksum    = 0;
    m_Header.sequenceNum = NetTest::Htonl(sequenceNum);
    m_Header.flags       = flags;

    m_Header.checksum = NetTest::Htons(CalcCheckSum(this, packetSize));
}

// CalcCheckSum
uint16_t NetMessage::CalcCheckSum(void* pData, uint32_t dataLen) NN_NOEXCEPT
{
    uint32_t  sum = 0;
    uint16_t* p;
    int len = dataLen;

    p = (uint16_t*) pData;
    while (0 < len)
    {
        sum += NetTest::Ntohs(*p++);
        len -= 2;
    }
    sum = (sum & 0xffff) + (sum >> 16);
    sum = (sum & 0xffff) + (sum >> 16);

    return (uint16_t) (sum ^ 0xffff);
}

// VerifyCheckSum
bool NetMessage::VerifyCheckSum(uint32_t packetLen) NN_NOEXCEPT
{
    return CalcCheckSum(this, packetLen) == 0;
}

// ValidateHeader
bool NetMessage::ValidateHeader(uint32_t recvLen, const nn::socket::SockAddrIn& remoteAddr) NN_NOEXCEPT
{
    if( recvLen < sizeof(NetHeader) )
    {
        NN_NETTEST_LOG("\n Recved message smaller than our header. Len: %d\n\n", recvLen);
        return false;
    }

    if( GetHeaderLen() != sizeof(NetHeader) )
    {
        NN_NETTEST_LOG("\nERROR: Recved message with unexpected header length! expted: %d, header len: %d\n\n", (int)sizeof(NetHeader), (int)GetHeaderLen());
        return false;
    }

    if( GetNodeIndex() >= PacketRateTest::GetParams().nodeCount )
    {
        NN_NETTEST_LOG("\nERROR: Recved message with out of range node index. index: %d\n\n", GetNodeIndex());
        return false;
    }

    if( PacketRateTest::g_pNodes[GetNodeIndex()].addr.sin_addr.S_addr != remoteAddr.sin_addr.S_addr )
    {
        NN_NETTEST_LOG("\nERROR: Recved message does not match ip address. index: %d, recved IP: %s, expected IP: %s\n\n", GetNodeIndex(), NetTest::InetNtoa(remoteAddr.sin_addr), PacketRateTest::g_Params.pAddrs[GetNodeIndex()].pIp);
        return false;
    }

    return true;
}

// TimedRecvFrom
ssize_t PacketRateTest::TimedRecvFrom(int socket, NetMessage& message, nn::socket::SockAddrIn* pAddr, nn::socket::SockLenT* pAddrLen, NetTest::Tick& recvdTick) NN_NOEXCEPT
{
    ssize_t rval;

    NetTest::Tick startRecv = NetTest::GetTick();
    rval = NetTest::RecvFrom(socket, &message, sizeof(message), nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr*>(pAddr), pAddrLen);
    recvdTick = NetTest::GetTick();

    if( rval > 0 )
    {
        uint32_t duration = (uint32_t)NetTest::TickToTime(recvdTick - startRecv).GetMicroSeconds();
        g_Stats.pSendSamples[g_Stats.recvCount] = duration;
        g_Stats.recvDurationSum += duration;

        ++g_Stats.recvCount;
        if( g_Stats.recvCount >= MaxSamples )
        {
            NN_NETTEST_LOG("ERROR: Sample buffer full! Try reducing REPORT_INT, increasing MaxSamples, or removing --REPORT_CALLTIME\n\n");
            return -1;
        }

        if( duration < g_Stats.minRecvMicroSec )
        {
            g_Stats.minRecvMicroSec = duration;
        }

        if( duration > g_Stats.maxRecvMicroSec )
        {
            g_Stats.maxRecvMicroSec = duration;
        }
    }

    return rval;
}

// NormalRecvFrom
ssize_t PacketRateTest::NormalRecvFrom(int socket, NetMessage& message, nn::socket::SockAddrIn* pAddr, nn::socket::SockLenT* pAddrLen, NetTest::Tick& recvdTick) NN_NOEXCEPT
{
    ssize_t rval = NetTest::RecvFrom(socket, &message, sizeof(message), nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr*>(pAddr), pAddrLen);
    recvdTick = NetTest::GetTick();
    return rval;
}

// TimedSendTo
ssize_t PacketRateTest::TimedSendTo(int socket, NetMessage& message, NodeInfo& peer, uint32_t packetSize) NN_NOEXCEPT
{
    ssize_t rval;
    NetTest::Tick endSend;

    NetTest::Tick startSend = NetTest::GetTick();
    rval = NetTest::SendTo(socket, &message, packetSize, nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr*>(&peer.addr), sizeof(peer.addr));
    endSend = NetTest::GetTick();

    if( rval > 0 )
    {
        uint32_t duration = (uint32_t)NetTest::TickToTime(endSend - startSend).GetMicroSeconds();

        g_Stats.pSendSamples[g_Stats.sendCount] = duration;
        ++g_Stats.sendCount;

        if( g_Stats.sendCount >= MaxSamples )
        {
            NN_NETTEST_LOG("ERROR: Sample buffer full! Try reducing REPORT_INT, increasing MaxSamples, or removing --REPORT_CALLTIME\n\n");
            return -1;
        }

        g_Stats.sendDurationSum += duration;

        if( duration < g_Stats.minSendMicroSec )
        {
            g_Stats.minSendMicroSec = duration;
        }

        if( duration > g_Stats.maxSendMicroSec )
        {
            g_Stats.maxSendMicroSec = duration;
        }
    }

    return rval;
}

// NormalSendTo
ssize_t PacketRateTest::NormalSendTo(int socket, NetMessage& message, NodeInfo& peer, uint32_t packetSize) NN_NOEXCEPT
{
    return NetTest::SendTo(socket, &message, packetSize, nn::socket::MsgFlag::Msg_None, reinterpret_cast<nn::socket::SockAddr*>(&peer.addr), sizeof(peer.addr));
}

// Print stats for peer.
void PeerStats::Print(uint32_t statsDurationMs, uint32_t frameCount) NN_NOEXCEPT
{
    int rtt = 0;
    if( NetTest::TickToTime(totalRttTicks).GetMicroSeconds() != 0 && ackRecvCount != 0 )
    {
        rtt = (int)NetTest::TickToTime(totalRttTicks).GetMicroSeconds() / ackRecvCount;
    }

    P2P_LOG("Packets sent last frame        : %d/%d\n", (int)frameSent, (int)PacketRateTest::GetParams().packetsPerFrame);
    P2P_LOG("Avg. packets sent per frame    : %d\n",    (int)(packetsSent / frameCount));
    P2P_LOG("Packets recvd last frame       : %d\n",    (int)frameRecvd);
    P2P_LOG("Avg. Packets Recvd per frame   : %d\n",    (int)(packetsRecved / frameCount));
    P2P_LOG("Send Rate                      : %d per sec\n",     (int)(packetsSent   / (statsDurationMs / 1000.0f)));
    P2P_LOG("Recv Rate                      : %d per sec\n",     (int)(packetsRecved / (statsDurationMs / 1000.0f)));
    P2P_LOG("Recv/Sent ratio                : %d%%\n", (int)(100.0f * ((float)packetsRecved / (float)packetsSent)));

    P2P_LOG("Recv Drop Rate(raw)            : %.2f%%\n",      (100.0f * ((float)(packetsDropped - packetsOutOfRange) / (float)(packetsRecved + packetsDropped))));
    P2P_LOG("Recv Drop Rate(unexpected IDs) : %.2f%%\n",      (100.0f * ((float)(packetsDropped) / (float)(packetsRecved + packetsDropped))));
    P2P_LOG("Missing Packets                : %d\n",          (int)packetsDropped);
    P2P_LOG("Out of range packets           : %d\n",          (int)packetsOutOfRange);
    P2P_LOG("Total RTT ACKS Recvd           : %d\n",          (int)ackRecvCount);
    P2P_LOG("Average RTT                    : %d MircoSec\n", rtt);

    if( PacketRateTest::GetParams().doVerifyData )
    {
        P2P_LOG("Packets failed checksum verification: %d\n", packetsBadData);
    }

    P2P_LOG("\n");

    Clear();
}

// Clears stats.
void PeerStats::Clear() NN_NOEXCEPT
{
    packetsSent       = 0;
    packetsRecved     = 0;
    packetsOutOfRange = 0;
    packetsDropped    = 0;
    packetsBadData    = 0;
    rttSeqNum         = 0;
    totalRttTicks     = NetTest::Tick(0);
    ackRecvCount      = 0;
}

// Params Constructor
Params::Params() NN_NOEXCEPT
    : nodeCount(0),
      nodeIndex(0xFF),
      packetSize(1450),
      packetsPerFrame(1),
      frameIntervalMs(16.66f),
      reportIntervalMs(4000),
      testDurationSec(30),
      rttFrameInterval(0),
      reportCallTime(false),
      doVerifyData(false),
      doTermOnBadData(false),
      reportNodeStats(false),
      reportCsv(false),
      noReport(false)
{
    memset(pAddrs, 0, sizeof(pAddrs));
}

// ParseCommandLine
bool Params::ParseCommandLine(int argC, const char * const * argV) NN_NOEXCEPT
{
    if( argC <= 1 )
    {
        NN_NETTEST_LOG("Parsing Error. Not enough arguments. argc: %d\n\n", argC);
        return false;
    }

    // loop through all command line arguments
    for(unsigned iArg = 1; static_cast<int>(iArg) < argC; ++iArg)
    {
        bool isParsed = false;

        // Search for a parser
        for(unsigned iParser = 0; iParser < Parser::ParserCount; ++iParser)
        {
            // Is this the right parser?
            if( strncmp(argV[iArg], Parser::g_pParsers[iParser]->GetName(), strlen(Parser::g_pParsers[iParser]->GetName())) == 0 )
            {
                if( Parser::g_pParsers[iParser]->Parse(*this, argC, argV, iArg) )
                {
                    isParsed = true;
                    break; // Stop searching for a parser.
                }
                else
                {
                    return false;
                }
            }
        }

        if( !isParsed )
        {
            NN_NETTEST_LOG("\nERROR: Could not find a parser for command line option: %s\n\n", argV[iArg]);
            return false;
        }
    }

    if( nodeIndex == 0xFF )
    {
        NN_NETTEST_LOG("\nERROR: --NODE_INDEX must be specified!\n Example: --NODE_INDEX 0\n\n");
        return false;
    }

    if( nodeCount < 2 )
    {
        NN_NETTEST_LOG("\nERROR: --ADDR_INFO must be specified with at lease TWO entries!\n Example: --ADDR_INFO 192.168.0.2:8000 192.168.0.3:8001\n\n");
        return false;
    }

    return true;
}

// CalcStandardDeviation
uint32_t CalcStandardDeviation(uint32_t* pSamples, uint32_t sampleCount, uint32_t average) NN_NOEXCEPT
{
    uint32_t diviationSum = 0;
    for(unsigned i = 0; i < sampleCount; ++i)
    {
        uint32_t diff = (uint32_t)((int32_t)pSamples[i] - average);
        diviationSum += diff * diff;
    }

    return (uint32_t)sqrt(diviationSum / sampleCount);
}

// Prints global statistics
void PacketRateTest::PrintStats(uint32_t statsDurationMs, uint32_t frameDurationMs) NN_NOEXCEPT
{
    int32_t packetsSent       = 0;
    int32_t packetsRecved     = 0;
    int32_t packetsDropped    = 0;
    int32_t packetsOutOfRange = 0;
    uint32_t ackRecvCount     = 0;
    NetTest::Tick totalRttTicks(0);

    for(unsigned iPeer = 0; iPeer < m_PeerCount; ++iPeer)
    {
        NodeInfo& peer = GetPeerByHandle(iPeer);

        packetsSent       += peer.packetStats.packetsSent;
        packetsRecved     += peer.packetStats.packetsRecved;
        packetsDropped    += peer.packetStats.packetsDropped;
        packetsOutOfRange += peer.packetStats.packetsOutOfRange;
        totalRttTicks     += peer.packetStats.totalRttTicks;
        ackRecvCount      += peer.packetStats.ackRecvCount;

        if( PacketRateTest::GetParams().reportNodeStats && ! PacketRateTest::GetParams().noReport)
        {
            P2P_LOG("Node                           : %d\n", peer.nodeIndex);
            peer.packetStats.Print(statsDurationMs, g_Stats.frameCount);
        }
        else
        {
            peer.packetStats.Clear();
        }
    }

    if( PacketRateTest::GetParams().reportCsv )
    {
        int rtt = 0;
        if( NetTest::TickToTime(totalRttTicks).GetMicroSeconds() != 0 && ackRecvCount != 0 )
        {
            rtt = (int)NetTest::TickToTime(totalRttTicks).GetMicroSeconds() / ackRecvCount;
        }

        CSV_LOG("%d,%d,%d,%.2f,%d\n", (int)g_Params.nodeIndex, (int)(packetsRecved / (statsDurationMs / 1000.0f)), (int)(packetsSent / (statsDurationMs / 1000.0f)), (float)(100.0f * ((float)(packetsDropped - packetsOutOfRange) / (float)(packetsRecved + packetsDropped))), rtt);
    }

    if( !PacketRateTest::GetParams().noReport )
    {
        P2P_LOG("Frame count                    : %d/%d\n", (int)g_Stats.frameCount, (int)(g_Params.reportIntervalMs / g_Params.frameIntervalMs));
        P2P_LOG("Avg. Frame duration            : %.1f/%.1f\n", g_Stats.frameDurationSumMs / (float)g_Stats.frameCount, g_Params.frameIntervalMs);
        P2P_LOG("Stats duration                 : %d/%d\n\n", (int)statsDurationMs, (int)g_Params.reportIntervalMs);

        P2P_LOG("Send Rate                      : %d/%d per second\n", (int)(packetsSent / (statsDurationMs / 1000.0f)), (int)(g_Params.packetsPerFrame * m_PeerCount * (1000.0f / g_Params.frameIntervalMs)));
        P2P_LOG("Recv Rate                      : %d per second\n", (int)(packetsRecved / (statsDurationMs / 1000.0f)));
        P2P_LOG("Recv Drop Rate(raw)            : %.2f%%\n", (100.0f * ((float)(packetsDropped - packetsOutOfRange) / (float)(packetsRecved + packetsDropped))));
        P2P_LOG("Recv Drop Rate(unexpected IDs) : %.2f%%\n", (100.0f * ((float)(packetsDropped) / (float)(packetsRecved + packetsDropped))));
        P2P_LOG("Recved unexpected packet IDs   : %d\n\n", packetsOutOfRange);

        if( PacketRateTest::GetParams().reportCallTime )
        {
            uint32_t sendAverage = g_Stats.sendDurationSum / g_Stats.sendCount;
            int32_t sendDeviation = CalcStandardDeviation(g_Stats.pSendSamples, g_Stats.sendCount, sendAverage);

            P2P_LOG("SendTo Avg. Duration   : %d Micro Seconds\n",   (int)sendAverage);
            P2P_LOG("SendTo Min. Duration   : %d Micro Seconds\n",   (int)g_Stats.minSendMicroSec);
            P2P_LOG("SendTo Max. Duration   : %d Micro Seconds\n",   (int)g_Stats.maxSendMicroSec);
            P2P_LOG("SendTo Deviation       : %d Micro Seconds\n\n", (int)sendDeviation);

            uint32_t recvAverage = g_Stats.recvDurationSum / g_Stats.recvCount;
            int32_t recvDeviation = CalcStandardDeviation(g_Stats.pRecvSamples, g_Stats.recvCount, recvAverage);

            P2P_LOG("RecvFrom Avg. Duration : %d Micro Seconds\n",   (int)recvAverage);
            P2P_LOG("RecvFrom Min. Duration : %d Micro Seconds\n",   (int)g_Stats.minRecvMicroSec);
            P2P_LOG("RecvFrom Max. Duration : %d Micro Seconds\n",   (int)g_Stats.maxRecvMicroSec);
            P2P_LOG("RecvFrom Deviation     : %d Micro Seconds\n\n", (int)recvDeviation);
        }
    }

    g_Stats.Clear();
}

// uint32_t PacketRateTest::GetRandom() NN_NOEXCEPT
// {
//     static uint32_t prev = 0;
//     return (prev = (prev ^ (uint32_t)(nn::os::GetSystemTick().GetInt64Value())));
// }

}}} // Namespaces
