﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/os/os_Thread.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <cstdlib> // For malloc
#include <new>     // For placement new
#include <util_ThreadManager.h>

#define NN_SERVER_LOG(format, ...) \
    do \
    {  \
        NN_LOG("[SASStress - Server]: " format, ##__VA_ARGS__); \
    } while ( NN_STATIC_CONDITION(false) )
#define NN_CLIENT_LOG(format, ...) \
    do \
    {  \
        NN_LOG("[SASStress - Client]: " format, ##__VA_ARGS__); \
    } while ( NN_STATIC_CONDITION(false) )
using namespace nnt::net::manualtests::utils;

namespace
{
    //NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
    NN_ALIGNAS(4096) nn::socket::ConfigDefaultWithMemory g_socketConfig;
    bool isClient = false;
    const char * ServerAddress = NULL;
    uint16_t ServerPort = 8110;
    uint32_t SendSize = 256 * 1024;
    // const uint32_t DefaultSocketNumber = 3;
    NN_ALIGNAS(4096) uint8_t g_threadManagerMem[sizeof(ThreadManager)];
    ThreadManager * g_threadManager;

    struct ServerArgs
    {
        int socket;
        uint32_t sendBufferSize;
        uint32_t recvBufferSize;
    };

    struct ClientArgs
    {
        int socket;
        uint32_t sendBufferSize;
        uint32_t recvBufferSize;
    };

    struct SendArgs
    {
        int socket;
    };
    struct ReceiveArgs
    {
        int socket;
    };
    nn::nifm::NetworkConnection * g_pNifmConnection;
}

void SendThread(void * args)
{
    SendArgs * sendArgs = (SendArgs *)args;
    ThreadManager::ThreadStatus currentStatus;
    char * buffer = (char *)malloc(SendSize);
    if (buffer == NULL)
    {
        g_threadManager->SetCurrentThreadStatus(ThreadManager::ThreadStatus_Error);
        return;
    }
    memset(buffer, 1, SendSize);
    uint64_t totalBytesSent = 0;
    uint32_t sendCount = 0;
    uint32_t wouldBlockCount = 0;

    do
    {
        int ret = nn::socket::Send(sendArgs->socket, buffer, SendSize, nn::socket::MsgFlag::Msg_None);
        if (ret < 0)
        {
            if (nn::socket::GetLastError() != nn::socket::Errno::EWouldBlock)
            {
                NN_SERVER_LOG("Failed to Send on socket as Non-Blocking. error: %d\n", nn::socket::GetLastError());
                break;
            }
            ++wouldBlockCount;
        }
        else if (ret == 0)
        {
            NN_SERVER_LOG("Connection Shut down gracefully.\n");
            break;
        }
        else
        {
            ++sendCount;
            totalBytesSent += ret;

            if (sendCount % 1000 == 0)
            {
                NN_SERVER_LOG("fd: %d, sends: %d, bytes sent: %d, EAGAIN: %d \n", sendArgs->socket, sendCount, totalBytesSent, wouldBlockCount);
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(25));
    } while (g_threadManager->GetCurrentThreadStatus(&currentStatus) != -1 && currentStatus == ThreadManager::ThreadStatus_Running);

    free(buffer);
}

void ServerThread(void * args)
{
    ServerArgs * serverArgs = (ServerArgs *)args;
    NN_UNUSED(serverArgs);
    ThreadManager::ThreadStatus currentStatus;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        int listenSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);

        if (listenSocket == -1)
        {
            NN_SERVER_LOG("Failed to open listen socket. error: %d\n", nn::socket::GetLastError());
            continue;
        }
        nn::socket::SockAddrIn saServer;
        saServer.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
        saServer.sin_port = nn::socket::InetHtons(ServerPort);
        saServer.sin_family = nn::socket::Family::Af_Inet;
        // bind the socket to an open port
        if (nn::socket::Bind(listenSocket, reinterpret_cast<nn::socket::SockAddr *>(&saServer), sizeof(saServer)) < 0)
        {
            NN_SERVER_LOG("Failed to Bind listen socket. error: %d\n", nn::socket::GetLastError());
            if (nn::socket::Close(listenSocket) < 0)
            {
                NN_SERVER_LOG("Failed to Close listen socket. error: %d\n", nn::socket::GetLastError());
            }
            continue;
        }
        nn::socket::SockLenT saServerLen = sizeof(saServer);
        if (nn::socket::GetSockName(listenSocket, reinterpret_cast<nn::socket::SockAddr *>(&saServer), &saServerLen) < 0)
        {
            NN_SERVER_LOG("Failed to GetSockName for listen socket. error: %d\n", nn::socket::GetLastError());
            if (nn::socket::Close(listenSocket) < 0)
            {
                NN_SERVER_LOG("Failed to Close listen socket. error: %d\n", nn::socket::GetLastError());
            }
            continue;
        }


        if (nn::socket::Listen(listenSocket, -1) < 0)
        {
            NN_SERVER_LOG("Failed to Listen on listen socket. error: %d\n", nn::socket::GetLastError());
            if (nn::socket::Close(listenSocket) < 0)
            {
                NN_SERVER_LOG("Failed to Close listen socket. error: %d\n", nn::socket::GetLastError());
            }
            continue;
        }

        do
        {
            nn::socket::SockAddrIn saClient = { 0 };
            nn::socket::SockLenT size = sizeof(saClient);
            NN_SERVER_LOG("Waiting for new connection on %s:%d\n",nn::socket::InetNtoa(saServer.sin_addr),nn::socket::InetNtohs(saServer.sin_port));
            int clientSocket = nn::socket::Accept(listenSocket, reinterpret_cast<nn::socket::SockAddr *>(&saClient), &size);
            if (clientSocket < 0)
            {
                NN_SERVER_LOG("Failed to Accept on listen socket. error: %d\n", nn::socket::GetLastError());
                break;
            }
            NN_SERVER_LOG("New connection on %s:%d from %s:%d!\n", nn::socket::InetNtoa(saServer.sin_addr), nn::socket::InetNtohs(saServer.sin_port),
                                                                   nn::socket::InetNtoa(saClient.sin_addr), nn::socket::InetNtohs(saClient.sin_port));
            int val = 1;
            nn::socket::SetSockOpt(clientSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::Tcp_NoDelay, &val, sizeof(val));
            if (nn::socket::Fcntl(clientSocket, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock) < 0)
            {
                NN_SERVER_LOG("Failed to set socket as Non-Blocking. error: %d\n", nn::socket::GetLastError());
                if (nn::socket::Close(clientSocket) < 0)
                {
                    NN_SERVER_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
                }
                break;
            }
            SendArgs * args = (SendArgs *)malloc(sizeof(SendArgs));
            if (args == NULL)
            {
                g_threadManager->SetCurrentThreadStatus(ThreadManager::ThreadStatus_Error);
                if (nn::socket::Close(clientSocket) < 0)
                {
                    NN_SERVER_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
                }
                break;
            }
            args->socket = clientSocket;
            nn::os::ThreadType * sendThread = g_threadManager->CreateThread(SendThread, args, nn::os::LowestThreadPriority);
            g_threadManager->StartAllThreads();
            if (g_threadManager->WaitThreadsWithTimeout(&sendThread, 1, nn::TimeSpan::FromSeconds(120)) == -1)
            {
                NN_SERVER_LOG("Waiting for thread Timed out!: %p\n", sendThread);
            }
            if (g_threadManager->DestroyThreadWithTimeout(sendThread, nn::TimeSpan::FromSeconds(3)) == -1)
            {
                NN_SERVER_LOG("Failed to Destroy Thread!: %p\n", sendThread);
                g_threadManager->SetCurrentThreadStatus(ThreadManager::ThreadStatus_Error);
                if (nn::socket::Close(clientSocket) < 0)
                {
                    NN_SERVER_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
                }
                break;
            }
            free(args);
            if (nn::socket::Close(clientSocket) < 0)
            {
                NN_SERVER_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
                break;
            }

        } while (g_threadManager->GetCurrentThreadStatus(&currentStatus) != -1 && currentStatus == ThreadManager::ThreadStatus_Running);
        if (nn::socket::Close(listenSocket) < 0)
        {
            NN_SERVER_LOG("Failed to Close listen socket. error: %d\n", nn::socket::GetLastError());
        }
    } while (g_threadManager->GetCurrentThreadStatus(&currentStatus) != -1 && currentStatus == ThreadManager::ThreadStatus_Running);


} //NOLINT(impl/function_size)

void ReceiveThread(void * args)
{
    ReceiveArgs * receiveArgs = (ReceiveArgs *)args;
    ThreadManager::ThreadStatus currentStatus;
    char * buffer = (char *)malloc(SendSize);
    if (buffer == NULL)
    {
        g_threadManager->SetCurrentThreadStatus(ThreadManager::ThreadStatus_Error);
        return;
    }
    uint64_t totalBytesReceived = 0;
    uint32_t receiveCount = 0;
    uint32_t wouldBlockCount = 0;
    do
    {
        ssize_t received = nn::socket::Recv(receiveArgs->socket, buffer, SendSize, nn::socket::MsgFlag::Msg_None);
        if (received < 0)
        {
            if (nn::socket::GetLastError() != nn::socket::Errno::EWouldBlock)
            {
                NN_CLIENT_LOG("Failed to Recv on socket. error: %d\n", nn::socket::GetLastError());
                break;
            }
            ++wouldBlockCount;
        }
        else if (received == 0)
        {
            NN_CLIENT_LOG("Connection Shut down gracefully.\n");
            break;
        }
        else
        {
            ++receiveCount;
            totalBytesReceived += received;

            if (receiveCount % 1000 == 0)
            {
                NN_CLIENT_LOG("fd: %d, receives: %d, bytes received: %d, EAGAIN: %d \n", receiveArgs->socket, receiveCount, totalBytesReceived, wouldBlockCount);
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(25));
    } while (g_threadManager->GetCurrentThreadStatus(&currentStatus) != -1 && currentStatus == ThreadManager::ThreadStatus_Running);

    free(buffer);

}
void ClientThread(void * args)
{
    ClientArgs * clientArgs = (ClientArgs *)args;
    NN_UNUSED(clientArgs);
    ThreadManager::ThreadStatus currentStatus;

    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        int clientSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        if (clientSocket == -1)
        {
            NN_CLIENT_LOG("Failed to open client socket. error: %d\n", nn::socket::GetLastError());
            continue;
        }

        nn::socket::SockAddrIn saServer;
        nn::socket::InetAton(ServerAddress,&saServer.sin_addr);
        NN_CLIENT_LOG("SETTING PORT: %d\n",ServerPort);
        saServer.sin_port = nn::socket::InetHtons(ServerPort);
        saServer.sin_family = nn::socket::Family::Af_Inet;
        NN_CLIENT_LOG("Connecting to server at %s:%d\n", nn::socket::InetNtoa(saServer.sin_addr), nn::socket::InetNtohs(saServer.sin_port));
        if (nn::socket::Connect(clientSocket, reinterpret_cast<nn::socket::SockAddr *>(&saServer), sizeof(saServer)) < 0)
        {
            NN_CLIENT_LOG("Failed to Connect to server. error: %d\n", nn::socket::GetLastError());
            if (nn::socket::Close(clientSocket) < 0)
            {
                NN_CLIENT_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
            }
            continue;
        }

        nn::socket::SockAddrIn saClient = { 0 };
        nn::socket::SockLenT saClientLen = sizeof(saServer);
        if (nn::socket::GetSockName(clientSocket, reinterpret_cast<nn::socket::SockAddr *>(&saServer), &saClientLen) < 0)
        {
            NN_CLIENT_LOG("Failed to GetSockName for client socket. error: %d\n", nn::socket::GetLastError());
            if (nn::socket::Close(clientSocket) < 0)
            {
                NN_CLIENT_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
            }
            continue;
        }
        NN_CLIENT_LOG("Connected to server at %s:%d on %s:%d\n", nn::socket::InetNtoa(saServer.sin_addr), nn::socket::InetNtohs(saServer.sin_port)
                                                               , nn::socket::InetNtoa(saClient.sin_addr), nn::socket::InetNtohs(saClient.sin_port));
        int val = 1;
        nn::socket::SetSockOpt(clientSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::Tcp_NoDelay, &val, sizeof(val));

        if (nn::socket::Fcntl(clientSocket, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock) < 0)
        {
            NN_CLIENT_LOG("Failed to set socket as Non-Blocking. error: %d\n", nn::socket::GetLastError());
            if (nn::socket::Close(clientSocket) < 0)
            {
                NN_CLIENT_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
            }
            break;
        }

        ReceiveArgs * args = (ReceiveArgs *)malloc(sizeof(ReceiveArgs));
        if (args == NULL)
        {
            g_threadManager->SetCurrentThreadStatus(ThreadManager::ThreadStatus_Error);
            if (nn::socket::Close(clientSocket) < 0)
            {
                NN_CLIENT_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
            }
            break;
        }
        args->socket = clientSocket;
        nn::os::ThreadType * receiveThread = g_threadManager->CreateThread(ReceiveThread, args,nn::os::LowestThreadPriority);
        g_threadManager->StartAllThreads();
        if (g_threadManager->WaitThreadsWithTimeout(&receiveThread, 1, nn::TimeSpan::FromSeconds(120)) == -1)
        {
            NN_CLIENT_LOG("ERROR: Waiting for thread Timed out!: %p\n", receiveThread);
        }
        if (g_threadManager->DestroyThreadWithTimeout(receiveThread, nn::TimeSpan::FromSeconds(3)) == -1)
        {
            NN_CLIENT_LOG("ERROR: Failed to Destroy Thread!: %p\n", receiveThread);
            g_threadManager->SetCurrentThreadStatus(ThreadManager::ThreadStatus_Error);
            if (nn::socket::Close(clientSocket) < 0)
            {
                NN_CLIENT_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
            }
            break;
        }

        free(args);
        if (nn::socket::Close(clientSocket) < 0)
        {
            NN_CLIENT_LOG("Failed to Close client socket. error: %d\n", nn::socket::GetLastError());
        }
    }
    while (g_threadManager->GetCurrentThreadStatus(&currentStatus) != -1 && currentStatus == ThreadManager::ThreadStatus_Running);

}
void ParseArgs(int argCount, char ** args)
{
    bool setIsClient = false;
    bool setPort = false;
    for (int i = 0; i < argCount; ++i)
    {
        if (strncmp(args[i], "ServerAddress=", sizeof("ServerAddress=") - 1) == 0)
        {
            ServerAddress = args[i] + sizeof("ServerAddress=") - 1;
            NN_LOG("ServerAddress = %s\n", ServerAddress);
        }
        else if (strncmp(args[i], "ServerPort=", sizeof("ServerPort=") - 1) == 0)
        {
            ServerPort = atoi(args[i] + sizeof("ServerPort=") - 1);
            setPort = true;
        }
        else if (strncmp(args[i], "SendSize=", sizeof("SendSize=") - 1) == 0)
        {
            SendSize = atoi(args[i] + sizeof("SendSize=") - 1);
        }
        else if (strcmp(args[i], "-c") == 0)
        {
            isClient = true;
            setIsClient = true;
        }
        else if (strcmp(args[i], "-s") == 0)
        {
            isClient = false;
            setIsClient = true;
        }
        else
        {
            NN_LOG("Warning! Unexpected Command Line Argument: %s\n", args[i]);
        }
    }
    if (!setIsClient)
    {
        NN_LOG("Warning! Client/Server choice was not made (-c or -s). Defaulting to server.\n");
    }

    if (isClient && ServerAddress == NULL)
    {
        NN_LOG("Warning! ServerAddress not set for client (ServerAddress=<Address string>).  Defaulting to loopback.\n");
        ServerAddress = "192.168.0.12";
    }

    if (isClient && !setPort)
    {
        NN_LOG("Warning! ServerPort not set for client (ServerPort=<Port Number>).  Defaulting to 8110.\n");
        ServerPort = 8110;
    }

}

void InterfaceManagementThread(void * args)
{
    ThreadManager::ThreadStatus currentStatus;
    nn::Result rval;
    rval = nn::nifm::Initialize();
    if (rval.IsFailure())
    {
        NN_LOG("Failed to Initialize NIFM!\n");
        return;
    }
    rval = nn::nifm::SetExclusiveClient(nn::nifm::GetClientId());
    if (rval.IsFailure())
    {
        NN_LOG("Failed to SetExclusiveClient on NIFM!\n");
        return;
    }

    do
    {
        g_pNifmConnection = new nn::nifm::NetworkConnection;
        if (nullptr == g_pNifmConnection)
        {
            NN_LOG("Failed to allocate nn::nifm::NetworkConnection!\n");
            break;
        }
        nn::nifm::RequestHandle requestHandle = g_pNifmConnection->GetRequestHandle();
        nn::nifm::SetRequestConnectionConfirmationOption(requestHandle, nn::nifm::ConnectionConfirmationOption_Prohibited);
        nn::nifm::SetRequestPersistent(requestHandle, true);

        g_pNifmConnection->SubmitRequest();

        while (!g_pNifmConnection->IsAvailable())
        {
            NN_LOG("Waiting for network interface...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        // NIFM: Network Interface is Online
        NN_LOG("====================================\n");
        NN_LOG("NIFM: Network ===>  O N L I N E <===\n");
        NN_LOG("====================================\n");



        while (g_pNifmConnection->IsAvailable()
            && g_threadManager->GetCurrentThreadStatus(&currentStatus) != -1
            && currentStatus == ThreadManager::ThreadStatus_Running)
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        g_pNifmConnection->CancelRequest();
        delete(g_pNifmConnection);

        // NIFM: Network Interface is Offline
        NN_LOG("====================================\n");
        NN_LOG("NIFM: Network ===>  D O W N     <===\n");
        NN_LOG("====================================\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
        NN_LOG("Submitting new request!\n");
    }
    while (g_threadManager->GetCurrentThreadStatus(&currentStatus) != -1 && currentStatus == ThreadManager::ThreadStatus_Running);
    NN_LOG("Exiting Management Thread...\n");
}
/*extern "C" void nninitStartup()
{
}

extern "C" void nndiagStartup()
{
}*/

extern "C" void nnMain()
{
    int argCount = nn::os::GetHostArgc();
    char ** args = nn::os::GetHostArgv();
    nn::os::ThreadType * thread;
    nn::os::ThreadType * imThread;
    ParseArgs(argCount - 1, &args[1]);

    g_threadManager = new (g_threadManagerMem) ThreadManager;
    if (!g_threadManager)
    {
        NN_LOG("Failed to construct Thread Manager!\n");
        return;
    }

    nn::Result rval = nn::socket::Initialize(g_socketConfig);
    if (rval.IsFailure())
    {
        NN_LOG("Failed to Initialize Socket Layer!\n");
        goto out;
    }

    imThread = g_threadManager->CreateThread(InterfaceManagementThread, NULL, nn::os::HighestThreadPriority);

    g_threadManager->StartAllThreads();

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    if (isClient)
    {
        ClientArgs args = { 0 };
        thread = g_threadManager->CreateThread(ClientThread,&args);
    }
    else
    {
        ServerArgs args = { 0 };
        thread = g_threadManager->CreateThread(ServerThread, &args);
    }

    g_threadManager->StartAllThreads();

    g_threadManager->WaitThreadWithTimeout(thread, nn::TimeSpan::FromSeconds(1200));
    g_threadManager->DestroyThread(thread);
    g_threadManager->DestroyThread(imThread);
    out:
    g_threadManager->~ThreadManager();
    return;
}
