﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy all Threads
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Shutdown_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Shutdown_Various_Tests)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, outAddr = { 0 };
    int                     tcpSock = -1, udpSock = -1, tcpServer = -1, udpServer = -1, acceptedSock = -1, rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx;
    bool                    isSuccess = true;
    char                    buf1[1024] = { 0 };

    const char * msgs[] = {
    "The rain in Spain ",
    "Flows ",
    "mainly on the plain."
    };

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////
    NN_LOG( "Testing descriptor:" );

    for ( idx = -2; idx < 40; idx++ )
    {
        NN_LOG( "%d, ", idx );
        if ( idx == g_echoServer->serverUDP || idx == g_echoServer->serverTCP )
        {
            continue;
        }

        rc = nn::socket::Shutdown( idx, nn::socket::ShutdownMethod::Shut_RdWr );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nShutdown() bad fds: %d should have failed with nn::socket::Errno::EBadf. Errno: %d\n", idx, myError );
                isSuccess = false;
            }
        }
        else
        {
            NN_LOG( "\nShutdown() succeeded but fds: %d. Expected to fail.\n", idx );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing nn::socket::Errno::EBadf Failed\n" );
    isSuccess = true;

    ////////////
    // Test incorrect how parameter
    ///////////
    NN_LOG( "Testing incorect how parameter: " );

    // Make TCP connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    for ( idx = -2; idx < 5; idx++ )
    {
        // Skip correct how parameters
        if ( idx >= static_cast<int>(nn::socket::ShutdownMethod::Shut_Rd) && idx <= static_cast<int>(nn::socket::ShutdownMethod::Shut_RdWr) )
        {
            continue;
        }

        NN_LOG( "%d, ", idx );

        rc = nn::socket::Shutdown( tcpSock, static_cast<nn::socket::ShutdownMethod>(idx) );
        if ( rc == 0 )
        {
            isSuccess = false;
            NN_LOG( "\nShutdown() Succeeded but expected to Fail. \n" );
        }
        else
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EInval )
            {
                isSuccess = false;
                NN_LOG( "\nShutdown() Expected Errno nn::socket::Errno::EInval. Errno=<%d>", myError );
            }
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing incorrect how parameters Failed!\n" );
    isSuccess = true;

    // Close TCP socket
    nn::socket::Close( tcpSock );
    tcpSock = -1;

    ////////////
    // Test correct how parameters
    ///////////
    NN_LOG( "Testing correct how parameters: " );
    for ( idx = static_cast<int>(nn::socket::ShutdownMethod::Shut_Rd); idx <= static_cast<int>(nn::socket::ShutdownMethod::Shut_RdWr); idx++ )
    {
        NN_LOG( "%d, ", idx );
        tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
        ERROR_IF( tcpSock < 0, "Socket() Failed! Errrno=<%d>\n", nn::socket::GetLastError() );

        rc = nn::socket::Shutdown( tcpSock, static_cast<nn::socket::ShutdownMethod>(idx) );
        ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }
    NN_LOG( "\n" );

    ////////////
    // Test nn::socket::Errno::ENotConn - Does not fail on NX and FreeBSD
    ///////////
    NN_LOG( "Testing nn::socket::Errno::ENotConn\n");

    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Socket() Failed! Errno=<%d>\n", nn::socket::GetLastError() );

    rc = nn::socket::Shutdown( tcpSock, nn::socket::ShutdownMethod::Shut_Wr );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    nn::socket::Close( tcpSock );
    tcpSock = -1;

    ////////////
    // Test nn::socket::Errno::EOpNotSupp - IpProto_Sctp not supported
    ///////////
    NN_LOG( "Testing nn::socket::Errno::EOpNotSupp\n" );

    // Create DGRAM socket with SCTP protocol to get nn::socket::Errno::EOpNotSupp but NX does not support SCTP protcol
    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, static_cast<nn::socket::Protocol>(nn::socket::ProtocolPrivate::IpProto_Sctp) );
    ERROR_IF_AND_COUNT( udpSock != -1, "Socket() Succeed but expected to Fail.\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EProtoNoSupport, "Socket() expected Errno nn::socket::Errno::EProtoNoSupport. Errno=<%d>\n", myError );

    ////////////
    // Shutdown RD - TCP
    ///////////
    NN_LOG( "Testing Shutdown RD - TCP\n" );

    // Make TCP server
    tcpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServer < 0, "Failed calling Socket for TCP Server Socket. Errno=<%d>\n", nn::socket::GetLastError() );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 8051 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!\n" );

    rc = nn::socket::Bind( tcpServer, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpServer, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Make TCP connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8051 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( tcpServer, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Accept() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server Write a message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Shutdown reading from client Socket
    rc = nn::socket::Shutdown( tcpSock, nn::socket::ShutdownMethod::Shut_Rd );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( acceptedSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Server Write a message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Read() Failed! Expected to Succeed with size 0. Size=<%d>\n", rc );

    // Close Sockets
    nn::socket::Close( tcpSock );
    tcpSock = -1;
    nn::socket::Close( tcpServer );
    tcpServer = -1;
    nn::socket::Close( acceptedSock );
    acceptedSock = -1;

    ////////////
    // Shutdown RD - UDP
    ///////////
    NN_LOG( "Testing Shutdown RD - UDP\n" );

    // Make UDP server
    udpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServer < 0, "Failed calling Socket for UDP Server Socket. Errno=<%d>\n", nn::socket::GetLastError() );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 8050 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!\n" );

    rc = nn::socket::Bind( udpServer, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server: Target Address
    memset( &outAddr, 0, sizeof(outAddr) );
    outAddr.sin_port        = nn::socket::InetHtons( 8051 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate TARGET IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server Connect:
    rc = nn::socket::Connect( udpServer, (nn::socket::SockAddr *) &outAddr, sizeof( outAddr ) );
    ERROR_IF( rc < 0, "Failed to connect UDP Server Socket to UDP socket! Errno=<%d>\n", nn::socket::GetLastError() );

    // Make UDP connection
    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8051, "127.0.0.1", 8050 );
    ERROR_IF( udpSock < -1, "Failed calling MakeUDPConnection!  Errno=<%d>\n", nn::socket::GetLastError() );

    // Server Write a message
    rc = nn::socket::Write( udpServer, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Shutdown reading from client Socket
    rc = nn::socket::Shutdown( udpSock, nn::socket::ShutdownMethod::Shut_Rd );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( udpServer, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Server Write a message
    rc = nn::socket::Write( udpServer, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Read() Failed! Expected to Succeed with size 0. Size=<%d>\n", rc );

    // Close Sockets
    nn::socket::Close( udpSock );
    udpSock = -1;
    nn::socket::Close( udpServer );
    udpServer = -1;

    ////////////
    // Shutdown WR - TCP
    ///////////
    NN_LOG( "Testing Shutdown WR - TCP\n" );

    // Make TCP server
    tcpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServer < 0, "Failed calling Socket for TCP Server Socket. Errno=<%d>\n", nn::socket::GetLastError() );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 8051 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!\n" );

    rc = nn::socket::Bind( tcpServer, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpServer, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Make TCP connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8051 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( tcpServer, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Accept() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( acceptedSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Shutdown writing from client Socket
    rc = nn::socket::Shutdown( tcpSock, nn::socket::ShutdownMethod::Shut_Wr );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server Write a message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Client Write a message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF_AND_COUNT( rc > -1, "Write() Succeeded! Expected to Fail\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EPipe, "Write() Errno expected nn::socket::Errno::EPipe. Actual Errno=<%d>\n", myError );


    // Close Sockets
    nn::socket::Close( tcpSock );
    tcpSock = -1;
    nn::socket::Close( tcpServer );
    tcpServer = -1;
    nn::socket::Close( acceptedSock );
    acceptedSock = -1;

    ////////////
    // Shutdown WR - UDP
    ///////////
    NN_LOG( "Testing Shutdown WR - UDP\n" );

    // Make UDP server
    udpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServer < 0, "Failed calling Socket for UDP Server Socket. Errno=<%d>\n", nn::socket::GetLastError() );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 8050 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!\n" );

    rc = nn::socket::Bind( udpServer, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server: Target Address
    memset( &outAddr, 0, sizeof(outAddr) );
    outAddr.sin_port        = nn::socket::InetHtons( 8051 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate TARGET IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server Connect:
    rc = nn::socket::Connect( udpServer, (nn::socket::SockAddr *) &outAddr, sizeof( outAddr ) );
    ERROR_IF( rc < 0, "Failed to connect UDP Server Socket to UDP socket! Errno=<%d>\n", nn::socket::GetLastError() );

    // Make UDP connection
    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8051, "127.0.0.1", 8050 );
    ERROR_IF( udpSock < -1, "Failed calling MakeUDPConnection!  Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( udpServer, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Shutdown reading from client Socket
    rc = nn::socket::Shutdown( udpSock, nn::socket::ShutdownMethod::Shut_Wr );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server Write a message
    rc = nn::socket::Write( udpServer, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Client Write a message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF_AND_COUNT( rc > -1, "Write() Succeeded! Expected to Fail\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EPipe, "Write() Errno expected nn::socket::Errno::EPipe. Actual Errno=<%d>\n", myError );

    // Close Sockets
    nn::socket::Close( udpSock );
    udpSock = -1;
    nn::socket::Close( udpServer );
    udpServer = -1;

    ////////////
    // Shutdown RDWR - TCP
    ///////////
    NN_LOG( "Testing Shutdown RDWR - TCP\n" );

    // Make TCP server
    tcpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServer < 0, "Failed calling Socket for TCP Server Socket. Errno=<%d>\n", nn::socket::GetLastError() );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 8051 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!\n" );

    rc = nn::socket::Bind( tcpServer, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpServer, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Make TCP connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8051 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( tcpServer, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Accept() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( acceptedSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Server Write a message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Shutdown writing from client Socket
    rc = nn::socket::Shutdown( tcpSock, nn::socket::ShutdownMethod::Shut_RdWr );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server Write a message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Read() Failed! Expected size 0. Actual=<%d>\n", rc );

    // Client Write a message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF_AND_COUNT( rc > -1, "Write() Succeeded! Expected to Fail\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EPipe, "Write() Errno expected nn::socket::Errno::EPipe. Actual Errno=<%d>\n", myError );


    // Close Sockets
    nn::socket::Close( tcpSock );
    tcpSock = -1;
    nn::socket::Close( tcpServer );
    tcpServer = -1;
    nn::socket::Close( acceptedSock );
    acceptedSock = -1;

    ////////////
    // Shutdown RDWR - UDP
    ///////////
    NN_LOG( "Testing Shutdown RDWR - UDP\n" );

    // Make UDP server
    udpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServer < 0, "Failed calling Socket for UDP Server Socket. Errno=<%d>\n", nn::socket::GetLastError() );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 8050 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!\n" );

    rc = nn::socket::Bind( udpServer, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server: Target Address
    memset( &outAddr, 0, sizeof(outAddr) );
    outAddr.sin_port        = nn::socket::InetHtons( 8051 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate TARGET IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server Connect:
    rc = nn::socket::Connect( udpServer, (nn::socket::SockAddr *) &outAddr, sizeof( outAddr ) );
    ERROR_IF( rc < 0, "Failed to connect UDP Server Socket to UDP socket! Errno=<%d>\n", nn::socket::GetLastError() );

    // Make UDP connection
    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8051, "127.0.0.1", 8050 );
    ERROR_IF( udpSock < -1, "Failed calling MakeUDPConnection!  Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( udpServer, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Server Write a message
    rc = nn::socket::Write( udpServer, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Shutdown reading from client Socket
    rc = nn::socket::Shutdown( udpSock, nn::socket::ShutdownMethod::Shut_RdWr );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server Write a message
    rc = nn::socket::Write( udpServer, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Read() Failed! Expected size 0. Actual=<%d>\n", rc );

    // Client Write a message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF_AND_COUNT( rc > -1, "Write() Succeeded! Expected to Fail\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EPipe, "Write() Errno expected nn::socket::Errno::EPipe. Actual Errno=<%d>\n", myError );

    // Close Sockets
    nn::socket::Close( udpSock );
    udpSock = -1;
    nn::socket::Close( udpServer );
    udpServer = -1;

    ////////////
    // Sequential Shutdown - TCP
    ///////////
    NN_LOG( "Testing Sequential Shutdown - TCP\n" );

    // Make TCP server
    tcpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServer < 0, "Failed calling Socket for TCP Server Socket. Errno=<%d>\n", nn::socket::GetLastError() );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 8051 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!\n" );

    rc = nn::socket::Bind( tcpServer, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    rc = nn::socket::Listen( tcpServer, 5 );
    ERROR_IF( rc < 0, "Listen() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Make TCP connection
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8051 );
    ERROR_IF( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( tcpServer, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Accept() failed for TCP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( acceptedSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Server Write a message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Shutdown writing from client Socket
    rc = nn::socket::Shutdown( tcpSock, nn::socket::ShutdownMethod::Shut_Rd );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( acceptedSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Server Write a message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Read() Failed! Expected to Succeed with size 0. Size=<%d>\n", rc );

    // Shutdown writing from client Socket
    rc = nn::socket::Shutdown( tcpSock, nn::socket::ShutdownMethod::Shut_Wr );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server Write a message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Read() Failed! Expected size 0. Actual=<%d>\n", rc );

    // Client Write a message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF_AND_COUNT( rc > -1, "Write() Succeeded! Expected to Fail\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EPipe, "Write() Errno expected nn::socket::Errno::EPipe. Actual Errno=<%d>\n", myError );

    // Close Sockets
    nn::socket::Close( tcpSock );
    tcpSock = -1;
    nn::socket::Close( tcpServer );
    tcpServer = -1;
    nn::socket::Close( acceptedSock );
    acceptedSock = -1;

    ////////////
    // Sequential Shutdown - UDP
    ///////////
    NN_LOG( "Testing Sequential Shutdown - UDP\n" );

    // Make UDP server
    udpServer = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServer < 0, "Failed calling Socket for UDP Server Socket. Errno=<%d>\n", nn::socket::GetLastError() );

    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port      = nn::socket::InetHtons( 8050 );
    localAddr.sin_family    = nn::socket::Family::Af_Inet;

    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!\n" );

    rc = nn::socket::Bind( udpServer, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() failed for UDP Server socket. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server: Target Address
    memset( &outAddr, 0, sizeof(outAddr) );
    outAddr.sin_port        = nn::socket::InetHtons( 8051 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate TARGET IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server Connect:
    rc = nn::socket::Connect( udpServer, (nn::socket::SockAddr *) &outAddr, sizeof( outAddr ) );
    ERROR_IF( rc < 0, "Failed to connect UDP Server Socket to UDP socket! Errno=<%d>\n", nn::socket::GetLastError() );

    // Make UDP connection
    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 8051, "127.0.0.1", 8050 );
    ERROR_IF( udpSock < -1, "Failed calling MakeUDPConnection!  Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( udpServer, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Server Write a message
    rc = nn::socket::Write( udpServer, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Shutdown reading from client Socket
    rc = nn::socket::Shutdown( udpSock, nn::socket::ShutdownMethod::Shut_Rd );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Client Write a message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Server Read a message
    rc = nn::socket::Read( udpServer, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Read() Failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );

    // Server Write a message
    rc = nn::socket::Write( udpServer, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Read() Failed! Expected to Succeed with size 0. Size=<%d>\n", rc );

    // Shutdown reading from client Socket
    rc = nn::socket::Shutdown( udpSock, nn::socket::ShutdownMethod::Shut_Wr );
    ERROR_IF_AND_COUNT( rc < 0, "Shutdown() Failed but expected to Succeed. Errno=<%d>\n", nn::socket::GetLastError() );

    // Server Write a message
    rc = nn::socket::Write( udpServer, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != strlen( msgs[0] ), "Write() failed! Expected size %d. Actual=<%d>\n", strlen( msgs[0] ), rc );

    // Client Read a message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Read() Failed! Expected size 0. Actual=<%d>\n", rc );

    // Client Write a message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF_AND_COUNT( rc > -1, "Write() Succeeded! Expected to Fail\n" );
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EPipe, "Write() Errno expected nn::socket::Errno::EPipe. Actual Errno=<%d>\n", myError );

    // Close Sockets
    nn::socket::Close( udpSock );
    udpSock = -1;
    nn::socket::Close( udpServer );
    udpServer = -1;

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( tcpServer > -1 )
    {
        nn::socket::Close( tcpServer );
        tcpServer = -1;
    }

    if ( udpServer > -1 )
    {
        nn::socket::Close( udpServer );
        udpServer = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }
}// NOLINT(impl/function_size)


TEST(ApiUnit,Shutdown_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,Shutdown_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
