﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//////////////////////////////////////////////////////////////
//
// I M P O R T A N T:  This testing is *ONLY* for NX.  These tests do not work on Windows
//
//////////////////////////////////////////////////////////////

#ifndef NN_BUILD_CONFIG_OS_WIN32

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;

static bool                 g_workBool = false;
static bool                 g_shouldWrite = false;
static bool                 g_shouldRead  = false;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy all Threads
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}


static void
PollPrintFlags( nn::socket::PollEvent event )
{
    NN_LOG( "POLL flags: [" );
    if ( (event & nn::socket::PollEvent::PollIn) != nn::socket::PollEvent::PollNone )     NN_LOG( "nn::socket::PollEvent::PollIn | ");
    if ( (event & nn::socket::PollEvent::PollRdNorm) != nn::socket::PollEvent::PollNone ) NN_LOG( "nn::socket::PollEvent::PollRdNorm | ");
    if ( (event & nn::socket::PollEvent::PollRdBand) != nn::socket::PollEvent::PollNone ) NN_LOG( "nn::socket::PollEvent::PollRdBand | ");
    if ( (event & nn::socket::PollEvent::PollPri) != nn::socket::PollEvent::PollNone )    NN_LOG( "nn::socket::PollEvent::PollPri | ");
    if ( (event & nn::socket::PollEvent::PollOut) != nn::socket::PollEvent::PollNone )    NN_LOG( "nn::socket::PollEvent::PollOut | ");
    if ( (event & nn::socket::PollEvent::PollWrNorm) != nn::socket::PollEvent::PollNone)  NN_LOG( "nn::socket::PollEvent::PollWrNorm | ");
    if ( (event & nn::socket::PollEvent::PollWrBand) != nn::socket::PollEvent::PollNone)  NN_LOG( "nn::socket::PollEvent::PollWrBand | ");
    if ( (event & nn::socket::PollEvent::PollErr) != nn::socket::PollEvent::PollNone)     NN_LOG( "nn::socket::PollEvent::PollErr | ");
    if ( (event & nn::socket::PollEvent::PollHup) != nn::socket::PollEvent::PollNone)     NN_LOG( "nn::socket::PollEvent::PollHup | ");
    if ( (event & nn::socket::PollEvent::PollNVal) != nn::socket::PollEvent::PollNone)    NN_LOG( "nn::socket::PollEvent::PollNVal | ");
    NN_LOG( "]\n" );
}


////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Poll_Initialize)
{
    InitializeTesting();
}

static void
PollUnblockReadThread( void * inArg )
{
    int * sockfds = (int *) inArg;
    int rc = 0;

    NN_LOG( "(Thread) - Start, Sleeping...\n" );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    NN_LOG( "(Thread) - Wakup! Writing 1 bytes\n" );

    rc = nn::socket::Write( *sockfds, "1", 1 );
    if ( rc != 1 )
    {
        NN_LOG( "##### FAILED - Return Code from Write is: %d\n", rc );
    }

    g_workBool = true;

    NN_LOG( "(Thread) - Exit\n" );
}


TEST(ApiUnit,Poll_Block_For_Infinte_Time_Test)
{
    nn::socket::PollFd  myPollFds[1];
    int                 clifds = -1, svrfds = -1, rc = 0 , thisThread = -1;
    bool                isSuccess = true;

    ////////////
    // Create UDP Client / Server connection
    ///////////

    clifds = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000,  "127.0.0.1", 9001 );
    ERROR_IF( clifds < 0, "MakeUDPConnection failed for client, errno: %d\n", nn::socket::GetLastError() );

    svrfds = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9001,  "127.0.0.1", 9000 );
    ERROR_IF( clifds < 0, "MakeUDPConnection failed for server, errno: %d\n", nn::socket::GetLastError() );


    ////////////
    // Create a thread to run the "Unblock" routine
    ////////////

    g_workBool = false;
    thisThread = g_pThreadedTest->CreateThread( &PollUnblockReadThread, (void *) &svrfds, kThreadPriority );
    if ( thisThread < 0 )
    {
        NN_LOG( "Failed to create and start Unblock (thread)!\n" );
        goto out;
    }

    ////////////
    // Test (Read) Unblocking Select
    ////////////

    myPollFds[0].fd      = clifds;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PollPrintFlags( myPollFds[0].events );

    NN_LOG( "Calling Poll with INFTIM (Infinite Timeout) - BLOCKING..\n" );
    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, 1, -1  ) );  // -1 == INFTIM (Infinite Time)
    NN_LOG( "Poll Unblocked - PASS\n" );

    if ( g_workBool == false )
    {
        NN_LOG( "===> Poll() unblocked - but Poll wasn't unblocked by thread!!  This was unexpected\n" );
        nTestsFailing++;
        goto out;
    }

    PollPrintFlags( myPollFds[0].revents );

    if ( (myPollFds[0].revents & nn::socket::PollEvent::PollIn) != nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "================\n");
        NN_LOG( "== Pass - Poll was unblocked by Write/nn::socket::PollEvent::PollIn\n" );
        NN_LOG( "================\n");
        nTestsPassing++;
    }

    // Cleanup test - FALL THRU

out:

    if ( thisThread > -1 )
    {
        // Wait DNS Server threads to die
        g_pThreadedTest->WaitThread( thisThread );

        // Destroy all Threads
        g_pThreadedTest->DestroyThread( thisThread );

        // Clear Thread
        thisThread = -1;
    }

    if ( clifds > -1 )
    {
        nn::socket::Close( clifds );
        clifds = -1;
    }

    if ( svrfds > -1 )
    {
        nn::socket::Close( svrfds );
        svrfds = -1;
    }

    return;
}


TEST(ApiUnit,Poll_Various_Tests)
{
    nn::socket::PollFd  myPollFds[1];
    int            tcpSock = -1, udpSock = -1, udpServSock = -1, listenSock = -1, acceptSock =-1, rc = 0;
    int            optval = -1;
    nn::socket::SockLenT      optlen = -1;
    bool           isSuccess = true, rval;
    uint64_t       unused1 = 0, unused2 = 0;
    unsigned char  buf1[128];

    memset( buf1, 0, sizeof( buf1 ) );

    ////////////
    // Make TCP Server
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9020, "127.0.0.1" );
    ERROR_IF_AND_COUNT( rval == false, "Failed calling MakeiListenSocket!  Errno:%d\n", nn::socket::GetLastError() );

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9020 );
    ERROR_IF_AND_COUNT( tcpSock < -1, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    acceptSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptSock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );


    ////////////
    // Make UDP Connection
    ///////////

    udpServSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 9001 );
    ERROR_IF_AND_COUNT( udpSock < -1, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9001, "127.0.0.1", 9000 );
    ERROR_IF_AND_COUNT( udpSock < -1, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );


    ////////////
    // Call POLL with a NULL decriptor ARRAY
    ///////////

    NN_LOG( "Call POLL with a NULL decriptor ARRAY\n" );

    PRINT_AND_CALL(rc = nn::socket::Poll( static_cast<nn::socket::PollFd*>(nullptr), 0, 0 ) );
    ERROR_IF_AND_COUNT( rc != 0, "Poll() returned: %d for Null Array of Fds - Errno: %d\n", rc, nn::socket::GetLastError() );


    ////////////
    // TCP: POLL FDs Array has Negative Length
    ///////////

    NN_LOG( "[TCP] - Poll FDS Array Length is (-1)\n" );

    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, -1, 0 ) );
    ERROR_IF_AND_COUNT( rc > -1 || nn::socket::GetLastError() != nn::socket::Errno::EFault,
                       "Poll() returned OK for -1 Array of FDs legnth!" );

    ////////////
    // UDP: POLL FDs Array has Negative Length
    ///////////

    NN_LOG( "[UDP] - Poll FDS Array Length is (-1)\n" );

    myPollFds[0].fd      = udpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, -1, 0 ) );
    ERROR_IF_AND_COUNT( rc > -1 || nn::socket::GetLastError() != nn::socket::Errno::EFault,
                       "Poll() returned OK for -1 Array of FDs legnth!" );


    ////////////
    // Place Normal data on TCP and UDP sockets
    ///////////

    rc = NATF::API::COMMON::WriteData( acceptSock, (unsigned char *) "Hello!", 6, unused1, unused2 );
    ERROR_IF_AND_COUNT( rc != 6, "Failed calling WriteData!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    rc = NATF::API::COMMON::WriteData( udpServSock, (unsigned char *) "Hello!", 6, unused1, unused2 );
    ERROR_IF_AND_COUNT( rc != 6, "Failed calling WriteData! rc: %d,  Errno:%d\n", rc, nn::socket::GetLastError() );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));


    ///////////
    // TCP: nn::socket::PollEvent::PollIn + nn::socket::PollEvent::PollRdNorm
    /////////

    NN_LOG( "[TCP] - Test nn::socket::PollEvent::PollIn / nn::socket::PollEvent::PollRdNorm\n" );

    // TCP nn::socket::PollEvent::PollIn
    memset( myPollFds, 0, sizeof( myPollFds) );
    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PollPrintFlags( myPollFds[0].events );
    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, 1, 0 ) );
    PollPrintFlags( myPollFds[0].revents );

    ERROR_IF_AND_COUNT( rc != 1, "Poll Return code %d did not equal 1 descriptor!\n", rc );
    ERROR_IF_AND_COUNT(  ( myPollFds[0].revents & nn::socket::PollEvent::PollIn ) == nn::socket::PollEvent::PollNone || ( myPollFds[0].revents & nn::socket::PollEvent::PollRdNorm ) == nn::socket::PollEvent::PollNone,
                        "Poll failed - I expected nn::socket::PollEvent::PollIn and nn::socket::PollEvent::PollRdNorm but got %d\n", myPollFds[0].revents );

    NN_LOG( "PASSED!\n" );

    ///////////
    // UDP: nn::socket::PollEvent::PollIn + nn::socket::PollEvent::PollRdNorm
    /////////

    NN_LOG( "[UDP] - Test nn::socket::PollEvent::PollIn / nn::socket::PollEvent::PollRdNorm\n" );

    // UDP nn::socket::PollEvent::PollIn
    memset( myPollFds, 0, sizeof( myPollFds) );
    myPollFds[0].fd      = udpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollNVal | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollErr;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PollPrintFlags( myPollFds[0].events );
    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, 1, 0 ) );
    PollPrintFlags( myPollFds[0].revents );

    ERROR_IF_AND_COUNT( rc != 1, "Poll Return code %d did not equal 1 descriptor (ready-to-Read)!\n", rc );
    ERROR_IF_AND_COUNT(  (myPollFds[0].revents & nn::socket::PollEvent::PollIn) == nn::socket::PollEvent::PollNone,
                        "Poll failed - I expected either nn::socket::PollEvent::PollIn and nn::socket::PollEvent::PollRdNorm but got %d\n", myPollFds[0].revents );

    NN_LOG( "PASSED!\n" );

    ///////////
    // Test UDP and TCP at same time
    /////////

    {
        nn::socket::PollFd myPoll[2];

        memset( myPoll, 0, sizeof( myPoll) );

        // TCP nn::socket::PollEvent::PollIn
        myPoll[0].fd      = tcpSock;
        myPoll[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollNVal | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollErr;
        myPoll[0].revents = nn::socket::PollEvent::PollNone;

        // UDP nn::socket::PollEvent::PollIn
        myPoll[1].fd      = udpSock;
        myPoll[1].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollNVal | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollErr;
        myPoll[1].revents = nn::socket::PollEvent::PollNone;

        NN_LOG( "[TCP]: " );
        PollPrintFlags( myPoll[0].events );
        NN_LOG( "[UDP]: " );
        PollPrintFlags( myPoll[1].events );

        PRINT_AND_CALL(rc = nn::socket::Poll( myPoll, 2, 0 ) );

        NN_LOG( "[TCP]: " );
        PollPrintFlags( myPoll[0].revents );
        NN_LOG( "[UDP]: " );
        PollPrintFlags( myPoll[1].revents );

        ERROR_IF_AND_COUNT( rc != 2, "Poll Return code %d did not equal 2 descriptors!\n", rc );
        ERROR_IF_AND_COUNT(  ( myPoll[0].revents & nn::socket::PollEvent::PollIn ) == nn::socket::PollEvent::PollNone || ( myPoll[0].revents & nn::socket::PollEvent::PollRdNorm ) == nn::socket::PollEvent::PollNone,
                        "[TCP] Poll failed - I expected nn::socket::PollEvent::PollIn and nn::socket::PollEvent::PollRdNorm but got %d\n", myPoll[0].revents );

        ERROR_IF_AND_COUNT(  ( myPoll[1].revents & nn::socket::PollEvent::PollIn ) == nn::socket::PollEvent::PollNone || ( myPoll[1].revents & nn::socket::PollEvent::PollRdNorm ) == nn::socket::PollEvent::PollNone,
                        "[UDP] Poll failed - I expected nn::socket::PollEvent::PollIn and nn::socket::PollEvent::PollRdNorm but got %d\n", myPoll[1].revents );

    }

    ////////////
    // Test: Place Out-Of-Band (OOB) data onto TCP socket
    ///////////

    NN_LOG( "[TCP] - Test: Place Out-Of-Band (OOB) data onto TCP socket\n" );

    rc = nn::socket::Send( acceptSock, (unsigned char *) "X", 1, nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc != 1, "Failed calling SendTo with nn::socket::MsgFlag::Msg_Oob!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    rc = nn::socket::Send( acceptSock, (unsigned char *) "Good Bye!", 9, nn::socket::MsgFlag::Msg_None);
    ERROR_IF_AND_COUNT( rc != 9, "Failed calling WriteData!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    ///////////
    // TCP: nn::socket::PollEvent::PollIn + nn::socket::PollEvent::PollRdNorm + nn::socket::PollEvent::PollRdBand + nn::socket::PollEvent::PollPri
    /////////

    NN_LOG( "[UDP] - Test nn::socket::PollEvent::PollIn / nn::socket::PollEvent::PollRdNorm\n" );

    // UDP nn::socket::PollEvent::PollIn
    memset( myPollFds, 0, sizeof( myPollFds) );
    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollRdBand | nn::socket::PollEvent::PollPri | nn::socket::PollEvent::PollNVal | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollErr;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    PollPrintFlags( myPollFds[0].events );
    PRINT_AND_CALL(rc = nn::socket::Poll( myPollFds, 1, 0 ) );
    PollPrintFlags( myPollFds[0].revents );

    ERROR_IF_AND_COUNT( rc != 1, "Poll Return code %d did not equal 1 descriptor (ready-to-Read)!\n", rc );
    ERROR_IF_AND_COUNT(  (myPollFds[0].revents & nn::socket::PollEvent::PollRdBand ) == nn::socket::PollEvent::PollNone,
                        "Poll failed - I expected nn::socket::PollEvent::PollRdBand but got %d\n", myPollFds[0].revents );

    ///////////
    // TCP: Read Hello!
    /////////

    rc = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    NN_LOG( "(Part 1) - Normal Data: %.*s\n", rc, buf1 );
    ERROR_IF_AND_COUNT( rc != 6, "Failed to receive 'Hello!' data!  rc: %d\n", rc );

    ///////////
    // TCP: Read OOB data
    /////////

    rc = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc != 1, "Failed to receive OOB data!  rc: %d\n", rc );

    NN_LOG( "(Out of Band) - OOB Data: %.*s\n", rc, buf1 );

    ///////////
    // TCP: Read Good Bye!
    /////////

    rc = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    NN_LOG( "(Part 2) - Normal Data: %.*s\n", rc, buf1 );
    ERROR_IF_AND_COUNT( rc != 9, "Failed to receive 'Good Bye!' data!  rc: %d\n", rc );

WRAP_FAILING_TEST( "SIGLO-57670", "AT&T System 5 release 4 (svr4) port of POLL to FreeBSD/NX missing nn::socket::PollEvent::PollWrBand\n" )
{

    ////////////
    // Test: Ready-To-Write (OOB) data onto TCP socket
    ///////////

    NN_LOG( "[TCP] - Test: Ready-To-Write Out-Of-Band (OOB) data onto TCP socket\n" );

    // Set up Poll
    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollWrBand | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    // Invoke Poll
    PollPrintFlags(  myPollFds[0].events );
    rc = nn::socket::Poll( myPollFds, 1, 0 );
    PollPrintFlags(  myPollFds[0].revents );
    ERROR_IF_AND_COUNT( rc < 1, "Poll Failed!  rc: %d", rc);

    if (  ( myPollFds[0].revents & nn::socket::PollEvent::PollWrBand ) == nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "Poll did not return nn::socket::PollEvent::PollWrBand - but I requeted it!  Errno: %d\n", nn::socket::GetLastError() );
        goto out;
    }

} // Wrap Failing Test

    ////////////
    // Test: (OOB) data is INLINE onto TCP socket
    ////////////

    NN_LOG( "[TCP] - Test: Place Out-Of-Band (OOB) data INLINE (aka Normal Data) for TCP socket\n" );

    // Close TCP Socket
    nn::socket::Close( tcpSock );
    tcpSock = -1;

    nn::socket::Close( acceptSock );
    acceptSock = -1;

    ////////////
    // Open New TCP Socket with nn::socket::Option::So_OobInline
    ///////////
    optval = 1;
    optlen = sizeof( optval );
    rval = NATF::API::COMMON::SetSocketOptAndConnect( tcpSock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_OobInline, &optval, optlen,
                                                      0, g_echoServer->kInterfaceIPv4Addr,
                                                      9020, g_echoServer->kInterfaceIPv4Addr, false, 0 );   // isUDP = false

    ERROR_IF_AND_COUNT( rval == false, "Failed calling SetSocketOptAndConnect()!  Errno:%d\n", nn::socket::GetLastError() );

    acceptSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( acceptSock < 0, "Failed calling Accept!  Errno:%d\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );


    ////////////
    // Set OBBINLINE Data
    ///////////

    rc = nn::socket::Send( acceptSock, (unsigned char *) "Hello!", 6, nn::socket::MsgFlag::Msg_None );
    ERROR_IF_AND_COUNT( rc != 6, "Failed calling WriteData!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    rc = nn::socket::Send( acceptSock, (unsigned char *) "X", 1, nn::socket::MsgFlag::Msg_Oob );
    ERROR_IF_AND_COUNT( rc != 1, "Failed calling SendTo with nn::socket::MsgFlag::Msg_Oob!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    rc = nn::socket::Send( acceptSock, (unsigned char *) "Good Bye!", 9, nn::socket::MsgFlag::Msg_None);
    ERROR_IF_AND_COUNT( rc != 9, "Failed calling WriteData!  rc: %d, Errno:%d\n", rc, nn::socket::GetLastError() );

    ///////////
    // TCP: Read Hello!XGood Bye!
    /////////

    rc = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    NN_LOG( "Normal Data: %.*s\n", rc, buf1 );
    ERROR_IF_AND_COUNT( rc != 6, "Failed to receive 'Hello!' data!  rc: %d\n", rc );

    rc = nn::socket::Recv( tcpSock, buf1, sizeof( buf1 ), nn::socket::MsgFlag::Msg_None );
    NN_LOG( "Normal Data: %.*s\n", rc, buf1 );
    ERROR_IF_AND_COUNT( rc != 10, "Failed to receive 'XGood Bye!' data!  rc: %d\n", rc );


    ////////////
    // Test: Socket HANGHUP during I/O Operation
    ///////////

    // Shutdown client
    nn::socket::Shutdown( tcpSock, nn::socket::ShutdownMethod::Shut_RdWr );

    NN_LOG( "[TCP] - Test: Socket [HANGUP] return flag\n" );

    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollWrBand | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    // Invoke Poll
    PollPrintFlags(  myPollFds[0].events );
    rc = nn::socket::Poll( myPollFds, 1, 0 );
    PollPrintFlags(  myPollFds[0].revents );
    ERROR_IF_AND_COUNT( rc < 1, "Poll Failed!  rc: %d", rc);

    if (  ( myPollFds[0].revents & nn::socket::PollEvent::PollHup ) == nn::socket::PollEvent::PollNone )
    {
        NN_LOG( "FAILED!  ==> Poll did not return nn::socket::PollEvent::PollHup - but I requeted it!  Errno: %d\n", nn::socket::GetLastError() );
        goto out;
    }

    // TCP Socket is closed
    tcpSock = -1;


    ////////////
    // Test: Operation on closed socket
    ///////////

    NN_LOG( "[TCP] - Test: Closed Socket should not be able to READ or WRITE\n" );

    myPollFds[0].fd      = tcpSock;
    myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollRdNorm | nn::socket::PollEvent::PollRdBand | nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollWrBand | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
    myPollFds[0].revents = nn::socket::PollEvent::PollNone;

    // Invoke Poll
    PollPrintFlags(  myPollFds[0].events );
    rc = nn::socket::Poll( myPollFds, 1, 0 );
    PollPrintFlags(  myPollFds[0].revents );
    ERROR_IF_AND_COUNT( rc != 0, "Poll Success was expected, but Pollactually failed!  rc: %d", rc);

    if (  ( myPollFds[0].revents & nn::socket::PollEvent::PollIn ) != nn::socket::PollEvent::PollNone  || ( myPollFds[0].revents & nn::socket::PollEvent::PollOut ) != nn::socket::PollEvent::PollNone )

    {
        NN_LOG( "FAILED!  ==> Poll Indicated that a closed socket can (READ or WRITE), result: 0x%x\n",
                myPollFds[0].revents );
        goto out;
    }

    // Fall Thru to cleanup

out:

    if ( udpServSock > 0 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }

    if ( acceptSock > 0 )
    {
        nn::socket::Close( acceptSock );
        acceptSock = -1;
    }

    if ( listenSock > 0 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }


    if ( tcpSock > 0 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > 0 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

}  // NOLINT(impl/function_size)


static void
PollTimeout(int in_milliseconds, nn::socket::Type socktype )
{
    nn::socket::PollFd  myPollFds[1];
    int       rc = -1, sockfds = -1;
    uint64_t  start_usec = 0, stop_usec = 0, delta_usec = 0, timeout_usec;
    uint64_t  smoothedTimeout = 0;
    bool      isSuccess = true;

    if ( socktype == nn::socket::Type::Sock_Stream )
    {
        NN_LOG( "Testing with: [TCP][Timeout Milliseconds = %ld]\n", in_milliseconds );
        sockfds = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
        ERROR_IF_AND_COUNT( sockfds < 0, "Unable to create nn::socket::Type::Sock_Stream, rc: %d, errno: %d\n", rc, nn::socket::GetLastError() );
    }
    else
    {
        NN_LOG( "Testing with: [UDP][Timeout Milliseconds = %ld]\n", in_milliseconds );
        sockfds = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 9000, "127.0.0.1", 8053 );
        ERROR_IF_AND_COUNT(  sockfds < 0, "Unable to create nn::socket::Type::Sock_Dgram, rc: %d, errno: %d\n", rc, nn::socket::GetLastError() );
    }

    // Top level Test
    for( int idx = 0; idx < 3; idx++ )
    {
        ////////////
        // POLL FDs Array
        ///////////

        myPollFds[0].fd      = sockfds;
        myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
        myPollFds[0].revents = nn::socket::PollEvent::PollNone;

        // Start time - MicroSeconds
        start_usec = NATF::API::COMMON::GetMicroSeconds();

        // Call Poll - but expect Poll to timeout
        rc = nn::socket::Poll( myPollFds, 1, in_milliseconds );
        ERROR_IF_AND_COUNT( rc != 0 , "FAILED: poill() return cocde is <%ld>, but I expected it to be (ZERO == 0)!\n", rc );

        // End time - MicroSecodns
        stop_usec = NATF::API::COMMON::GetMicroSeconds();

        // Delta time
        delta_usec = stop_usec - start_usec;

        /// Convert Timeout (MilliSeconds) to (Microseconds)
        timeout_usec = in_milliseconds * 1000;

        // Display result
        NN_LOG( "Start_usec <%llu>, Stop_usec <%llu>, Elasped_usec <%llu>, Timeout_usec <%llu>\n",
                    start_usec, stop_usec, delta_usec, timeout_usec );

        ///////////
        // RESULTS:
        ///////////

        // We don't allow select() to timeout QUICKER/FASTER than the timeout
        ERROR_IF_AND_COUNT( delta_usec < timeout_usec,
                           "FAILED: Poll() timed out after waiting <%llu>usec, but timeout is set to <%llu>usec!\n",
                           delta_usec, timeout_usec );

       smoothedTimeout = timeout_usec + 3000000;

       // system call should unblock before 3s delta
       ERROR_IF_AND_COUNT( delta_usec > smoothedTimeout,
       "FAILED: Poll() timed out after waiting <%llu>usecs, but expected\n", delta_usec, smoothedTimeout );
    }

out:

    if ( sockfds > -1 )
    {
        nn::socket::Close(sockfds);
        sockfds = -1;
    }

};



TEST(ApiUnit,Poll_notimeout)
{
    PollTimeout( 0, nn::socket::Type::Sock_Stream );
    PollTimeout( 0, nn::socket::Type::Sock_Dgram );
}

#ifdef ENABLE_FAILING_TESTS

TEST(ApiUnit,Poll_no1usec)
{
    PollTimeout( 1, nn::socket::Type::Sock_Stream );
    PollTimeout( 1, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t5usecs)
{
    PollTimeout( 5, nn::socket::Type::Sock_Stream );
    PollTimeout( 5, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t10usecs)
{
    PollTimeout( 10, nn::socket::Type::Sock_Stream );
    PollTimeout( 10, nn::socket::Type::Sock_Dgram );

}

TEST(ApiUnit,Poll_t100usecs)
{
    PollTimeout( 100, nn::socket::Type::Sock_Stream );
    PollTimeout( 100, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t1s)
{
    PollTimeout( 1000, nn::socket::Type::Sock_Stream );
    PollTimeout( 1000, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t5s)
{
    PollTimeout( 5000, nn::socket::Type::Sock_Stream );
    PollTimeout( 5000, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t10s)
{
    PollTimeout( 10000, nn::socket::Type::Sock_Stream );
    PollTimeout( 10000, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t11s)
{
    PollTimeout( 10001, nn::socket::Type::Sock_Stream );
    PollTimeout( 10001, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t15s)
{
    PollTimeout( 15000, nn::socket::Type::Sock_Stream );
    PollTimeout( 15000, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t19999s)
{
    PollTimeout( 1999, nn::socket::Type::Sock_Stream );
    PollTimeout( 1999, nn::socket::Type::Sock_Dgram );
}

TEST(ApiUnit,Poll_t51s)
{
    PollTimeout( 5001, nn::socket::Type::Sock_Stream );
    PollTimeout( 5001, nn::socket::Type::Sock_Dgram );
}

#endif // If Failing tests Enabled



struct PollPayload
{
    uint8_t   type;
    int       threadId;
    nn::Bit64 desiredCore;
    int       sockfds;
    size_t    maxBufSize;
    uint64_t  bytesSent;
    uint64_t  writeWouldBlock;
    uint64_t  writeNotFull;
    int       writerDelay;
};


static void
PollWriter( void * in_payload ) NN_NOEXCEPT
{
    nn::socket::PollFd      myPollFds[1];
    size_t                  rc = 0;
    unsigned char *         workPtr = NULL;
    uint8_t                 startPos = 0;
    bool                    isSuccess = true;
    struct PollPayload *  payload = (PollPayload *) in_payload;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, payload->desiredCore );

    NN_LOG( "Writer Thread: %d, Core: %lld, Socket: %d, MaxBufSize: %d - Starting!\n",
            payload->threadId, payload->desiredCore, payload->sockfds, payload->maxBufSize );

    // Make Random Noise
    startPos = payload->sockfds;
    rc = NATF::API::COMMON::MakeRandomNoise( &workPtr, payload->maxBufSize, startPos );
    ERROR_IF( rc < 0, "MakeRandomNoise failed!  Errno: %d\n", nn::socket::GetLastError() );

    // Wait for start..
    while( g_workBool == false )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( 5 ));
    }

    // While writer should run
    while( g_shouldWrite == true )
    {
        // Poll for Writing
        myPollFds[0].fd      = payload->sockfds;
        myPollFds[0].events  = nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
        myPollFds[0].revents = nn::socket::PollEvent::PollNone;

        rc = nn::socket::Poll( myPollFds, 1, 1000 );
        if (rc == 0 )
        {
             nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
             continue;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( payload->writerDelay ));

        if ( (myPollFds[0].revents & (nn::socket::PollEvent::PollOut | nn::socket::PollEvent::PollWrNorm)) != nn::socket::PollEvent::PollNone )
        {
            rc = NATF::API::COMMON::WriteData( payload->sockfds, workPtr, payload->maxBufSize, payload->writeWouldBlock, payload->writeNotFull );
            ERROR_IF( rc < 0, "Write failed!  Errno: %d\n", nn::socket::GetLastError() );

            // Fall Thru
         }

         payload->bytesSent = payload->bytesSent + rc;
    }

out:

    NN_LOG( "Writer Thread: %d, Core: %lld, Socket: %d - Exiting!\n",
                      payload->threadId, payload->desiredCore, payload->sockfds );

    // Clear Random Result
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }

    return;
}


static void
PollReader( void * in_payload ) NN_NOEXCEPT
{
    nn::socket::PollFd      myPollFds[1];
    size_t                  rc = 0;
    struct PollPayload *    payload = (PollPayload *) in_payload;
    bool                    isSuccess = true;
    unsigned char *         workPtr;

    // Float this thread to the right CPU
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, payload->desiredCore );

    NN_LOG( "Reader Thread: %d, Core: %lld, Socket: %d - Starting!\n",
                      payload->threadId, payload->desiredCore, payload->sockfds );

    // Allocate Buffer Ptr;
    workPtr = (unsigned char *) malloc( payload->maxBufSize );
    ERROR_IF( workPtr == NULL, "Work Pointer failed to allocate 4096 bytes!" );

    // Wait for start..
    while( g_workBool == false )
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds( 5 ));
    }

    // While reader sahould run
    while( g_shouldRead == true )
    {
        // Poll for Writing
        myPollFds[0].fd      = payload->sockfds;
        myPollFds[0].events  = nn::socket::PollEvent::PollIn | nn::socket::PollEvent::PollErr | nn::socket::PollEvent::PollHup | nn::socket::PollEvent::PollNVal;
        myPollFds[0].revents = nn::socket::PollEvent::PollNone;

        rc = nn::socket::Poll( myPollFds, 1, 1000 );
        if ( rc == 0 )
        {
             nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
             continue;
        }

        if ( (myPollFds[0].revents & nn::socket::PollEvent::PollIn) != nn::socket::PollEvent::PollNone )
        {
            rc = NATF::API::COMMON::ReadData( payload->sockfds, workPtr, payload->maxBufSize, payload->writeWouldBlock );
            ERROR_IF( rc < 0, "Read failed!  Errno: %d\n", nn::socket::GetLastError() );

            payload->bytesSent = payload->bytesSent + rc;
        }
    }

out:

    NN_LOG( "Reader Thread: %d, Core: %lld, Socket: %d - Exiting!\n",
                      payload->threadId, payload->desiredCore, payload->sockfds );

    // Clear Random Result
    if ( workPtr != NULL )
    {
        free( workPtr );
        workPtr = NULL;
    }
}


static int
PollRead_And_Write_UDP( int numberOfPairs, int  in_delaytimems, const char *  in_interface ) NN_NOEXCEPT
{
    struct PollPayload   payload[15];

    int               idx = 0;
    bool              isSuccess = true;
    size_t            maxSize = -1;
    int               core = 0, result = -1;

    memset( (char *) &payload, 0, sizeof( payload ) );
    for ( idx = 0; idx < numberOfPairs; idx++ )
    {
        payload[idx].threadId = -1;
        payload[idx].sockfds  = -1;
    }

    NN_LOG( "====> Testing Number of sockets: %d\n", numberOfPairs );

    ////////////////
    // Start 2 pairs of UDP sockets
    ///////////////

    // Indicate that threads should run..
    g_shouldWrite = true;
    g_shouldRead  = true;
    g_workBool    = false;

    // Start Threads
    maxSize = 9000;
    core = 1;
    for( idx = 0; idx < numberOfPairs; idx = idx + 2 )
    {
        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        /////////////////
        // Create Reader - Thread
        /////////////////

        payload[idx].type        = 'R';
        payload[idx].desiredCore = core;
        payload[idx].maxBufSize  = maxSize;
        payload[idx].sockfds     = NATF::API::COMMON::MakeUDPConnection( in_interface, ( 9000 + idx),
                                                                         in_interface, ( 9001 + idx) );
        ERROR_IF( payload[idx].sockfds < 0, "MakeUDPConnection failed" );

        payload[idx].threadId = g_pThreadedTest->CreateThread( &PollReader, (void *) &payload[idx], kThreadPriority );
        if ( payload[idx].threadId < 0 )
        {
            NN_LOG( "Failed to create a Reader (thread)!\n" );
            goto out;
        }

        /////////////////
        // Createa Writer - Thread
        /////////////////

        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        payload[idx + 1].type        = 'W';
        payload[idx + 1].writerDelay = in_delaytimems;
        payload[idx + 1].desiredCore = core;
        payload[idx + 1].maxBufSize  = maxSize;
        payload[idx + 1].sockfds     = NATF::API::COMMON::MakeUDPConnection( in_interface, (9001 + idx ),
                                                                             in_interface, (9000 + idx )  );
        ERROR_IF( payload[idx + 1].sockfds < 0, "MakeUDPConnection failed" );

        payload[idx + 1].threadId = g_pThreadedTest->CreateThread( &PollWriter, (void *) &payload[idx + 1], kThreadPriority );
        if ( payload[idx + 1].threadId < 0 )
        {
            NN_LOG( "Failed to create a Writer (thread)!\n" );
            goto out;
        }
    }
    // Give threads time to start
    NN_LOG( "====> Give threads time to start...\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Test Time 60 seconds
    NN_LOG( "====> Test running for next 60 seconds..." );
    g_workBool    = true;

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
    NN_LOG( "Done!\n\n" );

    // Stop Writers
    NN_LOG( "Stop Writers!\n" );
    g_shouldWrite = false;
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Stop Readers
    NN_LOG( "Stop Readers!\n" );
    g_shouldRead = false;
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Wait for each thread death..
    for ( idx = 0; idx < numberOfPairs; idx++ )
    {
        // Wait for thread to die
        g_pThreadedTest->WaitThread( payload[idx].threadId );

        // Destroy Thread
        g_pThreadedTest->DestroyThread( payload[idx].threadId );

        // Close the Socket
        nn::socket::Close( payload[idx].sockfds );

        // If Writer Thread
        if (  payload[idx].type == 'W' )
        {
             NN_LOG(
             "Type: %c, Thread : %d, Core: %d, Socket: %d, MaxBufSize: %d, bytes Sent: %lu, wouldBlock: %d, writeNotFull: %d, writeDelay: %dms\n",
                  payload[idx].type, payload[idx].threadId, payload[idx].desiredCore, payload[idx].sockfds,
                  payload[idx].maxBufSize, payload[idx].bytesSent, payload[idx].writeWouldBlock, payload[idx].writeNotFull,
                  payload[idx].writerDelay );
             NN_LOG(
             "          BytesPerSecond: %d, BitsPerSecond: %d\n",
                         (payload[idx].bytesSent / 60), (payload[idx].bytesSent * 8 / 60 ) );
        }
        else
        {
             NN_LOG(
             "Type: %c, Thread : %d, Core: %d, Socket: %d, MaxBufSize: %d, bytes Read: %lu, wouldBlock: %d, readNotFull: %d\n",
                  payload[idx].type, payload[idx].threadId, payload[idx].desiredCore, payload[idx].sockfds,
                  payload[idx].maxBufSize, payload[idx].bytesSent, payload[idx].writeWouldBlock, payload[idx].writeNotFull );
             NN_LOG(
             "          BytesPerSecond: %d, BitsPerSecond: %d\n",
                         (payload[idx].bytesSent / 60), (payload[idx].bytesSent * 8 / 60 ) );
        }
        payload[idx].threadId = -1;
        payload[idx].sockfds  = -1;
    }

    NN_LOG( "All threads stopped!\n" );

    result = 0;
    for ( idx = 0; idx < numberOfPairs; idx = idx + 2 )
    {
        // If Reader and Writer disagree..
        if ( payload[idx].bytesSent != payload[idx + 1].bytesSent )
        {
            result = -1;
        }
    }

    return( result );

out:


    for ( idx = 0; idx < numberOfPairs; idx++ )
    {
        if ( payload[idx].threadId > -1 )
        {
            // Waitthreads to die
            g_pThreadedTest->WaitThread( payload[idx].threadId );

            // Destroy Thread
            g_pThreadedTest->DestroyThread( payload[idx].threadId );

            payload[idx].threadId = -1;
        }

        if ( payload[idx].sockfds > -1 )
        {
            nn::socket::Close( payload[idx].sockfds );
            payload[idx].sockfds = -1;
        }
    }

    return( -1 );

}   // NOLINT(impl/function_size)


TEST(ApiUnit,Poll_Read_Write_UDP)
{
    PollRead_And_Write_UDP( 2, 0, "127.0.0.1" );

    PollRead_And_Write_UDP( 4, 0, "127.0.0.1" );

    PollRead_And_Write_UDP( 6, 0, "127.0.0.1" );
}


static int
PollRead_And_Write_TCP( int  numberOfPairs, int in_delaytimems, const char * in_interface )
{
    struct PollPayload   payload[15];

    int               idx = 0, listenSock = -1;
    bool              isSuccess = true;
    size_t            maxSize = -1;
    int               core = -1, result = -1;
    bool              rval;

    memset( (char *) &payload, 0, sizeof( payload ) );
    for ( idx = 0; idx < numberOfPairs; idx++ )
    {
        payload[idx].threadId = -1;
        payload[idx].sockfds  = -1;
    }

    NN_LOG( "====> Testing Number of sockets: %d\n", numberOfPairs );

    ////////////////
    // Start 2 pairs of UDP sockets
    ///////////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9000, in_interface );
    ERROR_IF( rval == false, "MakeListenSocket failed!" );

    // Indicate that threads should run..
    g_shouldWrite = true;
    g_shouldRead  = true;
    g_workBool    = false;   // Threads don't run read..

    // Start Threads
    maxSize = 65535;
    core = 1;
    for( idx = 0; idx < numberOfPairs; idx = idx + 2 )
    {
        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        /////////////////
        // Create Reader - Thread
        /////////////////

        payload[idx].type         = 'R';
        payload[idx].desiredCore  = core;
        payload[idx].maxBufSize   = maxSize;
        payload[idx].sockfds      = NATF::API::COMMON::MakeTCPConnection( in_interface, 9000 );

        payload[idx].threadId = g_pThreadedTest->CreateThread( &PollReader, (void *) &payload[idx], kThreadPriority );
        if ( payload[idx].threadId < 0 )
        {
            NN_LOG( "Failed to create a Reader (thread)!\n" );
            goto out;
        }

        /////////////////
        // Createa Writer - Thread
        /////////////////

        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        payload[idx + 1].type = 'W';
        payload[idx + 1].writerDelay = in_delaytimems;
        payload[idx + 1].desiredCore = core;
        payload[idx + 1].maxBufSize  = maxSize;
        payload[idx + 1].sockfds     = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );

        payload[idx + 1].threadId = g_pThreadedTest->CreateThread( &PollWriter, (void *) &payload[idx + 1], kThreadPriority );
        if ( payload[idx + 1].threadId < 0 )
        {
            NN_LOG( "Failed to create a Writer (thread)!\n" );
            goto out;
        }
    }
    NN_LOG( "====> Give threads time to start...\n" );
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Test Time 60 seconds
    NN_LOG( "====> Test running for next 60 seconds..." );
    g_workBool = true;

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));
    NN_LOG( "Done!\n\n" );

    // Stop Writers
    NN_LOG( "Stop Writers!\n" );
    g_shouldWrite = false;
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Stop Readers
    NN_LOG( "Stop Readers!\n" );
    g_shouldRead = false;
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    // Wait for each thread death..
    for ( idx = 0; idx < numberOfPairs; idx++ )
    {
        // Wait for thread to die
        g_pThreadedTest->WaitThread( payload[idx].threadId );

        // Destroy Thread
        g_pThreadedTest->DestroyThread( payload[idx].threadId );

        // Close the Socket
        nn::socket::Close( payload[idx].sockfds );

        // If Writer Thread
        if (  payload[idx].type == 'W' )
        {
             NN_LOG(
             "Type: %c, Thread : %d, Core: %d, Socket: %d, MaxBufSize: %d, bytes Sent: %lu, wouldBlock: %d, writeNotFull: %d, writerDelay: %dms\n",
                  payload[idx].type, payload[idx].threadId, payload[idx].desiredCore, payload[idx].sockfds,
                  payload[idx].maxBufSize, payload[idx].bytesSent, payload[idx].writeWouldBlock,
                  payload[idx].writeNotFull, payload[idx].writerDelay );
             NN_LOG(
             "          BytesPerSecond: %d, BitsPerSecond: %d\n",
                         (payload[idx].bytesSent / 60), (payload[idx].bytesSent * 8 / 60 ) );
        }
        else
        {
             NN_LOG( "Type: %c, Thread : %d, Core: %d, Socket: %d, MaxBufSize: %d, bytes Read: %lu, wouldBlock: %d, readNotFull: %d\n",
                  payload[idx].type, payload[idx].threadId, payload[idx].desiredCore, payload[idx].sockfds,
                  payload[idx].maxBufSize, payload[idx].bytesSent, payload[idx].writeWouldBlock, payload[idx].writeNotFull );
             NN_LOG(
             "          BytesPerSecond: %d, BitsPerSecond: %d\n",
                         (payload[idx].bytesSent / 60), (payload[idx].bytesSent * 8 / 60 ) );
        }

        payload[idx].threadId = -1;
        payload[idx].sockfds  = -1;
    }

    NN_LOG( "All threads stopped!\n" );

    result = 0;
    for ( idx = 0; idx < numberOfPairs; idx = idx + 2 )
    {
        // If Reader and Writer disagree..
        if ( payload[idx].bytesSent != payload[idx + 1].bytesSent )
        {
            result = -1;
        }
    }


out:

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

    for ( idx = 0; idx < numberOfPairs; idx++ )
    {
        if ( payload[idx].threadId > -1 )
        {
            // Waitthreads to die
            g_pThreadedTest->WaitThread( payload[idx].threadId );

            // Destroy Thread
            g_pThreadedTest->DestroyThread( payload[idx].threadId );

            payload[idx].threadId = -1;
        }

        if ( payload[idx].sockfds > -1 )
        {
            nn::socket::Close( payload[idx].sockfds );
            payload[idx].sockfds = -1;
        }
    }

    return( 0 );

}  // NOLINT(impl/function_size)



TEST(ApiUnit,Poll_Read_Write_TCP)
{
    PollRead_And_Write_TCP( 2, 0, "127.0.0.1" );

    PollRead_And_Write_TCP( 4, 0, "127.0.0.1" );

    PollRead_And_Write_TCP( 6, 0, "127.0.0.1" );
}


TEST(ApiUnit,Poll_Teardown)
{
    TeardownTesting();
}

#else // Windows

TEST(ApiUnit,Poll_Windows)
{
    NN_LOG( "=============================================\n" );
    NN_LOG( "===  N O   T E S T s   T O    R U N \n" );
    NN_LOG( "=============================================\n" );
}

#endif

}}  // Namespace: NATF::API
