﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the DNS Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );

    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait DNS Server threads to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy all Threads
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Win_Ioctl_Initialize)
{
    InitializeTesting();
}


TEST(ApiUnit,Win_Ioctl_Various_Tests)
{
    int                 tcpSock = -1, udpSock = -1, listenSock = -1, acceptedSock = -1, rc = -1;
    nn::socket::Errno   myError = nn::socket::Errno::ESuccess;
    int                 idx, optVal;
    bool                isSuccess = true, rval = true;
    ssize_t             rc1;

    const char * msgs[] = {
    "The rain in Spain ",
    "Flows ",
    "mainly on the plain."
    };


    ////////////////////
    ////  nn::socket::Errno::EBadf
    ////////////////////

    NN_LOG( "Testing descriptor: " );

    for ( idx = -2; idx < 40; idx++ )
    {
        NN_LOG( "%d, ", idx );
        if ( idx == g_echoServer->serverUDP || idx == g_echoServer->serverTCP )
        {
            continue;
        }

        rc = nn::socket::Ioctl( idx, nn::socket::IoctlCommand::FionRead, (void *) &optVal, sizeof( optVal ) );
        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nIoctl() bad fds: %d should have failed with nn::socket::Errno::EBadf. Errno=<%d>\n", idx, myError );
                isSuccess = false;
            }
        }
        else
        {
            NN_LOG( "\nIoctl() succeeded but fds: %d. Expected to fail.\n" );
            isSuccess = false;
        }
    }

    NN_LOG( "\n" );
    ERROR_IF_AND_COUNT( !isSuccess, "Testing nn::socket::Errno::EBadf Failed\n" );
    isSuccess = true;



WRAP_FAILING_TEST( "SIGLO-58423", "Socket API: [Win] TCP nn::socket::IoctlCommand::FionRead does not return number bytes read-to-read" )
{
    ////////////////////
    ////  TCP - nn::socket::IoctlCommand::FionRead
    ////////////////////

    NN_LOG( "[TCP] - nn::socket::IoctlCommand::FionRead\n" );

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 8053 );
    ERROR_IF( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    rc1 = nn::socket::Send( tcpSock, msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc1 != (ssize_t) strlen( msgs[0] ), "Failed to write: %d bytes to echo socket!\n", rc );

    // Give Server time to echo
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Apply Ioctl
    rc = nn::socket::Ioctl( tcpSock, nn::socket::IoctlCommand::FionRead, (void *) &optVal, sizeof( optVal ) );
    ERROR_IF_AND_COUNT( rc < 0, "Ioctl failed while calling nn::socket::IoctlCommand::FionRead - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "nn::socket::IoctlCommand::FionRead bytes: %d\n", optVal );

    // Bytes in queue should be what we sent
    ERROR_IF_AND_COUNT( optVal != (int) strlen( msgs[0] ), "I expected: %d bytes, actually got: %d bytes!\n", strlen( msgs[0] ), optVal );

    NN_LOG( "=============\n" );
    NN_LOG( "PASSED!  nn::socket::IoctlCommand::FionRead bytes is correct!\n" );
    NN_LOG( "=============\n" );

    nn::socket::Close( tcpSock );
    tcpSock = -1;

}

WRAP_FAILING_TEST( "SIGLO-58313", "Socket API: [Win] UDP nn::socket::IoctlCommand::FionRead places socket in bad state" )
{
    ////////////////////
    ////  UDP - nn::socket::IoctlCommand::FionRead
    ////////////////////

    NN_LOG( "[UDP] - nn::socket::IoctlCommand::FionRead\n" );

    udpSock = NATF::API::COMMON::MakeUDPConnection(  "127.0.0.1", 9012, "127.0.0.1", 8053 );
    ERROR_IF( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    rc1 = nn::socket::Send( udpSock, msgs[0], strlen( msgs[0] ), nn::socket::MsgFlag::Msg_None );
    ERROR_IF( rc1 != (ssize_t) strlen( msgs[0] ), "Failed to write: %d bytes to echo socket!\n", rc );

    // Give Server time to echo
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // Apply Ioctl
    rc = nn::socket::Ioctl( udpSock, nn::socket::IoctlCommand::FionRead, (void *) &optVal, sizeof( optVal ) );
    ERROR_IF_AND_COUNT( rc < 0, "Ioctl failed while calling nn::socket::IoctlCommand::FionRead - errno: %d\n", nn::socket::GetLastError() );
    NN_LOG( "nn::socket::IoctlCommand::FionRead bytes (raw): %d\n", optVal );
    NN_LOG( "nn::socket::IoctlCommand::FionRead bytes (smoothed): %d\n", optVal );

    // Bytes in queue should be what we sent
    if ( optVal != (int) strlen( msgs[0] )  )
    {
        NN_LOG( "I expected: %d bytes, actually got: %d bytes!\n", strlen( msgs[0] ), optVal );
        nTestsFailing++;
    }
    else
    {
        NN_LOG( "=============\n" );
        NN_LOG( "PASSED!  nn::socket::IoctlCommand::FionRead bytes is correct!\n" );
        NN_LOG( "=============\n" );
        nTestsPassing++;
    }

    nn::socket::Close( udpSock );
    udpSock = -1;


} // End Failing Test


    ////////////////////
    ////  TCP - nn::socket::IoctlCommand::FionWrite / nn::socket::IoctlCommand::FionSpace
    ////////////////////

    NN_LOG( "[TCP] - nn::socket::IoctlCommand::FionWrite / nn::socket::IoctlCommand::FionSpace\n" );


    ////////////
    // Create TCP connection
    ///////////

    rval = NATF::API::COMMON::MakeListenSocket( listenSock, 9060, "127.0.0.1" );
    ERROR_IF( rval == false, "Failed to create Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9060 );
    ERROR_IF( tcpSock < 0, "Failed to create Client Socket - Errno: %d\n", nn::socket::GetLastError() );

    acceptedSock = nn::socket::Accept( listenSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Failed calling Accept on Listen Socket - Errno: %d\n", nn::socket::GetLastError() );

    rc = nn::socket::Fcntl( tcpSock, nn::socket::FcntlCommand::F_SetFl, nn::socket::FcntlFlag::O_NonBlock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );


    //////////////////
    // FionSpace/FionWrite available on Windows?
    /////////////////

    // Try nn::socket::IoctlCommand::FionSpace Ioctl
    rc = nn::socket::Ioctl( tcpSock, nn::socket::IoctlCommand::FionSpace, (void *) &optVal, sizeof( optVal ) );
    ERROR_IF_AND_COUNT( rc != -1 || nn::socket::GetLastError() != nn::socket::Errno::EInval, "Ioctl expected to return -1 with errno EInval while calling nn::socket::IoctlCommand::FionSpace - rc: %d errno: %d\n", rc, nn::socket::GetLastError() );

    // Try nn::socket::IoctlCommand::FionSpace Ioctl
    rc = nn::socket::Ioctl( tcpSock, nn::socket::IoctlCommand::FionWrite, (void *) &optVal, sizeof( optVal ) );
    ERROR_IF_AND_COUNT( rc != -1 || nn::socket::GetLastError() != nn::socket::Errno::EInval, "Ioctl expected to return -1 with errno EInval while calling nn::socket::IoctlCommand::FionWrite - rc: %d errno: %d\n", rc, nn::socket::GetLastError() );


    nn::socket::Close( listenSock );
    listenSock = -1;

    nn::socket::Close( acceptedSock );
    acceptedSock = -1;

    nn::socket::Close( tcpSock );
    tcpSock = -1;

out:

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( listenSock > -1 )
    {
        nn::socket::Close( listenSock );
        listenSock = -1;
    }

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

}   // NOLINT(impl/function_size)


TEST(ApiUnit,Win_Ioctl_Teardown)
{
    TeardownTesting();
}

}}  // Namespace: NATF::API
