﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

////////////////////////////////////////////////////////////////
// DEFAULT: DISABLED

#define ENABLE_FAILING_TESTS

#include <nn/socket.h>
#include <nn/nn_Log.h>

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

namespace
{
    bool TestInetNtop_Family(nn::socket::Family af, bool isSupported)
    {
        bool isSuccess = true;
        const char *rval;

        nn::socket::SockAddrIn addr;
        char addrAscii[nn::socket::Inet_AddrStrlen];

        memset(&addr, 0, sizeof(addr));
        memset(&addrAscii, 0, sizeof(addrAscii));
        PRINT_AND_CALL(rval = nn::socket::InetNtop(af, &addr.sin_addr, addrAscii, nn::socket::Inet_AddrStrlen));

        if( isSupported )
        {
            ERROR_IF(nullptr == rval, "InetNtop failed when expected to succeed.");
            ERROR_IF(!NATF::API::VerifyIpAddrEquiv(addrAscii, &addr.sin_addr), "VerifyIpAddrEquiv failed.");
        }
        else
        {
            ERROR_IF(nullptr != rval, "InetNtop succeeded when expected to fail.");
        }

out:
        return isSuccess;
    }

    bool TestInetNtop_AddrAscii(char *addrAscii, bool isSupported)
    {
        bool isSuccess = true;
        const char *rval;

        nn::socket::SockAddrIn addr;

        memset(&addr, 0, sizeof(addr));
        addr.sin_addr.S_addr = 0x0100007F; // IpV4 loopback
        PRINT_AND_CALL(rval = nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, addrAscii, static_cast<nn::socket::SockLenT>(nn::socket::Inet_AddrStrlen)));

        if( isSupported )
        {
            ERROR_IF(nullptr == rval, "InetNtop failed when expected to succeed.");
            ERROR_IF(!NATF::API::VerifyIpAddrEquiv(addrAscii, &addr.sin_addr), "VerifyIpAddrEquiv failed.");
        }
        else
        {
            ERROR_IF(nullptr != rval, "InetNtop succeeded when expected to fail.");
        }

out:
        return isSuccess;
    }

    bool TestInetNtop_AddrNetwork(void *pAddrNetwork, bool isSupported)
    {
        bool isSuccess = true;
        const char *rval;

        char addrAscii[nn::socket::Inet_AddrStrlen];

        memset(&addrAscii, 0, sizeof(addrAscii));
        PRINT_AND_CALL(rval = nn::socket::InetNtop(nn::socket::Family::Af_Inet, pAddrNetwork, addrAscii, static_cast<nn::socket::SockLenT>(nn::socket::Inet_AddrStrlen)));

        if( isSupported )
        {
            ERROR_IF(nullptr == rval, "InetNtop failed when expected to succeed.");
            ERROR_IF(!NATF::API::VerifyIpAddrEquiv(addrAscii, reinterpret_cast<nn::socket::InAddr *>(pAddrNetwork)), "VerifyIpAddrEquiv failed.");
        }
        else
        {
            ERROR_IF(nullptr != rval, "InetNtop succeeded when expected to fail.");
        }

out:
        return isSuccess;
    }

    bool TestInetNtop_Size(int size, bool isSupported)
    {
        bool isSuccess = true;
        const char *rval;

        nn::socket::SockAddrIn addr;
        char addrAscii[nn::socket::Inet_AddrStrlen];

        memset(&addr, 0, sizeof(addr));
        memset(&addrAscii, 0, sizeof(addrAscii));
        addr.sin_addr.S_addr = 0x0100007F; // IpV4 loopback
        PRINT_AND_CALL(rval = nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, addrAscii, static_cast<nn::socket::SockLenT>(size)));

        if( isSupported )
        {
            ERROR_IF(nullptr == rval, "InetNtop failed when expected to succeed.");
            ERROR_IF(!NATF::API::VerifyIpAddrEquiv(addrAscii, &addr.sin_addr), "VerifyIpAddrEquiv failed.");
        }
        else
        {
            ERROR_IF(nullptr != rval, "InetNtop succeeded when expected to fail.");
        }

out:
        return isSuccess;
    }

    bool TestInetNtop_Main()
    {
        bool isSuccess = true;
        bool rval = false;

        nn::socket::SockAddrIn addrIpv4;
        char dstSupportsIpV4[nn::socket::Inet_AddrStrlen];
        char alternateAddrNetwork[4] = { 0x01, 0x00, 0x00, 0x7F };

        ////////////////////////////////////////////////////////////////
        // Function:
        // const char *inet_ntop(int af, const void *restrict src, char *restrict dst, nn::socket::SockLenT size);

        ////////////////////////////////////////////////////////////////
        // Argument:
        // int af

        PRINT_AND_CALL(rval = TestInetNtop_Family(nn::socket::Family::Af_Inet, true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

        PRINT_AND_CALL(rval = TestInetNtop_Family(nn::socket::Family::Pf_Inet, true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

#ifdef AF6_INET
        // NOTE: AF6_INET not defined
        PRINT_AND_CALL(rval = TestInetNtop_Family(AF6_INET, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
        EXPECT_ERRNO_NX(nn::socket::Errno::EAfNoSupport);
        EXPECT_ERRNO_WIN(nn::socket::Errno::EAfNoSupport);
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);
#endif

//"[WIN] InetNtop succeeded when expected to fail (family PF_INET6)"
        PRINT_AND_CALL(rval = TestInetNtop_Family(nn::socket::Family::Pf_Inet6, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
        EXPECT_ERRNO_NX(nn::socket::Errno::EAfNoSupport);
        EXPECT_ERRNO_WIN(nn::socket::Errno::EAfNoSupport);
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);

//"[NX] InetNtop errno: Expected nn::socket::Errno::EAfNoSupport, Returned nn::socket::Errno::EInval (family -1)"
        PRINT_AND_CALL(rval = TestInetNtop_Family(static_cast<nn::socket::Family>(-1), false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
        EXPECT_ERRNO_NX(nn::socket::Errno::EAfNoSupport);
        EXPECT_ERRNO_WIN(nn::socket::Errno::EAfNoSupport);
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);

//"[NX] InetNtop errno: Expected nn::socket::Errno::EAfNoSupport, Returned nn::socket::Errno::EInval (family nn::socket::Family::Af_Unspec)"
        PRINT_AND_CALL(rval = TestInetNtop_Family(nn::socket::Family::Af_Unspec, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
        EXPECT_ERRNO_NX(nn::socket::Errno::EAfNoSupport);
        EXPECT_ERRNO_WIN(nn::socket::Errno::EAfNoSupport);
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);

//"[NX] InetNtop errno: Expected nn::socket::Errno::EAfNoSupport, Returned nn::socket::Errno::EInval (family PF_IPX)"
        PRINT_AND_CALL(rval = TestInetNtop_Family(static_cast<nn::socket::Family>(PF_IPX), false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
        EXPECT_ERRNO_NX(nn::socket::Errno::EAfNoSupport);
        EXPECT_ERRNO_WIN(nn::socket::Errno::EAfNoSupport);
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);

        PRINT_AND_CALL(rval = TestInetNtop_Family(static_cast<nn::socket::Family>(-255), false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

        PRINT_AND_CALL(rval = TestInetNtop_Family(static_cast<nn::socket::Family>(0), false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

        PRINT_AND_CALL(rval = TestInetNtop_Family(static_cast<nn::socket::Family>(255), false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

        ////////////////////////////////////////////////////////////////
        // Argument:
        // const void *restrict src

        memset(&addrIpv4, 0, sizeof(addrIpv4));
        addrIpv4.sin_addr.S_addr = 0x0100007F; // IpV4 loopback
        PRINT_AND_CALL(rval = TestInetNtop_AddrNetwork(&addrIpv4.sin_addr, true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

#ifdef sockaddr_in6
{
        // NOTE: sockaddr_in6 not defined
        struct sockaddr_in6 addrIpv6;
        memset(&addrIpv6, 0, sizeof(addrIpv6));
        strncpy(&addrIpv6.sin6_addr.s6_addr, "::1"); // IpV6 loopback
        PRINT_AND_CALL(rval = TestInetNtop_AddrNetwork(&addrIpv6.sin6_addr, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
}
#endif

//"[NX] InetNtop exception: Data access to an invalid memory region (addrNetwork nullptr)"
        PRINT_AND_CALL(rval = TestInetNtop_AddrNetwork(nullptr, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

        PRINT_AND_CALL(rval = TestInetNtop_AddrNetwork(reinterpret_cast<void*>(alternateAddrNetwork), true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

        ////////////////////////////////////////////////////////////////
        // Argument:
        // char *restrict dst

//"[NX] InetNtop exception: Data access to an invalid memory region (addrAscii nullptr)"
        PRINT_AND_CALL(rval = TestInetNtop_AddrAscii(nullptr, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

        memset(dstSupportsIpV4, 0, nn::socket::Inet_AddrStrlen);
        PRINT_AND_CALL(rval = TestInetNtop_AddrAscii(dstSupportsIpV4, true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

#ifdef INET6_ADDRSTRLEN
{
        // Note: [NX] INET6_ADDRSTRLEN not defined
        char dstSupportsIpV6[INET6_ADDRSTRLEN];
        memset(dstSupportsIpV6, 0, INET6_ADDRSTRLEN);
        PRINT_AND_CALL(rval = TestInetNtop_AddrAscii(dstSupportsIpV6, true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
}
#endif

        ////////////////////////////////////////////////////////////////
        // Argument:
        // nn::socket::SockLenT size

        PRINT_AND_CALL(rval = TestInetNtop_Size(nn::socket::Inet_AddrStrlen, true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

#ifdef INET6_ADDRSTRLEN
        // Note: [NX] INET6_ADDRSTRLEN not defined
        PRINT_AND_CALL(rval = TestInetNtop_Size(INET6_ADDRSTRLEN, true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
#endif

//"SIGLONTD-6673: [WIN] InetNtop succeeded when expected to fail (size 0)"
        PRINT_AND_CALL(rval = TestInetNtop_Size(0, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
        EXPECT_ERRNO_NX(nn::socket::Errno::ENoSpc);
        EXPECT_ERRNO_WIN(nn::socket::Errno::ENoSpc);
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);

//"[NX] InetNtop exception: Data access to an invalid memory region (size -1)"
        PRINT_AND_CALL(rval = TestInetNtop_Size(-1, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
        EXPECT_ERRNO_NX(nn::socket::Errno::ENoSpc);
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);

//"[WIN, NX] InetNtop errno: Expected ENOSPC, Returned WIN ENOMSG & NX nn::socket::Errno::EInval (size 3)"
        PRINT_AND_CALL(rval = TestInetNtop_Size(3, false));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");
        EXPECT_ERRNO_NX(nn::socket::Errno::ENoSpc);
        EXPECT_ERRNO_WIN(nn::socket::Errno::ENoSpc);
        nn::socket::SetLastError(nn::socket::Errno::ESuccess);

//"[WIN, NX] InetNtop failed when expected to succeed (size 10)" -- 8 is too small
        // 127.0.0.1'\0'
        // 123456789 10
        PRINT_AND_CALL(rval = TestInetNtop_Size(10, true));
        ERROR_IF_AND_COUNT(rval != true, "InetNtop() test failed");

out:
        return isSuccess;
    } // NOLINT(impl/function_size)

} // unnamed

namespace NATF {
namespace API {

TEST(ApiUnit, TestInetNtop_Setup)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Socket), "TestSetup failed");

out:
    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(ApiUnit, TestInetNtop_Main)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    INITIALIZE_TEST_COUNTS;

    NN_LOG("Calling TestInetNtop_Main...\n");
    isSuccess = ::TestInetNtop_Main();

    PRINT_TEST_COUNTS;

    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

TEST(ApiUnit, TestInetNtop_Teardown)
{
    bool isSuccess = true;
    NN_LOG("In\n\n");

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed");

out:
    EXPECT_EQ(isSuccess, true);
    NN_LOG("Out\n\n");
}

}} // NATF::API
