﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>
#include <nn/nifm.h>

namespace NATF {
namespace API {
namespace COMMON {

//*********************
//*  G L O B A L S
//*********************

//*********************
//*  Functions
//*********************

int
SetNonBlocking( int  sockfds )
{
    int   rc = -1, opts = -1;

    // Client: Fcntl - nn::socket::FcntlFlag::O_NonBlock
    opts = nn::socket::Fcntl( sockfds, nn::socket::FcntlCommand::F_GetFl );
    opts |= static_cast<int>(nn::socket::FcntlFlag::O_NonBlock);
    rc = nn::socket::Fcntl( sockfds, nn::socket::FcntlCommand::F_SetFl, static_cast<nn::socket::FcntlFlag>(opts) );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to set non-blocking flag option.  Errno: %d\n", nn::socket::GetLastError() );
        return( -1 );
    }

    return( 0 );
}

int
SetBlocking( int  sockfds )
{
    int   rc = -1, opts = -1;

    // Client: Fcntl - nn::socket::FcntlFlag::O_NonBlock
    opts = nn::socket::Fcntl( sockfds, nn::socket::FcntlCommand::F_GetFl );
    opts = opts & static_cast<int>(~nn::socket::FcntlFlag::O_NonBlock);
    rc = nn::socket::Fcntl( sockfds, nn::socket::FcntlCommand::F_SetFl, static_cast<nn::socket::FcntlFlag>(opts) );
    if ( rc < 0 )
    {
        NN_LOG( "Failed to set blocking flag option.  Errno: %d\n", nn::socket::GetLastError() );
        return( -1 );
    }

    return( 0 );
}

int
MakeUDPConnection(  const char *  sourceHost, uint16_t   sourcePort,
                    const char *  targetHost, uint16_t   targetPort ) NN_NOEXCEPT
{
    nn::socket::SockAddrIn   localAddr = { 0 }, remoteAddr = { 0 };
    int           clisock = -1, rc = -1;
    bool          isSuccess = true;

    // Client: Create UDP Socket
    clisock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( clisock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Client: Set Non-Blocking
    rc = NATF::API::COMMON::SetNonBlocking( clisock );
    ERROR_IF( rc < 0, "Fcntl() failed setting nn::socket::FcntlFlag::O_NonBlock!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Client: Target Address
    memset(&localAddr, 0, sizeof(localAddr));
    localAddr.sin_port        = nn::socket::InetHtons(sourcePort);
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, sourceHost, &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client: Bind (self)
    rc = nn::socket::Bind( clisock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind() Failed but Success was expected!" );

    // Client: Target Address
    memset(&remoteAddr, 0, sizeof(remoteAddr));
    remoteAddr.sin_port        = nn::socket::InetHtons(targetPort);
    remoteAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, targetHost, &remoteAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client: UDP Connect
    rc = nn::socket::Connect( clisock, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), sizeof(remoteAddr) );
    if (rc < 0 )
    {
        if ( nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock || nn::socket::GetLastError() == nn::socket::Errno::EAgain )
        {
            ;   // Do nothing
        }
        else
        {
            ERROR_IF( rc < 0, "UDP Connect Failed - errno=<%d>\n\n", nn::socket::GetLastError() );
        }
    }

    return( clisock );

out:

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    return( -1 );
}

int
MakeTCPConnection( const char *  targetHost, uint16_t   targetPort ) NN_NOEXCEPT
{
    nn::socket::SockAddrIn   remoteAddr;
    int           clisock = -1, rc, opt = 0;
    bool          isSuccess = true;

    // Client: Socket
    clisock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( clisock < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Client: Address
    memset(&remoteAddr, 0, sizeof(remoteAddr));
    remoteAddr.sin_port        = nn::socket::InetHtons(targetPort);
    remoteAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Turn IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, targetHost, &remoteAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    NN_LOG( "Establishing a TCP Connection to [%s:%d]\n", targetHost, targetPort );

    // Client: Connect
    rc = nn::socket::Connect( clisock, reinterpret_cast<nn::socket::SockAddr *>(&remoteAddr), sizeof(remoteAddr) );
#ifndef NN_BUILD_CONFIG_OS_WIN32
    ERROR_IF( rc < 0, "Connect Failed - errno=<%d>\n\n", nn::socket::GetLastError() );
#else
    if (rc < 0)
    {
        if (nn::socket::GetLastError() != nn::socket::Errno::ESuccess)
        {
            NN_LOG( "Windows Connect failed with errno=<%d>\n", nn::socket::GetLastError());
            goto out;
        }
    }
#endif

    NN_LOG( "TCP Connected to: [%s:%d]\n",  targetHost, targetPort );

    opt = 1;

#ifndef NN_BUILD_CONFIG_OS_WIN32
    // Disable (David) Nagels algorimthum
    rc = nn::socket::SetSockOpt( clisock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, (char *) &opt, sizeof( opt ) );
    ERROR_IF(rc < 0, "Failed to turn off Nagle algorithum!" );

#endif

    rc = NATF::API::COMMON::SetNonBlocking( clisock );
    ERROR_IF(rc < 0, "SetNonBlocking on TCP Socket Failed!  Clisock=<%d>\n", clisock );

    return( clisock );

out:

    if ( clisock > -1 )
    {
        nn::socket::Close( clisock );
        clisock = -1;
    }

    return( -1 );
}


int
MakeRandomNoise( unsigned char ** rtrnPtr, size_t rtrnPtrSize, uint8_t & startpos ) NN_NOEXCEPT
{
    unsigned char *   myPtr;
    size_t            idx;

    myPtr = (unsigned char *) malloc( rtrnPtrSize );
    if ( myPtr == NULL )
    {
        NN_LOG( "Failed to allocate PTr of size: %d\n", rtrnPtrSize );
        return( -1 );
    }

    for( idx = 0; idx < static_cast<int>(rtrnPtrSize); ++idx )
    {
        startpos++;
        if ( startpos > 255 )
        {
            startpos = 0;
        }

        myPtr[idx] = (unsigned char) startpos;
    }

    // Save return value
    *rtrnPtr = myPtr;

    return( 0 );
}


int
WriteData( int sockfds, unsigned char * outBuff, size_t outBuffLen,
           uint64_t &   writeWouldBlock, uint64_t &  writeNotFull ) NN_NOEXCEPT
{
    int    bytesSent, bytesLeft, rc = 0;

    bytesSent = 0;
    bytesLeft = static_cast<int>(outBuffLen);
    while( bytesLeft > 0 )
    {
        rc = nn::socket::Write( sockfds, &outBuff[bytesSent], bytesLeft );
        if ( rc < 0 )
        {
            if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
            {
                writeWouldBlock++;
                break;
            }
            NN_LOG( "Client socket:  %d, nn::socket::Write() Failed - Errno: %d\n",
                         sockfds, nn::socket::GetLastError() );
            goto out;
        }

        if ( rc != static_cast<int>(outBuffLen) )
        {
            writeNotFull++;
        }

        bytesSent = bytesSent + rc;
        bytesLeft = bytesLeft - rc;
    }

    if ( bytesSent > 0 )
    {
        return( bytesSent );
    }
    else
    {
        return( rc );
    }

out:

    return( -1 );
}

int
ReadData( int sockfds, unsigned char * outBuff, size_t outBuffLen, uint64_t & readWouldBlock )
{
    int bytesRead, bytesLeft, rc = 0;

    bytesRead  = 0;
    bytesLeft  = (int) outBuffLen;
    while( bytesLeft > 0 )
    {
        rc = nn::socket::Read( sockfds, &outBuff[bytesRead], bytesLeft );
        if ( rc < 0 )
        {
            if ( nn::socket::GetLastError() == nn::socket::Errno::EAgain || nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock )
            {
                readWouldBlock++;
                break;
            }

            NN_LOG( "Client Socket: %d, nn::socket::Read() Failed - Errno: %d\n",
                    sockfds, nn::socket::GetLastError() );
            goto out;
        }

        bytesRead = bytesRead + rc;
        bytesLeft = bytesLeft - rc;
    }

    if ( bytesRead > 0 )
    {
        return( bytesRead );
    }
    else
    {
        return( rc );
    }

out:

    return( -1 );
}


bool
MakeListenSocket(int & sockfds, int inPort, const char * inInterface) NN_NOEXCEPT
{
    nn::socket::SockAddrIn   localAddr = { 0 };
    int                      rc = -1;
    int                      optind = 1;
    bool                     isSuccess = true;

    sockfds = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    ERROR_IF( sockfds < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Client: Local Address
    memset(&localAddr, 0, sizeof( localAddr ));
    localAddr.sin_port        = nn::socket::InetHtons( static_cast<uint16_t>(inPort) );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton(nn::socket::Family::Af_Inet, inInterface, &localAddr.sin_addr.S_addr);
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Set Reuse Socket
    rc = nn::socket::SetSockOpt(sockfds, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, (char *) &optind, sizeof( optind ) );
    ERROR_IF( rc < 0, "SetSockOpt() Failed but Success was expected!" );

    // Client: Bind (self)
    rc = nn::socket::Bind(sockfds, (nn::socket::SockAddr *) &localAddr, sizeof(localAddr));
    ERROR_IF( rc < 0, "Bind() returned: %d but was unexpected!", rc );

    // For TCP - Add a listen queue
    rc = nn::socket::Listen(sockfds, 5 );
    ERROR_IF( rc < 0, "Listen() returned: %d but errno: %d!", rc, nn::socket::GetLastError());

    return( true );

out:

    return( false);

}


bool
SetSocketOptAndConnect( int & sockfds, nn::socket::Level socklevel, nn::socket::Option sockopt, void * opt, nn::socket::SockLenT & optlen,
                        int localPort,  const char * localInterface,
                        int targetPort, const char * targetInterface,
                        bool isUDP, int expectedRC ) NN_NOEXCEPT
{
    nn::socket::SockAddrIn   localAddr, targetAddr;
    bool                 isSuccess = true;
    int                  rc;

    // Create Socket
    if ( isUDP == true )
    {
        sockfds = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    }
    else
    {
        sockfds = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    }

    ERROR_IF( sockfds < 0, "Socket() failed!  Errno=<%d>\n\n", nn::socket::GetLastError() );

    // Tell echo server to REUSE it's TCP address port
    PRINT_AND_CALL( rc = nn::socket::SetSockOpt( sockfds, socklevel, sockopt, opt, optlen ) );
    ERROR_IF( rc < 0, "SetSockOpt failed: rc: %d, socklevel: %d, sockopt: %d, errno: %d",
                           rc, socklevel, sockopt, nn::socket::GetLastError() );

    // Is Option on?
    PRINT_AND_CALL( rc = nn::socket::GetSockOpt( sockfds, socklevel, sockopt, opt, &optlen ) );
    ERROR_IF( rc < 0, "GetSockOpt failed! rc: %d, socklevel: %d, sockopt: %d, optlen: %d, errno: %d\n",
                       rc, socklevel, sockopt, optlen, nn::socket::GetLastError() );

#ifndef NN_BUILD_CONFIG_OS_WIN32

    // 1s complement?
    if (  ( socklevel == nn::socket::Level::Sol_Socket ) &&
          ( sockopt == nn::socket::Option::So_Broadcast )  )
    {
        // This converts a directed broadcast to a limited broadcast.
        // See "nn::socket::Option::Ip_OnesBcast option" on the following page for details
        // https://www.freebsd.org/cgi/man.cgi?query=ip&apropos=0&sektion=0&manpath=FreeBSD+10.1-RELEASE&arch=default&format=html
        int enableFlag = 1;
        if( nn::socket::SetSockOpt( sockfds, nn::socket::Level::Sol_Ip,
                                                    nn::socket::Option::Ip_OnesBcast, &enableFlag, sizeof(enableFlag)) >= 0)
        {
           NN_LOG("SetSockOpt nn::socket::Option::Ip_OnesBcast OK\n");
        }
    }

#endif

    if ( isUDP == true )  // UDP
    {
        // Client: Set Local Address
        memset(&localAddr, 0, sizeof( localAddr ));
        localAddr.sin_port        = nn::socket::InetHtons( static_cast<uint16_t>(localPort) );
        localAddr.sin_family      = nn::socket::Family::Af_Inet;

        // Client: Translate Target IP Address into Network Address
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, localInterface, &localAddr.sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        // Client: Bind (self)
        rc = nn::socket::Bind( sockfds, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc != expectedRC, "Bind() returned: %d but expectedRC: %d was expected!", rc, expectedRC );
    }

    // Client: Target Address
    memset(&targetAddr, 0, sizeof( targetAddr ));
    targetAddr.sin_port        = nn::socket::InetHtons( static_cast<uint16_t>(targetPort) );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client:  Connect
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, targetInterface, &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client: UDP Connect
    rc = nn::socket::Connect( sockfds, reinterpret_cast<nn::socket::SockAddr *>(&targetAddr), sizeof(targetAddr) );
    if (rc < 0 )
    {
        if ( nn::socket::GetLastError() == nn::socket::Errno::EWouldBlock || nn::socket::GetLastError() == nn::socket::Errno::EAgain )
        {
            ;   // Do nothing
        }
        else
        {
            ERROR_IF( rc < 0, "Connect Failed - errno=<%d>\n\n", nn::socket::GetLastError() );
        }
    }

    return(true);

out:

    return(false);
}

uint64_t
GetTicksInSeconds() NN_NOEXCEPT
{
    nn::os::Tick now_in_ticks = nn::os::GetSystemTick();
    return( nn::os::ConvertToTimeSpan( now_in_ticks ).GetSeconds() );
}

uint64_t
GetMicroSeconds() NN_NOEXCEPT
{
    nn::os::Tick now_in_ticks = nn::os::GetSystemTick();
    return( nn::os::ConvertToTimeSpan( now_in_ticks ).GetMicroSeconds() );
}

bool
CheckBooleanSetSockOpt(int sockfds, nn::socket::Level inSocklevel, nn::socket::Option inSockopt, bool isEnabled, int inValue) NN_NOEXCEPT
{
    bool                  isSuccess = true;
    int                   rc = 0, intoptval = 0;
    nn::socket::SockLenT  intoptlen = 0;

    intoptval = inValue;
    intoptlen = sizeof(intoptval);

    PRINT_AND_CALL(rc = nn::socket::SetSockOpt(sockfds, inSocklevel, inSockopt, &intoptval, intoptlen));
    ERROR_IF_AND_COUNT(rc < 0, "SetSockOpt [ON] - Failed - errno [%d]\n",  nn::socket::GetLastError() );

    intoptval = -1;   // Unimportant Values
    intoptlen = sizeof(intoptval);
    PRINT_AND_CALL(rc = nn::socket::GetSockOpt(sockfds, inSocklevel, inSockopt, reinterpret_cast<void *>(&intoptval), &intoptlen));
    NN_LOG("GetSockOpt Returns: intoptval: %d, intoptlen: %d\n", intoptval, intoptlen);
    ERROR_IF_AND_COUNT(rc < 0, "GetSockOpt [ON] - Failed - errno [%d]\n",  nn::socket::GetLastError() );
    ERROR_IF_AND_COUNT(intoptlen != sizeof(int), "GetSockOpt [ON] - Failed - intoptlen: %d, expected: %d\n", intoptlen, sizeof(int));

    if (isEnabled == true)
    {
        ERROR_IF_AND_COUNT(intoptval < 1, "GetSockOpt [ON] - Failed - intoptval: %d, expected: %d\n", intoptval, 1);
    }
    else
    {
        ERROR_IF_AND_COUNT(intoptval > 0, "GetSockOpt [OFF] - Failed - intoptval: %d, expected: %d\n", intoptval, 0);
    }

    return(true);

out:

    return(false);
}

bool
CheckNumericSetSockOpt(int sockfds, nn::socket::Level inSocklevel, nn::socket::Option inSockopt, bool isValid, int inValue) NN_NOEXCEPT
{
    bool                  isSuccess = true;
    int                   rc = 0, intoptval = 0;
    nn::socket::SockLenT  intoptlen = 0;

    intoptval = inValue;
    intoptlen = sizeof(intoptval);

    PRINT_AND_CALL(rc = nn::socket::SetSockOpt(sockfds, inSocklevel, inSockopt, &intoptval, intoptlen));
    if (isValid == true)
    {
        ERROR_IF_AND_COUNT(rc < 0, "SetSockOpt [VALUE: %d] - Failed - errno [%d]\n", inValue, nn::socket::GetLastError() );
    }

    intoptval = -1;   // Unimportant Values
    intoptlen = sizeof(intoptval);
    PRINT_AND_CALL(rc = nn::socket::GetSockOpt(sockfds, inSocklevel, inSockopt, reinterpret_cast<void *>(&intoptval), &intoptlen));
    NN_LOG("GetSockOpt Returns: intoptval: %d, intoptlen: %d\n", intoptval, intoptlen);
    ERROR_IF_AND_COUNT(intoptlen != sizeof(int), "GetSockOpt [Value: %d] - Failed - intoptlen: %d, expected: %d\n", inValue, intoptlen, sizeof(int));

    if (isValid == true)
    {
        ERROR_IF_AND_COUNT(rc < 0, "GetSockOpt [Value: %d] - Failed - errno [%d]\n",  inValue, nn::socket::GetLastError() );
        ERROR_IF_AND_COUNT(intoptval != inValue, "GetSockOpt [Value: %d] - Failed - intoptval: %d, expected: %d\n", inValue, intoptval, inValue);
    }
    else
    {
        ERROR_IF_AND_COUNT(intoptval == inValue, "GetSockOpt [Value: %d] - Failed - intoptval: %d, expected: %d\n", inValue, intoptval, inValue);
    }

    return(true);

out:

    return(false);
}

bool
CheckTimevalSetSockOpt(int sockfds, nn::socket::Level inSocklevel, nn::socket::Option inSockopt, bool isValid, nn::socket::TimeVal inValue) NN_NOEXCEPT
{
    nn::socket::TimeVal   intoptval = { 0 };
    nn::socket::SockLenT  intoptlen = 0;
    int                   rc = 0;
    bool                  isSuccess = true;

    intoptval.tv_sec  = inValue.tv_sec;
    intoptval.tv_usec = inValue.tv_usec;
    intoptlen = sizeof(intoptval);

    PRINT_AND_CALL(rc = nn::socket::SetSockOpt(sockfds, inSocklevel, inSockopt, &intoptval, intoptlen));
    if (isValid == true)
    {
        ERROR_IF_AND_COUNT(rc < 0, "SetSockOpt [VALUE: tv_sec: %d, tv_usec: %d] - Failed - errno [%d]\n",
                           inValue.tv_sec, inValue.tv_usec, nn::socket::GetLastError() );
    }

    intoptval = { 0 };   // Unimportant Values
    intoptlen = sizeof(intoptval);
    PRINT_AND_CALL(rc = nn::socket::GetSockOpt(sockfds, inSocklevel, inSockopt, reinterpret_cast<void *>(&intoptval), &intoptlen));
    NN_LOG("GetSockOpt Returns: intoptval: %d, intoptlen: %d\n", intoptval, intoptlen);
    ERROR_IF_AND_COUNT(intoptlen != sizeof(nn::socket::TimeVal), "GetSockOpt [VALUE: tv_sec: %d, tv_usec: %d] - Failed - intoptlen: %d, expected: %d\n",
                       inValue.tv_sec, inValue.tv_usec, intoptlen, sizeof(nn::socket::TimeVal));

    if (isValid == true)
    {
        ERROR_IF_AND_COUNT(rc < 0, "GetSockOpt [VALUE: tv_sec: %d, tv_usec: %d] - Failed - errno [%d]\n",
                           inValue.tv_sec, inValue.tv_usec, nn::socket::GetLastError() );
        ERROR_IF_AND_COUNT(intoptval.tv_sec != inValue.tv_sec && intoptval.tv_usec != inValue.tv_usec,
                           "GetSockOpt [VALUE: tv_sec: %d, tv_usec: %d] - Failed - Got unexpected: [tv_sec: %d, tv_usec: %d]\n",
                           inValue.tv_sec, inValue.tv_usec, intoptval.tv_sec, intoptval.tv_usec);
    }
    else
    {
        ERROR_IF_AND_COUNT(intoptval.tv_sec == inValue.tv_sec && intoptval.tv_usec == inValue.tv_usec,
                           "GetSockOpt [VALUE: tv_sec: %d, tv_usec: %d] - Failed - Got unexpected: [tv_sec: %d, tv_usec: %d]\n",
                           inValue.tv_sec, inValue.tv_usec, intoptval.tv_sec, intoptval.tv_usec);
    }

    return(true);

out:

    return(false);
}

}}}; // NATF::API::COMMON

