﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//////////////////////////////////////////////////////////////
// DEFAULT: ENABLE_FAILING_TESTS - DISABLED
//
// #define ENABLE_FAILING_TESTS

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc
#include <cmath>      // abs

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nn/socket/socket_ApiPrivate.h>

#include <nnt/nntest.h>

#include <nn/os/os_Thread.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo
#include <nn/nifm.h>

#include "Unit/testNet_ThreadedTest.h"          // Threads for testing
#include "Unit/testNet_EchoServer.h"            // TCP / UDP echo server
#include "Unit/testNet_CommonFunctions.h"       // Common Functions

namespace NATF {
namespace API {

//*********************
//*  G L O B A L S
//*********************

static const int            kThreadPriority = 10;       // Thread Priority
static const int            kMaxNumOfSockets = 3;       // Max test sockets

static ThreadedController * g_pThreadedTest = NULL;
static int                  g_echoServerThreadId = -1;
static EchoServer *         g_echoServer = NULL;

static bool                 g_shouldRun = false;
static const int            g_kNumWorkerThreads = 10;


static void
StartEchoServer( void * inArg )
{
    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread starting\n" );

    EchoServer * echoServer = (EchoServer *) inArg;

    // Start the Echo Server Main Loop
    echoServer->Start( (void *) NULL);

    // Tell Log we are done
    NN_LOG( "[Echo Server]: Thread exiting\n" );
}


static bool
InitializeTesting()
{
    bool        isSuccess = true;

    NN_LOG( "In\n\n" );

    ///////////////////////////
    //// Test Counts: Initialize
    ///////////////////////////

    INITIALIZE_TEST_COUNTS;


    ///////////////////////////
    //// NIFM Library: Initialize
    ///////////////////////////

    ERROR_IF(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket), "TestSetup failed.");

    // NIFM: Network Interface is Online
    NN_LOG( "====================================\n" );
    NN_LOG( "NIFM: Network ===>  O N L I N E <===\n" );
    NN_LOG( "====================================\n" );


    ///////////////////////////
    ////  Allocate Echo Server
    ///////////////////////////

    g_echoServer = new EchoServer();
    if ( g_echoServer == NULL )
    {
        NN_LOG( "Failed to allocate an EchoServer.  Can't start Loopback Echo Server\n" );
        goto out;
    }

    NN_LOG( "===> Starting [Echo Server]\n" );

    // Create: ThreadedTest
    g_pThreadedTest = new ThreadedController();
    if ( g_pThreadedTest == NULL )
    {
        NN_LOG( "new fail allocating a 'new' ThreadedController\n" );
        goto out;
    }

    // Allocate 1 slot
    g_pThreadedTest->Initialize( kMaxNumOfSockets );
    // Create a thread to run the Server in

    g_echoServerThreadId = g_pThreadedTest->CreateThread( &StartEchoServer, (void *) g_echoServer, kThreadPriority );
    if (  g_echoServerThreadId < 0 )
    {
        NN_LOG( "Failed to create and start Echo Server (thread)!\n" );
        goto out;
    }

    NN_LOG( "Successfully started Echo Server" );

    // Give Thread time to start
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

    return( true );

out:

    // If the loopback DNS server is allocated
    if ( g_echoServer != NULL )
    {
        delete g_echoServer;
        g_echoServer = NULL;
    }

    // If the loopback DNS server is allocated
    if ( g_pThreadedTest != NULL )
    {
        delete g_pThreadedTest;
        g_pThreadedTest = NULL;
    }

    return( false );
}


static bool
TeardownTesting()
{
     bool    isSuccess = true;

    ///////////////////
    //// Stop DNS Server
    ////////////////////

    NN_LOG( "Stopping Echo Server\n" );

    // If the echo server is allocated
    if ( g_echoServer != NULL )
    {
        g_echoServer->Stop();
    }

    // Wait for Echo Server thread to die
    g_pThreadedTest->WaitThread( g_echoServerThreadId );

    // Destroy Echo Server Thread
    g_pThreadedTest->DestroyThread( g_echoServerThreadId );

    // Delete Echo Server
    delete g_echoServer;
    g_echoServer = NULL;

    // Delete Threaded Test
    delete g_pThreadedTest;
    g_pThreadedTest = NULL;
    g_echoServerThreadId = -1;

    NN_LOG( "Echo Server successfully stopped!\n" );


    ////////////////////
    ////  Stop Testing
    ////////////////////

    ERROR_IF(!NATF::API::TestTeardown(), "TestTeardown failed.");

    ////////////////////
    ////  Print Test Counts
    ////////////////////

    PRINT_TEST_COUNTS;

    EXPECT_EQ( isSuccess, true );

    NN_LOG( "Out\n\n" );

    return( true );

out:

    return( false );
}



////////////////
//
// B E G I N   T E S T I N G
//
////////////////

TEST(ApiUnit,Win_Bind_Initialize)
{
    InitializeTesting();
}

#ifndef ADDRESS_FAMILY
#define ADDRESS_FAMILY uint8_t
#endif

TEST(ApiUnit,Win_Bind_Various_Tests)
{
    nn::socket::SockAddrIn  localAddr = { 0 }, targetAddr = { 0 }, outAddr = { 0 };
    int                     tcpSock = -1, udpSock = -1, udpServSock = -1, tcpServSock = -1, acceptedSock = -1;
    int                     rc = -1;
    nn::socket::Errno       myError = nn::socket::Errno::ESuccess;
    int                     idx;
    bool                    isSuccess = true;
    char                    buf1[1024];

    const char * msgs[] = {
    "The rain in Spain ",
    "Flows ",
    "mainly on the plain."
    };


    ////////////
    //  Establish UDP Target
    ///////////

    // Client: Target Address
    memset(&outAddr, 0, sizeof(outAddr));
    outAddr.sin_port        = nn::socket::InetHtons( 8053 );
    outAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &outAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    ////////////
    // Test nn::socket::Errno::EBadf
    ///////////


    NN_LOG( "Test Bind does not react to bad file descriptors [nn::socket::Errno::EBadf]\n\n" );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7000 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    NN_LOG( "Testing descriptor:" );
    for ( idx = -2; idx < 40; idx++ )
    {
        if  ( ( idx == g_echoServer->serverTCP )  ||
              ( idx == g_echoServer->serverUDP )  )
        {
            NN_LOG( "(Skip: %d) ", idx );
            continue;
        }

        NN_LOG( "%d, ", idx );

        rc = nn::socket::Bind( idx, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc > 0, "Bind() succeeded but should have failed processing descriptor: %d\n", idx );

        if ( rc < 0 )
        {
            myError = nn::socket::GetLastError();
            if ( myError != nn::socket::Errno::EBadf )
            {
                NN_LOG( "\nBind() bad fds: %d should have failed with nn::socket::Errno::EBadf, but actually failed with: %d\n", idx, myError );
                goto out;
            }

            continue;  // Jump to top
        }

        if ( rc == 0 )
        {
            NN_LOG( "\nBind() fds: %d:  Succeeded but sent (no bytes)!\n", idx );
            goto out;
        }
    }

    NN_LOG( "\n" );
    nTestsPassing++;


    /////////////
    // Simple TCP Bind
    /////////////

    NN_LOG( "Test simple TCP Bind:\n" );

    // Server: Create TCP Socket
    tcpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7000 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server: Bind
    rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF_AND_COUNT( rc < 0, "Failed calling Bind for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: Listen
    rc = nn::socket::Listen( tcpServSock, 5 );
    ERROR_IF( rc < 0, "Failed calling Listen for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Client: Connect
    tcpSock = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 7000 );
    ERROR_IF_AND_COUNT( tcpSock < 0, "Failed calling MakeTCPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    // Server: Accept
    acceptedSock = nn::socket::Accept( tcpServSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF( acceptedSock < 0, "Failed calling Accept for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Client: Write a Message
    rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to write message of size: %u\n", strlen( msgs[0] ) );

    // Server Read a Message
    rc = nn::socket::Read( acceptedSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to read message of size: %u, actually got: %d\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc ) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );
    NN_LOG( "Server Recv buf1 = [%.*s]\n", rc, buf1);

    // Server: Write a Message
    rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to write message of size: %u\n", strlen( msgs[0] ) );

    // Client: Read a Message
    rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to read message of size: %u, actually got: %d\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc ) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );
    NN_LOG( "Client Recv buf1 = [%.*s]\n", rc, buf1);

    // Close Sockets
    nn::socket::Close( tcpSock );
    tcpSock = -1;
    nn::socket::Close( acceptedSock );
    acceptedSock = -1;
    nn::socket::Close( tcpServSock );
    tcpServSock = -1;


    nTestsPassing++;

    /////////////
    // Simple UDP Bind
    /////////////

    NN_LOG( "Test simple UDP Bind:\n" );

    // Server: Create UDP Socket
    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServSock < 0, "Failed calling Socket for UDP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7000 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate SELF IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server: Bind
    rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF_AND_COUNT( rc < 0, "Failed calling Bind for UDP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: Target Address
    memset( &targetAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port        = nn::socket::InetHtons( 7001 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate TARGET IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &targetAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server Connect:
    rc = nn::socket::Connect( udpServSock, (nn::socket::SockAddr *) &targetAddr, sizeof( targetAddr ) );
    ERROR_IF( rc < 0, "Failed to connect UDP Server Socket to UDP socket !  Errno: %d\n", nn::socket::GetLastError() );

    // Client: Make UDP Connecton
    udpSock = NATF::API::COMMON::MakeUDPConnection( "127.0.0.1", 7001, "127.0.0.1", 7000 );
    ERROR_IF_AND_COUNT( udpSock < 0, "Failed calling MakeUDPConnection!  Errno:%d\n", nn::socket::GetLastError() );

    // Client: Write a Message
    rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to write message of size: %u\n", strlen( msgs[0] ) );

    // Server Read a Message
    rc = nn::socket::Read( udpServSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to read message of size: %u, actually got: %d\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc ) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );
    NN_LOG( "Server Recv buf1 = [%.*s]\n", rc, buf1);

    // Server: Write a Message
    rc = nn::socket::Write( udpServSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
    ERROR_IF( rc < 0, "Failed to write 18 bytes, errno: %d\n", nn::socket::GetLastError() );
    ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to write message of size: %u, actually wrote: %d\n", strlen( msgs[0] ), rc );

    // Client: Read a Message
    rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
    ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to read message of size: %u, actually got: %d\n", strlen( msgs[0] ), rc );
    ERROR_IF( strncmp( buf1, msgs[0], rc ) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );
    NN_LOG( "Client Recv buf1 = [%.*s]\n", rc, buf1);

    // Close Sockets
    nn::socket::Close( udpSock );
    udpSock = -1;
    nn::socket::Close( udpServSock );
    udpServSock = -1;

    nTestsPassing++;

    /////////////
    // TCP Bind - nullptr for address
    /////////////

    NN_LOG( "TCP Bind - nullptr for Address:  Errno == nn::socket::Errno::EInval\n" );

    // Client: Make TCP Socket
    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Failed to make TCP socket - Errno: %d\n", nn::socket::GetLastError() );

    // Client: Bind with NULL address
    rc = nn::socket::Bind( tcpSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc > -1, "Bind succeed but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "I expected to receive nn::socket::Errno::EInval but actually got errno: %d\n", myError );

    // Close Sockets
    nn::socket::Close( tcpSock );
    tcpSock = -1;

    /////////////
    // UDP Bind - nullptr for address
    /////////////

    NN_LOG( "UDP Bind - nullptr for Address: Errno == nn::socket::Errno::EInval\n" );

    // Client: Make UDP Socket
    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Failed to make TCP socket - Errno: %d\n", nn::socket::GetLastError() );

    // Client: Bind with NULL address
    rc = nn::socket::Bind( udpSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
    ERROR_IF_AND_COUNT( rc > -1, "UDP Bind succeed but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "I expected to receive nn::socket::Errno::EInval but actually got errno: %d\n", myError );

    // Close Sockets
    nn::socket::Close( udpSock );
    udpSock = -1;

    /////////////
    // TCP Bind - nullptr for address and bad length
    /////////////

    NN_LOG( "TCP Bind - nullptr for Address, Len = 1: Errno == nn::socket::Errno::EInval\n" );

    // Client: Make TCP Socket
    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Failed to make TCP socket - Errno: %d\n", nn::socket::GetLastError() );

    // Client: Bind with NULL address
    rc = nn::socket::Bind( tcpSock, static_cast<nn::socket::SockAddr*>(nullptr), 1 );
    ERROR_IF_AND_COUNT( rc > -1, "Bind succeed but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "I expected to receive nn::socket::Errno::EInval but actually got errno: %d\n", myError );

    // Close Sockets
    nn::socket::Close( tcpSock );
    tcpSock = -1;


    /////////////
    // UDP Bind - nullptr for address and bad length
    /////////////

    NN_LOG( "UDP Bind - nullptr for Address, Len = 1:  Errno == nn::socket::Errno::EInval\n" );

    // Client: Make UDP Socket
    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Failed to make TCP socket - Errno: %d\n", nn::socket::GetLastError() );

    // Client: Bind with NULL address
    rc = nn::socket::Bind( udpSock, static_cast<nn::socket::SockAddr*>(nullptr), 1 );
    ERROR_IF_AND_COUNT( rc > -1, "UDP Bind succeed but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "I expected to receive nn::socket::Errno::EInval but actually got errno: %d\n", myError );

    // Close Sockets
    nn::socket::Close( udpSock );
    udpSock = -1;


    /////////////
    // TCP Bind - Not enough Address for nn::socket::Family::Af_Inet
    /////////////

    NN_LOG( "TCP Bind - not enough Address for nn::socket::Family::Af_Inet - Errno == nn::socket::Errno::EInval\n" );

    // Client: Make UDP Socket
    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Failed to make TCP socket - Errno: %d\n", nn::socket::GetLastError() );

    // Client: Bind with NULL address
    rc = nn::socket::Bind( tcpSock, (nn::socket::SockAddr *) &localAddr, 2 );
    ERROR_IF_AND_COUNT( rc > -1, "UDP Bind succeed but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "I expected to receive nn::socket::Errno::EInval but actually got errno: %d\n", myError );

    // Close Sockets
    nn::socket::Close( tcpSock );
    tcpSock = -1;


    /////////////
    // UDP Bind - Not enough Address for nn::socket::Family::Af_Inet
    /////////////

    NN_LOG( "UDP Bind - not enough Address for nn::socket::Family::Af_Inet - Errno == nn::socket::Errno::EInval\n" );

    // Client: Make UDP Socket
    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Failed to make TCP socket - Errno: %d\n", nn::socket::GetLastError() );

    // Client: Bind with NULL address
    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, 2 );
    ERROR_IF_AND_COUNT( rc > -1, "UDP Bind succeed but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "I expected to receive nn::socket::Errno::EInval but actually got errno: %d\n", myError );

    // Close Sockets
    nn::socket::Close( udpSock );
    udpSock = -1;


    ///////////////////
    // TCP - Unsupported Address Types
    ///////////////////

    NN_LOG( "TCP - Unsupported Address Types:  ERRNO = nn::socket::Errno::EAfNoSupport\n" );

    // Server: Create TCP Socket
    tcpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7000 );
    localAddr.sin_family      = nn::socket::Family::Af_Unspec;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    NN_LOG( "Testing Address FAMILY [" );

    for( idx = 0; idx < 255; idx++ )
    {
        if ( idx == static_cast<int>(nn::socket::Family::Af_Unspec) ) continue;
        if ( idx == static_cast<int>(nn::socket::Family::Af_Inet) )   continue;

        NN_LOG( "%d, ", idx );

        // Host Order - Set Address Family
        localAddr.sin_family = static_cast<nn::socket::Family>(idx);

        // Server: Bind
        rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc > -1, "Succeeded calling BIND for TCP Server Socket at Idx: %d, but was supposed to FAIL!\n", idx );

        myError = nn::socket::GetLastError();
        //ERROR_IF( myError != nn::socket::Errno::EAddrNotAvail, "I expected to receive nn::socket::Errno::EAddrNotAvail but actually got errno: %d\n", myError );
        ERROR_IF( myError != nn::socket::Errno::EAfNoSupport, "I expected to receive (Errno: nn::socket::Errno::EAfNoSupport) - but actually got errno: %d\n", myError );
    }

    NN_LOG( "]\n" );
    nTestsPassing++;

    // Close TCP socket
    nn::socket::Close( tcpServSock );
    tcpServSock = -1;


    ///////////////////
    // UDP - Unsupported Address Types
    ///////////////////

    NN_LOG( "UDP - Unsupported Address Types:  ERRNO = nn::socket::Errno::EAfNoSupport\n" );

    // Server: Create UDP Socket
    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( udpServSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7000 );
    localAddr.sin_family      = nn::socket::Family::Af_Unspec;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    NN_LOG( "Testing Address FAMILY [" );

    for( idx = 0; idx < 255; idx++ )
    {
        if ( idx == static_cast<int>(nn::socket::Family::Af_Unspec) ) continue;
        if ( idx == static_cast<int>(nn::socket::Family::Af_Inet) )   continue;

        NN_LOG( "%d, ", idx );

        // Host Order - Set Address Family
        localAddr.sin_family = static_cast<nn::socket::Family>(idx);

        // Server: Bind
        rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc > -1, "Succeeded calling BIND for TCP Server Socket at Idx: %d, but was supposed to FAIL!\n", idx );

        myError = nn::socket::GetLastError();
        ERROR_IF( myError != nn::socket::Errno::EAfNoSupport, "I expected to receive (errno == nn::socket::Errno::EAfNoSupport) but actually got errno: %d\n", myError );
    }

    NN_LOG( "]\n" );
    nTestsPassing++;

    // Close UDP socket
    nn::socket::Close( udpServSock );
    udpServSock = -1;


    /////////////
    // TCP Bind - to an IP Address not on localhost
    /////////////

    NN_LOG( "TCP Bind to an IP address not on localhost:  Errno == nn::socket::Errno::EAddrNotAvail\n" );

    // Server: Create TCP Socket
    tcpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7000 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "192.203.230.10", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server: Bind
    rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc > -1, "Bind Succeed but should have failed" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAddrNotAvail, "I expected nn::socket::Errno::EAddrNotAvail - but I actually got errno: %d!", myError );

    // Close TCP socket
    nn::socket::Close( tcpServSock );
    tcpServSock = -1;


    /////////////
    // UDP Bind - to an IP Address not on localhost
    /////////////

    NN_LOG( "UDP Bind to an IP address not on localhost:  Errno == nn::socket::Errno::EAddrNotAvail\n" );

    // Server: Create UDP Socket
    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServSock < 0, "Failed calling Socket for UDP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7000 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "192.203.230.10", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server: Bind
    rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc > -1, "Bind Succeed but should have failed" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAddrNotAvail, "I expected nn::socket::Errno::EAddrNotAvail - but I actually got errno: %d!", myError );

    // Close TCP socket
    nn::socket::Close( udpServSock );
    udpServSock = -1;



    ///////////////////
    // TCP - Double Bind to same IP and Port
    ///////////////////

    NN_LOG( "TCP - Double Bind (Same Address and Port):  ERRNO = nn::socket::Errno::EInval\n" );

    // Server: Create TCP Socket
    tcpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7050 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // First Bind:
    rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind failed but Success was expected! errno: %d", nn::socket::GetLastError() );

    // Second Bind:
    rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc > -1, "Bind success but failure was expected!" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "I expected nn::socket::Errno::EInval - but I actually got errno: %d!", myError );

    // Close TCP socket
    nn::socket::Close( tcpServSock );
    tcpServSock = -1;


    ///////////////////
    // UDP - Double Bind to same IP and Port
    ///////////////////

    NN_LOG( "UDP - Double Bind (Same Address and Port):  ERRNO = nn::socket::Errno::EInval\n" );

    // Server: Create TCP Socket
    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp  );
    ERROR_IF( udpServSock < 0, "Failed calling Socket for UDP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7050 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // First Bind:
    rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind failed but success was expected!" );

    // Second Bind:
    rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc > -1, "Bind success but failure was expected!" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EInval, "I expected nn::socket::Errno::EInval - but I actually got errno: %d!", myError );

    // Close UDP socket
    nn::socket::Close( udpServSock );
    udpServSock = -1;


    ///////////////////
    // TCP - 2x Sockets Bind to same IP and Port
    ///////////////////

    NN_LOG( "TCP - Two different sockets Bind to (Same Address and Port):  ERRNO = nn::socket::Errno::EAddrInUse\n" );

    // Server: Create TCP Socket
    tcpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7050 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // First Bind:
    rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind failed but Success was expected! errno: %d", nn::socket::GetLastError() );

    // Client: Create TCP Socket
    tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Client: (SELF) Address
    memset( &localAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port        = nn::socket::InetHtons( 7050 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client Second Bind:
    rc = nn::socket::Bind( tcpSock, (nn::socket::SockAddr *) &targetAddr, sizeof( localAddr ) );
    ERROR_IF( rc> -1, "Bind suceeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAddrInUse, "I expected nn::socket::Errno::EAddrInUse - but I actually got errno: %d!", myError );

    // Close TCP socket
    nn::socket::Close( tcpServSock );
    tcpServSock = -1;
    nn::socket::Close( tcpSock );
    tcpSock = -1;

    ///////////////////
    // UDP - 2x Sockets Bind to same IP and Port
    ///////////////////

    NN_LOG( "UDP - 2x different sockets Bind to (Same Address and Port):  ERRNO = nn::socket::Errno::EAddrInUse\n" );

    // Server: Create UDP Socket
    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 7050 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // First Bind:
    rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind failed but Success was expected! errno: %d", nn::socket::GetLastError() );

    // Client: Create UDP Socket
    udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Client: (SELF) Address
    memset( &localAddr, 0, sizeof(targetAddr) );
    targetAddr.sin_port        = nn::socket::InetHtons( 7050 );
    targetAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Client: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Client Second Bind:
    rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &targetAddr, sizeof( localAddr ) );
    ERROR_IF( rc> -1, "Bind suceeded but should have failed!\n" );

    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EAddrInUse, "I expected nn::socket::Errno::EAddrInUse - but I actually got errno: %d!", myError );

    // Close TCP socket
    nn::socket::Close( udpServSock );
    udpServSock = -1;
    nn::socket::Close( udpSock );
    udpSock = -1;

    ///////////////////
    // TCP - Bind to a reserved port
    ///////////////////

    NN_LOG( "TCP - Bind to a reserved port (80 == HTTP):  Works for WINSOCK!\n" );

    // Server: Create TCP Socket
    tcpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
    ERROR_IF( tcpServSock < 0, "Failed calling Socket for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 80 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server: Bind
    rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind failed with errno: %d\n", nn::socket::GetLastError() );

    // Close TCP socket
    nn::socket::Close( tcpServSock );
    tcpServSock = -1;


    ///////////////////
    // UDP - Bind to a reserved port - ** WORKS ** Under Windows
    ///////////////////

    NN_LOG( "UDP - Bind to a reserved port (80):  -- Works for WINSOCK\n" );

    // Server: Create UDP Socket
    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
    ERROR_IF( udpServSock < 0, "Failed calling Socket for UDP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 80 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server: Bind
    rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc < 0, "Bind failed with errno: %d\n", nn::socket::GetLastError() );

    // Close TCP socket
    nn::socket::Close( udpServSock );
    udpServSock = -1;


    ///////////////////
    // TCP - Bind to a closed socket - Errno == nn::socket::Errno::EInval
    ///////////////////

    NN_LOG( "TCP - Bind to a closed socket - Errno == nn::socket::Errno::EInval\n" );

    // Server: Create TCP Socket
    tcpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );

    // Close the socket
    nn::socket::Close( tcpServSock );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 80 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server: Bind
    rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc > -1, "Bind Succeeded but fail was expected!" );

    // Server: check errno
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EBadf, "I expected nn::socket::Errno::EInval - but I actually got errno: %d!", myError );

    // Close TCP socket
    tcpServSock = -1;


    ///////////////////
    // UDP - Bind to a closed socket - Errno == nn::socket::Errno::EInval
    ///////////////////

    NN_LOG( "UDP - Bind to a closed socket - Errno == nn::socket::Errno::EBadf\n" );

    // Server: Create TCP Socket
    udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );

    // Close the socket
    nn::socket::Close( udpServSock );

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 80 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    // Server: Bind
    rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
    ERROR_IF( rc > -1, "Bind Succeeded but fail was expected!" );

    // Server: check errno
    myError = nn::socket::GetLastError();
    ERROR_IF_AND_COUNT( myError != nn::socket::Errno::EBadf, "I expected nn::socket::Errno::EBadf - but I actually got errno: %d!", myError );

    // Close TCP socket
    udpServSock = -1;

    ///////////////////
    // TCP - Bind to NIFM Interface
    ///////////////////

    NN_LOG( "TCP - Bind to NIFM Interface\n" );

    {
        nn::socket::InAddr myIp   = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw   = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

        // Server Sock
        tcpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
        ERROR_IF( tcpServSock < 0, "Failed to create Socket - errno: %d\n", nn::socket::GetLastError() );

        // Server - Local Address
        memset( &localAddr, 0, sizeof(localAddr));
        localAddr.sin_port        = nn::socket::InetHtons( 9300 );
        localAddr.sin_family      = nn::socket::Family::Af_Inet;

        NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
        localAddr.sin_addr.S_addr = myIp.S_addr;

        // Server: Bind
        rc = nn::socket::Bind( tcpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF_AND_COUNT( rc < 0, "Failed calling Bind for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

        // Server Listen
        rc = nn::socket::Listen( tcpServSock, 5 );
        ERROR_IF( rc < 0, "Failed calling Listen for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

        // Server Sock
        tcpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );
        ERROR_IF( tcpSock < 0, "Failed to create Socket - errno: %d\n", nn::socket::GetLastError() );

        // Client Connect:
        rc = nn::socket::Connect( tcpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc < 0, "Client Connet to Servcer failed -- errno: d\n", nn::socket::GetLastError() );

        // Server: Accept
        acceptedSock = nn::socket::Accept( tcpServSock, static_cast<nn::socket::SockAddr*>(nullptr), 0 );
        ERROR_IF( acceptedSock < 0, "Failed calling Accept for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

        // Client: Write a Message
        rc = nn::socket::Write( tcpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
        ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to write message of size: %u\n", strlen( msgs[0] ) );

        // Server Read a Message
        rc = nn::socket::Read( acceptedSock, (unsigned char *) buf1, sizeof( buf1 ) );
        ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to read message of size: %u, actually got: %d\n", strlen( msgs[0] ), rc );
        ERROR_IF( strncmp( buf1, msgs[0], rc ) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );
        NN_LOG( "Server Recv buf1 = [%.*s]\n", rc, buf1);

        // Server: Write a Message
        rc = nn::socket::Write( acceptedSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
        ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to write message of size: %u\n", strlen( msgs[0] ) );

        // Client: Read a Message
        rc = nn::socket::Read( tcpSock, (unsigned char *) buf1, sizeof( buf1 ) );
        ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to read message of size: %u, actually got: %d\n", strlen( msgs[0] ), rc );
        ERROR_IF( strncmp( buf1, msgs[0], rc ) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );
        NN_LOG( "Client Recv buf1 = [%.*s]\n", rc, buf1);

        // Close Sockets
        nn::socket::Close( tcpSock );
        tcpSock = -1;
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
        nn::socket::Close( tcpServSock );
        tcpServSock = -1;

        nTestsPassing++;
    }

    ///////////////////
    // UDP - Bind to NIFM Interface
    ///////////////////

    NN_LOG( "UDP - Bind to NIFM Interface\n" );

    {
        nn::socket::InAddr myIp   = { 0 };
        nn::socket::InAddr myMask = { 0 };
        nn::socket::InAddr myGw   = { 0 };
        nn::socket::InAddr myDns1 = { 0 };
        nn::socket::InAddr myDns2 = { 0 };

       // Ask NIFM for our current (primary interface) IP Address
       nn::nifm::GetCurrentIpConfigInfo( &myIp, &myMask, &myGw, &myDns1, &myDns2 );

       // UDP Server Sock
       udpServSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
       ERROR_IF( udpServSock < 0, "Failed to create Socket - errno: %d\n", nn::socket::GetLastError() );

       // UDP Server - Local Address
       memset( &localAddr, 0, sizeof(localAddr));
       localAddr.sin_port        = nn::socket::InetHtons( 9250 );
       localAddr.sin_family      = nn::socket::Family::Af_Inet;

       NN_LOG( "My IP Address: %08x\n", myIp.S_addr );
       localAddr.sin_addr.S_addr = myIp.S_addr;

       // Server: Bind
       rc = nn::socket::Bind( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
       ERROR_IF_AND_COUNT( rc < 0, "Failed calling Bind for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

       // Establish Target Port
       localAddr.sin_port        = nn::socket::InetHtons( 9251 );

       // Server Connect:
       rc = nn::socket::Connect( udpServSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
       ERROR_IF( rc < 0, "Failed to connect UDP Server Socket to UDP socket !  Errno: %d\n", nn::socket::GetLastError() );

       // UDP Client Sock
       udpSock = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
       ERROR_IF( udpSock < 0, "Failed to create Socket - errno: %d\n", nn::socket::GetLastError() );

       // Establish Source Port
       localAddr.sin_port        = nn::socket::InetHtons( 9251 );

       // Client: Bind
       rc = nn::socket::Bind( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
       ERROR_IF_AND_COUNT( rc < 0, "Failed calling Bind for TCP Server Socket, Errno: %d\n", nn::socket::GetLastError() );

       // Establish Target Port
       localAddr.sin_port        = nn::socket::InetHtons( 9250 );

       // Client Connect:
       rc = nn::socket::Connect( udpSock, (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
       ERROR_IF( rc < 0, "Failed to connect UDP Server Socket to UDP socket !  Errno: %d\n", nn::socket::GetLastError() );

       // Client: Write a Message
       rc = nn::socket::Write( udpSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
       ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to write message of size: %u\n", strlen( msgs[0] ) );

       // Server Read a Message
       rc = nn::socket::Read( udpServSock, (unsigned char *) buf1, sizeof( buf1 ) );
       ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to read message of size: %u, actually got: %d\n", strlen( msgs[0] ), rc );
       ERROR_IF( strncmp( buf1, msgs[0], rc ) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );
       NN_LOG( "Server Recv buf1 = [%.*s]\n", rc, buf1);

       // Server: Write a Message
       rc = nn::socket::Write( udpServSock, (unsigned char *) msgs[0], strlen( msgs[0] ) );
       ERROR_IF( rc < 0, "Failed to write 18 bytes, errno: %d\n", nn::socket::GetLastError() );
       ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to write message of size: %u, actually wrote: %d\n", strlen( msgs[0] ), rc );

       // Client: Read a Message
       rc = nn::socket::Read( udpSock, (unsigned char *) buf1, sizeof( buf1 ) );
       ERROR_IF( rc != (int) strlen( msgs[0] ), "Failed to read message of size: %u, actually got: %d\n", strlen( msgs[0] ), rc );
       ERROR_IF( strncmp( buf1, msgs[0], rc ) != 0, "Did not receive message [%.*s], Got [%.*s]\n", rc, msgs[0], rc, buf1 );
       NN_LOG( "Client Recv buf1 = [%.*s]\n", rc, buf1);

       // Close Sockets
       nn::socket::Close( udpSock );
       udpSock = -1;
       nn::socket::Close( udpServSock );
       udpServSock = -1;

        nTestsPassing++;
    }


out:

    if ( acceptedSock > -1 )
    {
        nn::socket::Close( acceptedSock );
        acceptedSock = -1;
    }

    if ( tcpSock > -1 )
    {
        nn::socket::Close( tcpSock );
        tcpSock = -1;
    }

    if ( udpSock > -1 )
    {
        nn::socket::Close( udpSock );
        udpSock = -1;
    }

    if ( tcpServSock > -1 )
    {
        nn::socket::Close( tcpServSock );
        tcpServSock = -1;
    }

    if ( udpServSock > -1 )
    {
        nn::socket::Close( udpServSock );
        udpServSock = -1;
    }

}   // NOLINT(impl/function_size)


TEST(ApiUnit,Win_Bind_With_SO_REUSEADDR)
{

WRAP_FAILING_TEST( "SIGLO-69331", "Socket API: Socket API: Windows nn::socket::Option::So_ReuseAddr not working" )
{
    nn::socket::TimeVal  myTimeout;
    nn::socket::SockAddrIn   localAddr = { 0 };
    int                  sockArr[15], cliArr[15];
    int                  rc = -1, n;
    int                  idx, idx1, bestFds = 0;
    bool                 isSuccess = true;
    int                  optVal;
    nn::socket::FdSet    readSet, writeSet, exceptSet;

    // Clear Socket Array
    for( idx = 0; idx < 15; idx++ )
    {
        sockArr[idx] = -1;
    }

    // For I = 1 to 15
    for( idx = 0; idx < 15; idx++ )
    {
        sockArr[idx] = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp );

        // Server: (SELF) Address
        memset( &localAddr, 0, sizeof(localAddr) );
        localAddr.sin_port        = nn::socket::InetHtons( 9000 );
        localAddr.sin_family      = nn::socket::Family::Af_Inet;

        // Server: Translate Target IP Address into Network Address
        rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
        ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

        // Server: Set nn::socket::Option::So_ReusePort
        optVal = 1;
        rc = nn::socket::SetSockOpt( sockArr[idx], nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, (void *) &optVal, sizeof( optVal ) );
        ERROR_IF( rc < 0, "SetSockOpt nn::socket::Option::So_ReuseAddr failed!" );

        // Server: Bind
        rc = nn::socket::Bind( sockArr[idx], (nn::socket::SockAddr *) &localAddr, sizeof( localAddr ) );
        ERROR_IF( rc < 0, "Bind failed with errno: %d\n", nn::socket::GetLastError() );

        // Create Listen Port
        rc = nn::socket::Listen( sockArr[idx], 1 );
        ERROR_IF( rc < 0, "Listen Failed: Errno: %d\n", nn::socket::GetLastError() );
    }

    nn::socket::FdSetZero( &readSet );
    nn::socket::FdSetZero( &writeSet );
    nn::socket::FdSetZero( &exceptSet );

    for( idx1 = 0; idx1 < 3; idx1++ )
    {
        for( idx = 0; idx < 5; idx++ )
        {
            cliArr[idx] = NATF::API::COMMON::MakeTCPConnection( "127.0.0.1", 9000 );
            ERROR_IF( cliArr[idx] < 0, "Connect Failed: Errno: %d\n", nn::socket::GetLastError() );
        }

        bestFds = 0;
        for( idx = 0; idx < 15; idx++ )
        {
            if ( sockArr[idx] > -1 )
            {
                if ( bestFds < sockArr[idx] ) bestFds = sockArr[idx];

                nn::socket::FdSetSet( sockArr[idx], &readSet );
                nn::socket::FdSetSet( sockArr[idx], &writeSet );
                nn::socket::FdSetSet( sockArr[idx], &exceptSet );
            }
        }

        myTimeout.tv_sec  = 1;
        myTimeout.tv_usec = 0;

        // Call Select
        n = nn::socket::Select( bestFds + 1, &readSet, &writeSet, &exceptSet, &myTimeout );
        NN_LOG( "N = %d\n", n );
        ERROR_IF( n < 0,  "Select Failed: errno: %d\n", nn::socket::GetLastError() );
        ERROR_IF( n == 0, "Select timed out - but was not supposed to!\n" );

        for( idx = 0; idx < 15; idx++ )
        {
            if ( nn::socket::FdSetIsSet( sockArr[idx], &readSet ) != 0 )
            {
                NN_LOG( "TCP Listen Socket: %d is ready to read!\n", sockArr[idx] );
            }

            if ( nn::socket::FdSetIsSet( sockArr[idx], &writeSet ) != 0 )
            {
                NN_LOG( "TCP Listen Socket: %d is ready to write!\n", sockArr[idx] );
            }

            if ( nn::socket::FdSetIsSet( sockArr[idx], &exceptSet ) != 0 )
            {
                NN_LOG( "TCP Listen Socket: %d has an except set!\n", sockArr[idx] );
            }
        }

        for( idx = 0; idx < 15; idx++ )
        {
            if ( cliArr[idx] > -1 )
            {
                nn::socket::Close( cliArr[idx] );
                cliArr[idx] = -1;
            }
        }

    }

    nTestsPassing++;

out:

    // Clear Socket Array
    for( idx = 0; idx < 15; idx++ )
    {
        nn::socket::Close( sockArr[idx] );
        sockArr[idx] = -1;
    }

} // Wrap Failing Test Case


}

namespace {

struct PAYLOAD_API_TEST_BIND
{
    int                 threadId;
    nn::socket::SockAddrIn  sockAddr;
    int                 desiredCore;
};

}  // Anonymous Namespace


void
Win_Bind_Contest_Thread( void * inArg )
{
    int                            rc = 0, sockfds = -1;
    int                            idx;
    bool                           isSuccess = true;
    struct PAYLOAD_API_TEST_BIND * payload;

    payload = (struct PAYLOAD_API_TEST_BIND *) inArg;

    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), nn::os::IdealCoreUseDefaultValue, payload->desiredCore );

    idx      = 9000;
    while ( g_shouldRun == true )
    {
        // Pick a port inrange
        idx++;
        if ( idx > 9015 ) idx = 9000;

        // Set the Port
        payload->sockAddr.sin_port = nn::socket::InetHtons( (uint16_t) idx );

        // Give me a socket...
        sockfds = nn::socket::Socket( nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp );
        ERROR_IF( sockfds < 0, "Failed to create UDP socket - Errno: %d\n", nn::socket::GetLastError() );

        rc = nn::socket::Bind( sockfds, (nn::socket::SockAddr *) &payload->sockAddr, sizeof( nn::socket::SockAddrIn ) );
        if ( rc < 0 )
        {
            /*NN_LOG( "[%d]: Lose, Port [%d] with Errno: %d\n", payload->threadId,
                                                   nn::socket::InetNtohs( payload->sockAddr.sin_port ),
                                                   nn::socket::GetLastError() );*/
            nn::socket::Close( sockfds );
            sockfds = -1;

            continue;
        }

        NN_LOG( "[%d] Win !  Bound to Port [%d]\n", payload->threadId,
                                                    nn::socket::InetNtohs( payload->sockAddr.sin_port )  );

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

        nn::socket::Close( sockfds );
        sockfds = -1;
    }

    nTestsPassing++;

out:

    if ( sockfds > -1 )
    {
        nn::socket::Close( sockfds );
        sockfds = -1;
    }

}


TEST(ApiUnit,Win_Bind_Contest)
{
    nn::socket::SockAddrIn            localAddr = { 0 };
    int                           rc = -1;
    int                           idx;
    int                           core = 1;
    bool                          isSuccess = true;
    struct PAYLOAD_API_TEST_BIND  payload[g_kNumWorkerThreads];

    // Server: (SELF) Address
    memset( &localAddr, 0, sizeof(localAddr) );
    localAddr.sin_port        = nn::socket::InetHtons( 9050 );
    localAddr.sin_family      = nn::socket::Family::Af_Inet;

    // Server: Translate Target IP Address into Network Address
    rc = nn::socket::InetPton( nn::socket::Family::Af_Inet, "127.0.0.1", &localAddr.sin_addr.S_addr );
    ERROR_IF( rc != 1, "InetPton() Failed but Success was expected!" );

    g_shouldRun = true;

    for( idx = 0; idx < g_kNumWorkerThreads; idx++ )
    {
        memcpy( &payload[idx].sockAddr, &localAddr, sizeof( nn::socket::SockAddrIn ) );

        // Desired Core
        core = core * 2;
        if ( core > 8 ) core = 1;

        // Save Core
        payload[idx].desiredCore = core;

        payload[idx].threadId = g_pThreadedTest->CreateThread( &Win_Bind_Contest_Thread, (void *) &localAddr, kThreadPriority );
        if ( payload[idx].threadId < 0 )
        {
            NN_LOG( "Failed to create and start Bind Contest Thread (thread)!\n" );
            goto out;
        }
    }

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));

    NN_LOG( "Stopping Threads!\n" );
    g_shouldRun = false;

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    for( idx = 0; idx < g_kNumWorkerThreads; idx++ )
    {
        g_pThreadedTest->WaitThread( payload[idx].threadId );
        g_pThreadedTest->DestroyThread( payload[idx].threadId );
        payload[idx].threadId = -1;
    }

    nTestsPassing++;

out:

    for( idx = 0; idx < g_kNumWorkerThreads; idx++ )
    {
        if ( payload[idx].threadId > -1 )
        {
            g_pThreadedTest->WaitThread( payload[idx].threadId );
            g_pThreadedTest->DestroyThread( payload[idx].threadId );
        }
    }
}

TEST(ApiUnit,Win_Bind_Teardown)
{
    TeardownTesting();
}

}}  // Namespace: NATF::API
