﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket.h> // nn::socket
#include <nn/nn_Log.h> // NN_LOG
#include <nn/util/util_Uuid.h> // nn::util::Uuid

#include <cstdlib> // strtol
#include <cstdio> // sprintf
#include <cstring> // strcpy

#include "Unit/testNet_ApiUnitCommon.h"


namespace NATF {
namespace API {

    const char *namesUrlCharValidStatesEnum[] =
    {
        "UrlCharValid_GenDelims",
        "UrlCharValid_SubDelims",
        "UrlCharValid_Unreserved",
        "UrlCharValid_NeedPctEncoding"
    };

    int nTestsPassing;
    int nTestsFailing;
    int nTestsSkipped;

    bool ConvertToValidFormat(char *buffer, int len) NN_NOEXCEPT
    {
        bool isSuccess = true;

        char *pMatch = nullptr;

        ERROR_IF(nullptr == buffer, "Precondition failed: buffer null");

        while( nullptr != (pMatch = reinterpret_cast<char *>(memchr(buffer, ' ', len))) )
        {
            *pMatch = '_';
        }

    out:
        return isSuccess;
    }

    bool BuildAndPrintJiraMessage(const char *name, const char *value) NN_NOEXCEPT
    {
        bool isSuccess = true;
        int rval;
        char printBuffer[256];

        const char baseMessage[] = "##teamcity[failingTestJira name='%s' value='%s']\n";

        char nameBuffer[32];
        char valueBuffer[128];

        ERROR_IF(nullptr == name, "Precondition failed: name null");
        ERROR_IF(nullptr == value, "Precondition failed: value null");

        strncpy(nameBuffer, name, 31);
        nameBuffer[31] = '\0';
        strncpy(valueBuffer, value, 127);
        valueBuffer[127] = '\0';

        ERROR_IF(!ConvertToValidFormat(nameBuffer, 32), "Step failed: ConvertToValidFormat(name) failed");
        ERROR_IF(!ConvertToValidFormat(valueBuffer, 128), "Step failed: ConvertToValidFormat(value) failed");

        rval = sprintf(printBuffer, baseMessage, nameBuffer, valueBuffer);
        ERROR_IF(0 > rval, "Step failed: sprintf failed");

        NN_LOG(printBuffer);

    out:
        return isSuccess;
    }

    bool VerifyIpAddrEquiv(const char *addrAscii, const nn::socket::InAddr *pAddrNetwork) NN_NOEXCEPT
    {
        bool isSuccess = true;

        char srcCpy[256];
        char *strtokSrc;

        ERROR_IF(nullptr == addrAscii, "VerifyIpAddrEquiv precondition not met: addrAscii null");
        ERROR_IF(nullptr == pAddrNetwork, "VerifyIpAddrEquiv precondition not met: pAddrNetwork null");

        strncpy(srcCpy, addrAscii, 255);
        srcCpy[255] = '\0';
        strtokSrc = &srcCpy[0];
        for( int i = 0; i < 4; ++i )
        {
            char *srcToken = strtok(strtokSrc, ".");
            if( nullptr == srcToken )
            {
                NN_LOG("VerifyIpAddrEquiv i=%d, addrAscii=%s\n", i, addrAscii);
                ERROR_IF(NN_STATIC_CONDITION(true), "VerifyIpAddrEquiv failed (addrAscii is missing a period).");
            }

            long int srcTokenLint = strtol(srcToken, nullptr, 10);
            long int dstTokenLint = (pAddrNetwork->S_addr >> (i * 8)) & 0xFF;
            if( srcTokenLint != dstTokenLint )
            {
                NN_LOG("VerifyIpAddrEquiv i=%d, addrAscii=%s, pAddrNetwork->S_addr=0x%X, srcTokenLint=%d, dstTokenLint=%d\n", i, addrAscii, pAddrNetwork->S_addr, srcTokenLint, dstTokenLint);
                ERROR_IF(NN_STATIC_CONDITION(true), "VerifyIpAddrEquiv failed (result mismatch).");
            }

            strtokSrc = nullptr;
        }

out:
        return isSuccess;
    }

    bool VerifyUrlCharValid(int *rval, char c) NN_NOEXCEPT
    {
        bool isSuccess = true;

        // RFC3986: Uniform Resource Identifier (URI): Generic Syntax
        // See http://www.ietf.org/rfc/rfc3986.txt

        // RFC3986 Section 2.2 Reserved Characters
        const char charsGenDelims[] = ":/?#[]@";
        const char charsSubDelims[] = "!$&'()*+,;=";

        // RFC3986 Section 2.3 Unreserved Characters
        const char charsUnreserved[] = "-._~"; // also, A-Z,a-z,0-9

        ERROR_IF(nullptr == rval, "VerifyUrlCharValid: rval nullptr");

        if( nullptr != memchr(charsGenDelims, c, strlen(charsGenDelims)) )
        {
            *rval = static_cast<int>(UrlCharValid_GenDelims);
        }
        else if( nullptr != memchr(charsSubDelims, c, strlen(charsSubDelims)) )
        {
            *rval = static_cast<int>(UrlCharValid_SubDelims);
        }
        else if( (nullptr != memchr(charsUnreserved, c, strlen(charsUnreserved)))
            || (('0' <= c) && ('9' >= c))
            || (('a' <= c) && ('z' >= c))
            || (('A' <= c) && ('Z' >= c)) )
        {
            *rval = static_cast<int>(UrlCharValid_Unreserved);
        }
        else
        {
            *rval = static_cast<int>(UrlCharValid_NeedPctEncoding);
        }

out:
        return isSuccess;
    }

    bool FindStrInList(bool *rval, const char** list, int listSize, const char *str) NN_NOEXCEPT
    {
        bool isSuccess = true;

        char nameBuffer[512];

        ERROR_IF(nullptr == rval, "FindStrInList precondition failed (rval nullptr)");
        ERROR_IF(nullptr == list, "FindStrInList precondition failed (list nullptr)");
        for( int i = 0; i < listSize; ++i )
        {
            ERROR_IF(0 > sprintf(nameBuffer, "FindStrInList precondition failed (list[%d] nullptr)", i), "FindStrInList: sprintf failed");
            ERROR_IF(nullptr == list[i], nameBuffer);
        }
        ERROR_IF(nullptr == str, "FindStrInList precondition failed (str nullptr)");

        *rval = false;
        for( int i = 0; i < listSize; ++i )
        {
            if( 0 == strcmp(list[i], str) )
            {
                *rval = true;
                break;
            }
        }

out:
        return isSuccess;
    }

    bool VerifyHostNetAddrEquiv(uint16_t hostAddr, uint16_t networkAddr) NN_NOEXCEPT
    {
        bool isSuccess = true;

        int numberOfBytes = sizeof(hostAddr);
        char printBuffer[512];

        for( int i = 0; i < numberOfBytes; ++i )
        {
            char hostByte = 0xFF & (hostAddr >> (8 * i));
            char networkByte = 0xFF & (networkAddr >> (8 * (numberOfBytes - 1 - i)));

            ERROR_IF(0 > sprintf(printBuffer, "VerifyHostNetAddrEquiv: hostAddr=%d, networkAddr=%d, i=%d, hostByte=%c, networkByte=%c", hostAddr, networkAddr, i, hostByte, networkByte), "sprintf failed");
            ERROR_IF(hostByte != networkByte, printBuffer);
        }

out:
        return isSuccess;
    }

    bool VerifyHostNetAddrEquiv(uint32_t hostAddr, uint32_t networkAddr) NN_NOEXCEPT
    {
        bool isSuccess = true;

        int numberOfBytes = sizeof(hostAddr);
        char printBuffer[512];

        for( int i = 0; i < numberOfBytes; ++i )
        {
            char hostByte = 0xFF & (hostAddr >> (8 * i));
            char networkByte = 0xFF & (networkAddr >> (8 * (numberOfBytes - 1 - i)));

            ERROR_IF(0 > sprintf(printBuffer, "VerifyHostNetAddrEquiv: hostAddr=%d, networkAddr=%d, i=%d, hostByte=%c, networkByte=%c", hostAddr, networkAddr, i, hostByte, networkByte), "sprintf failed");
            ERROR_IF(hostByte != networkByte, printBuffer);
        }

out:
        return isSuccess;
    }

    bool PrintSockaddrIn(nn::socket::SockAddrIn *pSockaddrIn) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("==PrintSockaddrIn: In==\n");

        ERROR_IF(nullptr == pSockaddrIn, "Precondition failed (pHostent nullptr)");

        NN_LOG("pSockaddrIn->sin_family: value=%d\n", pSockaddrIn->sin_family);
        NN_LOG("pSockaddrIn->sin_port: value=%d\n", pSockaddrIn->sin_port);
        NN_LOG("pSockaddrIn->sin_addr.S_addr: value=%d\n", pSockaddrIn->sin_addr.S_addr);

out:
        NN_LOG("==PrintSockaddrIn: Out==\n");
        return isSuccess;
    }

    bool PrintSockaddrIn(sockaddr_in *pSockaddrIn) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("==PrintSockaddrIn: In==\n");

        ERROR_IF(nullptr == pSockaddrIn, "Precondition failed (pHostent nullptr)");

        NN_LOG("pSockaddrIn->sin_family: value=%d\n", pSockaddrIn->sin_family);
        NN_LOG("pSockaddrIn->sin_port: value=%d\n", pSockaddrIn->sin_port);
        NN_LOG("pSockaddrIn->sin_addr.S_addr: value=%d\n", pSockaddrIn->sin_addr.s_addr);

out:
        NN_LOG("==PrintSockaddrIn: Out==\n");
        return isSuccess;
    }

    bool PrintHostent(nn::socket::HostEnt *pHostent) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("==PrintHostent: In==\n");

        ERROR_IF(nullptr == pHostent, "Precondition failed (pHostent nullptr)");

        NN_LOG("pHostent->h_name: len=%d, str=%s\n", strlen(pHostent->h_name), pHostent->h_name);
        if( nullptr != pHostent->h_aliases )
        {
            for( int i = 0; nullptr != pHostent->h_aliases[i]; ++i )
            {
                NN_LOG("pHostent->h_aliases[%d]: len=%d, str=%s\n", i, strlen(pHostent->h_aliases[i]), pHostent->h_aliases[i]);
            }
        }
        NN_LOG("pHostent->h_addrtype=%d\n", pHostent->h_addrtype);
        NN_LOG("pHostent->h_length=%d\n", pHostent->h_length);
        if( nullptr != pHostent->h_addr_list )
        {
            for( int i = 0; nullptr != pHostent->h_addr_list[i]; ++i )
            {
                NN_LOG("pHostent->h_addr_list[%d]: len=%d, str=%s\n", i, strlen(pHostent->h_addr_list[i]), pHostent->h_addr_list[i]);
            }
        }

out:
        NN_LOG("==PrintHostent: Out==\n");
        return isSuccess;
    }

    bool PrintHostent(hostent *pHostent) NN_NOEXCEPT
    {
        bool isSuccess = true;

        NN_LOG("==PrintHostent: In==\n");

        ERROR_IF(nullptr == pHostent, "Precondition failed (pHostent nullptr)");

        NN_LOG("pHostent->h_name: len=%d, str=%s\n", strlen(pHostent->h_name), pHostent->h_name);
        if( nullptr != pHostent->h_aliases )
        {
            for( int i = 0; nullptr != pHostent->h_aliases[i]; ++i )
            {
                NN_LOG("pHostent->h_aliases[%d]: len=%d, str=%s\n", i, strlen(pHostent->h_aliases[i]), pHostent->h_aliases[i]);
            }
        }
        NN_LOG("pHostent->h_addrtype=%d\n", pHostent->h_addrtype);
        NN_LOG("pHostent->h_length=%d\n", pHostent->h_length);
        if( nullptr != pHostent->h_addr_list )
        {
            for( int i = 0; nullptr != pHostent->h_addr_list[i]; ++i )
            {
                NN_LOG("pHostent->h_addr_list[%d]: len=%d, str=%s\n", i, strlen(pHostent->h_addr_list[i]), pHostent->h_addr_list[i]);
            }
        }

out:
        NN_LOG("==PrintHostent: Out==\n");
        return isSuccess;
    }

    bool SetupSockaddrIn(nn::socket::SockAddrIn *pAddr, nn::socket::Family family, uint16_t portHostFormat, int addrHostFormat) NN_NOEXCEPT
    {
        bool isSuccess = true;

        ERROR_IF(nullptr == pAddr, "SetupSockaddrIn precondition failed: pAddr null");

        pAddr->sin_family      = family;
        pAddr->sin_port        = nn::socket::InetHtons(portHostFormat);
        pAddr->sin_addr.S_addr = nn::socket::InetHtonl(addrHostFormat);

out:
        return isSuccess;
    }

    bool SetupSockaddrIn(nn::socket::SockAddrIn *pAddr, nn::socket::Family family, uint16_t portHostFormat, const char *addrAsciiFormat) NN_NOEXCEPT
    {
        bool isSuccess = true;

        ERROR_IF(nullptr == pAddr, "SetupSockaddrIn precondition failed: pAddr null");

        pAddr->sin_family      = family;
        pAddr->sin_port        = nn::socket::InetHtons(portHostFormat);
        ERROR_IF(1 > nn::socket::InetPton(family, addrAsciiFormat, &pAddr->sin_addr), "SetupSockaddrIn: InetPton failed");
        memset(pAddr->sin_zero, '\0', sizeof(pAddr->sin_zero));

out:
        return isSuccess;
    }

    bool CloseAllSockets() NN_NOEXCEPT
    {
        bool isSuccess = true;

        for( int i = 0; i < 32; ++i )
        {
            if( -1 == nn::socket::Close(i) )
            {
                EXPECT_ERRNO_NX(nn::socket::Errno::ENotSock);
                EXPECT_ERRNO_WIN(nn::socket::Errno::ENotSock);
            }
        }

out:
        return isSuccess;
    }

}} // namespace NATF::API
