﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/socket.h>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nnt.h>

#include "testNet_ApiCommon.h"
#include "Complex/testNet_SelectUnitData.h"
#include "Complex/testNet_UnitCommon.h"
#include "Complex/testNet_SelectUnitServer.h"
#include "Complex/testNet_SelectUnitClient.h"

namespace NATF {
namespace API {

NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

void RunUnitTest(int numRead, int numWrite, int numException, SocketControlFlags controlFlags, uint64_t timeoutValue)
{
    nn::Result result;

    result = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
                                    nn::socket::DefaultSocketMemoryPoolSize,
                                    nn::socket::MinSocketAllocatorSize,
                                    nn::socket::DefaultConcurrencyLimit);

    if( result.IsFailure() )
    {
        NN_LOG("\nError: nn::socket::Initialize() failed. Err Desc: %d\n\n", result.GetDescription());
        goto bail;
    }
    else
    {
        // we use the AutoReleaseObject to ensure that if there is a failure (i.e. an exception) that we
        // release allocated resources (sockets, threads, etc.) if an exception is thrown
        AutoReleaseObject<SimpleValidator> validator(new SimpleValidator(), false);
        AutoReleaseObject<SelectUnitData> unitData(new SelectUnitData(ADDRESS_FAMILY_FLAGS_AF_INET, SOCK_TYPE_FLAGS_SOCK_STREAM, PROTOCOL_FLAGS_DEFAULT,
                                                                      controlFlags,
                                                                      numRead, numWrite, numException,
                                                                      &(*validator), timeoutValue),
                                                   false);
        AutoReleaseObject<SelectUnitServer> server(new SelectUnitServer(&(*unitData)), false);
        AutoReleaseObject<SelectUnitClient> client(new SelectUnitClient(&(*unitData)), false);

        (*server).SetPeer(&(*client));
        (*client).SetPeer(&(*server));

        AutoReleaseObject<TestManagerThread> testManagerThread(new TestManagerThread(&(*unitData), &(*server), &(*client)), false);

        (*testManagerThread).Start();
        (*testManagerThread).WaitForDone();

        if ( true == (*validator).DidFail() )
        {
            ADD_FAILURE();
        };
    };

bail:
    if (result.IsSuccess())
    {
        result = nn::socket::Finalize();
        if( result.IsFailure() )
        {
            NN_LOG("\nError: nn::socket::Finalize() failed. Further test results might be polluted. Err Desc: %d\n\n", result.GetDescription());
            goto bail;
        };
    };

    return;
}

// TEST WITH NO FILE DESCRIPTORS
TEST(SelectUnit, test_nb_0_0_0_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

// TEST WITH NO FILE DESCRIPTORS
TEST(SelectUnit, test0_0_0_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

// TESTS WITH 1 FILE DESCRIPTOR
// with NULL (i.e. forever) timeout
TEST(SelectUnit, test1_0_0_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 0, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test0_1_0_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 1, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test1_1_0_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 1, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test0_0_1_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 1, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test1_0_1_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 0, 1, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test0_1_1_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 1, 1, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test1_1_1_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 1, 1, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

// with 500ms timeout
TEST(SelectUnit, test1_0_0_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 0, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test0_1_0_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 1, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test1_1_0_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 1, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test0_0_1_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 1, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test1_0_1_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 0, 1, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test0_1_1_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 1, 1, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test1_1_1_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 1, 1, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

// TESTS WITH 2 FILE DESCRIPTORS
// with NULL (i.e. forever) timeout
TEST(SelectUnit, test2_0_0_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 0, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test0_2_0_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 2, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test2_2_0_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 2, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test0_0_2_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 2, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test2_0_2_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 0, 2, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test0_2_2_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 2, 2, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

TEST(SelectUnit, test2_2_2_default_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 2, 2, SOCKET_CONTROL_FLAGS_DEFAULT, 0);
};

// with 500ms timeout
TEST(SelectUnit, test2_0_0_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 0, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test0_2_0_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 2, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test2_2_0_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 2, 0, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test0_0_2_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 2, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test2_0_2_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 0, 2, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test0_2_2_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 2, 2, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

TEST(SelectUnit, test2_2_2_default_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 2, 2, SOCKET_CONTROL_FLAGS_DEFAULT, 500);
};

// NONBLOCKING TESTS WITH 1 FILE DESCRIPTOR
// with NULL (i.e. forever) timeout
TEST(SelectUnit, test1_0_0_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 0, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test0_1_0_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 1, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test1_1_0_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 1, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test0_0_1_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 1, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test1_0_1_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 0, 1, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test0_1_1_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 1, 1, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test1_1_1_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 1, 1, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

// with 500ms timeout
TEST(SelectUnit, test1_0_0_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 0, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test0_1_0_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 1, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test1_1_0_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 1, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test0_0_1_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 1, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test1_0_1_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 0, 1, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test0_1_1_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 1, 1, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test1_1_1_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(1, 1, 1, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};


// NONBLOCKING TESTS WITH 2 FILE DESCRIPTORS
// with NULL (i.e. forever) timeout
TEST(SelectUnit, test2_0_0_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 0, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test0_2_0_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 2, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test2_2_0_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 2, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test0_0_2_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 2, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test2_0_2_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 0, 2, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test0_2_2_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 2, 2, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

TEST(SelectUnit, test2_2_2_nonblocking_NULLtv)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 2, 2, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 0);
};

// with 500ms timeout

TEST(SelectUnit, test2_0_0_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 0, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test0_2_0_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 2, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test2_2_0_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 2, 0, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test0_0_2_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 0, 2, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test2_0_2_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 0, 2, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test0_2_2_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(0, 2, 2, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};

TEST(SelectUnit, test2_2_2_nonblocking_500ms)
{
    UNIT_TEST_TRACE("");
    RunUnitTest(2, 2, 2, SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING, 500);
};





}} // namespace NATF::API
