﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Complex/testNet_DnsBurn.h"
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#include <nn/socket/socket_ApiPrivate.h>
#include <nn/socket/socket_ResolverOptionsPrivate.h>
#include <nn/socket/resolver/resolver_Client.h>

namespace NATF {
namespace API {

std::list<int> g_CancelList;
std::mutex g_CancelListMutex;

void AddCancelHandleToList(int cancelHandle)
{
    std::lock_guard<std::mutex> lock(g_CancelListMutex);
    g_CancelList.push_back(cancelHandle);
}

void RemoveCancelHandleFromList(int cancelHandle)
{
    std::lock_guard<std::mutex> lock(g_CancelListMutex);
    std::list<int>::iterator iterator;
    for (iterator = g_CancelList.begin(); iterator != g_CancelList.end(); ++iterator)
    {
        std::cout << *iterator;
    }
}

void SleepOrFail(unsigned seconds, SimpleValidator& validator, bool doCancel=false)
{
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        if (true == validator.DidFail())
        {
            NN_LOG("Error\n", seconds);
            break;
        };

        NN_LOG("Seconds remaining: %d...\n", seconds);
    }
    while ( 0 != --seconds );

    if (true == doCancel)
    {
        std::lock_guard<std::mutex> lock(g_CancelListMutex);
        std::list<int>::iterator iterator;
        for (iterator = g_CancelList.begin(); iterator != g_CancelList.end(); ++iterator)
        {
            AttemptCancel( (*iterator), "Main Thread");
        };

        g_CancelList.clear();
    };

    return;
};

void AttemptCancel(int & cancelHandleInOut, const char* why)
{
    nn::socket::ResolverOption option = {
        .key = nn::socket::ResolverOptionKey::SetCancelHandleInteger,
        .type = nn::socket::ResolverOptionType::Integer,
        .size = sizeof(int),
        .data.integerValue = cancelHandleInOut
    };
    int rc = nn::socket::ResolverSetOption(option);
    NN_LOG("Cancel(%08x) for: %s, rc: %d, errno: %s (%d)\n",
           cancelHandleInOut, why, rc, strerror(errno), errno);
    cancelHandleInOut = 0;
};

/**
 * @param clientHostname the hostname for lookup
 * @param clientUseGetaddrinfo if true then use GetAddrInfo if false use GetHostByName
 * @param secondsToBurn the number of seconds that the test should run for
 * @param serverDelay amount of time to introduce between server read and response
 * @note the server is currently single-threaded, so balance the delay over the number of working threads
 * @param forceServerTcp set the truncation bit (TC=1) on UDP queries to force a TCP lookup
 * @param workingBufferSize set the SF client working buffer size; 0 for default
 */
template <unsigned numberOfWorkerThreads>
void RunDnsBurnUnitTest(const char* clientHostname,
                        bool clientUseGetAddrinfo,
                        unsigned secondsToBurn,
                        unsigned serverDelay,
                        bool forceServerTcp,
                        unsigned workingBufferSize)
{
    /* flag value for cleanup: did we acquire access to NIFM?*/
    bool shouldFinalizeTestSetup = false;
    int rc = -1;
    nn::Result result;
    nn::socket::SockAddrIn addr;

    DnsBurnTestGetServerSocketAddress(addr);
    nn::socket::PrivateDnsAddressArrayArgument arg = {
        .count = 1,
        .addresses[0] = addr
    };

    nn::socket::ResolverOption option = {
        .key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                              nn::socket::ResolverOptionPrivateKey::SetDnsServerAddressesPointer)),
        .type = nn::socket::ResolverOptionType::Pointer,
        .size = sizeof(arg),
        .data.pointerValue = reinterpret_cast<char*>(&arg)
    };

    if(!NATF::API::TestSetup(NATF::API::TestSetupOptions_Nifm | NATF::API::TestSetupOptions_Socket))
    {
        NN_LOG("Error: TestSetup failed.\n");
        goto bail;
    };
    shouldFinalizeTestSetup = true;


    if ( -1 == (rc = nn::socket::ResolverSetOption(option)))
    {
        NN_LOG("Error: Unable to SetDnsAddressesPrivate\n");
        goto bail;
    }

    if (0 != workingBufferSize)
    {
        option = {
            .key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                  nn::socket::ResolverOptionLocalKey::GetHostByNameBufferSizeUnsigned64)),
            .type = nn::socket::ResolverOptionType::Pointer,
            .size = sizeof(arg),
            .data.unsigned64Value = workingBufferSize
        };

        if (true == clientUseGetAddrinfo)
        {
            option.key = static_cast<nn::socket::ResolverOptionKey>(static_cast<uint32_t>(
                                                                        nn::socket::ResolverOptionLocalKey::GetAddrInfoBufferSizeUnsigned64));
        };

        if (-1 == (rc = nn::socket::ResolverSetOption(option)))
        {
            NN_LOG("Unable to set working buffer size\n");
        };
    }
    else
    {
        // shared object across multiple threads, auto release it out of scope
        AutoReleaseObject<SimpleValidator> validator(new SimpleValidator(), false);
        AutoReleaseObject<DnsBurnTestThread> serverThread(new DnsBurnTestThread(), false);
        AutoReleaseObject<DnsBurnTestThread> workerThreads[numberOfWorkerThreads];

        // initialize and start the server
        (*serverThread).InitializeServer(&(*validator),
                                         serverDelay,
                                         numberOfWorkerThreads,
                                         forceServerTcp);
        (*serverThread).Start();

        NN_LOG("Giving the server a few seconds to warm up...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

        // client thread object released when out of scope
        for (unsigned idx=0; idx < numberOfWorkerThreads ; ++idx)
        {
            workerThreads[idx] = new DnsBurnTestThread();
            (*workerThreads[idx]).InitializeClient(idx + 1,
                                                   &(*validator),
                                                   clientHostname,
                                                   clientUseGetAddrinfo,
                                                   numberOfWorkerThreads);
            (*workerThreads[idx]).Start();
        };

        SleepOrFail(secondsToBurn, (*validator) );

        // interrupt the worker threads
        for (unsigned idx=0; idx < numberOfWorkerThreads; ++idx)
        {
            (*workerThreads[idx]).Interrupt();
        };

        // wait for them to complete
        for (unsigned idx=0; idx < numberOfWorkerThreads; ++idx)
        {
            (*workerThreads[idx]).WaitForDone();
        };

        (*serverThread).Interrupt();
        (*serverThread).WaitForDone();

        // if we haven't already failed then we should have had at least 1
        // successful resolve
        if ( false == (*validator).DidFail() )
        {
            // non-fatal errors are treated as an attempt that did not succeed
            // we that every thread to have at least a single successful resolve
            // if this doesn't happen then it strongly suggests resource starvation
            for (unsigned idx=0; idx < numberOfWorkerThreads; ++idx)
            {
                unsigned total, success, failure;
                (*workerThreads[idx]).GetFinalStatistics(total, success, failure);

                SVALIDATE_FAIL((&(*validator)), success == 0,
                               "Thread %d: no successful resolutions "
                               "within %u seconds "
                               "total: %u, success: %u, failure: %u\n",
                               idx, secondsToBurn, total, success, failure);
            };
        };

        // get the failure code
        if ( true == (*validator).DidFail())
        {
            ADD_FAILURE();
        };
    }; // autorelease objects release

bail:
    if (true == shouldFinalizeTestSetup)
    {
        TestTeardown();
    };

    return;
}; //NOLINT(impl/function_size)

/**
 * server cancel / expire tests
 *
 * - client cancel exposed by setting client delay low and server delay higher;
 *          the main thread will Cancel the handle when done
 *
 * - server expire exposed by setting the server delay higher than the resolver
 *          cancel expiration time (8 seconds, but let's say 9 for rounding error)
 *          and the client delay higher still
 */

// gai
TEST(DnsBurn, 1ThreadsGetAddrInfoCancelUdp)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<1>(g_Host, true, 5, 1, false, 0);
};

TEST(DnsBurn, 1ThreadsGetAddrInfoExpireUdp)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<1>(g_Host, true, 10, 9, false, 0);
};

TEST(DnsBurn, 1ThreadsGetAddrInfoCancelTcp)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<1>(g_Host, true, 5, 9, true, 0);
};

TEST(DnsBurn, 1ThreadsGetAddrInfoExpireTcp)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<1>(g_Host, true, 10, 9, true, 0);
};

// ghbn
TEST(DnsBurn, 1ThreadsGetHostByNameCancelUdp)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<1>(g_Host, false, 5, 9, false, 0);
};

TEST(DnsBurn, 1ThreadsGetHostByNameExpireUdp)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<1>(g_Host, false, 10, 9, false, 0);
};

TEST(DnsBurn, 1ThreadsGetHostByNameCancelTcp)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<1>(g_Host, false, 5, 9, true, 0);
};

TEST(DnsBurn, 1ThreadsGetHostByNameExpireTcp)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<1>(g_Host, false, 10, 9, true, 0);
};

/**
 * 4-Thread Tests
 */
TEST(DnsBurn, 4ThreadsGetAddrInfo10SecondClient0SecondServerDelayUdpOnlyDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<4>(g_Host, true, 10, 0, false, 0);
};

TEST(DnsBurn, 4ThreadsGetHostByName10SecondClient0SecondServerDelayUdpOnlyDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<4>(g_Host, false, 10, 0, false, 0);
};

TEST(DnsBurn, 4ThreadsGetAddrInfo10SecondClient0SecondServerDelayForceServerTcpDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<4>(g_Host, true, 10, 0, true, 0);
};

TEST(DnsBurn, 4ThreadsGetHostByName10SecondClient0SecondServerDelayForceServerTcpDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<4>(g_Host, false, 10, 0, true, 0);
};

/**
 * 8 thread tests
 */
TEST(DnsBurn, 8ThreadsGetAddrInfo10SecondClient0SecondServerDelayUdpOnlyDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<8>(g_Host, true, 10, 0, false, 0);
};

TEST(DnsBurn, 8ThreadsGetHostByName10SecondClient0SecondServerDelayUdpOnlyDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<8>(g_Host, false, 10, 0, false, 0);
};

TEST(DnsBurn, 8ThreadsGetAddrInfo10SecondClient0SecondServerDelayTruncateUdpForceTcpDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<8>(g_Host, true, 10, 0, true, 0);
};

TEST(DnsBurn, 8ThreadsGetHostByName10SecondClient0SecondServerDelayTruncateUdpForceTcpDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<8>(g_Host, false, 10, 0, true, 0);
};

/**
 * 16 threads
 */
TEST(DnsBurn, 16ThreadsGetAddrInfo20SecondClient0SecondServerDelayUdpOnlyDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<16>(g_Host, true, 20, 0, false, 0);
};

TEST(DnsBurn, 16ThreadsGetHostByName20SecondClient0SecondServerDelayUdpOnlyDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<16>(g_Host, false, 20, 0, false, 0);
};

TEST(DnsBurn, 16ThreadsGetAddrInfo20SecondClient0SecondServerDelayTruncateUdpForceTcpDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<16>(g_Host, true, 20, 0, true, 0);
};

TEST(DnsBurn, 16ThreadsGetHostByName20SecondClient0SecondServerDelayTruncateUdpForceTcpDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<16>(g_Host, false, 20, 0, true, 0);
};

/**
 * 32 threads
 */
TEST(DnsBurn, 32ThreadsGetAddrInfo30SecondClient0SecondServerDelayUdpOnlyDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<32>(g_Host, true, 30, 0, false, 0);
};

TEST(DnsBurn, 32ThreadsGetHostByName30SecondClient0SecondServerDelayTruncateUdpOnlyDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<32>(g_Host, false, 30, 0, false, 0);
};

TEST(DnsBurn, 32ThreadsGetAddrInfo30SecondClient0SecondServerDelayTruncateUdpForceTcpDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<32>(g_Host, true, 30, 0, true, 0);
};

TEST(DnsBurn, 32ThreadsGetHostByName30SecondClient0SecondServerDelayTruncateUdpForceTcpDefaultWorkingBufferSize)
{
    UNIT_TEST_TRACE("");
    RunDnsBurnUnitTest<32>(g_Host, false, 30, 0, true, 0);
};

} // API
} // NATF
