﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Utils/InitApis.h"
#include "NetTest_Port.h"
#include "curl/curl.h"
#include <nn/ssl.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/nifm/nifm_ApiForTest.h>

namespace NATF {
namespace Utils {

    InitApi::InitApi(InitApiFlags apisToInit) NN_NOEXCEPT :
        m_apisToInit(apisToInit),
        m_apisAreInit(InitApiFlags::InitApiFlags_None),
        m_pNifmConnection(nullptr) {}

    InitApi::~InitApi() NN_NOEXCEPT
    {
        if( m_apisAreInit != InitApiFlags::InitApiFlags_None )
        {
            NATF_LOG(" * WARNING: Library Apis were not finalized! Finalizing through InitApi destructor\n\n");
            Finalize();
        }
    }

    bool InitApi::Init(const nn::util::Uuid& netProfile) NN_NOEXCEPT
    {
        // Init NIFM
        if( (m_apisToInit & InitApiFlags::InitApiFlags_Nifm) != InitApiFlags::InitApiFlags_None )
        {
            // if socket library is used to communicate with external nodes, we must initialize
            //   network interface manager (NIFM) first
            NATF_LOG("Initializing nn::nifm...\n");
            nn::Result result = nn::nifm::Initialize();
            if( result.IsFailure() )
            {
                NATF_LOG(" * Error: Failed to Initialize nn::nifm. Desc: %d\n\n", result.GetDescription());
                return false;
            }

            result = nn::nifm::SetExclusiveClient(nn::nifm::GetClientId());
            if( result.IsFailure() )
            {
                NATF_LOG(" * Error: Failed to set exclusive client for nn::nifm. Desc: %d\n\n", result.GetDescription());
                return false;
            }

            m_apisAreInit |= InitApiFlags::InitApiFlags_Nifm;
        }

        // Init Network
        if( (m_apisToInit & InitApiFlags::InitApiFlags_Network) != InitApiFlags::InitApiFlags_None )
        {
            m_pNifmConnection = new nn::nifm::NetworkConnection;
            if( nullptr == m_pNifmConnection )
            {
                NATF_LOG("Failed to allocate nn::nifm::NetworkConnection.\n");
                return false;
            }
            NATF_LOG("Getting request handle.\n");
            nn::nifm::RequestHandle requestHandle = m_pNifmConnection->GetRequestHandle();
            nn::nifm::SetRequestConnectionConfirmationOption(requestHandle,nn::nifm::ConnectionConfirmationOption_Prohibited);
            nn::nifm::SetRequestPersistent(requestHandle, true);

            if( netProfile != nn::util::InvalidUuid )
            {
                NATF_LOG("Setting Uuid.\n");
                if( SetRequestNetworkProfileId(requestHandle, netProfile).IsFailure() )
                {
                    char pUuidSring[nn::util::Uuid::StringSize];
                    NATF_LOG("Failed to set network profile id! Uuid: %s\n\n", netProfile.ToString(pUuidSring, sizeof(pUuidSring)));
                    return false;
                }
            }

            // Submit asynchronous request to NIFM
            m_pNifmConnection->SubmitRequest();

            // Wait while NIFM brings up the interface
            while(!m_pNifmConnection->IsAvailable())
            {
                NATF_LOG("Waiting for network interface to become available...\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            m_apisAreInit |= InitApiFlags::InitApiFlags_Network;
        }

        // Init nn::Socket
        if( (m_apisToInit & InitApiFlags::InitApiFlags_Socket) != InitApiFlags::InitApiFlags_None )
        {
            NATF_LOG("Initializing nn::socket...\n");
            if( !NetTest::Init() )
            {
                NATF_LOG(" * Error: Failed to Initialize socket API.\n\n");
                Finalize();
                return false;
            }
            m_apisAreInit |= InitApiFlags::InitApiFlags_Socket;
        }

        // Init nn::Ssl
        if( (m_apisToInit & InitApiFlags::InitApiFlags_Ssl) != InitApiFlags::InitApiFlags_None )
        {
            NATF_LOG("Initializing SSL library...\n");
            nn::Result result = nn::ssl::Initialize();
            if(result.IsFailure())
            {
                NATF_LOG("Failed to initialize SSL library. Module: %d, Desc: %d\n", result.GetModule(), result.GetDescription());
                Finalize();
                return false;
            }
            m_apisAreInit |= InitApiFlags::InitApiFlags_Ssl;
        }

        // Init LibCurl
        if( (m_apisToInit & InitApiFlags::InitApiFlags_Curl) != InitApiFlags::InitApiFlags_None )
        {
            CURLcode curlErr = curl_global_init(CURL_GLOBAL_ALL);
            if( curlErr != 0 )
            {
                NATF_LOG(" * Error: curl_global_init failed with: %d\n\n", (int)curlErr);
                Finalize();
                return false;
            }
            m_apisAreInit |= InitApiFlags::InitApiFlags_Curl;
        }

        return true;
    }

    void InitApi::Finalize() NN_NOEXCEPT
    {
        // Finalize LibCurl
        if( (m_apisAreInit & InitApiFlags::InitApiFlags_Curl) != InitApiFlags::InitApiFlags_None )
        {
            NATF_LOG("Cleaning up LibCurl\n");
            curl_global_cleanup();
        }

        // Finalize nn::Ssl
        if( (m_apisAreInit & InitApiFlags::InitApiFlags_Ssl) != InitApiFlags::InitApiFlags_None )
        {
            NATF_LOG("Finalizing SSL\n");
            nn::ssl::Finalize();
        }

        // Finalize nn::Socket
        if( (m_apisAreInit & InitApiFlags::InitApiFlags_Socket) != InitApiFlags::InitApiFlags_None )
        {
            NATF_LOG("Finalizing nn::socket\n");
            NetTest::Finalize();
        }

        // Delete NIFM Connection obj
        if( m_pNifmConnection )
        {
            NATF_LOG("Canceling nifm connection request\n");
            m_pNifmConnection->CancelRequest();

            delete m_pNifmConnection;
            m_pNifmConnection = nullptr;
        }

        m_apisAreInit = InitApiFlags::InitApiFlags_None;
    }

}} // Namespaces NATF::Utils
