﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Utils/CommandLineParser.h"
#include <cstring>
#include <nn/util/util_Uuid.h>

namespace NATF {
namespace Utils {

// Constructor
Parser::Parser(const char* pName) NN_NOEXCEPT :
    m_isUsed(false),
    m_pNext(nullptr),
    m_pName(pName) {}

// Parse one float from command line args.
bool Parser::ParseFloat(float& fltOut, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    float value = -1;

    int rval = sscanf(&argV[iArg][strlen(GetName())], "=%f", &value);
    if( rval != 1 )
    {
        NATF_LOG("Failed to parse %s. rval: %d\n\n", GetName(), rval);
        return false;
    }

    fltOut = value;
    NATF_LOG("Parsed %s: %d.%d\n", GetName(), (int)value, int(value * 100.0f) % 100);

    return true;
}

// Parse one int from command line args.
bool Parser::ParseInt32(int& intOut, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int value = -1;

    int rval = sscanf(&argV[iArg][strlen(GetName())], "=%d", &value);
    if( rval != 1 )
    {
        NATF_LOG("Failed to parse %s. rval: %d\n\n", GetName(), rval);
        return false;
    }

    intOut = value;
    NATF_LOG("Parsed %s: %d\n", GetName(), value);
    return true;
}

// Parse ip address from command line args
bool Parser::ParseIpAddress(char* pBufferOut, uint32_t bufferSize, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int ip_1 = -1;
    int ip_2 = -1;
    int ip_3 = -1;
    int ip_4 = -1;

    int rval = sscanf(&argV[iArg][strlen(GetName())], "=%d.%d.%d.%d", &ip_1, &ip_2, &ip_3, &ip_4);
    if( rval != 4 )
    {
        NATF_LOG("Failed to parse %s. rval: %d\n\n", GetName(), rval);
        return false;
    }

    if( ip_1 < 0 || ip_1 > 255 ||
        ip_2 < 0 || ip_2 > 255 ||
        ip_3 < 0 || ip_3 > 255 ||
        ip_4 < 0 || ip_4 > 255 )
    {
        NATF_LOG("Invalid ip address: %s\n\n", argV[iArg]);
        return false;
    }

    rval = NETTEST_SNPRINTF(pBufferOut, bufferSize, "%d.%d.%d.%d", ip_1, ip_2, ip_3, ip_4);
    if( rval <= 0 || static_cast<uint32_t>(rval) >= bufferSize )
    {
        NATF_LOG("Failed to write ip address to buffer. %s: Buffer Size: %d\n\n", argV[iArg], bufferSize);
    }

    NATF_LOG("Parsed %s: %s\n", GetName(), pBufferOut);
    return true;
}

// Parse one int from command line args.
bool Parser::ParseMd5Hash(MD5Hash::Result& hashOut, int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    uint32_t v1,  v2,  v3,  v4,
             v5,  v6,  v7,  v8,
             v9,  v10, v11, v12,
             v13, v14, v15, v16;

    int rval = sscanf(&argV[iArg][strlen(GetName())],
                 "=%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x,%x",
                 &v1, &v2, &v3, &v4, &v5, &v6, &v7, &v8, &v9, &v10, &v11, &v12, &v13, &v14, &v15, &v16);

    if( rval != 16 )
    {
        NATF_LOG("Failed to parse %s. rval: %d\n\n", argV[iArg], rval);
        return false;
    }

    hashOut.Set((uint8_t)v1,  (uint8_t)v2,  (uint8_t)v3,  (uint8_t)v4,
                (uint8_t)v5,  (uint8_t)v6,  (uint8_t)v7,  (uint8_t)v8,
                (uint8_t)v9,  (uint8_t)v10, (uint8_t)v11, (uint8_t)v12,
                (uint8_t)v13, (uint8_t)v14, (uint8_t)v15, (uint8_t)v16);

    NATF_LOG("Parsed %s\n", argV[iArg]);
    return true;
}

// Constructor
ParserGroup::ParserGroup() NN_NOEXCEPT
    : m_pHead(nullptr) {}

// Destructor
ParserGroup::~ParserGroup() NN_NOEXCEPT
{
    while( m_pHead )
    {
        Parser* pTemp = m_pHead;
        m_pHead = m_pHead->m_pNext;
        delete pTemp;
    }
}

// Parse
bool ParserGroup::Parse(int argC, const char * const * argV) NN_NOEXCEPT
{
    // loop through all command line arguments
    for(int iArg = 1; iArg < argC; ++iArg)
    {
        bool isParsed = false;

        // Search for a parser
        Parser* m_pCurrent = m_pHead;
        while(m_pCurrent)
        {
            // Is this the right parser?
            if( strncmp(argV[iArg], m_pCurrent->GetName(), strlen(m_pCurrent->GetName())) == 0 )
            {
                // Has this parser already been used?
                if( m_pCurrent->m_isUsed )
                {
                    NATF_LOG(" * Error: Duplicate parameter! Param: %s\n\n", m_pCurrent->GetName());
                    return false;
                }

                // Parse this parameter
                if( m_pCurrent->Parse(argC, argV, iArg) )
                {
                    m_pCurrent->m_isUsed = true;
                    isParsed = true;
                    break; // Stop searching for a parser.
                }
                else
                {
                    return false;
                }
            }

            m_pCurrent = m_pCurrent->m_pNext;
        }

        if( !isParsed )
        {
            NATF_LOG("\nERROR: Could not find a parser for command line option: %s\n\n", argV[iArg]);
            return false;
        }
    }

    bool haveAllRequiredParams = true;

    // Set default values for any parameters not passed.
    Parser* m_pCurrent = m_pHead;
    while(m_pCurrent)
    {
        // Was this parameter passed?
        if( false == m_pCurrent->m_isUsed )
        {
            // Try to set a default value. Did we fail?
            if( false == m_pCurrent->SetDefaultValue() )
            {
                NATF_LOG(" * Error: Required parameter not passed: %s\n\n", m_pCurrent->GetName());
                haveAllRequiredParams = false;
            }
        }

        m_pCurrent = m_pCurrent->m_pNext;
    }

    return haveAllRequiredParams;
}


//*******************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv IpParser Parser vvvvvvvvvvvvvvvvvvvvvvvvv
IpParser::IpParser(const char* pParamName, const char* pDefaultIp, char* pIpStringBuf, uint32_t bufferLen) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultIp(pDefaultIp),
    m_pIpStringBuf(pIpStringBuf),
    m_bufferLen(bufferLen) {}

bool IpParser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    return ParseIpAddress(m_pIpStringBuf, m_bufferLen, argC, argV, iArg);
}

bool IpParser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultIp )
    {
        int ret = NETTEST_SNPRINTF(m_pIpStringBuf, m_bufferLen, "%s", m_pDefaultIp);
        if( ret <= 0 || static_cast<uint32_t>(ret) >= m_bufferLen )
        {
            NATF_LOG(" * Error: Failed to set default ip parameter.\n\n");
            return false;
        }

        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ IpParser Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//*******************************************************************


//**********************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv Int32Parser Parser vvvvvvvvvvvvvvvvvvvvvvvvv
Int32Parser::Int32Parser(const char* pParamName, const int32_t* pDefaultVal, int32_t& outVal) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultVal(pDefaultVal),
    m_outVal(outVal) {}

bool Int32Parser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    return ParseInt32(m_outVal, argC, argV, iArg);
}

bool Int32Parser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultVal )
    {
        NATF_LOG("%s not set. Using default value: %d\n", GetName(), *m_pDefaultVal);
        m_outVal = *m_pDefaultVal;
        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ Int32Parser Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//***********************************************************************


//**********************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv Int16Parser Parser vvvvvvvvvvvvvvvvvvvvvvvvv
Int16Parser::Int16Parser(const char* pParamName, const int16_t* pDefaultVal, int16_t& outVal) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultVal(pDefaultVal),
    m_outVal(outVal) {}

bool Int16Parser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int32_t tempVal = 0;

    if( !ParseInt32(tempVal, argC, argV, iArg) )
    {
        return false;
    }

    const int32_t maxInt16 = (0xFFFF >> 1);
    const int32_t minInt16 = (-1 * maxInt16) - 1;
    if( tempVal > maxInt16 || tempVal < minInt16 )
    {
        NATF_LOG(" Param %s must be within the int16 range: %d, %d\n\n", GetName(), minInt16, maxInt16);
        return false;
    }

    m_outVal = static_cast<int16_t>(tempVal);
    return true;
}

bool Int16Parser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultVal )
    {
        NATF_LOG("%s not set. Using default value: %d\n", GetName(), (int32_t)(*m_pDefaultVal));
        m_outVal = *m_pDefaultVal;
        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ Int16Parser Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//***********************************************************************


//**********************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv UInt32Parser Parser vvvvvvvvvvvvvvvvvvvvvvvvv
UInt32Parser::UInt32Parser(const char* pParamName, const uint32_t* pDefaultVal, uint32_t& outVal) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultVal(pDefaultVal),
    m_outVal(outVal) {}

bool UInt32Parser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int32_t tempVal = 0;

    if( !ParseInt32(tempVal, argC, argV, iArg) )
    {
        return false;
    }

    if( tempVal < 0 )
    {
        NATF_LOG(" Param %s must not negative\n\n", GetName());
        return false;
    }

    m_outVal = static_cast<uint32_t>(tempVal);
    return true;
}

bool UInt32Parser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultVal )
    {
        NATF_LOG("%s not set. Using default value: %u\n", GetName(), *m_pDefaultVal);
        m_outVal = *m_pDefaultVal;
        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ UInt32Parser Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//***********************************************************************


//**********************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv UInt16Parser Parser vvvvvvvvvvvvvvvvvvvvvvvvv
UInt16Parser::UInt16Parser(const char* pParamName, const uint16_t* pDefaultVal, uint16_t& outVal) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultVal(pDefaultVal),
    m_outVal(outVal) {}

bool UInt16Parser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    int32_t tempVal = 0;

    if( !ParseInt32(tempVal, argC, argV, iArg) )
    {
        return false;
    }

    if( tempVal < 0 || tempVal > 0xFFFF)
    {
        NATF_LOG(" Param %s must be within a uint16 range: 0, %u\n\n", GetName(), 0xFFFF);
        return false;
    }

    m_outVal = static_cast<uint16_t>(tempVal);
    return true;
}

bool UInt16Parser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultVal )
    {
        NATF_LOG("%s not set. Using default value: %d\n", GetName(), (int32_t)(*m_pDefaultVal));
        m_outVal = *m_pDefaultVal;
        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ UInt16Parser Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//***********************************************************************


//************************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv Md5HashParser Parser vvvvvvvvvvvvvvvvvvvvvvvvv
Md5HashParser::Md5HashParser(const char* pParamName, const MD5Hash::Result* pDefaultVal, MD5Hash::Result& outVal) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultVal(pDefaultVal),
    m_outVal(outVal) {}

bool Md5HashParser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    return ParseMd5Hash(m_outVal, argC, argV, iArg);
}

bool Md5HashParser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultVal )
    {
        NATF_LOG("%s not set. Using default value.\n", GetName());
        m_outVal = *m_pDefaultVal;
        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ Md5HashParser Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//************************************************************************


//*****************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv String Parser vvvvvvvvvvvvvvvvvvvvvvvvv
StringParser::StringParser(const char* pParamName, const char* pDefaultVal, char* pOutBuffer, uint32_t bufferLen) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultVal(pDefaultVal),
    m_pOutBuffer(pOutBuffer),
    m_bufferLen(bufferLen) {}

bool StringParser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    // Make sure there is an '=' after parameter name.
    if( argV[iArg][strlen(GetName())] != '=' )
    {
        NATF_LOG("Failed to parse %s. Bad format. Expecting '=' after %s\n\n", argV[iArg], GetName());
        return false;
    }

    const char* pParamValue = &argV[iArg][strlen(GetName()) + 1]; // +1 is for '='
    int rval = NETTEST_SNPRINTF(m_pOutBuffer, m_bufferLen, "%s", pParamValue);
    if( rval < 0 || static_cast<uint32_t>(rval) >= m_bufferLen )
    {
        NATF_LOG("Failed to write string param to buffer. Param: %s, Value: %s, Buffer Len: %d\n\n", GetName(), pParamValue, m_bufferLen);
        return false;
    }
    else
    {
        NATF_LOG("Parsed: %s\n", argV[iArg]);
        return true;
    }
}

bool StringParser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultVal )
    {
        NATF_LOG("%s not set. Using default value: %s\n", GetName(), m_pDefaultVal);
        int rval = NETTEST_SNPRINTF(m_pOutBuffer, m_bufferLen, "%s", m_pDefaultVal);
        if( rval < 0 || static_cast<uint32_t>(rval) >= m_bufferLen )
        {
            NATF_LOG("Failed to write string param to buffer. Param: %s, Value: %s, Buffer Len: %d\n\n", GetName(), m_pDefaultVal, m_bufferLen);
            return false;
        }

        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ String Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//*****************************************************************


//*********************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv UuidParser Parser vvvvvvvvvvvvvvvvvvvvvvvvv
UuidParser::UuidParser(const char* pParamName, const nn::util::Uuid* pDefaultVal, nn::util::Uuid& outVal) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultVal(pDefaultVal),
    m_outVal(outVal) {}

bool UuidParser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    nn::util::Uuid val;
    char pUuidSting[nn::util::Uuid::StringSize];

    // Make sure there is an '=' after parameter name.
    if( argV[iArg][strlen(GetName())] != '=' )
    {
        NATF_LOG("Failed to parse %s. Bad format. Expecting '=' after %s\n\n", argV[iArg], GetName());
        return false;
    }

    const char* pParamValue = &argV[iArg][strlen(GetName()) + 1]; // +1 is for '='
    val.FromString(pParamValue);
    if( strcmp(val.ToString(pUuidSting, sizeof(pUuidSting)), pParamValue) != 0 )
    {
        NATF_LOG("Uuid bad format: %s\n\n", argV[iArg]);
        return false;
    }

    m_outVal.FromString(pParamValue);
    NATF_LOG("Parsed: %s\n", argV[iArg]);
    return true;
}

bool UuidParser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultVal )
    {
        NATF_LOG("%s not set. Using default value.\n", GetName());
        memcpy(m_outVal.data, m_pDefaultVal->data, sizeof(m_outVal.data));
        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ UuidParser Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//*********************************************************************


//*****************************************************************
//vvvvvvvvvvvvvvvvvvvvvvvvv Bool Parser vvvvvvvvvvvvvvvvvvvvvvvvv
BoolParser::BoolParser(const char* pParamName, const bool* pDefaultVal, bool& outVal) NN_NOEXCEPT :
    Parser(pParamName),
    m_pDefaultVal(pDefaultVal),
    m_outVal(outVal) {}

bool BoolParser::Parse(int argC, const char * const * argV, unsigned iArg) const NN_NOEXCEPT
{
    const uint32_t MaxBoolStrLen = 6; // strlen("false") + 1 = 6
    char m_pStrBuffer[MaxBoolStrLen];

    // Make sure there is an '=' after parameter name.
    if( argV[iArg][strlen(GetName())] != '=' )
    {
        NATF_LOG("Failed to parse %s. Bad format. Expecting '=' after %s\n\n", argV[iArg], GetName());
        return false;
    }

    const char* pParamValue = &argV[iArg][strlen(GetName()) + 1]; // +1 is for '='
    int rval = NETTEST_SNPRINTF(m_pStrBuffer, sizeof(m_pStrBuffer), "%s", pParamValue);
    if( rval <= 0 || static_cast<uint32_t>(rval) >= sizeof(m_pStrBuffer) )
    {
        NATF_LOG("Bad format! Valid values: true, false. Param: %s, Value: %s\n\n", GetName(), pParamValue);
        return false;
    }

    NetTest::StrUpr(m_pStrBuffer); // Make parameter uppercase so we can make it case-insensitive.
    if( strcmp(m_pStrBuffer, "TRUE") == 0 )
    {
        m_outVal = true;
    }
    else if( strcmp(m_pStrBuffer, "FALSE") == 0 )
    {
        m_outVal = false;
    }
    else
    {
        NATF_LOG("Bad format! Valid values: true, false. Param: %s, Value: %s\n\n", GetName(), pParamValue);
        return false;
    }

    NATF_LOG("Parsed: %s\n", argV[iArg]);
    return true;
}

bool BoolParser::SetDefaultValue() NN_NOEXCEPT
{
    if( m_pDefaultVal )
    {
        if( *m_pDefaultVal )
        {
            NATF_LOG("%s not set. Using default value: true\n", GetName());
        }
        else
        {
            NATF_LOG("%s not set. Using default value: false\n", GetName());
        }

        m_outVal = *m_pDefaultVal;
        return true;
    }
    else
    {
        return false;
    }
};
//^^^^^^^^^^^^^^^^^^^^^^^^^ Bool Parser ^^^^^^^^^^^^^^^^^^^^^^^^^
//*****************************************************************

}} // Namespaces NATF::Utils
