﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Tests/CurlHttpDownloadMultiTest.h"

#include <new> // For placement new
#include <cstring>
#include <cstdlib> // For Malloc

#include <curl/curl.h>
#include <nn/nifm.h>
#include <nn/fs.h>

namespace NATF {
namespace Tests {

    const Utils::InitApiFlags CurlHttpDownloadMulti::InitFlags = Utils::InitApiFlags::InitApiFlags_Nifm | Utils::InitApiFlags::InitApiFlags_Network | Utils::InitApiFlags::InitApiFlags_Socket | Utils::InitApiFlags::InitApiFlags_Curl;

    // Constructor
    CurlHttpDownloadMulti::CurlHttpDownloadMulti(const char* pTestName, const nn::util::Uuid& netProfile, uint32_t timeoutMs, uint32_t connectionCount, const NATF::Modules::LibCurl::Params& params) NN_NOEXCEPT
        :
          BaseTest(pTestName, false, InitFlags, netProfile),
          m_connectionCount(connectionCount),
          m_timeoutMs(timeoutMs),
          m_pModules(nullptr),
          m_params(params)
    {
        m_params.publishUpSpeedToTeamCity = false;
    }

    // Destructor
    CurlHttpDownloadMulti::~CurlHttpDownloadMulti() NN_NOEXCEPT
    {
        FreeModules();
    }

    // Init
    bool CurlHttpDownloadMulti::Init() NN_NOEXCEPT
    {
        nn::Result result = nn::fs::MountHostRoot();
        if( result.IsFailure() )
        {
            Log(" * Warning: Failed to mount host file system!\n\n");
        }

        return true;
    }

    // Cleanup
    bool CurlHttpDownloadMulti::Cleanup() NN_NOEXCEPT
    {
        FreeModules();
        nn::fs::UnmountHostRoot();

        return true;
    }

    // Config
    bool CurlHttpDownloadMulti::Config() NN_NOEXCEPT
    {
        if( m_connectionCount < 0 || m_connectionCount > 999 )
        {
            LogError(" Connection count must be less than 999.\n\n");
            return false;
        }

        if( !AllocateModules() )
        {
            return false;
        }

        for(int32_t i = 0; i < m_connectionCount; ++i)
        {
            char pThreadName[] = "CurlDownload_000";
            size_t nameLength = strlen(pThreadName);
            NETTEST_SNPRINTF(&pThreadName[nameLength - 3], 3, "%d", i + 1);

            TestThread* pServerThread = CreateTestThread(pThreadName, m_timeoutMs);
            if( !pServerThread )
            {
                return false;
            }

            pServerThread->AddModule(m_pModules[i]);
        }

        return true;
    }

    // AllocateModules
    bool CurlHttpDownloadMulti::AllocateModules() NN_NOEXCEPT
    {
        // Allocate Server modules
        m_pModules = (Modules::LibCurl*)malloc(sizeof(Modules::LibCurl) * m_connectionCount);
        if( !m_pModules )
        {
            Log(" Failed to allocate memory!\n\n");
            FreeModules();
            return false;
        }
        else
        {
            // Initialize all server modules with placement new.
            for(int32_t i = 0; i < m_connectionCount; ++i)
            {
                new (&m_pModules[i]) Modules::LibCurl(m_params);
            }
        }

        return true;
    }

    // FreeModules
    void CurlHttpDownloadMulti::FreeModules() NN_NOEXCEPT
    {
        if( m_pModules )
        {
            // Call destructors for modules.
            Log("Calling destructors on modules...\n");
            for(int32_t i = 0; i < m_connectionCount; ++i)
            {
                (&m_pModules[i])->~LibCurl();
            }

            Log("Deallocating modules...\n");
            free(m_pModules);
            m_pModules = nullptr;
        }
    }

} // namespace Tests
} // namespace NATF
