﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NetTest_Port.h"

#include "Modules/LibCurlModule.h"
#include "Utils/InternetTestHosts.h"

#include <cstdio>
#include <cstdlib>

#include <curl/curl.h>
#include <nn/ssl.h>
#include <nn/fs.h>

#include <string>

namespace NATF {
namespace Modules {

    class File
    {
    public:

        File() : m_pData(nullptr) {}

        char* OpenAndRead(const std::string& path, uint32_t& outFileSize)
        {
            outFileSize = 0;

            if( m_pData )
            {
                delete [] m_pData;
                m_pData = nullptr;
            }

            nn::Result result = nn::fs::OpenFile(&m_Handle, path.c_str(), nn::fs::OpenMode_Read);
            if( result.IsFailure() )
            {
                return nullptr;
            }

            int64_t fileSize = 0;
            result = GetSize(&fileSize);
            if( result.IsFailure() || fileSize <= 0 )
            {
                return nullptr;
            }

            if( fileSize > (0xFFFFFFFFFFFFFFFF >> ((8 - sizeof(uint32_t)) * 8)) )
            {
                NATF_LOG("Error! File size too large to pre-allocate!\n\n");
                return nullptr;
            }

            m_pData = new char[static_cast<uint32_t>(fileSize)];
            if( !m_pData )
            {
                return nullptr;
            }

            int64_t totalBytesRead = 0;

            while( totalBytesRead < fileSize )
            {
                size_t bytesRead = 0;
                size_t maxRead = static_cast<size_t>(fileSize - totalBytesRead); // Truncation is ok.
                result = Read(&bytesRead, m_pData, totalBytesRead, maxRead);
                if( result.IsFailure() )
                {
                    delete [] m_pData;
                    m_pData = nullptr;
                    return nullptr;
                }

                totalBytesRead += bytesRead;
            }

            // Overflow was already checked above
            outFileSize = static_cast<uint32_t>(fileSize);
            return m_pData;
        }

        nn::Result Open(const std::string& path, int mode)
        {
            return nn::fs::OpenFile(&m_Handle, path.c_str(), mode);
        }

        nn::Result Read(void* buffer, int64_t offset, size_t length)
        {
            return nn::fs::ReadFile(m_Handle, offset, buffer, length, nn::fs::ReadOption());
        }

        nn::Result Read(size_t* outValue, void* buffer, int64_t offset, size_t length)
        {
            return nn::fs::ReadFile(outValue, m_Handle, offset, buffer, length, nn::fs::ReadOption());
        }

        nn::Result Write(const void* buffer, int64_t offset, size_t length)
        {
            return nn::fs::WriteFile(m_Handle, offset, buffer, length, nn::fs::WriteOption());
        }

        nn::Result SetSize(int64_t size)
        {
            return nn::fs::SetFileSize(m_Handle, size);
        }

        nn::Result GetSize(int64_t* outValue)
        {
            return nn::fs::GetFileSize(outValue, m_Handle);
        }

        nn::Result Flush()
        {
            return nn::fs::FlushFile(m_Handle);
        }

        ~File()
        {
            nn::fs::CloseFile(m_Handle);
            if( m_pData )
            {
                delete [] m_pData;
                m_pData = nullptr;
            }
        }

    private:
        nn::fs::FileHandle m_Handle;
        char* m_pData;
    };

// Helper class
//   Prevents the need of including curl.h inside of LibCurlModule.h
class LibCurl::Helper
{
public:

    static size_t CurlSslConnectionBeforeFunction(CURL* pCurl, void *pSslConnection, void *pUserData) NN_NOEXCEPT;
    static size_t CurlSslConnectionAfterFunction(CURL* pCurl, void *pSslConnection, void *pUserData, curl_handshake_info handshakeInfo) NN_NOEXCEPT;
    static size_t CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT;
    static size_t CurlWriteFunctionWithHash(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT;
    static size_t CurlWriteFunctionNoHash(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT;
    static size_t ReadCallback(void *ptr, size_t size, size_t nmemb, void *stream) NN_NOEXCEPT;
    static size_t CurlSetSockOpts(void *pData, curl_socket_t curlfd, curlsocktype purpose) NN_NOEXCEPT;
    static void PrintResults(LibCurl* pThis, CURL* pCurlEasy, double testDuration) NN_NOEXCEPT;
    static bool CheckReturnCodes(LibCurl* pThis, CurlEasyData* pCurlData, CURLcode curlErr) NN_NOEXCEPT;
    static void ConfigureGeneralOps(LibCurl* pThis, CurlEasyData* pEasyData) NN_NOEXCEPT;
    static bool ConfigureSpecifcOps(LibCurl* pThis, CurlEasyData* pEasyData, CURLoption& curlUpSize) NN_NOEXCEPT;
    static bool ConfigureUploadOps(LibCurl* pThis, CurlEasyData* pEasyData, CURLoption curlUpSize) NN_NOEXCEPT;
    static void ConfigureProxy(LibCurl* pThis, CurlEasyData* pEasyData) NN_NOEXCEPT;
    static bool EasyPerform(LibCurl* pThis, CurlEasyData* pEasyData) NN_NOEXCEPT;
    static bool MultiPerform(LibCurl* pThis, CURLM* pCurlMulti, CurlEasyData* pEasyData, uint32_t easyCount) NN_NOEXCEPT;
    static void CleanUp(LibCurl* pThis, CURLM*& pCurlMulti, CurlEasyData*& pCurlEasy, uint32_t easyCount) NN_NOEXCEPT;
    static int GetPollParams(LibCurl* pThis, CURLM* pCurlMulti, NetTest::PollFd* pPollParams) NN_NOEXCEPT;
};

LibCurl::Params::Params() NN_NOEXCEPT :
    multiApiConnectionCount(0),
    expectedCurlReturn(0),
    httpMethod(HttpMethod_Get),
    expectedHttpResponse(200),
    expectedAuthReturn(0),
    uploadSize(0),
    useSelect(false),
    doVerifyPeer(true),
    doVerifyHostname(true),
    doVerifyDate(true),
    isAutoUrlEnabled(false),
    proxyPort(0),
    proxyAuthMethod(CURLAUTH_NONE),
    useInternalPki(false),
    publishDownSpeedToTeamCity(false),
    publishUpSpeedToTeamCity(false),
    sessionCacheMode(nn::ssl::Connection::SessionCacheMode::SessionCacheMode_None),
    verifyResultArrayLen(4),
    sendBuffer(32 * 1024),
    recvBuffer(1024 * 256),
    curlHandleLoop(1),
    curlPerformLoop(1),
    curlHostErrRetryCount(3),
    timeBetweenHostErrRetryMs(200),
    checkHash(true),
    isExpectingBufferTooShort(false),
    pUploadData(nullptr)
{
    memset(pUrl, 0, sizeof(pUrl));
    memset(pClientCertPath, 0, sizeof(pClientCertPath));
    memset(pServerCertPath, 0, sizeof(pServerCertPath));
    memset(pProxyServer, 0, sizeof(pProxyServer));
    memset(pProxyUserPwd, 0, sizeof(pProxyUserPwd));
    memset(pHttpHeaderOption, 0, sizeof(pHttpHeaderOption));
    memset(pExpectedCipherName, 0, sizeof(pExpectedCipherName));
    memset(pExpectedSslVersion, 0, sizeof(pExpectedSslVersion));
}

class LibCurl::CurlEasyData
{
public:

    CurlEasyData() NN_NOEXCEPT :
        index(0),
        pThis(nullptr),
        pCurl(nullptr),
        isFinished(false),
        percent10Done(0),
        bytesSent(0),
        pHeaderList(nullptr)
    {
        pErrorBuffer[0] = '\0';
    }

    ~CurlEasyData() NN_NOEXCEPT
    {
        if( pHeaderList )
        {
            curl_slist_free_all(pHeaderList);
            pHeaderList = nullptr;
        }
    }

    uint32_t index;
    LibCurl* pThis;
    CURL* pCurl;
    bool isFinished;
    MD5Hash md5Hash;
    MD5Hash::Result calculatedHash;
    int32_t percent10Done;
    uint64_t bytesSent;
    struct curl_slist* pHeaderList;
    char pErrorBuffer[CURL_ERROR_SIZE];
};

// Contructor
LibCurl::LibCurl(const Params& params) NN_NOEXCEPT
    : BaseModule(params.useSelect),
      m_params(params),
      m_isBufferTooShortError(false)
{ }

// GetName
const char* LibCurl::GetName() const NN_NOEXCEPT
{ return "LibCurl"; }

// CurlWriteFunctionWithHash
size_t LibCurl::Helper::CurlWriteFunctionWithHash(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
{
    CurlEasyData* pEasyData = (CurlEasyData*)userdata;

    size_t count = blobsize*blobcount;
    if (count > 0)
    {
        //NN_LOG("%.*s", count, pData);
        pEasyData->md5Hash.Update((unsigned char*)pData, (unsigned)count);
    }

    return count;
}

// CurlWriteFunctionNoHash
size_t LibCurl::Helper::CurlWriteFunctionNoHash(char *pData, size_t blobsize, size_t blobcount, void *userdata) NN_NOEXCEPT
{
    return blobsize*blobcount;
}


// ReadCallback
size_t LibCurl::Helper::ReadCallback(void *ptr, size_t size, size_t nmemb, void *stream) NN_NOEXCEPT
{
    CurlEasyData* pEasyData = (CurlEasyData*)stream;

    size_t bytesToSend = 0;
    if( pEasyData->bytesSent + size * nmemb > pEasyData->pThis->m_params.uploadSize )
    {
        // Its ok if it gets truncated, this function will be called again to finish the rest.
        bytesToSend = static_cast<size_t>(pEasyData->pThis->m_params.uploadSize - pEasyData->bytesSent);
    }
    else
    {
        bytesToSend = size * nmemb;
    }

    if( bytesToSend == 0 )
    {
        return 0;
    }

    if(pEasyData->pThis->m_params.pUploadData)
    {
        memcpy(ptr, &pEasyData->pThis->m_params.pUploadData[pEasyData->bytesSent], bytesToSend);
    }
    else
    {
        char val = static_cast<char>(pEasyData->bytesSent);
        for(size_t i = 0; i < bytesToSend; ++i)
        {
            ((char*)ptr)[i] = val;
            ++val;
        }
    }

    pEasyData->bytesSent += bytesToSend;
    int32_t percent10Done = static_cast<int32_t>((float)pEasyData->bytesSent / (float)pEasyData->pThis->m_params.uploadSize * 10.0f);

    // Have we completed at least another 10 percent?
    if( percent10Done > pEasyData->percent10Done )
    {
        pEasyData->percent10Done = percent10Done;
        pEasyData->pThis->Log(" Sent: %d%%\n", pEasyData->percent10Done * 10);
    }

    return bytesToSend;
}

// CurlSetSockOpts
size_t LibCurl::Helper::CurlSetSockOpts(void *pData, curl_socket_t curlfd, curlsocktype purpose) NN_NOEXCEPT
{
    int                 rval;
    nn::socket::Errno   error = nn::socket::Errno::ESuccess;
    LibCurl*            pThis = (LibCurl*)pData;
    int                 nSendBuf;
    int                 nRecvBuf;
    NetTest::SockLen    paramSize;

    if( pThis->m_params.sendBuffer > 0 )
    {
        rval = NetTest::SetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &pThis->m_params.sendBuffer, sizeof(pThis->m_params.sendBuffer));
        if( rval != 0 )
        {
            error = NetTest::GetLastError();
            pThis->Log(" * SetSockOpt on nn::socket::Option::So_SndBuf failed with error: %d\n\n", error);
            return 1;
        }
    }

    nSendBuf = 0;
    paramSize = sizeof(nSendBuf);
    rval = NetTest::GetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &nSendBuf, &paramSize);
    if( rval != 0 )
    {
        error = NetTest::GetLastError();
        pThis->Log(" * GetSockOpt on nn::socket::Option::So_SndBuf failed with error: %d\n\n", error);
        return 1;
    }

    if( pThis->m_params.recvBuffer > 0 )
    {
        rval = NetTest::SetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &pThis->m_params.recvBuffer, sizeof(pThis->m_params.recvBuffer));
        if( rval != 0 )
        {
            error = NetTest::GetLastError();
            pThis->Log(" * SetSockOpt on nn::socket::Option::So_RcvBuf failed with error: %d\n\n", error);
            return 1;
        }
    }

    nRecvBuf = 0;
    paramSize = sizeof(nRecvBuf);
    rval = NetTest::GetSockOpt(static_cast<int>(curlfd), nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &nRecvBuf, &paramSize);
    if( rval != 0 )
    {
        error = NetTest::GetLastError();
        pThis->Log(" * GetSockOpt on nn::socket::Option::So_RcvBuf failed with error: %d\n\n", error);
        return 1;
    }

    pThis->Log("SockOpts: nn::socket::Option::So_SndBuf: %d, nn::socket::Option::So_RcvBuf: %d\n\n", nSendBuf, nRecvBuf);

    return 0;
}

size_t LibCurl::Helper::CurlSslConnectionBeforeFunction(CURL* pCurl, void *pSslConnection, void *pUserData) NN_NOEXCEPT
{
    NN_UNUSED(pCurl);

    nn::Result result = nn::ResultSuccess();
    nn::ssl::Connection* pConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnection);
    LibCurl* pThis = reinterpret_cast<LibCurl*>(pUserData);

    pThis->Log("\n ** CurlSslConnectionBeforeFunction\n\n");

    if(pThis->m_params.sessionCacheMode != nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId)
    {
        result = pConnection->SetSessionCacheMode(pThis->m_params.sessionCacheMode);
        if(result.IsFailure())
        {
            pThis->LogError(" * Failed to SetSessionCacheMode. Desc: %d\n\n", result.GetDescription());
            return 1;
        }
    }

    result = pConnection->SetServerCertBuffer(pThis->m_pServerCertBuffer, sizeof(pThis->m_pServerCertBuffer));
    if(result.IsFailure())
    {
        pThis->LogError(" * Failed to SetServerCertBuffer. Desc: %d\n\n", result.GetDescription());
        return 1;
    }

    return 0;
}

size_t LibCurl::Helper::CurlSslConnectionAfterFunction(CURL* pCurl, void *pSslConnectionPtr, void *pUserData, curl_handshake_info handshakeInfo) NN_NOEXCEPT
{
    NN_UNUSED(pCurl);
    size_t ret = 0;

    LibCurl* pThis = reinterpret_cast<LibCurl*>(pUserData);
    nn::ssl::Connection* pSslConnection = reinterpret_cast<nn::ssl::Connection*>(pSslConnectionPtr);

    pThis->Log("\n ** CurlSslConnectionAfterFunction\n\n");

    if(handshakeInfo.isHandshakeSuccess)
    {
        pThis->Log("Handshake success!\n");
        pThis->Log("Server Cert Data Len: %u\n", handshakeInfo.certDataSize);
        pThis->Log("Server Cert Count: %u\n\n", handshakeInfo.certCount);
    }
    else
    {
        pThis->Log("Handshake failure!\n");
    }

    nn::Result* pErrorArray = new nn::Result[pThis->m_params.verifyResultArrayLen];
    if(pErrorArray != nullptr)
    {
        uint32_t errorsWritten = 0;
        uint32_t totalErrors = 0;

        nn::Result result = pSslConnection->GetVerifyCertErrors(pErrorArray, &errorsWritten, &totalErrors, pThis->m_params.verifyResultArrayLen);
        if(result.IsFailure())
        {
            if(nn::ssl::ResultBufferTooShort::Includes(result))
            {
                pThis->Log("nn::ssl::GetVerifyErrors() returned ResultBufferTooShort! ErrorsWritten: %u TotalErrors: %u\n\n", errorsWritten, totalErrors);
                pThis->m_isBufferTooShortError = true;
            }
            else
            {
                pThis->LogError("nn::ssl::GetVerifyErrors() failed! Module: %d, Desc: %d\n\n", result.GetModule(), result.GetDescription());
                ret = static_cast<size_t>(-1); // Fail LibCurl.
            }
        }
        else
        {
            pThis->Log("ErrorsWritten: %u TotalErrors: %u\n", errorsWritten, totalErrors);
            for(uint32_t iError = 0; iError < errorsWritten; ++iError)
            {
                pThis->Log("Verify error %u: desc: %d\n", iError, pErrorArray[iError].GetDescription());
            }
        }

        delete [] pErrorArray;
        pErrorArray = nullptr;
    }
    else
    {
        pThis->LogError("CurlSslConnectionAfterFunction: Failed to allocate Verify Result Array\n\n");
        ret = static_cast<size_t>(-1); // Fail LibCurl.
    }

    if(handshakeInfo.isHandshakeSuccess)
    {
        nn::ssl::Connection::CipherInfo cipherInfo;

        nn::Result result = pSslConnection->GetCipherInfo(&cipherInfo);
        if(result.IsFailure())
        {
            pThis->LogError("CurlSslConnectionAfterFunction: Failed to get cipher info. Desc: %d\n\n", result.GetDescription());
            return static_cast<size_t>(-1); // Fail LibCurl.
        }

        pThis->Log("Cipher Spec: %s\n", cipherInfo.cipherName);
        pThis->Log("Ssl Version: %s\n", cipherInfo.versionName);

        // If we are expecting a specific cipher spec, then check it.
        if(pThis->m_params.pExpectedCipherName[0] != '\0')
        {
            uint32_t minBufLen = (sizeof(pThis->m_params.pExpectedCipherName) < sizeof(cipherInfo.cipherName)) ? sizeof(pThis->m_params.pExpectedCipherName) : sizeof(cipherInfo.cipherName);
            if(strncmp(pThis->m_params.pExpectedCipherName, cipherInfo.cipherName, minBufLen) != 0)
            {
                pThis->LogError("CurlSslConnectionAfterFunction: Cipher spec not as expected. Expected: %s\n\n", pThis->m_params.pExpectedCipherName);
                return static_cast<size_t>(-1); // Fail LibCurl.
            }
        }

        // If we are expecting a specific Ssl version, then check it.
        if(pThis->m_params.pExpectedSslVersion[0] != '\0')
        {
            uint32_t minBufLen = (sizeof(pThis->m_params.pExpectedSslVersion) < sizeof(cipherInfo.versionName)) ? sizeof(pThis->m_params.pExpectedSslVersion) : sizeof(cipherInfo.versionName);
            if(strncmp(pThis->m_params.pExpectedSslVersion, cipherInfo.versionName, minBufLen) != 0)
            {
                pThis->LogError("CurlSslConnectionAfterFunction: Ssl version not as expected. Expected: %s\n\n", pThis->m_params.pExpectedSslVersion);
                return static_cast<size_t>(-1); // Fail LibCurl.
            }
        }
    }
    else
    {
        if(pThis->m_params.pExpectedCipherName[0] != '\0')
        {
            pThis->LogError("CurlSslConnectionAfterFunction: Could not verify cipher name due to handshake failure!\n\n");
            return static_cast<size_t>(-1); // Fail LibCurl.
        }

        if(pThis->m_params.pExpectedSslVersion[0] != '\0')
        {
            pThis->LogError("CurlSslConnectionAfterFunction: Could not verify ssl version due to handshake failure!\n\n");
            return static_cast<size_t>(-1); // Fail LibCurl.
        }
    }

    return ret;
}

// CurlSslFunction
size_t LibCurl::Helper::CurlSslContextFunction(CURL* pCurl, void* pSslContext, void* pUserData) NN_NOEXCEPT
{
    size_t returnVal = 0;
    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSslContext);
    LibCurl* pThis = reinterpret_cast<LibCurl*>(pUserData);

    nn::Result result = pContext->Create(nn::ssl::Context::SslVersion_Auto);
    if( result.IsFailure() )
    {
        pThis->LogError("Failed to create context! Desc: %d\n\n", result.GetDescription());
        return static_cast<size_t>(-1);
    }

    do
    {
        if(strcmp(pThis->m_params.pClientCertPath, "") != 0)
        {
            File clientCertFile;
            uint32_t fileSize = 0;

            char* pClientCert = clientCertFile.OpenAndRead(pThis->m_params.pClientCertPath, fileSize);
            if( !pClientCert )
            {
                pThis->LogError(" * Error: Failed to open and/or read cert file data!\n\n");
                returnVal = static_cast<size_t>(-1);
                break;
            }

            nn::ssl::CertStoreId clientCertId;
            result = pContext->ImportClientPki( &clientCertId,
                                                pClientCert,
                                                nullptr,
                                                fileSize,
                                                0);

            if( result.IsFailure() )
            {
                pThis->LogError(" * Error: Failed to import client cert!\n\n");
                returnVal = static_cast<size_t>(-1);
                break;
            }
        }

        if(strcmp(pThis->m_params.pServerCertPath, "") != 0)
        {
            // Loop through all the certificates. Are seperated by ','
            const char* pWordStart = &pThis->m_params.pServerCertPath[0];
            const char* pWordEnd = pWordStart;
            while(*pWordStart != '\0' && *pWordEnd != '\0')
            {
                // Seek to the end of the word.
                pWordEnd = pWordStart + 1;
                while(*pWordEnd != '\0' && *pWordEnd != ',')
                {
                    ++pWordEnd;
                }

                // Make string from single cert path
                std::string certPath;
                certPath.assign(pWordStart, pWordEnd - pWordStart);
                pThis->Log("Importing Server Cert: %s\n", certPath.c_str());

                File serverCertFile;
                uint32_t fileSize = 0;
                char* pServerCert = serverCertFile.OpenAndRead(certPath, fileSize);
                if( !pServerCert )
                {
                    pThis->LogError(" * Error: Failed to open and/or read cert file data!\n\n");
                    returnVal = static_cast<size_t>(-1);
                    break;
                }

                nn::ssl::CertStoreId serverCertId;
                result = pContext->ImportServerPki( &serverCertId,
                                        pServerCert,
                                        fileSize,
                                        nn::ssl::CertificateFormat_Pem);

                if( result.IsFailure() )
                {
                    pThis->LogError(" * Error: Failed to import server cert!\n\n");
                    returnVal = static_cast<size_t>(-1);
                    break;
                }

                pWordStart = pWordEnd + 1;
            }
        }

        if(pThis->m_params.useInternalPki)
        {
            nn::ssl::CertStoreId certStoreId;
            result = pContext->RegisterInternalPki(&certStoreId, nn::ssl::Context::InternalPki_DeviceClientCertDefault);
            if (result.IsFailure())
            {
                pThis->LogError(" * Error: Failed to register internal Pki. Desc: %d\n\n", result.GetDescription());
                returnVal = static_cast<size_t>(-1);
                break;
            }
        }
    } while(NN_STATIC_CONDITION(false));

    return returnVal;
}

// PrintResults
void LibCurl::Helper::PrintResults(LibCurl* pThis, CURL* pCurlEasy, double testDurationSec) NN_NOEXCEPT
{
    double downloadSize  = 0.0f;
    double downloadSpeed = 0.0f;
    double uploadSize    = 0.0f;
    double uploadSpeed   = 0.0f;

    bool isSsl = (strncmp(pThis->m_params.pUrl, "https", 5) == 0);

    CURLcode err1 = curl_easy_getinfo(pCurlEasy, CURLINFO_SIZE_DOWNLOAD,  &downloadSize);
    CURLcode err2 = curl_easy_getinfo(pCurlEasy, CURLINFO_SPEED_DOWNLOAD, &downloadSpeed);
    CURLcode err3 = curl_easy_getinfo(pCurlEasy, CURLINFO_SIZE_UPLOAD,    &uploadSize);
    CURLcode err4 = curl_easy_getinfo(pCurlEasy, CURLINFO_SPEED_UPLOAD,   &uploadSpeed);

    if( (err1 | err2 | err3 | err4) != CURLE_OK )
    {
        pThis->Log(" *** WARNING: Could not get libcurl performance details.\n\n");
    }
    else
    {
        char pDnSize[32];
        char pDnSpeed[32];
        char pUpSize[32];
        char pUpSpeed[32];
        char pTime[32];

        // Convert from bytes/sec -> mbits/sec
        downloadSpeed *= (8.0 / 1000000.0);
        uploadSpeed   *= (8.0 / 1000000.0);

        NETTEST_SNPRINTF(pDnSize,  32, "%.3f", (float)(downloadSize / (1024.0 * 1024.0)));
        NETTEST_SNPRINTF(pDnSpeed, 32, "%.3f", (float)downloadSpeed);
        NETTEST_SNPRINTF(pUpSize,  32, "%.3f", (float)(uploadSize / (1024.0 * 1024.0)));
        NETTEST_SNPRINTF(pUpSpeed, 32, "%.3f", (float)uploadSpeed);
        NETTEST_SNPRINTF(pTime,    32, "%.3f", (float)testDurationSec);

        pThis->Log(" ** Duration: %s sec\n\n", pTime);

        pThis->Log(" ** Upload Size: %s MB\n", pUpSize);
        pThis->Log(" ** Upload Throughput: %s Mbits/Sec\n\n", pUpSpeed);

        pThis->Log(" ** Download Size: %s MB\n", pDnSize);
        pThis->Log(" ** Download Throughput: %s Mbits/Sec\n\n", pDnSpeed);

        if(pThis->m_params.publishDownSpeedToTeamCity)
        {
            if( isSsl )
            {
                NN_LOG("##teamcity[buildStatisticValue key='LibCurl HTTPS DL (Mbps)' value='%s']\n", pDnSpeed);
            }
            else
            {
                NN_LOG("##teamcity[buildStatisticValue key='LibCurl HTTP DL (Mbps)' value='%s']\n", pDnSpeed);
            }
        }

        if(pThis->m_params.publishUpSpeedToTeamCity)
        {
            if( isSsl )
            {
                NN_LOG("##teamcity[buildStatisticValue key='LibCurl HTTPS UL (Mbps)' value='%s']\n", pUpSpeed);
            }
            else
            {
                NN_LOG("##teamcity[buildStatisticValue key='LibCurl HTTP UL (Mbps)' value='%s']\n", pUpSpeed);
            }
        }
    }
}

// CheckReturnCodes
bool LibCurl::Helper::CheckReturnCodes(LibCurl* pThis, CurlEasyData* pCurlData, CURLcode curlErr) NN_NOEXCEPT
{
    bool isSuccess = true;
    long response = 0;
    nn::Result tmpResult;

    curl_easy_getinfo(pCurlData->pCurl, CURLINFO_SSL_HANDSHAKE_RESULT, &response);

    nn::Result callResult = nn::ssl::GetSslResultFromValue(&tmpResult, reinterpret_cast<char*>(&response), sizeof(response));
    if(callResult.IsFailure())
    {
        pThis->LogError("\n ERROR: GetSslResultFromValue failed! Module: %d, Desc: %d\n\n", callResult.GetModule(), callResult.GetDescription());
        return false;
    }

    if(tmpResult.IsFailure())
    {
        pThis->Log("CURLINFO_SSL_HANDSHAKE_RESULT error value: %d, Module: %d, Desc: %d\n", response, tmpResult.GetModule(), tmpResult.GetDescription());
    }

    curl_easy_getinfo(pCurlData->pCurl, CURLINFO_SSL_VERIFYRESULT, &response);

    callResult = nn::ssl::GetSslResultFromValue(&tmpResult, reinterpret_cast<char*>(&response), sizeof(response));
    if(callResult.IsFailure())
    {
        pThis->LogError("\n ERROR: GetSslResultFromValue failed! Module: %d, Desc: %d\n\n", callResult.GetModule(), callResult.GetDescription());
        return false;
    }

    if(tmpResult.IsFailure())
    {
        pThis->Log("CURLINFO_SSL_VERIFYRESULT error value: %d, Module: %d, Desc: %d\n", response, tmpResult.GetModule(), tmpResult.GetDescription());
    }

    if( curlErr != pThis->m_params.expectedCurlReturn )
    {
        pThis->LogError("CurlEasy[%d] * ERROR: Expected return miss-match. Error: %d, Expected: %d\n\n", pCurlData->index, curlErr, pThis->m_params.expectedCurlReturn);
        pThis->Log(" * ERROR Details: %s\n\n", pCurlData->pErrorBuffer);

        isSuccess = false;
    }

    if( pThis->m_params.expectedAuthReturn != response )
    {
        pThis->LogError("CurlEasy[%d] * ERROR: Expected auth response miss-match. Response: %d, Expected: %d\n\n", pCurlData->index, response, pThis->m_params.expectedAuthReturn);
        pThis->Log(" * ERROR Details: %s\n\n", pCurlData->pErrorBuffer);

        isSuccess = false;
    }

    if(pThis->m_params.isExpectingBufferTooShort != pThis->m_isBufferTooShortError)
    {
        pThis->LogError("CurlEasy[%d] * ERROR: Unexpected BufferTooShort result: %s\n\n", (pThis->m_isBufferTooShortError) ? "true" : "false");

        isSuccess = false;
    }

    curl_easy_getinfo(pCurlData->pCurl, CURLINFO_RESPONSE_CODE, &response);

    if( pThis->m_params.expectedHttpResponse >= 0 && pThis->m_params.expectedHttpResponse != response )
    {
        pThis->LogError("CurlEasy[%d] * ERROR: Expected response miss-match. Response: %d, Expected: %d\n\n", pCurlData->index, response, pThis->m_params.expectedHttpResponse);
        pThis->Log(" * ERROR Details: %s\n\n", pCurlData->pErrorBuffer);

        isSuccess = false;
    }

    // Compare the two hash values
    if( curlErr == 0 && pThis->m_params.checkHash && pThis->m_params.httpMethod == HttpMethod_Get )
    {
        pCurlData->md5Hash.Final(pCurlData->calculatedHash);

        if( !pThis->CheckHashValues(pCurlData) )
        {
            isSuccess = false;
        }
    }

    return isSuccess;
}

// VerifyParams
bool LibCurl::VerifyParams() NN_NOEXCEPT
{
    if( m_params.httpMethod != HttpMethod_Get &&
             m_params.httpMethod != HttpMethod_Post &&
             m_params.httpMethod != HttpMethod_Put)
    {
        LogError(" * Unknown HttpMethod. Passed: %d\n\n", static_cast<int>(m_params.httpMethod));
        return false;
    }

    return true;
}

// ConfigureGeneralOps
void LibCurl::Helper::ConfigureGeneralOps(LibCurl* pThis, CurlEasyData* pEasyData) NN_NOEXCEPT
{
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_URL, pThis->m_params.pUrl);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_FOLLOWLOCATION, 1);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_VERBOSE, 1);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_CONNECTTIMEOUT_MS, 120 * 1000); // 2 min
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_ERRORBUFFER, pEasyData->pErrorBuffer);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SOCKOPTFUNCTION, Helper::CurlSetSockOpts);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SOCKOPTDATA, pThis);

    if(pThis->m_params.doVerifyPeer == false || pThis->m_params.doVerifyHostname == false)
    {
        curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_SKIP_DEFAULT_VERIFY, 1L);
    }

    if( pThis->m_params.doVerifyHostname )
    {
        curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_VERIFYHOST, 2L);
    }
    else
    {
        curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_VERIFYHOST, 0L);
    }

    if(pThis->m_params.sessionCacheMode == nn::ssl::Connection::SessionCacheMode::SessionCacheMode_SessionId)
    {
        curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_SESSIONID_CACHE, 1L);
    }
    else
    {
        curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_SESSIONID_CACHE, 0L);
    }

    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_VERIFYPEER, static_cast<long>(pThis->m_params.doVerifyPeer));
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_VERIFYDATE, static_cast<long>(pThis->m_params.doVerifyDate));
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_CONN_FUNCTION_BEFORE, CurlSslConnectionBeforeFunction);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_CONN_FUNCTION_AFTER, CurlSslConnectionAfterFunction);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_CONN_DATA, pThis);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_CTX_FUNCTION, CurlSslContextFunction);
    curl_easy_setopt(pEasyData->pCurl, CURLOPT_SSL_CTX_DATA, pThis);
}

// ConfigureSpecifcOps
bool LibCurl::Helper::ConfigureSpecifcOps(LibCurl* pThis, CurlEasyData* pEasyData, CURLoption& curlUpSize) NN_NOEXCEPT
{
    switch( pThis->m_params.httpMethod )
    {
    case HttpMethod_Get:
        if(pThis->m_params.checkHash)
        {
            curl_easy_setopt(pEasyData->pCurl, CURLOPT_WRITEFUNCTION, Helper::CurlWriteFunctionWithHash);
        }
        else
        {
            curl_easy_setopt(pEasyData->pCurl, CURLOPT_WRITEFUNCTION, Helper::CurlWriteFunctionNoHash);
        }

        curl_easy_setopt(pEasyData->pCurl, CURLOPT_WRITEDATA, pEasyData);
        break;

    case HttpMethod_Put:
        {
            curl_easy_setopt(pEasyData->pCurl, CURLOPT_UPLOAD, 1L);
            curlUpSize = CURLOPT_INFILESIZE_LARGE;
        } break;

    case HttpMethod_Post:
        {
            curl_easy_setopt(pEasyData->pCurl, CURLOPT_POST, 1L);
            curlUpSize = CURLOPT_POSTFIELDSIZE_LARGE;
        } break;

    default:
        pThis->LogError(" * ERROR: http method not supported.\n\n");
        return false;
    }

    return true;
}

// ConfigureUploadOps
bool LibCurl::Helper::ConfigureUploadOps(LibCurl* pThis, CurlEasyData* pEasyData, CURLoption curlUpSize) NN_NOEXCEPT
{
    pEasyData->pHeaderList = nullptr;

    if( pThis->m_params.httpMethod == HttpMethod_Put || pThis->m_params.httpMethod == HttpMethod_Post )
    {
        curl_easy_setopt(pEasyData->pCurl, CURLOPT_READFUNCTION, Helper::ReadCallback);
        curl_easy_setopt(pEasyData->pCurl, CURLOPT_READDATA, pEasyData);
        curl_easy_setopt(pEasyData->pCurl, CURLOPT_FORBID_REUSE, 1); // **** TEMP ****
        curl_off_t nBigSize = pThis->m_params.uploadSize;
        curl_easy_setopt(pEasyData->pCurl, curlUpSize, nBigSize);

        // We turn off "Expect: 100-continue" We want to just send it.
        pEasyData->pHeaderList = curl_slist_append(pEasyData->pHeaderList, "Expect:");
    }

    if(strcmp(pThis->m_params.pHttpHeaderOption, "") != 0)
    {
        pEasyData->pHeaderList = curl_slist_append(pEasyData->pHeaderList, pThis->m_params.pHttpHeaderOption);
    }

    curl_easy_setopt(pEasyData->pCurl, CURLOPT_HTTPHEADER, pEasyData->pHeaderList);

    return true;
}

// ConfigureProxy
void LibCurl::Helper::ConfigureProxy(LibCurl* pThis, CurlEasyData* pEasyData) NN_NOEXCEPT
{
    if( strcmp(pThis->m_params.pProxyServer, "") != 0 )
    {
        pThis->Log(" Using proxy: %s\n", pThis->m_params.pProxyServer);

        /**
        * Configure proxy if needed
        */
        curl_easy_setopt(pEasyData->pCurl,
                       CURLOPT_PROXYAUTOCONFIG, 0);
        curl_easy_setopt(pEasyData->pCurl,
                       CURLOPT_PROXY,
                       pThis->m_params.pProxyServer);
        curl_easy_setopt(pEasyData->pCurl,
                       CURLOPT_PROXYPORT,
                       pThis->m_params.proxyPort);
        curl_easy_setopt(pEasyData->pCurl,
                       CURLOPT_PROXYAUTH,
                       pThis->m_params.proxyAuthMethod);

        if( strcmp(pThis->m_params.pProxyUserPwd, "") != 0 )
        {
            curl_easy_setopt(pEasyData->pCurl,
                            CURLOPT_PROXYUSERPWD,
                            pThis->m_params.pProxyUserPwd);
        }
    }
}

// CheckHashValues
bool LibCurl::CheckHashValues(CurlEasyData* pEasyData) NN_NOEXCEPT
{
    if( memcmp(m_params.expectedHash.m_pHash, pEasyData->calculatedHash.m_pHash, MD5Hash::HASH_SIZE) == 0 )
    {
        Log(" * SUCCESS! Hash match! *\n");

        Log("Hash: %.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x\n"
           , m_params.expectedHash.m_pHash[0]
           , m_params.expectedHash.m_pHash[1]
           , m_params.expectedHash.m_pHash[2]
           , m_params.expectedHash.m_pHash[3]
           , m_params.expectedHash.m_pHash[4]
           , m_params.expectedHash.m_pHash[5]
           , m_params.expectedHash.m_pHash[6]
           , m_params.expectedHash.m_pHash[7]
           , m_params.expectedHash.m_pHash[8]
           , m_params.expectedHash.m_pHash[9]
           , m_params.expectedHash.m_pHash[10]
           , m_params.expectedHash.m_pHash[11]
           , m_params.expectedHash.m_pHash[12]
           , m_params.expectedHash.m_pHash[13]
           , m_params.expectedHash.m_pHash[14]
           , m_params.expectedHash.m_pHash[15] );

        return true;
        }
        else
        {
        Log(" * FAILURE! HASH MISS-MATCH! *\n");

        Log("Hash expected hash: %.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x\n"
           , m_params.expectedHash.m_pHash[0]
           , m_params.expectedHash.m_pHash[1]
           , m_params.expectedHash.m_pHash[2]
           , m_params.expectedHash.m_pHash[3]
           , m_params.expectedHash.m_pHash[4]
           , m_params.expectedHash.m_pHash[5]
           , m_params.expectedHash.m_pHash[6]
           , m_params.expectedHash.m_pHash[7]
           , m_params.expectedHash.m_pHash[8]
           , m_params.expectedHash.m_pHash[9]
           , m_params.expectedHash.m_pHash[10]
           , m_params.expectedHash.m_pHash[11]
           , m_params.expectedHash.m_pHash[12]
           , m_params.expectedHash.m_pHash[13]
           , m_params.expectedHash.m_pHash[14]
           , m_params.expectedHash.m_pHash[15] );

        Log("Hash calculated hash:  %.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x,%.2x\n"
           , pEasyData->calculatedHash.m_pHash[0]
           , pEasyData->calculatedHash.m_pHash[1]
           , pEasyData->calculatedHash.m_pHash[2]
           , pEasyData->calculatedHash.m_pHash[3]
           , pEasyData->calculatedHash.m_pHash[4]
           , pEasyData->calculatedHash.m_pHash[5]
           , pEasyData->calculatedHash.m_pHash[6]
           , pEasyData->calculatedHash.m_pHash[7]
           , pEasyData->calculatedHash.m_pHash[8]
           , pEasyData->calculatedHash.m_pHash[9]
           , pEasyData->calculatedHash.m_pHash[10]
           , pEasyData->calculatedHash.m_pHash[11]
           , pEasyData->calculatedHash.m_pHash[12]
           , pEasyData->calculatedHash.m_pHash[13]
           , pEasyData->calculatedHash.m_pHash[14]
           , pEasyData->calculatedHash.m_pHash[15] );

        return false;
    }
}

// EasyPerform
bool LibCurl::Helper::EasyPerform(LibCurl* pThis, CurlEasyData* pEasyData) NN_NOEXCEPT
{
    bool isSuccess = false;

    pThis->Log(" Making request: %s\n", pThis->m_params.pUrl);

    for(int32_t iTryCount = 0; iTryCount < pThis->m_params.curlHostErrRetryCount; ++iTryCount)
    {
        NetTest::Tick startTime = NetTest::GetTick();
        CURLcode curlErr = curl_easy_perform(pEasyData->pCurl);
        NetTest::Tick endTime = NetTest::GetTick();

        pThis->Log(" curl_easy_perform - rval: %d\n", curlErr);
        if( !Helper::CheckReturnCodes(pThis, pEasyData, curlErr) )
        {
            // If there was a host resolver error, try again.
            if(curlErr == CURLE_COULDNT_RESOLVE_HOST)
            {
                NetTest::SleepMs(pThis->m_params.timeBetweenHostErrRetryMs);
                continue;
            }

            break;
        }

        isSuccess = true;

        if( curlErr == CURLE_OK )
        {
            double durationSec  = NetTest::TickToTime(endTime - startTime).GetMilliSeconds() / 1000.0;

            Helper::PrintResults(pThis, pEasyData->pCurl, durationSec);
        }

        break;
    }

    return isSuccess;
}

// GetPollParams
//  TODO: We are using Poll until the select bug SIGLONTD-3700 is resolved
//        This means we have to loop through all potential sockets to find the exact values.
int LibCurl::Helper::GetPollParams(LibCurl* pThis, CURLM* pCurlMulti, NetTest::PollFd* pPollParams) NN_NOEXCEPT
{
    int maxfd = -1;
    nn::socket::FdSet readSet;
    nn::socket::FdSet writeSet;
    nn::socket::FdSet errSet;

    nn::socket::FdSetZero(&readSet);
    nn::socket::FdSetZero(&writeSet);
    nn::socket::FdSetZero(&errSet);

    memset(pPollParams, 0, sizeof(NetTest::PollFd) * pThis->m_params.multiApiConnectionCount);

    //Get file descriptors from the transfers
    curl_multi_fdset(pCurlMulti, &readSet, &writeSet, &errSet, &maxfd);

    int socketCount = 0;
    for(int socketFd = 0; socketFd <= maxfd; ++socketFd)
    {
        pPollParams[socketCount].events = nn::socket::PollEvent::PollNone;

        if( nn::socket::FdSetIsSet(socketFd, &readSet) )
        {
            pPollParams[socketCount].events |= nn::socket::PollEvent::PollRdNorm;
        }

        if( nn::socket::FdSetIsSet(socketFd, &writeSet) )
        {
            pPollParams[socketCount].events |= nn::socket::PollEvent::PollWrNorm;
        }

        if( nn::socket::FdSetIsSet(socketFd, &errSet) )
        {
            pPollParams[socketCount].events |= nn::socket::PollEvent::PollErr;
        }

        if( pPollParams[socketCount].events != nn::socket::PollEvent::PollNone )
        {
            pPollParams[socketCount].fd = socketFd;
            ++socketCount;
            if( socketCount == (int)pThis->m_params.multiApiConnectionCount )
            {
                break;
            }
        }
    }

    return socketCount;
}

// MultiPerform
bool LibCurl::Helper::MultiPerform(LibCurl* pThis, CURLM* pCurlMulti, CurlEasyData* pEasyData, uint32_t easyCount) NN_NOEXCEPT
{
    bool isSuccess = true;

    // we start some action by calling perform right away
    CURLMcode status;
    int stillRunning; // keep number of running handles

    // Maximum number of sockets will be m_params.multiApiConnectionCount
    NetTest::PollFd* pPollParams = new NetTest::PollFd[pThis->m_params.multiApiConnectionCount];
    if( nullptr == pPollParams )
    {
        pThis->LogError(" Failed to allocate PollFd array.\n\n");
        return false;
    }

    pThis->Log(" Making request: %s\n", pThis->m_params.pUrl);
    NetTest::Tick startTime = NetTest::GetTick();
    do
    {
        status = curl_multi_perform(pCurlMulti, &stillRunning);
    } while( status == CURLM_CALL_MULTI_PERFORM );

    while( stillRunning )
    {
        int rval = 0; // Poll() return code. Set to timeout by default.
        long curlTimeOutMs = -1;

        curl_multi_timeout(pCurlMulti, &curlTimeOutMs);

        if( curlTimeOutMs > 0 )
        {
            if( curlTimeOutMs > MaxPollTimeoutSec * 1000 )
            {
                curlTimeOutMs = MaxPollTimeoutSec * 1000;
            }

            int paramCount = Helper::GetPollParams(pThis, pCurlMulti, pPollParams);
            if( paramCount > 0 )
            {
                rval = pThis->SelectOrPoll(pPollParams, paramCount, curlTimeOutMs);
            }
        }

        switch(rval)
        {
        case -1:
            // select error
            stillRunning = 0;
            pThis->LogError("ERROR: poll() returned error. Err: %d\n\n", nn::socket::GetLastError());
            isSuccess = false;
            break;
        case 0:
        default:
            // timeout or readable/writable sockets
            do
            {
                status = curl_multi_perform(pCurlMulti, &stillRunning);
            } while(status == CURLM_CALL_MULTI_PERFORM);
            break;
        }
    }
    NetTest::Tick endTime = NetTest::GetTick();

    if( status != 0 )
    {
        isSuccess = false;
    }

    pThis->Log("Multi perform has finished. status: %d\n", status);

    struct CURLMsg* pCurlMsg = nullptr;
    do
    {
        int msgq = 0;
        pCurlMsg = curl_multi_info_read(pCurlMulti, &msgq);
        if(pCurlMsg && (pCurlMsg->msg == CURLMSG_DONE))
        {
            for(uint32_t iEasy = 0; iEasy < easyCount; ++iEasy)
            {
                if( pEasyData[iEasy].pCurl == pCurlMsg->easy_handle )
                {
                    pEasyData[iEasy].isFinished = true;
                    if( !Helper::CheckReturnCodes(pThis, &pEasyData[iEasy], pCurlMsg->data.result) )
                    {
                        isSuccess = false;
                    }

                    if( pCurlMsg->data.result == CURLE_OK )
                    {
                        double durationSec  = NetTest::TickToTime(endTime - startTime).GetMilliSeconds() / 1000.0;

                        Helper::PrintResults(pThis, reinterpret_cast<CURL*>(&pEasyData[iEasy]), durationSec);
                    }
                    else if( pThis->m_params.expectedCurlReturn == CURLE_OK )
                    {
                        pThis->LogError(" * Failed fetching URL %s with error %u.\n\n", pThis->m_params.pUrl, pCurlMsg->data.result);
                        pThis->Log(" * Details: %s\n\n", pEasyData[iEasy].pErrorBuffer);
                        isSuccess = false;
                        break;
                    }

                    break;
                }
            }
        }
    } while(pCurlMsg);

    if( pPollParams )
    {
        delete [] pPollParams;
        pPollParams = nullptr;
    }

    return isSuccess;
}

// CleanUp
void LibCurl::Helper::CleanUp(LibCurl* pThis, CURLM*& pCurlMulti, CurlEasyData*& pCurlEasy, uint32_t easyCount) NN_NOEXCEPT
{
    pThis->Log("Cleaning up.\n");

    if( pCurlMulti )
    {
        if( pCurlEasy )
        {
            for( uint32_t iEasy = 0; iEasy < easyCount; ++iEasy )
            {
                if( pCurlEasy[iEasy].pCurl )
                {
                    curl_multi_remove_handle(pCurlMulti, pCurlEasy[iEasy].pCurl);
                    curl_easy_cleanup(pCurlEasy[iEasy].pCurl);
                    pCurlEasy[iEasy].pCurl = nullptr;
                }
                else
                {
                    break;
                }
            }
        }

        curl_multi_cleanup(pCurlMulti);
        pCurlMulti = nullptr;
    }
    else if( pCurlEasy && pCurlEasy[0].pCurl )
    {
        curl_easy_cleanup(pCurlEasy[0].pCurl);
        pCurlEasy[0].pCurl = nullptr;
    }

    if( pCurlEasy )
    {
        delete [] pCurlEasy;
        pCurlEasy = nullptr;
    }
}

// Run
bool LibCurl::Run() NN_NOEXCEPT
{
    bool isSuccess = true;
    CURLM* pCurlMulti = nullptr;
    CurlEasyData* pCurlEasy = nullptr;
    uint32_t easyCount = 1;

    if( !VerifyParams() )
    {
        return false;
    }

    for(uint32_t iCurlHandle = 0; iCurlHandle < (uint32_t)m_params.curlHandleLoop; ++iCurlHandle)
    {
        Log("CurlHandle Loop: %d\n", iCurlHandle);

        if( m_params.multiApiConnectionCount > 0 )
        {
            pCurlMulti = curl_multi_init();
            if( nullptr == pCurlMulti )
            {
                LogError(" * Error: Failed to create Curl Multi object.\n\n");
                return false;
            }

            easyCount = m_params.multiApiConnectionCount;
        }

        pCurlEasy = new CurlEasyData[easyCount];
        for(uint32_t iEasy = 0; iEasy < easyCount; ++iEasy)
        {
            pCurlEasy[iEasy].pThis = this;
            pCurlEasy[iEasy].index = iEasy;
        }

        uint32_t iEasyCurl = 0;
        do
        {
            if(m_params.isAutoUrlEnabled)
            {
                uint32_t hostIndex = (iEasyCurl + iCurlHandle) % (sizeof(Utils::g_pInternetTestHosts) / sizeof(Utils::g_pInternetTestHosts[0]));
                NETTEST_SNPRINTF(m_params.pUrl, sizeof(m_params.pUrl), "%s", Utils::g_pInternetTestHosts[hostIndex]);
            }

            pCurlEasy[iEasyCurl].pCurl = curl_easy_init();

            if( nullptr == pCurlEasy[iEasyCurl].pCurl ) // Failed to create curl handle
            {
                LogError(" * ERROR: Could not create libcurl easy handle.\n\n");
                isSuccess = false;
                break;
            }

            CURLoption curlSizeOpt = (CURLoption)0;

            Helper::ConfigureGeneralOps(this, &pCurlEasy[iEasyCurl]);

            if( strncmp("https", m_params.pUrl, 5) == 0 )
            {
                Log(" Using SSL\n");
            }

            if( !Helper::ConfigureSpecifcOps(this, &pCurlEasy[iEasyCurl], curlSizeOpt) )
            {
                isSuccess = false;
                break;
            }

            if( !Helper::ConfigureUploadOps(this, &pCurlEasy[iEasyCurl], curlSizeOpt) )
            {
                isSuccess = false;
                break;
            }

            Helper::ConfigureProxy(this, &pCurlEasy[iEasyCurl]);

            if( pCurlMulti )
            {
                curl_multi_add_handle(pCurlMulti, pCurlEasy[iEasyCurl].pCurl);
            }
        } while( ++iEasyCurl < easyCount );

        if( isSuccess )
        {
            // Use the same pCurlEasy[iEasyCurl].pCurl handles "m_params.curlPerformLoop" many times.
            for(uint32_t iPerform = 0; iPerform < (uint32_t)m_params.curlPerformLoop; ++iPerform)
            {
                Log("Loops: %d, %d\n", iCurlHandle, iPerform);

                if( pCurlMulti )
                {
                    isSuccess = Helper::MultiPerform(this, pCurlMulti, pCurlEasy, easyCount);
                }
                else
                {
                    isSuccess = Helper::EasyPerform(this, pCurlEasy);
                }

                if( !isSuccess )
                {
                    break;
                }
            } // Curl Perform Loop
        }

        Helper::CleanUp(this, pCurlMulti, pCurlEasy, easyCount);
    } // Curl Handle Loop

    return isSuccess;
}

}} // Namespace NATF::Modules
