﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NetTest_Port.h"
#include "Modules/IoctlModule.h"

#include <cstring> // for memset
#include <cstdlib> // for malloc

#define NET_EXPECT_OUT(CONDITION) \
do                                \
{                                 \
    if(!(CONDITION))              \
    {                             \
        if( NetTest::GetLastError() != expectedErr )    \
        {                                              \
            LogError("Condition: %s, Failed : errno: %d\n", \
                 #CONDITION, NetTest::GetLastError());  \
            isSuccess = false;                         \
            goto out;                                  \
        }                                              \
    }                        \
                             \
} while(false)

namespace NATF {
namespace Modules {

// ClientTcp
IoctlModule::AppErr IoctlModule::ClientTcp(unsigned& totalSent, unsigned& totalRecvd, const char* pSendBuffer, char* pRecvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT
{

    int rval,dataWrite,dataSpace;

    rval = NetTest::Send(m_params.socketFd, pSendBuffer, m_params.bufferSize, m_params.xferOptions);
    if (rval < 0 )
    {
        nn::socket::Errno err = NetTest::GetLastError();
        if (err == expectedErr)
        {
            rval = 0;
            return AppErr_Continue;
        }
        else
        {
            LogError("Error send: rval: %d, errno: %d\n", rval, err);
            return AppErr_Out;
        }
    }

    rval = NetTest::Ioctl(m_params.socketFd, nn::socket::IoctlCommand::FionWrite,&dataWrite,sizeof(dataWrite));
    if( rval < 0 )
    {
        nn::socket::Errno err = NetTest::GetLastError();
        LogError("Error Ioctl nn::socket::IoctlCommand::FionWrite: %d, errno: %d\n", rval, err);
        return AppErr_Out;
    }

    rval = NetTest::Ioctl(m_params.socketFd, nn::socket::IoctlCommand::FionSpace,&dataSpace,sizeof(dataSpace));
    if( rval < 0 )
    {
        nn::socket::Errno err = NetTest::GetLastError();
        LogError("Error Ioctl nn::socket::IoctlCommand::FionSpace: %d, errno: %d\n", rval, err);
        return AppErr_Out;
    }

    if(!m_params.clientHalf && dataWrite > m_params.socketSendSize / 2)
    {
        m_params.clientHalf = true;
        Log("Send Queue Half Full: Bytes In Send Queue (nn::socket::IoctlCommand::FionWrite): %d,Space In Send Queue (nn::socket::IoctlCommand::FionSpace): %d, Data Packet Size: %d\n",dataWrite,dataSpace,m_params.bufferSize);
    }
    if(!m_params.clientFull && dataWrite > m_params.socketSendSize - m_params.bufferSize)
    {
        m_params.clientFull = true;
        Log("Send Queue Full: Bytes In Send Queue (nn::socket::IoctlCommand::FionWrite): %d,Space In Send Queue (nn::socket::IoctlCommand::FionSpace): %d, Data Packet Size: %d\n",dataWrite,dataSpace,m_params.bufferSize);
    }

    return AppErr_Ok;
}

// ServerTcp
IoctlModule::AppErr IoctlModule::ServerTcp(unsigned& totalSent, unsigned& totalRecvd, const char* pSendBuffer, char* pRecvBuffer, nn::socket::Errno expectedErr) NN_NOEXCEPT
{

    int rval,data;
    rval = NetTest::Ioctl(m_params.socketFd, nn::socket::IoctlCommand::FionRead,&data,sizeof(data));
    if( rval < 0 )
    {
        nn::socket::Errno err = NetTest::GetLastError();
        LogError("Error Ioctl nn::socket::IoctlCommand::FionRead: %d, errno: %d\n", rval, err);
        return AppErr_Out;
    }

    if(!m_params.serverFull && ((m_params.socketRecvSize - data) < (int)m_params.bufferSize))
    {
        m_params.serverFull = true;
        Log("Recieve Queue Full. nn::socket::IoctlCommand::FionRead value: %d, Specified Receive Queue Size: %d, Data Packet Size: %d \n",data,m_params.socketRecvSize,m_params.bufferSize);
    }



    return AppErr_Ok;
}
// SetOpt
bool IoctlModule::SetOpt() NN_NOEXCEPT
{
    bool isSuccess = true;
    int rval = 0;
    int value = 0;

    if( m_params.isNonBlocking )
    {
        rval = NetTest::Fcntl(m_params.socketFd, static_cast<int>(nn::socket::FcntlCommand::F_SetFl), nn::socket::FcntlFlag::O_NonBlock);
        if (rval != 0)
        {
            LogError("Error: Fcntl(nn::socket::FcntlFlag::O_NonBlock) failed. rval: %d\n", rval);
            isSuccess = false;
            goto out;
        }
    }
    if( m_params.socketSendSize > 0 )
    {
        value = m_params.socketSendSize;
        rval = NetTest::SetSockOpt(m_params.socketFd, nn::socket::Level::Sol_Socket,
                nn::socket::Option::So_SndBuf, &value, sizeof(int));
        if (rval < 0)
        {
            Log("Warning: setsockopt(nn::socket::Option::So_SndBuf) failed! rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
        }
    }
    if( m_params.socketRecvSize > 0 )
    {
        value = m_params.socketRecvSize;
        rval = NetTest::SetSockOpt(m_params.socketFd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &value, sizeof(int));
        if( rval < 0 )
        {
            Log("Warning: setsockopt(nn::socket::Option::So_RcvBuf) failed! rval: %d, errno: %d\n\n", rval, NetTest::GetLastError());
        }
    }

out:
    return isSuccess;
}

// Init
bool IoctlModule::Init(nn::socket::Errno expectedErr) NN_NOEXCEPT
{
    bool isSuccess = true;

    if( (m_params.initType == InitType_All) || (m_params.initType == InitType_Init) )
    {
        if( !NetTest::Init() )
        {
            if( nn::socket::Errno::ESuccess != expectedErr )
            {
                Log("Expected NetTest::Init() failure.\n");
            } else
            {
                LogError("Error: NetTest::Init!\n");
                isSuccess = false;
                goto out;
            }
        }
    }
    if( (m_params.initType == InitType_All) || (m_params.initType == InitType_Socket) )
    {
        int rval = NetTest::Socket(m_params.socketFamily, m_params.socketType, m_params.socketProtocol);
        NET_EXPECT_OUT(rval >= 0);

        Log("Socket created: %d\n", rval);
        m_params.socketFd = rval;
        SetOpt();
    }
out:

    return isSuccess;
}

// Setup
bool IoctlModule::Setup(nn::socket::Errno expectedErr) NN_NOEXCEPT
{
    bool isSuccess = true;
    int rval;

    memset((void *) &m_params.remoteAddr, 0, sizeof(m_params.remoteAddr));
    m_params.addrLen = sizeof(m_params.remoteAddr);

    if( m_params.appType == AppType_Client )
    {
        m_params.remoteAddr.sin_family = nn::socket::Family::Af_Inet;
        m_params.remoteAddr.sin_port = NetTest::Htons(m_params.port);
        rval = NetTest::InetAddrPton(nn::socket::Family::Af_Inet, m_params.pIp, &m_params.remoteAddr.sin_addr);
        NET_EXPECT_OUT(rval >= 0);

        /* Wait for server to start */
        Log("Connecting...\n");
        rval = NetTest::Connect (m_params.socketFd, (NetTest::SockAddr *)&m_params.remoteAddr,  sizeof(m_params.remoteAddr));
        NET_EXPECT_OUT(rval >= 0);
    } else
    {
        NetTest::SockAddrIn localAddr;
        memset ((void *) &localAddr, 0, sizeof(NetTest::SockAddrIn));
        localAddr.sin_family = nn::socket::Family::Af_Inet;
        localAddr.sin_port = NetTest::Htons(m_params.port);
        localAddr.sin_addr.S_addr = nn::socket::InAddr_Any;

        rval = NetTest::Bind(m_params.socketFd, (NetTest::SockAddr *)&localAddr, sizeof(localAddr));
        NET_EXPECT_OUT(rval == 0);
        if( m_params.socketType == nn::socket::Type::Sock_Stream )
        {
            rval = NetTest::Listen (m_params.socketFd, m_params.backLog);
            NET_EXPECT_OUT(rval == 0);

            Log("Listening for connection on %d. backlog: %d\n", m_params.socketFd, m_params.backLog);
            m_params.addrLen = sizeof(m_params.remoteAddr);

            Log("Accepting...\n");
            rval = NetTest::Accept(m_params.socketFd, (NetTest::SockAddr *)&m_params.remoteAddr, &m_params.addrLen);
            NET_EXPECT_OUT(rval >= 0);

            NetTest::Close(m_params.socketFd);
            m_params.socketFd = rval;

            Log("Accept socket(%d) over port %d successful\n", rval, (int)NetTest::Ntohs(m_params.remoteAddr.sin_port));
        }
    }
out:

    return isSuccess;
}

// Teardown
bool IoctlModule::Teardown(nn::socket::Errno expectedErr) NN_NOEXCEPT
{
    bool isSuccess = true;
    int rval;

    if( (m_params.teardownType == TeardownType_All) || (m_params.teardownType == TeardownType_Shutdown) )
    {
        rval = NetTest::Shutdown(m_params.socketFd, m_params.shutdownType);
        NET_EXPECT_OUT(rval == 0);
    }
    if( (m_params.teardownType == TeardownType_All) || (m_params.teardownType == TeardownType_Close) )
    {
        rval = NetTest::Close(m_params.socketFd);
        NET_EXPECT_OUT(rval == 0);
    }

out:

    return isSuccess;
}

// DataXfer

bool IoctlModule::DataXfer(nn::socket::Errno expectedErr) NN_NOEXCEPT
{
    bool isSuccess = true;
    char sendBuffer[MaxBufSize];
    char recvBuffer[MaxBufSize];
    NetTest::Tick end, start;
    NetTest::Time duration;
    unsigned totalSent  = 0;
    unsigned totalRecvd = 0;
    int rval = 0;
    nn::socket::Errno err  = nn::socket::Errno::ESuccess;

    if( m_params.bufferSize > MaxBufSize )
    {
        LogError("bufferSize is too large, Max size: %d\n", MaxBufSize);
        isSuccess = false;
        goto out;
    }

    memset((void *)sendBuffer, 'a', m_params.bufferSize);
    memset((void *)recvBuffer, 'b', m_params.bufferSize);

    rval = NetTest::Fcntl(m_params.socketFd, static_cast<int>(nn::socket::FcntlCommand::F_SetFl), nn::socket::FcntlFlag::O_NonBlock);
    if( rval != 0 )
    {
        err = NetTest::GetLastError();
        LogError("Error: fcntl(nn::socket::FcntlCommand::F_SetFl) failed! rval: %d errno: %d\n", rval, err);
        isSuccess = false;
        goto out;
    }

    if(IoctlCheck("Initial") == AppErr_Out)
    {
        isSuccess = false;
        goto out;
    }

    start = NetTest::GetTick();
    while( NetTest::TickToTime(NetTest::GetTick() - start).GetSeconds() <= m_params.duration )
    {
        AppErr appErr = AppErr_Ok;

        if( m_params.appType == AppType_Client )
        {
            appErr = ClientTcp(totalSent, totalRecvd, sendBuffer, recvBuffer, expectedErr);
        }
        else if( m_params.appType == AppType_Server )
        {
            appErr = ServerTcp(totalSent, totalRecvd, sendBuffer, recvBuffer, expectedErr);
        }
        else
        {
            LogError("Error: Unknown AppType! AppType: %i\n", m_params.appType);
            isSuccess = false;
            goto out;
        }

        switch( appErr )
        {
        case AppErr_Ok:
        case AppErr_Continue:
            break;
        case AppErr_Out:
            isSuccess = false;
            goto out;
        default:
            LogError("Error! Unknown return error!\n");
            isSuccess = false;
            goto out;
        }

        if(m_params.appType == AppType_Client && m_params.clientHalf && m_params.clientFull){
            goto out;
        }
        else if(m_params.serverFull)
        {
            goto out;
        }


        NetTest::SleepMs(1);
    }

    isSuccess = false;

out:

    IoctlCheck("Final");

    return isSuccess;
}

IoctlModule::AppErr IoctlModule::IoctlCheck(const char * checkID)
{
    if( m_params.appType == AppType_Client )
    {
        int rval,data;
        rval = NetTest::Ioctl(m_params.socketFd, nn::socket::IoctlCommand::FionWrite,&data,sizeof(data));
        if( rval < 0 )
        {
            LogError("Error Ioctl nn::socket::IoctlCommand::FionWrite: %d, errno: %d\n", rval, NetTest::GetLastError());
            return AppErr_Out;
        }
        Log("%s Bytes In Client Send Queue (nn::socket::IoctlCommand::FionWrite): %d\n",checkID,data);

        rval = NetTest::Ioctl(m_params.socketFd, nn::socket::IoctlCommand::FionSpace,&data,sizeof(data));
        if( rval < 0 )
        {
            LogError("Error Ioctl nn::socket::IoctlCommand::FionSpace: %d, errno: %d\n", rval, NetTest::GetLastError());
            return AppErr_Out;
        }
        Log("%s Space Remaning In Client Send Queue (nn::socket::IoctlCommand::FionSpace): %d\n",checkID,data);
    }
    else
    {
        int rval,data;
        rval = NetTest::Ioctl(m_params.socketFd, nn::socket::IoctlCommand::FionRead,&data,sizeof(data));
        if( rval < 0 )
        {
            LogError("Error Ioctl nn::socket::IoctlCommand::FionRead: %d, errno: %d\n", rval, NetTest::GetLastError());
            return AppErr_Out;
        }
        Log("%s Bytes In Server Recieve Queue (nn::socket::IoctlCommand::FionRead): %d\n",checkID,data);
    }
    return AppErr_Ok;
}
// Constructor
IoctlModule::IoctlModule(AppType appType, const char* pIp, unsigned short port, InitType initType) NN_NOEXCEPT
{
    memset(&m_params, 0, sizeof(m_params));

    m_params.isNonBlocking       = false;
    m_params.socketRecvSize      = 128 * 1024;
    m_params.socketSendSize      = 128 * 1024;
    m_params.appType             = appType;
    m_params.pIp                 = pIp;
    m_params.port                = port;
    m_params.backLog             = 2;
    m_params.shutdownType        = nn::socket::ShutdownMethod::Shut_RdWr;
    m_params.duration            = 5; // Seconds
    m_params.bufferSize          = 1024;
    m_params.initType            = initType;

    m_params.socketFd            = -1;
    m_params.socketProtocol      = nn::socket::Protocol::IpProto_Tcp;
    m_params.socketFamily        = nn::socket::Family::Af_Inet;
    m_params.xferOptions         = nn::socket::MsgFlag::Msg_None;
    m_params.expectedInitErr     = nn::socket::Errno::ESuccess;
    m_params.expectedSetupErr    = nn::socket::Errno::ESuccess;
    m_params.expectedXferErr     = nn::socket::Errno::EWouldBlock;
    m_params.expectedTeardownErr = nn::socket::Errno::ENotConn;

    m_params.socketType   = nn::socket::Type::Sock_Stream;
    m_params.teardownType = TeardownType_All;

    m_params.serverFull          = false;
    m_params.clientHalf          = false;
    m_params.clientFull          = false;

}

// Run
bool IoctlModule::Run() NN_NOEXCEPT
{
    bool isSuccess = true;
    const int sleepCountSec = 2;

    if( m_params.appType == AppType_Client )
    {
        Log("Sleeping %d Seconds...\n", sleepCountSec);
        NetTest::SleepMs(sleepCountSec * 1000);
        Log("Done Sleeping!\n");
    }

    Log("Calling Init...\n");
    if( !Init(m_params.expectedInitErr) )
    {
        isSuccess = false;
        LogError("SoTestInit failed\n");
        goto out;
    }

    Log("Calling Setup...\n");
    if( !Setup(m_params.expectedSetupErr) )
    {
        isSuccess = false;
        LogError("SoTestSetup\n");
        goto out;
    }

    Log("Calling DataXfer...\n");
    if( !DataXfer(m_params.expectedXferErr) )
    {
        isSuccess = false;
        LogError("SoTestDataXfer, errno: %d\n", NetTest::GetLastError());
        goto out;
    }
    Log("Data xfer complete\n");

    Log("Calling Teardown...\n");
    if( !Teardown(m_params.expectedTeardownErr) )
    {
        LogError("Error: SoTestTeardown\n");
        isSuccess = false;
    }
out:

    return isSuccess;
}

// GetName
const char* IoctlModule::GetName() const NN_NOEXCEPT
{ return "IoctlModule"; }

}
}
