﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <poll.h>
#include <cstdarg>

namespace NetTest {

/* Types */

typedef sockaddr_in SockAddrIn;
typedef sockaddr    SockAddr;
typedef pollfd      PollFd;
typedef socklen_t   SockLen;
typedef fd_set      FdSet;
typedef timeval     TimeVal;
typedef hostent     HostEnt;

/* APIs */

inline bool Init()
{ return true; }

inline void Finalize()
{ return; }

inline int Socket(int family, int socketType, int proto)
{ return socket(family, socketType, proto); }

inline int Close(int fd)
{ return close(fd); }

inline int Shutdown(int fd, int how)
{ return shutdown(fd, how); }

inline int Bind(int fd, const SockAddr* pAddr, SockLen addrLen)
{ return bind(fd, pAddr, addrLen); }

inline int Listen(int fd, int backLog)
{ return listen(fd, backLog); }

inline int Accept(int fd, SockAddr* pAddr, SockLen* pAddrLen)
{ return accept(fd, pAddr, pAddrLen); }

inline int Connect(int fd, const SockAddr* pAddr, SockLen addrLen)
{ return connect(fd, pAddr, addrLen); }

inline int Send(int fd, const void* pBuf, uint32_t bufLen, int flags)
{ return send(fd, pBuf, bufLen, flags); }

inline int SendTo(int fd, const void* pBuf, uint32_t bufLen, int flags, SockAddr* pAddr, SockLen addrLen)
{ return sendto(fd, pBuf, bufLen, flags, pAddr, addrLen); }

inline int Recv(int fd, void* pBuf, uint32_t bufLen, int flags)
{ return recv(fd, pBuf, bufLen, flags); }

inline int RecvFrom(int fd, void* pBuf, uint32_t bufLen, int flags, SockAddr* pAddr, SockLen* pAddrLen)
{ return recvfrom(fd, pBuf, bufLen, flags, pAddr, pAddrLen); }

inline int GetSockName(int fd, SockAddr* pAddr, SockLen* pAddrLen)
{ return getsockname(fd, pAddr, pAddrLen); }

inline int GetPeerName(int fd, SockAddr* pAddr, SockLen* pAddrLen)
{ return getpeername(fd, pAddr, pAddrLen); }

inline int GetSockOpt(int fd, int level, int optName, void* pVal, SockLen* pValLen)
{ return getsockopt(fd, level, optName, pVal, pValLen); }

inline int SetSockOpt(int fd, int level, int optName, const void* pVal, SockLen valLen)
{ return setsockopt(fd, level, optName, pVal, valLen); }

inline int Fcntl(int fd, int command, ...)
{
    int     iocmd;
    va_list args;
    va_start(args, command);
    iocmd = va_arg(args, int);
    va_end(args);
    return fcntl(fd, command, iocmd);
}

inline int Ioctl(int socket, uint32_t command, void* pData, size_t dataLength)
{ return ioctl(socket,command, pData, dataLength); }

inline int Poll(PollFd* pFds, nfds_t fdCount, int timeout)
{ return poll(pFds, fdCount, timeout); }

inline int Select(int fdCount, FdSet* pReadFds, FdSet* pWriteFds, FdSet* pErrFds, TimeVal* pTimeVal)
{ return select(fdCount, pReadFds, pWriteFds, pErrFds, pTimeVal); }

inline int GetLastError()
{ return errno; }

inline int InetAddrPton(int family, const char* pStr, void* pDest)
{ return inet_pton(family, pStr, pDest); }

inline HostEnt* GetHostByName(const char* pName) NN_NOEXCEPT
{ return gethostbyname(pName); }

inline int16_t Htons(int16_t val)
{ return htons(val); }

inline int16_t Ntohs(int16_t val)
{ return ntohs(val); }

inline int32_t Htonl(int32_t val)
{ return htonl(val); }

inline int32_t Ntohl(int32_t val)
{ return ntohl(val); }

} // namespace NetTest
