﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "testNet_ApiCommon.h"
#include "Unit/testNet_ApiUnitCommon.h"

#include <cstdio>     // sprintf
#include <cctype>     // isprint
#include <cstdlib>    // malloc

#include <nn/os.h>
#include <nn/nn_Log.h>

#include <nn/socket.h>
#include <nnt/nntest.h>

namespace NATF {
namespace API {

/////////////////////////
//  C O N S T A N T S
/////////////////////////



////////////////////////////////////////////////
//// D N S     H E A D E R      R E C O R D
//////////////////////////////////////////////////

#ifndef DNS_HEADER_REC

struct DNS_HEADER_REC {

    uint16_t  Id :16;              /* query identification number */

#ifdef NN_BUILD_CONFIG_OS_WIN32                // BIG_ENDIAN

            /* fields in third byte */
    uint8_t   QR: 1;        /* response flag */
    uint8_t   opcode: 4;    /* purpose of message */
    uint8_t   AA: 1;        /* authoritive answer */
    uint8_t   TC: 1;        /* truncated message */
    uint8_t   RD: 1;        /* recursion desired */
            /* fields in fourth byte */
    uint8_t   RA: 1;        /* recursion available */
    uint8_t   unused :1;    /* unused bits (MBZ as of 4.9.3a3) */
    uint8_t   AD: 1;        /* authentic data from named */
    uint8_t   CD: 1;        /* checking disabled by resolver */
    uint8_t   rcode :4;     /* response code */

#else                            // LITTLE ENDIAN
            /* fields in third byte */
    uint8_t   RD :1;        /* recursion desired */
    uint8_t   TC :1;        /* truncated message */
    uint8_t   AA :1;        /* authoritive answer */
    uint8_t   opcode :4;    /* purpose of message */
    uint8_t   QR :1;        /* response flag */
            /* fields in fourth byte */
    uint8_t   rcode :4;     /* response code */
    uint8_t   CD: 1;        /* checking disabled by resolver */
    uint8_t   AD: 1;        /* authentic data from named */
    uint8_t   unused :1;    /* unused bits (MBZ as of 4.9.3a3) */
    uint8_t   RA :1;        /* recursion available */

#endif
            /* remaining bytes */
    uint16_t  qdcount :16;  /* number of question entries */
    uint16_t  ancount :16;  /* number of answer entries */
    uint16_t  nscount :16;  /* number of authority entries */
    uint16_t  arcount :16;  /* number of resource entries */

};

#endif

/////////////////////////
// LoopbackDnsServer
/////////////////////////

class LoopbackDnsServer
{
    public:

        //////////////
        //  Constants
        //////////////

        static const uint16_t    kDnsServerUDPPort = 8053;    // UDP port to listen on
        static const uint16_t    kDnsServerTCPPort = 8053;    // TCP port to listen on
        static const nn::Bit64   kDnsServerCore = 1;          // Run on NX Core 1


        //////////////
        //  Buffer overlays
        //////////////

        struct DNS_HEADER_REC *   pDnsHeader;

        //////////////
        //  Switches
        /////////////

        bool                      dnsServerShouldRun;          // true if DNS server (should run)
        bool                      dnsServerRunning;            // true if DNS server (is running)

        //////////////
        //  Response Buffers
        //////////////

        unsigned char *           UDPResponse;         // Pointer to a UDP Response to send back to BIONIC
        size_t                    UDPResponseLen;

        unsigned char *           TCPResponse;         // Pointer to a UDP Response to send back to BIONIC
        size_t                    TCPResponseLen;

        //////////////
        //  Socket Address
        //////////////

        char                      udpHost[128];
        uint16_t                  udpPort;

        char                      tcpHost[128];
        uint16_t                  tcpPort;


        //////////////
        //  Socket Fds
        //////////////

        int                       serverUDP;
        int                       serverTCP;
        int                       acceptedTCP;

        ////////////////////////////
        //  Socket Address Structs
        ////////////////////////////

        nn::socket::SockAddrIn        udpAddr;
        nn::socket::SockLenT                 udpAddrSize;

        nn::socket::SockAddrIn        tcpAddr;
        nn::socket::SockLenT                 tcpAddrSize;

        nn::socket::SockAddrIn        recvAddr;
        nn::socket::SockLenT                 recvAddrSize;

        ////////////////////////////
        //  DNS method Callbacks
        ////////////////////////////


        int (*DNSRequestCB)     (void *thisObj, unsigned char *, size_t &, bool *);
        int (*DNSResponseCB)    (void *thisObj, unsigned char *, size_t &, unsigned char *, size_t &, bool *, bool *);

        // Constructor
        LoopbackDnsServer();

        // Constructor + args
        LoopbackDnsServer( const char * in_udpHost, uint16_t in_udpPort,
                           const char * in_tcpHost, uint16_t in_tcpPort );

        // Destructor
        virtual ~LoopbackDnsServer();

        // Set UDP Response message
        void SetUDPResponse( unsigned char * dnsResp, size_t & dnsLen )
        {
            UDPResponse = dnsResp;
            UDPResponseLen  = dnsLen;
        }

        // Set TCP Response message
        void SetTCPResponse( unsigned char * dnsResp, size_t & dnsLen )
        {
            TCPResponse = dnsResp;
            TCPResponseLen  = dnsLen;
        }

        // Set DNS Request Message CB
        void SetDNSRequestCB( int (*newDNSRequestCB)
                              (void *thisObj, unsigned char *, size_t &, bool *) )
        {
            DNSRequestCB = *newDNSRequestCB;
        }

        // Set DNS Response Message CB
        void SetDNSResponseCB( int (*newDNSResponseCB)
                  (void * thisObj, unsigned char *, size_t &, unsigned char *, size_t &, bool *, bool *) )
        {
            DNSResponseCB = *newDNSResponseCB;
        }

        // Set location of DNS Resolver
        bool SetDnsResolver( const char *  dnsHost, uint16_t   dnsPort );

        // Make DNS Request CB
        //
        // Important: static
        static int MakeDnsRequestCB( void *               thisObj,
                                     unsigned char *      dnsRqst,
                                     size_t &             dnsRqstLen,
                                     bool *               isUDP );


        // Make DNS Response CB
        static int MakeDnsResponseCB( void *              thisObj,
                                      unsigned char *     dnsRqst,
                                      size_t &            dnsRqstLen,
                                      unsigned char *     dnsResp,
                                      size_t &            dnsRespLen,
                                      bool *              isUDP,
                                      bool *              shouldContinue );

        void PrintHex( unsigned char *    buff,
                       size_t &           buffLen,
                       bool *             isReq,
                       bool *             isUDP );

        int ProcessDns( unsigned char *    dnsRqst,
                        size_t &           dnsRqstLen,
                        unsigned char *    dnsResp,
                        size_t &           dnsRespLen,
                        bool *             isUDP,
                        bool *             shouldContinue );

        // Thread start DNS Server
        void Start( void *  arg);

        // Stop DNS Server
        int Stop();
};

}}   // namespace: NATF/API
