﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/socket_Api.h>

#include "Complex/testNet_UnitCommon.h" // SimpleValidator

namespace NATF {
namespace API {

/**
 * @brief Address Family flags indicate the protocol type used for this socket
 */
enum AddressFamilyFlags
{
    /**
     * @brief default flags are equal to unspecified
     */
    ADDRESS_FAMILY_FLAGS_DEFAULT = 1,

    /**
     * @brief nn::socket::Family::Af_Unspec; nn::socket::Family::Af_Unspec or unspecified protocol
     */
    ADDRESS_FAMILY_FLAGS_AF_UNSPEC = 1,

    /**
     * @brief nn::socket::Family::Af_Inet; nn::socket::Family::Af_Inet or internet socket protocol
     */
    ADDRESS_FAMILY_FLAGS_AF_INET = 2,

    /**
     * @brief max, used for checking
     */
    ADDRESS_FAMILY_FLAGS_MAX = 4,

    /**
     * @brief bit max for shift count
     */
    ADDRESS_FAMILY_FLAGS_NUMBITS = 2,
};

/**
 * @brief socket type flags indicate the type of socket to create
 */
enum SocketTypeFlags
{
    /**
     * @brief Default (unusable) flags
     */
    SOCK_TYPE_FLAGS_DEFAULT = 0,

    /**
     * @brief nn::socket::Type::Sock_Stream, TCP mutually exclusive to ST_SOCK_DGRAM
     */
    SOCK_TYPE_FLAGS_SOCK_STREAM = 1,

    /**
     * @brief nn::socket::Type::Sock_Dgram, UDP; mutually exclusive with ST_SOCK_DGRAM
     */
    SOCK_TYPE_FLAGS_SOCK_DGRAM = 2,

    /**
     * @brief bit max for shift count
     */
    SOCKET_TYPE_FLAGS_NUMBITS = 2,
};

/**
 * @brief Protocol flags given to socket, presently just default
 */
enum ProtocolFlags
{
    /**
     * @brief default (i.e. unused) flags
     */
    PROTOCOL_FLAGS_DEFAULT = 1,

    /**
     * @brief max, used for checking
     */
    PROTOCOL_FLAGS_MAX = 2,

    /**
     * @brief bit max for shift count
     */
    PROTOCOL_FLAGS_NUMBITS = 1,
};

/**
 * @brief socket IO flags used for ioctl / fcntl
 */
enum SocketControlFlags
{
    /**
     * @brief default (i.e. unused) flags
     */
    SOCKET_CONTROL_FLAGS_DEFAULT = 0,

    /**
     * @brief set non-blocking option on this socket
     */
    SOCKET_CONTROL_FLAGS_SOCK_NONBLOCKING = 2,

    /**
     * @brief max, used for checking
     */
    SOCKET_CONTROL_FLAGS_MAX = 3,

    /**
     * @brief max, used for checking
     */
    SOCKET_CONTROL_FLAGS_NUMBITS = 2,
};

/**
 * @brief SelectUnitData assigns sockets to specific sets for specific tests
 * that the actor (i.e. ACTOR_CLIENT/ACTOR_SERVER) operates on. Use the
 * SocketContainerListType to retrieve the set of SocketContainers (ie. sockets with metadata)
 * from SelectUnitData
 */
enum SocketContainerTypeFlags
{
    /** @brief we do not know what kind of socket this is */
    SOCKET_CONTAINER_TYPE_EMBRYONIC = 1,

    /** this is a listening (i.e. server) socket */
    SOCKET_CONTAINER_TYPE_LISTEN    = 2,

    /** @brief indicates a read-type socket */
    SOCKET_CONTAINER_TYPE_READ      = 4,

    /** @brief indicates a write-type socket */
    SOCKET_CONTAINER_TYPE_WRITE     = 8,

    /** @brief indicates an exception occurred, such as oob data */
    SOCKET_CONTAINER_TYPE_EXCEPTION = 16,

    /** @brief indicates the socket is marked for shutdown */
    SOCKET_CONTAINER_TYPE_SHUTDOWN = 32,

    /** @brief indicates the socket is marked for error*/
    SOCKET_CONTAINER_TYPE_ERROR = 64,
};

/**
 * @brief provide the actor as a key to get the proper set of sockets
 * When retrieving the socket container list from SelectUnitData
 */
enum SocketContainerActor
{
    SOCKET_CONTAINER_ACTOR_EMBRYONIC = 0,
    SOCKET_CONTAINER_ACTOR_CLIENT    = 1,
    SOCKET_CONTAINER_ACTOR_SERVER    = 2,
};

const char* SocketContainerActorAsString(const SocketContainerActor role);

/**
 * @brief API Unit tests use this structure to contain basic socket metadata
 * socket-related data are const (i.e. address family, socket type, protocol, and the socket)
 * because they are set at socket creation time and cannot be reset.
 *
 * The validator is const because socket data needs to depend on using it for validation and
 * must not change the validator.
 *
 * On the other hand other other data (socketControlFlags and returnValue) are not const
 * because they can be changed by the underling functions (fcntl/ioctl for socketControlFlags
 * and returnCode)
 */
struct SocketContainer
{
public:
    /**
     * @brief ctor
     */
    SocketContainer();

    /**
     * @brief the address family used for the socket
     */
    AddressFamilyFlags m_AddressFamilyFlags;

    /**
     * @brief the socket type flags
     */
    SocketTypeFlags m_SocketTypeFlags;

    /**
     * @brief the protocol flags
     */
    ProtocolFlags m_ProtocolFlags;

    /**
     * @brief the socket file descriptor
     */
    int m_Socket;

    /**
     * @brief if this socket is a listening socket then this is the backlog
     */
    uint32_t m_Backlog;

    /**
     * @brief the control flags for the socket
     */
    SocketControlFlags m_SocketControlFlags;

    /**
     * @brief the container type
     */
    SocketContainerTypeFlags m_ContainerTypeFlags;

    /**
     * @brief the current state of the socket
     */
    SocketContainerTypeFlags m_CurrentState;

    /**
     * @brief the role this socket plays
     */
    SocketContainerActor m_Role;

    /**
     * @brief get the size of a SocketContainer for network transfer
     */
    static size_t SizeOf();

    /**
     * @brief turn this SocketContainer into a buffer for network transfer
     * @param pBufferInOut a pointer to a buffer, this pointer is modified
     * @param bufferSize the size of the buffer (left)
     * @return 0 on success, error otherwise
     */
    int ToNetworkBuffer(uint8_t *& pBufferInOut, size_t bufferSize) const;

    /**
     * @brief turn this network buffer (created with ToNetworkBuffer) to a SocketContainer
     * @param pBufferInOut a pointer to a buffer, this pointer is modified
     * @param bufferSize the size of the buffer (left)
     * @return 0 on success, error otherwise
     */
    static int FromNetworkBuffer(SocketContainer& socketContainerOut, uint8_t * pBufferIn, size_t bufferSize);

    /**
     * @brief comparitor
     */
    bool operator==(const SocketContainer& o) const;
};

/**
 * @brief an interface that is used to manage the socket container list
 */
class ISocketContainerListManager
{
public:
    /** @brief get a copy of the socket container list */
    virtual void GetSocketContainerListCopy(std::list<SocketContainer>& socketContainerListOut) const = 0;

    /** @brief add a socket container to the list */
    virtual void AddSocketContainerToList(const SocketContainer& socketContainerIn) = 0;

    /** @brief remove a socket container from the list */
    virtual void RemoveSocketContainerFromList(const SocketContainer& socketContainerIn) = 0;
};

}}; // namespace NATF::API
