﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/ae.h>
#include <nn/init.h>
#include <nn/la/la_Api.h>
#include <nn/la/la_CommonArgumentsReader.h>
#include <nn/nd/nd_ApiForSystem.h>
#include <nn/nd/nd_NeighborInfoForSystem.h>
#include <nn/os.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include "testNd_Util.h"

#include <nnt.h>

using namespace nn;

namespace
{

const char* DummyEncounterSystemDataMagicString = "DUMMY_ENCOUNTER: ";

void ConsumeNeighborInfo(const account::Uid& uid) NN_NOEXCEPT
{
    int receivableCount = nd::GetReceivableNeighborInfoCountForSystem(uid);
    if( receivableCount == 0 )
    {
        NN_LOG("[DummyEncounter] No data is receivable for uid<%016x-%016x>.\n", uid._data[0], uid._data[1]);
        return;
    }
    NN_LOG("[DummyEncounter] Receivable %d.\n", receivableCount);
    for( int i = 0; i < receivableCount; i++ )
    {
        nd::NeighborInfoForSystem neighborInfo;
        std::memset(&neighborInfo, 0, sizeof(nd::NeighborInfoForSystem));
        int neighborUserDataCount = nd::ReceiveNeighborInfoForSystem(&neighborInfo, 1, uid);
        if( neighborUserDataCount == 0 )
        {
            break;
        }
        nnt::nd::DumpNeighborInfoForSystem(neighborInfo);
        if( std::memcmp(DummyEncounterSystemDataMagicString, neighborInfo.systemData, std::strlen(DummyEncounterSystemDataMagicString)) == 0 )
        {
            NN_LOG("[DummyEncounter] %s\n", reinterpret_cast<char*>(neighborInfo.systemData));
        }
    }
}

void WaitAndConsumeNeighborInfo(const account::Uid& uid) NN_NOEXCEPT
{
    auto pUpdateEvent = nd::GetNeighborInfoUpdateEventForSystem();

    while( NN_STATIC_CONDITION(true) )
    {
        pUpdateEvent->Wait();
        pUpdateEvent->Clear();
        ConsumeNeighborInfo(uid);
    }
}

} //~<anonymous>

void LibraryAppletEncounterMain(const nn::ae::LibraryAppletSelfInfo& info) NN_NOEXCEPT
{
    NN_LOG("[DummyEncounter] ===== DummyEncounter Start =====\n");
    NN_UTIL_SCOPE_EXIT{ NN_LOG("[DummyEncounter] ===== DummyEncounter End =======\n"); };

    nn::la::CommonArgumentsReader commonArgumentsReader;

    ASSERT_TRUE(commonArgumentsReader.TryPopFromInChannel());

    /* 引数は未使用。
        NN_LOG("CommonArguments:\n");
        NN_LOG("  laVersion = %d\n", commonArgumentsReader.GetLaVersion());
        NN_LOG("  colorIndex = %d\n", commonArgumentsReader.GetColorIndex());
        NN_LOG("  isPlayStartupSound = %s\n", commonArgumentsReader.IsPlayStartupSound() ? "true" : "false");
    */

    NN_ABORT_UNLESS_RESULT_SUCCESS(time::Initialize());
    account::InitializeForSystemService();

    account::Uid uid;
    if( !nnt::nd::GetAccountUid(&uid, 0) )
    {
        NN_LOG("[DummyEncounter] No account is registered. DummyEncounter can not work without a user.\n");
        return;
    }

    nd::InitializeForSystem();

    // 適当な送信データの設定。
    settings::system::SerialNumber serialNumber;
    settings::system::GetSerialNumber(&serialNumber);
    char systemData[nd::SystemDataSizeMax];
    int l = util::SNPrintf(systemData, sizeof(systemData), "%sHello from uid<%016llx-%016llx> in %s.\n",
        DummyEncounterSystemDataMagicString, uid._data[0], uid._data[1], serialNumber.string);

    nd::SetSystemData(uid, systemData, static_cast<size_t>(l));
    nd::SetSender(uid);

    // 起動前に貯まっていた分の消化。
    ConsumeNeighborInfo(uid);
    // 新しいデータを受信次第ログに表示する。無限ループ。
    WaitAndConsumeNeighborInfo(uid);
}

extern "C" void nninitStartup()
{
}

extern "C" void nnMain()
{
    nn::ae::InvokeLibraryAppletMain(LibraryAppletEncounterMain);
}
