﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/account.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/mem.h>
#include <nn/nd/nd_Api.h>
#include <nn/nd/nd_ApiForDebug.h>
#include <nn/nd/nd_ApiForSystem.h>
#include <nn/ndd.h>
#include <nn/os.h>
#include <nn/time.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "testNd_Util.h"

using namespace nn;

namespace nnt { namespace nd {

namespace {
    const int ThreadCountMax = 4;
    const size_t ThreadStackSize = 8 * 1024;
    os::ThreadType g_ThreadType[ThreadCountMax] = {};
    NN_ALIGNAS(os::ThreadStackAlignment) Bit8 g_ThreadStack[ThreadCountMax][ThreadStackSize];

    template <typename T = int>
    T Rand(T mod = 0) NN_NOEXCEPT
    {
        T t;
        os::GenerateRandomBytes(&t, sizeof(T));
        if( mod == 0 )
        {
            return t;
        }
        else
        {
            // mod を使うのなら （0 ~ mod -1) の正の値が欲しいのだろうという忖度。
            return (t > 0) ? (t % mod) : (-t % mod);
        }
    }

    settings::LanguageCode GetRandomLanguage() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(settings::LanguageCode, s_LanguageCode[32]);
        NN_FUNCTION_LOCAL_STATIC(int, s_LangCount, = nn::settings::GetAvailableLanguageCodes(s_LanguageCode, NN_ARRAY_SIZE(s_LanguageCode)));
        NN_SDK_ASSERT_LESS_EQUAL(s_LangCount, NN_ARRAY_SIZE(s_LanguageCode));
        return s_LanguageCode[Rand(s_LangCount)];
    }
} // ~nnt::nd::<anonymous>

bool AreEqualNeighborInfoForSystemExceptReceviedTimePoint(const nn::nd::NeighborInfoForSystem& lhs, const nn::nd::NeighborInfoForSystem& rhs) NN_NOEXCEPT
{
    return true
        // lhs.receivedTimePoint == rhs.receivedTimePoint
        && lhs.systemLanguageCode == rhs.systemLanguageCode
        && lhs.localUserId == rhs.localUserId
        && lhs.networkUserId == rhs.networkUserId
        && util::Strnlen(lhs.nickName.name, static_cast<int>(sizeof(lhs.nickName.name))) == util::Strnlen(rhs.nickName.name, static_cast<int>(sizeof(rhs.nickName.name)))
        && util::Strncmp(lhs.nickName.name, rhs.nickName.name, util::Strnlen(lhs.nickName.name, static_cast<int>(sizeof(lhs.nickName.name)))) == 0
        && lhs.recentlyPlayedApplicationCount == rhs.recentlyPlayedApplicationCount
        && std::memcmp(lhs.recentlyPlayedApplication, rhs.recentlyPlayedApplication, sizeof(ncm::ApplicationId) * lhs.recentlyPlayedApplicationCount) == 0
        && lhs.systemDataSize == rhs.systemDataSize
        && std::memcmp(lhs.systemData, rhs.systemData, lhs.systemDataSize) == 0;
}

bool GetAccountUid(account::Uid* pOut, int index) NN_NOEXCEPT
{
    account::Uid uid[account::UserCountMax];
    int accountCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&accountCount, uid, NN_ARRAY_SIZE(uid)));
    if( !(index < accountCount) )
    {
        return false;
    }
    *pOut = uid[index];
    return true;
}

bool GetFirstAccountUid(account::Uid* pOut) NN_NOEXCEPT
{
    int accountCount;
    NN_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&accountCount, pOut, 1));
    return accountCount >= 1;
}

int RunOnThread(void(*f)(void*), void* arg) NN_NOEXCEPT
{
    for( int i = 0; i < ThreadCountMax; i++ )
    {
        if( g_ThreadType[i]._state == os::ThreadType::State_NotInitialized )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(os::CreateThread(&g_ThreadType[i], f, arg, g_ThreadStack[i], ThreadStackSize, os::DefaultThreadPriority));
            os::StartThread(&g_ThreadType[i]);
            return i;
        }
    }
    NN_ABORT("Unable to create more than %d threads\n", ThreadCountMax);
}

void DestroyThread(int i) NN_NOEXCEPT
{
    os::DestroyThread(&g_ThreadType[i]);
}

void MakeDummyNeighborInfoForsystemWithRandomData(nn::nd::NeighborInfoForSystem* pOut) NN_NOEXCEPT
{
    std::memset(pOut, 0, sizeof(nn::nd::NeighborInfoForSystem));

    pOut->systemLanguageCode = GetRandomLanguage();
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&pOut->receivedTimePoint));
    pOut->receivedTimePoint -= nn::TimeSpan::FromSeconds(Rand<int64_t>(24 * 60 * 60));
    auto uuid = util::GenerateUuid();
    std::memcpy(pOut->localUserId._data, uuid.data, sizeof(pOut->localUserId._data));
    if( Rand<>(2) == 0 || true )
    {
        pOut->networkUserId = util::nullopt;
    }
    else
    {
        pOut->networkUserId = nn::nd::NetworkUserId{ Rand<Bit64>() };
    }
    util::SNPrintf(pOut->nickName.name, sizeof(pOut->nickName), "DummyNickName%d", Rand<>());

    pOut->recentlyPlayedApplicationCount = Rand(3) + 1;
    for( int i = 0; i < pOut->recentlyPlayedApplicationCount; i++ )
    {
        pOut->recentlyPlayedApplication[i].value = Rand<Bit64>();
    }

    pOut->systemDataSize = Rand<size_t>(256);
    os::GenerateRandomBytes(pOut->systemData, pOut->systemDataSize);
}

void DumpNeighborInfoForSystem(const nn::nd::NeighborInfoForSystem& info) NN_NOEXCEPT
{
    NN_LOG("----------------------------------------\n");
    time::SteadyClockTimePoint currentTimePoint;
    NN_ABORT_UNLESS_RESULT_SUCCESS(time::StandardSteadyClock::GetCurrentTimePoint(&currentTimePoint));
    int64_t seconds;
    auto getTimeSpanResult = time::GetSpanBetween(&seconds, info.receivedTimePoint, currentTimePoint);
    if( getTimeSpanResult.IsSuccess() )
    {
        NN_LOG("Received %lld seconds ago (%lld vs %lld)\n", seconds, info.receivedTimePoint.value, currentTimePoint.value);
    }
    else
    {
        NN_LOG("Unagled to get recevied time : %03d-%04d (0x%08x)\n", getTimeSpanResult.GetModule(), getTimeSpanResult.GetDescription(), getTimeSpanResult.GetInnerValueForDebug());
    }
    NN_LOG("Language : %s\n", info.systemLanguageCode.string);
    util::Uuid uuid;
    std::memcpy(uuid.data, info.localUserId._data, sizeof(uuid.data));
    char idStr[util::Uuid::StringSize];
    NN_LOG("LocalUserId : %s\n", uuid.ToString(idStr, sizeof(idStr)));
    if( info.networkUserId )
    {
        NN_LOG("NetworkUserId : %016x\n", info.networkUserId->_id);
    }
    else
    {
        NN_LOG("NetworkUserId : None\n");
    }
    NN_LOG("Nickname : %s\n", info.nickName.name);
    NN_LOG("Recently Played Application (%d)\n", info.recentlyPlayedApplicationCount);
    for( int i = 0; i < info.recentlyPlayedApplicationCount; i++ )
    {
        NN_LOG("    0x%016llx\n", info.recentlyPlayedApplication[i].value);
    }
    NN_LOG("System Data (%zu)\n", info.systemDataSize);
    for( size_t i = 0; i < info.systemDataSize; i++ )
    {
        NN_LOG("%02x ", info.systemData[i]);
        if( i % 16 == 15 )
        {
            NN_LOG("\n");
        }
    }
    NN_LOG("\n");
}

}} // ~nnt::nd
