﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <string>
#include <random>
#include <chrono>
#include <set>
#include <tuple>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/htc.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/kvdb/kvdb_InMemoryKeyValueStore.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentMetaDatabaseImpl.h>
#include <nn/ncm/ncm_IContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nnt/fsUtil/testFs_util_function.h>

namespace {
    uint8_t g_HeapMemory[48 * 1024 * 1024];
    nn::mem::StandardAllocator g_Allocator;

    typedef std::string String;

    void InitializeHeap()
    {
        g_Allocator.Initialize(g_HeapMemory, sizeof(g_HeapMemory));
    }

    void* Allocate(size_t size)
    {
        return g_Allocator.Allocate(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        g_Allocator.Free(p);
    }

    // TODO: Windows Temp ではなく、ツリーのテスト出力用のディレクトリに書きたいところ
    String CreateTemporaryRootPath()
    {
        return String(nnt::fs::util::GetHostTemporaryPath())
            .append("\\")
            .append("SIGLO_NCM_TEST")
            .append("_")
            .append("InMemoryKeyValueStoreTest");
    }

    nn::Result ShowDirectoryTreeRecursively(const char* rootPath, int indent)
    {
        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, rootPath, static_cast<nn::fs::OpenDirectoryMode>(nn::fs::OpenDirectoryMode_File | nn::fs::OpenDirectoryMode_Directory)));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(dir); };

        for (;;)
        {
            nn::fs::DirectoryEntry entry;
            int64_t numEntries;
            NN_RESULT_DO(nn::fs::ReadDirectory(&numEntries, &entry, dir, 1));
            if (numEntries == 0)
            {
                break;
            }

            for (auto i = 0; i < indent; i++)
            {
                NN_LOG("  ");
            }
            switch (entry.directoryEntryType)
            {
            case nn::fs::DirectoryEntryType_Directory:
                {
                    NN_LOG("%s/\n", entry.name);
                    ShowDirectoryTreeRecursively((std::string(rootPath) + "/" + entry.name).c_str(), indent + 1);
                }
                break;
            case nn::fs::DirectoryEntryType_File:
                {
                    NN_LOG("%s %lld bytes\n", entry.name, entry.fileSize);
                }
                break;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        NN_RESULT_SUCCESS;
    }

    nn::Result ShowDirectoryTree(const char* rootPath)
    {
        NN_LOG("Show directory tree %s\n", rootPath);

        return ShowDirectoryTreeRecursively(rootPath, 0);
    }

    class InMemoryKeyValueStoreTest : public testing::Test
    {
    protected:
        virtual void SetUp()
        {
            auto rootPath = CreateTemporaryRootPath();
            NN_ABORT_UNLESS(nn::fs::CreateDirectory(rootPath.c_str()).IsSuccess());
            m_RootPath = rootPath;
        }

        virtual void TearDown()
        {
            NN_ABORT_UNLESS(nn::fs::DeleteDirectoryRecursively(GetRootPath()).IsSuccess());
        }

        static void SetUpTestCase()
        {
            InitializeHeap();
            nn::fs::SetAllocator(Allocate, Deallocate);
            NN_ABORT_UNLESS(nn::fs::MountHostRoot().IsSuccess());
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }

        const char* GetRootPath()
        {
            return m_RootPath.c_str();
        }

        std::vector<nn::ncm::ContentMetaKey> GetRandomContentMetaKeyList(int count)
        {
            std::vector<nn::ncm::ContentMetaKey> list;
            for (int i = 0; i < count; i++)
            {
                nn::ncm::ContentMetaKey key = {};
                key.type = static_cast<nn::ncm::ContentMetaType>(m_RandomEngine());
                key.id = m_RandomEngine();
                key.version = static_cast<uint16_t>(m_RandomEngine());

                list.push_back(key);
            }

            return list;
        }

        std::vector<std::string> GetRandomDataList(int count, size_t minLength, size_t maxLength)
        {
            std::vector<std::string> list;
            for (int i = 0; i < count; i++)
            {
                auto size = std::uniform_int_distribution<size_t>(minLength, maxLength)(m_RandomEngine);
                std::string data;
                for (size_t i = 0; i < size; i++)
                {
                    char c = static_cast<char>(m_RandomEngine());
                    data.append(1, c);
                }

                list.push_back(data);
            }

            return list;
        }

        std::string     m_RootPath;
        std::mt19937_64 m_RandomEngine;
    };

    template < typename ContainerType, typename EntryType >
    bool Contains(const ContainerType& container, EntryType entry)
    {
        for (auto& x : container)
        {
            if (x == entry)
            {
                return true;
            }
        }

        return false;
    }

    template<typename ContainerType>
    bool IsEquivalent(const ContainerType& l, const ContainerType& r)
    {
        if (l.size() != r.size())
        {
            return false;
        }

        for (auto& x : l)
        {
            if (!Contains(r, x))
            {
                return false;
            }
        }

        return true;
    }
}

TEST_F(InMemoryKeyValueStoreTest, ContentMetaKeyValueStore)
{
    NN_LOG("root path: %s\n", GetRootPath());

    const int TestDataCount = 10000;

    nn::ncm::ContentMetaKeyValueStore kvs;
    NNT_EXPECT_RESULT_SUCCESS(kvs.Initialize(TestDataCount, nn::sf::GetNewDeleteMemoryResource()));
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());

    auto contentMetaKeyList = GetRandomContentMetaKeyList(TestDataCount);
    auto valueList = GetRandomDataList(TestDataCount, 16, 512);

    for (int i = 0; i < TestDataCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(kvs.Put(contentMetaKeyList[i], valueList[i].c_str(), valueList[i].size()));
    }

    auto CheckFunc = [&]()
    {
        for (int i = 0; i < TestDataCount; i++)
        {
            size_t valueSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(kvs.GetSize(&valueSize, contentMetaKeyList[i]));
            EXPECT_EQ(valueList[i].size(), valueSize);

            nn::kvdb::AutoBuffer value;
            NNT_EXPECT_RESULT_SUCCESS(value.Initialize(valueSize));
            size_t readSize;
            NNT_EXPECT_RESULT_SUCCESS(kvs.Get(&readSize, contentMetaKeyList[i], value.Get(), value.GetSize()));
            EXPECT_TRUE(std::memcmp(valueList[i].c_str(), value.Get(), value.GetSize()) == 0);
        }
    };

    NN_LOG("Check before Save()\n");
    CheckFunc();

    NNT_EXPECT_RESULT_SUCCESS(kvs.Save());

    NN_LOG("Check after Save()\n");
    CheckFunc();

    NN_LOG("kvs.Load()\n");
    NNT_EXPECT_RESULT_SUCCESS(kvs.Load());

    NN_LOG("Check after Load()\n");
    CheckFunc();

    NNT_EXPECT_RESULT_SUCCESS(ShowDirectoryTree(GetRootPath()));
}


void* operator new(size_t size) NN_NOEXCEPT
{
    return Allocate(size);
}

void operator delete(void* pv) NN_NOEXCEPT
{
    Deallocate(pv, 0);
}

void* operator new[](size_t size) NN_NOEXCEPT
{
    return Allocate(size);
}

void operator delete[](void* pv) NN_NOEXCEPT
{
    Deallocate(pv, 0);
}
