﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#ifndef SFAUDIO_SOURCE_H_

#define SFAUDIO_SOURCE_H_

#include <media/stagefright/MediaSource.h>
#include <media/stagefright/MediaBufferGroup.h>

#include <cstring>
#include <utils/Timers.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_ThreadApi.h>

#include <nn/audio.h>
#include <nn/applet/applet_Apis.h>
#include <nn/audio/audio_FinalOutputRecorder.h>

struct SfAudioSource : public android::MediaSource {

    // Note that the "channels" parameter _is_ the number of channels,
    // _not_ a bitmask of audio_channels_t constants.
    SfAudioSource(int32_t sampleRate, int32_t numChannels);

    android::status_t initCheck() const;
    virtual android::status_t start(android::MetaData *params = NULL);
    virtual android::status_t stop();

    virtual android::sp<android::MetaData> getFormat();

    // Returns the maximum amplitude since last call.
    int16_t getMaxAmplitude();

    virtual android::status_t read(
            android::MediaBuffer **out, const ReadOptions *options = NULL);

protected:
    virtual ~SfAudioSource();


private:
    enum {
        kMaxBufferSize = 2048,

        // After the initial mute, we raise the volume linearly
        // over kAutoRampDurationUs.
        kAutoRampDurationUs = 300000,

        // This is the initial mute duration to suppress
        // the video recording signal tone
        kAutoRampStartUs = 0,
    };

    android::Mutex mLock;
    android::status_t mInitCheck;
    enum { kBufferSize = 4096 };

    bool mStarted;
    int32_t mSampleRate;
    int32_t mNumChannels;
    size_t mPhase;

    bool mTrackMaxAmplitude;
    int64_t mStartTimeUs;
    int16_t mMaxAmplitude;
    int64_t mNumFramesReceived;

    void trackMaxAmplitude(int16_t *data, int nSamples);

    // This is used to raise the volume from mute to the
    // actual level linearly.
    void rampVolume(
        int32_t startFrame, int32_t rampDurationFrames,
        uint8_t *data,   size_t bytes);

    android::MediaBufferGroup *mGroup;
    static const int numBuffers = 2;
    void *inBuffers[numBuffers];

    // For connection with NN's Audio APIs
    nn::audio::FinalOutputRecorder gameRecord;
    nn::audio::FinalOutputRecorderBuffer finalOutputRecorderBuffer[numBuffers];
    nn::os::SystemEvent gameRecordBufferEvent;

    int numDiffs;
};

#endif // SFAUDIO_SOURCE_H_
