﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
 /*
 * Copyright 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


#define LOG_TAG "NvVideoEncoder"
#include "NvVideoEncoder.h"
#include <OMX_Video.h>
using namespace android;

NvVideoEncoder::NvVideoEncoder( uint32_t width, uint32_t height,
                                uint32_t framerate, uint32_t noOfRecFrame)
    : kMimeTypeAvc("video/avc"),
      gVideoWidth(width),        // default width+height
      gVideoHeight(height),
      gBitRate(8000000),     // 8Mbps
      gFrameRate(framerate),
      gNumFrameRecord(noOfRecFrame),
      gVideoDone(false),
      mStopCalled(false) {


}

NvVideoEncoder:: ~NvVideoEncoder() {
    //destroy
    if (mVideoThread.get()) {
        NN_LOG_V("ScoopVideo: Closing video thread..\n");
        mVideoThread->join();
        mVideoThread.clear();
        mVideoThread = NULL;
        NN_LOG_V("ScoopVideo: video thread closed\n");
    }
}

status_t NvVideoEncoder::start(NvMediaRecorder *aRecorder, fnEvent evenClk) {
    mVideoThread = new NvVideoEncoder::VideoRecordThread();
    mMediaRecorder = aRecorder;
    mVideoThread->mEncoder = this;
    mEventCallback = evenClk;
    mVideoThread->run( "VideoRecordThread" );
    return NO_ERROR;
}
status_t NvVideoEncoder::stop() {
    NN_LOG_V("ScoopVideo :: Stopped called");
    mStopCalled = true;
    mCondition.signal();
    return NO_ERROR;
}

void NvVideoEncoder::VideoRecordThread::requestExit() {
    Thread::requestExit();
}
bool NvVideoEncoder::VideoRecordThread::threadLoop()
{
    Mutex::Autolock autoLock(mLock);
    status_t err = NO_ERROR;
    NN_LOG_V("ScoopVideo:: preparing video encoder\n");
    sp<MediaCodec> videoEnc = mEncoder->prepareVideoEncoder();

    NN_LOG_V("ScoopVideo:: video encoder is ready\n");
    if (err != NO_ERROR){
        videoEnc->release();
        return false;
    }

    err = mEncoder->runVideoEncoder(videoEnc, /*muxer*/NULL);

    NN_LOG_V("ScoopVideo:: video encoder run compelete");

    if (err != NO_ERROR){
        NN_LOG_E("encoder run fails\n");
        return false;
    }

    mEncoder->mCondition.wait(mLock);
    NN_LOG_V("ScoopVideo:: exit video thread\n");

    videoEnc->release();
    videoEnc.clear();

    return false;
}

sp<MediaCodec> NvVideoEncoder::prepareVideoEncoder(){

    status_t err;
    sp<AMessage> format = new AMessage;

    format->setInt32("width", gVideoWidth);
    format->setInt32("height", gVideoHeight);
    format->setString("mime", kMimeTypeAvc);
    format->setInt32("encoder", true);
    format->setInt32("color-format", OMX_COLOR_FormatAndroidOpaque);
    //format->setInt32("color-format", OMX_COLOR_FormatYUV420Planar);
    format->setInt32("bitrate", gBitRate);
    format->setFloat("frame-rate", gFrameRate);
    format->setInt32("i-frame-interval", 1);
    format->setInt32("prepend-sps-pps-to-idr-frames", 1);
    format->setInt32("profile", OMX_VIDEO_AVCProfileHigh);
    format->setInt32("level", OMX_VIDEO_AVCLevel41);
    NN_LOG_V("ScoopVideo:: New Looper\n");
    sp<ALooper> looper = new ALooper;
    looper->setName("screenrecord_looper");
    looper->start();
    NN_LOG_V("ScoopVideo:: Creating codec\n");
    sp<MediaCodec> mVideoEncoder = MediaCodec::CreateByType(looper, kMimeTypeAvc, true);

    if (mVideoEncoder == NULL) {
        NN_LOG_V( "ScoopVideo:: ERROR: unable to create %s codec instance\n",
                kMimeTypeAvc);
        return mVideoEncoder;
    }
    //mVideoEncoder->incStrong(&sRefBaseOwner);
    NN_LOG_V("ScoopVideo:: configuring\n");
    err = mVideoEncoder->configure(format, NULL, NULL,
            MediaCodec::CONFIGURE_FLAG_ENCODE);
    if (err != NO_ERROR) {
        NN_LOG_V( "ScoopVideo:: ERROR: unable to configure %s codec at %dx%d (err=%d)\n",
                kMimeTypeAvc, gVideoWidth, gVideoHeight, err);
        mVideoEncoder->release();
        return mVideoEncoder;
    }

    NN_LOG_V("ScoopVideo:: create encoder input surface\n");
    err = mVideoEncoder->createInputSurface(&mProducer);
    if (err != NO_ERROR) {
        NN_LOG_V("ScoopVideo:: ERROR: unable to create encoder input surface (err=%d)\n", err);
        mVideoEncoder->release();
        return mVideoEncoder;
    }
    NN_LOG_V("ScoopVideo:: Producer return from createInputSurface %p\n",mProducer.get());

    NN_LOG_V("ScoopVideo:: Starting codec\n");
    err = mVideoEncoder->start();
    if (err != NO_ERROR) {
        NN_LOG_V( "ScoopVideo:: ERROR: unable to start codec (err=%d)\n", err);
        mVideoEncoder->release();
        return mVideoEncoder;
    }
    NN_LOG_V("ScoopVideo:: Codec prepared\n");
    //mVideoEncoder->incStrong(&sRefBaseOwner);
    return mVideoEncoder;
}

status_t NvVideoEncoder::runVideoEncoder(const sp<MediaCodec>& encoder,
        const sp<MediaMuxer>& muxer/*, FILE* rawFp*/) {
    static int kTimeout = 250000;   // be responsive on signal
    status_t err;
    int trackIdx = -1;
    uint32_t nFrames = 0;
    int64_t baseTs = 0;

    NN_LOG_V("ScoopVideo:: runVideoEncoder \n");
    Vector<sp<ABuffer> > buffers;
    err = encoder->getOutputBuffers(&buffers);
    if (err != NO_ERROR) {
        NN_LOG_V("ScoopVideo:: Unable to get output buffers (err=%d)\n", err);
        return err;
    }

    while (nFrames <= gNumFrameRecord) {
        size_t bufIndex, offset, size;
        int64_t ptsUsec;
        uint32_t flags;

        NN_LOG_V("ScoopVideo:: Calling dequeueOutputBuffer\n");
        err = encoder->dequeueOutputBuffer(&bufIndex, &offset, &size, &ptsUsec,
                &flags, kTimeout);
        NN_LOG_V("ScoopVideo:: dequeueOutputBuffer returned %d\n", err);
        switch (err) {
        case NO_ERROR:
            // got a buffer
            if ((flags & MediaCodec::BUFFER_FLAG_CODECCONFIG) != 0) {
                NN_LOG_V("ScoopVideo:: Got codec config buffer (%zu bytes)\n", size);
            }
            if (size != 0) {
                if (baseTs == 0) {
                    baseTs = ptsUsec;
                    NN_LOG_V("ScoopVideo:: base timestamp of video is set to %lld", baseTs);
                }
                ptsUsec = ptsUsec - baseTs;
                NN_LOG_V("ScoopVideo:: Got data in buffer %zu, size=%zu, pts=%lld baseTs - %lld\n",bufIndex, size, ptsUsec, baseTs);

                 {
                        err = mMediaRecorder->feedVideoData(buffers[bufIndex],ptsUsec, flags);
                        if (err != NO_ERROR) {
                            NN_LOG_V("ScoopVideo:: Failed feeding data to video recorder (err=%d)\n\n", err);
                            return err;
                        }
                }
                nFrames++;
            }
            err = encoder->releaseOutputBuffer(bufIndex);
            if (err != NO_ERROR) {
                NN_LOG_V("ScoopVideo:: Unable to release output buffer (err=%d)\n",
                        err);
                return err;
            }
            if ((flags & MediaCodec::BUFFER_FLAG_EOS) != 0) {
                // Not expecting EOS from SurfaceFlinger.  Go with it.
                NN_LOG_V("ScoopVideo:: Received end-of-stream\n");
            }
            break;
        case -EAGAIN:                       // INFO_TRY_AGAIN_LATER
            NN_LOG_V("ScoopVideo:: Got -EAGAIN, looping nFrames %d\n",nFrames);
            nFrames++;
            break;
        case INFO_FORMAT_CHANGED:           // INFO_OUTPUT_FORMAT_CHANGED
            {
                // Format includes CSD, which we must provide to muxer.
                NN_LOG_V("ScoopVideo:: Video Encoder format changed\n");
                sp<AMessage> newFormat;
                encoder->getOutputFormat(&newFormat);

                mMediaRecorder->setEncoderFormat(newFormat, false /*isAudio*/, &trackIdx);
                NN_LOG_V("ScoopVideo:: Video Encoder track index - %d\n", trackIdx);
            }
            break;
        case INFO_OUTPUT_BUFFERS_CHANGED:   // INFO_OUTPUT_BUFFERS_CHANGED
            // Not expected for an encoder; handle it anyway.
            NN_LOG_V("ScoopVideo:: Encoder buffers changed\n");
            err = encoder->getOutputBuffers(&buffers);
            if (err != NO_ERROR) {
                NN_LOG_V("ScoopVideo:: Unable to get new output buffers (err=%d)\n", err);
                return err;
            }
            break;
        case INVALID_OPERATION:
            NN_LOG_V("ScoopVideo:: dequeueOutputBuffer returned INVALID_OPERATION\n");
            return err;
        default:
            NN_LOG_V("ScoopVideo:: Got weird result %d from dequeueOutputBuffer\n", err);
            return err;
        }
    }
    gVideoDone = true;

    NN_LOG_V("ScoopVideo:: Sending EOS of input stream..\n");
    encoder->signalEndOfInputStream();
    mEventCallback(1); //SV_VIDEO_END

    return NO_ERROR;
}//NOLINT(impl/function_size)
