﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "SfAudioSink.h"
#include <utils/Timers.h>
#include <nn/audio.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
using namespace android;



#define Now() ns2ms( systemTime() )

#ifdef _DEBUG
//#define CAPTURE
#endif

#ifdef CAPTURE
#if defined(WIN32)
#include "WinStorage.h"
#define PlatformStorage WinStorage
#elif defined(CAFE)
#include "CafeStorage.h"
#define PlatformStorage CafeStorage
#elif defined(NN_PLATFORM_CTR)
#include "CTRStorage.h"
#define PlatformStorage CTRStorage
#endif
#endif

// moved here and made them static since const function does not
// support non-static data members as lvalue.
static unsigned int mTime0;
static unsigned int mFrames0;
static const unsigned int BUF_SIZE = 4 * 32768;
ssize_t gAudioBuffer[BUF_SIZE];


SfAudioSink::SfAudioSink()
{
    //MO_LOG_CHANNEL_REGISTER(Audio);
    mBufferSize = 12288 * 4; //1200*4; // DEFAULT_AUDIOSINK_BUFFERSIZE
    mChannels = 0;
    mSampleRate = 0;

    mFrameSize = sizeof(short);
    mTime0 = 0;
    mFrames0 = 0;
    mFramesWritten = 0;

    mAwesomeCallback = NULL;
    mAwesomeCallbackUser = NULL;
}

SfAudioSink::~SfAudioSink()
{
}

bool SfAudioSink::ready() const
{
    return true;
}

bool SfAudioSink::realtime() const
{
    return false;
}

ssize_t SfAudioSink::bufferSize() const
{
    return mBufferSize;
}

ssize_t SfAudioSink::frameCount() const
{
    return mBufferSize / mFrameSize;
}

ssize_t SfAudioSink::channelCount() const
{
    return mChannels;
}

ssize_t SfAudioSink::frameSize() const
{
    return mFrameSize;
}

uint32_t SfAudioSink::latency() const
{
    if( mAwesomeCallback )
        return 30; //500;
    return 0;
}

float SfAudioSink::msecsPerFrame() const
{
    return 1000.f / mSampleRate;
}

android::status_t SfAudioSink::getFramesWritten(uint32_t *frameswritten) const
{
    // Not implemented yet
    return INVALID_OPERATION;
}

android::status_t SfAudioSink::getPosition(uint32_t *position) const
{
    unsigned int now = Now();
    unsigned int dt = (now - mTime0);
//  if( dt < latency() )
//      return INVALID_OPERATION;
    int64_t nSamplesFromTime = dt * (mSampleRate  / 1000);
    unsigned nSamples = mFrames0 + (int32_t)nSamplesFromTime;
    // NN_SDK_LOG("getPos: %d / %d\n", nSamples, mFramesWritten );
    if( nSamples > mFramesWritten )
    {
        nSamples = mFramesWritten;
        mTime0 = now;
        mFrames0 = mFramesWritten;
    }
    *position = nSamples;
    return 0;
}


android::status_t SfAudioSink::getTimestamp(AudioTimestamp &ts) const
{
    // not implemented yet
    return INVALID_OPERATION;
}

uint32_t SfAudioSink::getSampleRate() const
{
    // not implemented yet
    return 0;
}

audio_stream_type_t SfAudioSink::getAudioStreamType() const
{
    // not implemented yet
    return (audio_stream_type_t)0;

}




int SfAudioSink::getSessionId() const
{
    return 0;
}

status_t SfAudioSink::open(
    uint32_t sampleRate,
    int channelCount,
    audio_channel_mask_t channelMask,
    audio_format_t format,
    int bufferCount,
    MediaPlayerBase::AudioSink::AudioCallback cb,
    void *cookie,
    audio_output_flags_t flags,
    const audio_offload_info_t *offloadInfo )
{
    NN_SDK_LOG("open() %d %d cb=%p\n", sampleRate, channelCount, cb);
    mSampleRate = sampleRate;
    mChannels = channelCount;
    mFrameSize = mChannels*sizeof(short);
    mAwesomeCallback = cb;
    mAwesomeCallbackUser = cookie;
     //nn::audio::AudioOutParameter parameter = { mSampleRate };
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.sampleRate = mSampleRate;
    // connect the nn audio APIs
    numDiffs = 0;
    // abort unless we can open it in the current sampling rate
    NN_ABORT_UNLESS(
        nn::audio::OpenDefaultAudioOut(&audioOut, parameter).IsSuccess(),
       "Failed to open AudioOut."
    );

    NN_ABORT_UNLESS(channelCount == nn::audio::GetAudioOutChannelCount(&audioOut),
        "Default AudioOut and input channelCount differ"
        );

    return 0;
}

NN_ALIGNAS(4096) static char  outBuff0[8192];       // fix to allocate instead?  global storage not exactly good...
NN_ALIGNAS(4096) static char  outBuff1[8192];
NN_ALIGNAS(4096) static char  outBuff2[8192];
//NN_ALIGNAS(4096) static char  outBuff3[8192];     // is it 3 or 5?
//NN_ALIGNAS(4096) static char  outBuff4[8192];

status_t SfAudioSink::start()
{
    NN_SDK_LOG("start()\n");
    mTime0 = Now();
    mFrames0 = mFramesWritten;

    int channelCount = nn::audio::GetAudioOutChannelCount(&audioOut);
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);
    nn::audio::SampleFormat sampleFormat = nn::audio::GetAudioOutSampleFormat(&audioOut);
    NN_LOG("  Name: %s\n", nn::audio::GetAudioOutName(&audioOut));
    NN_LOG("  ChannelCount: %d\n", channelCount);
    NN_LOG("  SampleRate: %d\n", sampleRate);

    NN_ASSERT(sampleFormat == nn::audio::SampleFormat_PcmInt16);

    NN_LOG("AudioOut buffersize %d numBuffers %d \n", defaultBufferSize , numBuffers);

    outBuffers[0] = (char*)(((long)outBuff0) + 4095 & (~4095));     // round up to nearest 4K boundary
    outBuffers[1] = (char*)(((long)outBuff1) + 4095 & (~4095));     // (Shouldn't be necessary with NN_ALIGNAS(4096)???
    outBuffers[2] = (char*)(((long)outBuff2) + 4095 & (~4095));

    const size_t bufferSize = nn::util::align_up(defaultBufferSize, nn::audio::AudioOutBuffer::SizeGranularity);
    // initialize membuffers and zero them
    for (int i = 0; i < numBuffers; ++i)
    {
        NN_ASSERT(outBuffers[i]);
        memset(outBuffers[i],0,bufferSize);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffers[i], bufferSize, defaultBufferSize);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
    }


    // start audio out
    NN_ABORT_UNLESS(
        nn::audio::StartAudioOut(&audioOut).IsSuccess(),
        "Failed to start playback."
    );

    if( mAwesomeCallback )
    {
        mThread = new CallbackThread();
        mThread->mAudioSink = this;
        getPosition( &mLastCbFrames );
        mLastCbFrames -= mSampleRate*latency() / 1000;
        mThread->run( "AudioCallback" );
    }
    bufferCopy = false;
    bufferOffset = 0;
    amountToCopy = 0;
    return 0;
}

ssize_t SfAudioSink::write(const void* buffer, size_t size)
{
#ifdef CAPTURE
    mCapture.AddElements( (const char*)buffer, size );
#endif

    size_t dataSize = size;

    if (((size < defaultBufferSize) || (amountToCopy > 0)) && (amountToCopy < defaultBufferSize)) {
        bufferCopy = true;
        memcpy(decodeBuffer + amountToCopy, buffer, size);
        amountToCopy  += size;
        size_t nFrames = size / frameSize();
        mFramesWritten += nFrames;
        return size;
    }

    // get a released buffer and write the new output
    nn::audio::AudioOutBuffer* audioOutBuffer = nullptr;
    audioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
    if (!audioOutBuffer)
    {
      //  NN_SDK_LOG(" write call return 0  \n");
        return 0;
    }

    void* outBuffer = nn::audio::GetAudioOutBufferDataPointer(audioOutBuffer);
    size_t outSize = nn::audio::GetAudioOutBufferDataSize(audioOutBuffer);

    if (bufferCopy)
        dataSize = amountToCopy;

    if (outSize != dataSize) {
        numDiffs++;
        const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);
        nn::audio::SetAudioOutBufferInfo(audioOutBuffer, outBuffer, bufferSize, dataSize);
    }
    if (bufferCopy) {
        memcpy(outBuffer, decodeBuffer, dataSize);
        bufferCopy = false;
        amountToCopy = 0;
        nn::audio::AppendAudioOutBuffer(&audioOut, audioOutBuffer);
        return 0;
    } else {
        memcpy(outBuffer, buffer, dataSize);
        nn::audio::AppendAudioOutBuffer(&audioOut, audioOutBuffer);
    }


    size_t nFrames = size / frameSize();
    mFramesWritten += nFrames;
    return size;
}

void SfAudioSink::stop()
{
    NN_SDK_LOG("SfAudioSink::stop()\n");
    // stop audio out
    nn::audio::StopAudioOut(&audioOut) ;
    nn::audio::CloseAudioOut(&audioOut);

    bufferCopy = false;
    bufferOffset = 0;
    amountToCopy = 0;

    mTime0 = 0;
    mFrames0 = 0;
    mFramesWritten = 0;
    // delete the buffers allocated
   /* for (int i = 0; i < numBuffers; ++i)
    {
        delete[] outBuffers[i];
    }*/

    NN_SDK_LOG("stop()\n");
}

void SfAudioSink::flush()
{
    NN_SDK_LOG("flush()\n");
}

void SfAudioSink::pause()
{
    NN_SDK_LOG("pause()\n");
    if( mThread.get() )
    {
        mThread->requestExitAndWait();
        mThread = NULL;
    }
    mTime0 = Now();
    mFrames0 = mFramesWritten;
}

void SfAudioSink::close()
{
    NN_SDK_LOG(" SfAudioSink::close()\n");
    // NN audioOut closure
    //nn::audio::CloseAudioOut(&audioOut);


    NN_SDK_LOG("close()\n");
    if( mThread.get() )
    {
        mThread->requestExitAndWait();
        mThread = NULL;
    }

#ifdef CAPTURE
    if (mCapture.Count>0)
    {
        PlatformStorage store;
        MoFile* f = store.OpenFile( "audio_dump.raw", 'c', NULL );
        if( f )
        {
            f->Write( mCapture.Elements, mCapture.Count );
            f->Close();
            delete f;
        }
    }
#endif
}

bool SfAudioSink::CallbackThread::threadLoop()
{
    mAudioSink->ThreadStep();
    return true;
}

void SfAudioSink::ThreadStep()
{

    uint32_t nFrames;
    if( getPosition( &nFrames ) )
        return;

    int consumed = nFrames - mLastCbFrames;
    mLastCbFrames = nFrames;
    if( consumed<=0 )
        return;
    consumed *= frameSize();
    //mDummy.Count = 0;
    //void* p = mDummy.Prepare( consumed );
    void *p  = gAudioBuffer;
    void* cookie = NULL;
    cb_event_t event = (cb_event_t) 0;
    //cz_android_5.1 new callback function() with different set of parameters
    //int actual = (*mAwesomeCallback)( this, p, consumed, mAwesomeCallbackUser );
    int actual = (*mAwesomeCallback)( this, p, consumed,  cookie, event);
    NN_SDK_LOG("callback(%d) -> %d\n", consumed, actual );
    write( p, actual );
}
