﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
//Siglo networking is unstable, causing many false problems
//#define ENABLE_SOCKET
#include <algorithm>
#include <cinttypes>
#include <cstddef>
#include <cstdlib>
#include <string>
#include <vector>
#include <utility>
#define JPEG_INTERNAL_OPTIONS // For RGB_PIXELSIZE
#include <jpeglib.h>
#include <mm_MemoryManagement.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#if defined(SOFTWARE)
namespace nn
{
namespace nvjpg
{
using namespace nn::image::detail::jpeg;
}
}
#endif

#ifdef MOUNT_SDCARD
#include <nn/fs/fs_SdCardForDebug.h>
#endif

#ifdef ENABLE_SOCKET
#include <nn/nifm.h>
#include <nn/socket.h>
#endif

// System heap setup
namespace // 'unnamed'
{
nn::lmem::HeapHandle g_FsHeap;
}

void *FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void *p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

void FsCreateHeap()
{
    const int heapSize = 512 * 1024;
    static uint8_t s_FsHeapBuffer[heapSize];
    g_FsHeap = nn::lmem::CreateExpHeap(s_FsHeapBuffer, sizeof(s_FsHeapBuffer), nn::lmem::CreationOption_DebugFill);
}

void FsDestroyHeap()
{
    nn::lmem::DestroyExpHeap(g_FsHeap);
    g_FsHeap = 0;
}

namespace // 'unnamed'
{
nn::mem::StandardAllocator g_MultimediaAllocator;
}

void MMCreateAllocator()
{
    const int allocatorSize = 512 << 20;
    static char s_MMAllocatorBuffer[allocatorSize];
    g_MultimediaAllocator.Initialize(s_MMAllocatorBuffer, sizeof(s_MMAllocatorBuffer));
}

void MMDestroyAllocator()
{
    g_MultimediaAllocator.Finalize();
}

void *MMAlloc(size_t size, size_t alignment, void*)
{
    return g_MultimediaAllocator.Allocate(size, alignment);
}

void MMFree(void *p, void*)
{
    g_MultimediaAllocator.Free(p);
}

void *MMRealloc(void *p, size_t newSize, void*)
{
    return g_MultimediaAllocator.Reallocate(p, newSize);
}

// Process startup setup
extern "C" void nninitStartup()
{
    const size_t heapSize = 128 << 20;
    nn::Result result = nn::os::SetMemoryHeapSize(heapSize);
    EXPECT_TRUE(result.IsSuccess());

    uintptr_t address;
    const size_t mallocSize = 32 << 20;
    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    EXPECT_TRUE(result.IsSuccess());

    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

extern "C" int *__errno()
{
    static int i = 0;
    return &i;
}

// Test "fixture"
struct NvJpgSanityTest : public ::testing::Test
{
    static void SetUpTestCase();
    static void TearDownTestCase();

    virtual void SetUp();
    virtual void TearDown();

    static bool sdcardMounted;
    static const char *jpegPath;
    static std::vector<nn::nvjpg::JOCTET> data;
};

bool NvJpgSanityTest::sdcardMounted = false;
const char *NvJpgSanityTest::jpegPath = 0;
std::vector<nn::nvjpg::JOCTET> NvJpgSanityTest::data;
#ifdef ENABLE_SOCKET
NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];
#endif

void NvJpgSanityTest::SetUpTestCase()
{
    FsCreateHeap();
    MMCreateAllocator();
    NN_SDK_LOG("Calling nv::mm::SetAllocator\n");
    nv::mm::SetAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    NN_SDK_LOG("Calling nv::SetGraphicsAllocator\n");
    nv::SetGraphicsAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    nv::SetGraphicsServiceName("nvdrv:t");
    NN_SDK_LOG("Calling nv::SetGraphicsDevtoolsAllocator\n");
    nv::SetGraphicsDevtoolsAllocator(MMAlloc, MMFree, MMRealloc, nullptr);
    const int mmFirmwareMemorySize = 8 << 20;
    NN_ALIGNAS(4096) static char s_mmFirmwareMemory[mmFirmwareMemorySize];
    NN_SDK_LOG("Calling nv::InitializeGraphics\n");
    nv::InitializeGraphics(s_mmFirmwareMemory, sizeof(s_mmFirmwareMemory));
    const int argc = nn::os::GetHostArgc();
    const char * const * const argv = nn::os::GetHostArgv();
    NN_SDK_LOG("\n NvJpgSanity Entering Main\n");
    if(argc <= 1)
    {
        NN_SDK_LOG("\n NvJpgSanity::Usage - %s  <jpeg-file-path>\n", argv[0]);
        return;
    }
    jpegPath = argv[1];
#ifdef MOUNT_SDCARD
    nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
    if(resultSdcardMount.IsFailure())
    {
        NN_SDK_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
            resultSdcardMount.GetModule(),
            resultSdcardMount.GetDescription());
        return;
    }
    sdcardMounted = true;
#endif

#ifdef ENABLE_SOCKET
    nn::nifm::Initialize();
    nn::nifm::SubmitNetworkRequest();
    while(nn::nifm::IsNetworkRequestOnHold())
    {
        NN_SDK_LOG("Network request on hold\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    if(!nn::nifm::IsNetworkAvailable())
    {
        NN_SDK_LOG("Network initialization failed\n");
        nn::nifm::CancelNetworkRequest();
        return;
    }
    else
    {
        nn::Result res = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
                                                nn::socket::DefaultSocketMemoryPoolSize,
                                                nn::socket::DefaultSocketAllocatorSize,
                                                nn::socket::DefaultConcurrencyLimit);
        if(res.IsFailure())
        {
            nn::nifm::CancelNetworkRequest();
            NN_SDK_LOG("nn::socket::Initialize failed!\n");
            return;
        }
    }
    NN_SDK_LOG("Network initialized (supposedly)\n");
#endif

    const nn::Result resultHostMount = nn::fs::MountHostRoot();
    if(resultHostMount.IsFailure())
    {
        NN_SDK_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
            resultHostMount.GetModule(),
            resultHostMount.GetDescription());
        return;
    }

    const std::string token(jpegPath, 0, std::string(jpegPath).find(':'));
    if((token == "http") || (token == "https") || (token == "file"))
    {
    }
    // Check whether media exists if it is a local file
    else
    {
        nn::fs::FileHandle fileHandle{};
        const nn::Result result = nn::fs::OpenFile(&fileHandle, jpegPath, nn::fs::OpenMode_Read);
        if(result.IsFailure())
        {
            NN_SDK_LOG("\n Failed to open %s\n\n Exiting NvJpgSanity Test\n", jpegPath);
        }
        else
        {
            int64_t fileSize = 0;
            nn::fs::GetFileSize(&fileSize, fileHandle);
            data.resize(fileSize);
            nn::fs::ReadFile(fileHandle, 0, &data[0], fileSize);
            nn::fs::CloseFile(fileHandle);
            NN_SDK_LOG("\n NvJpgSanity:: data (%" PRIi64 ") : %x%x%x%x\n", fileSize, data[0], data[1], data[2], data[3]);
        }
    }
    NN_SDK_LOG("\n NvJpgSanity:: Input File Name : %s\n", jpegPath);
}

void NvJpgSanityTest::TearDownTestCase()
{
#ifdef ENABLE_SOCKET
    nn::socket::Finalize();
    nn::nifm::CancelNetworkRequest();
#endif

#ifdef MOUNT_SDCARD
    nn::fs::Unmount("sdcard");
#endif

    nn::fs::UnmountHostRoot();

    MMDestroyAllocator();
    FsDestroyHeap();
}

void NvJpgSanityTest::SetUp()
{
}

void NvJpgSanityTest::TearDown()
{
    decltype(data) swaptrick{};
    std::swap(data, swaptrick);
}

const char *color_space_to_string(nn::nvjpg::J_COLOR_SPACE color_space)
{
    const char *ret = "<unknown-color-space>";
    switch(color_space)
    {
    default:
    case nn::nvjpg::JCS_UNKNOWN:
        ret = "JCS_UNKNOWN";
        break;
    case nn::nvjpg::JCS_GRAYSCALE:
        ret = "JCS_GRAYSCALE";
        break;
    case nn::nvjpg::JCS_RGB:
        ret = "JCS_RGB";
        break;
    case nn::nvjpg::JCS_YCbCr:
        ret = "JCS_YCbCr";
        break;
    case nn::nvjpg::JCS_CMYK:
        ret = "JCS_CMYK";
        break;
    case nn::nvjpg::JCS_YCCK:
        ret = "JCS_YCCK";
        break;
#if !defined(SOFTWARE)
    case nn::nvjpg::JCS_RGBA_8888:
        ret = "JCS_RGBA_8888";
        break;
    case nn::nvjpg::JCS_RGB_565:
        ret = "JCS_RGB_565";
        break;
#endif
    }
    return ret;
}

int color_space_to_component_count(nn::nvjpg::J_COLOR_SPACE color_space)
{
    int ret = -1;
    switch(color_space)
    {
    default:
    case nn::nvjpg::JCS_UNKNOWN:
        break;
    case nn::nvjpg::JCS_GRAYSCALE:
        ret = 1;
        break;
    case nn::nvjpg::JCS_RGB:
        ret = RGB_PIXELSIZE;
        break;
    case nn::nvjpg::JCS_YCbCr:
        ret = 3;
        break;
    case nn::nvjpg::JCS_CMYK:
    case nn::nvjpg::JCS_YCCK:
        ret = 4;
        break;
#if !defined(SOFTWARE)
    case nn::nvjpg::JCS_RGBA_8888:
        ret = 4;
        break;
    case nn::nvjpg::JCS_RGB_565:
        ret = 3;
        break;
#endif
    }
    return ret;
}

void init_source(nn::nvjpg::j_decompress_ptr cinfo)
{
    ASSERT_FALSE(NvJpgSanityTest::data.empty());
    cinfo->src->next_input_byte = &NvJpgSanityTest::data[0];
    cinfo->src->bytes_in_buffer = NvJpgSanityTest::data.size();
}

nn::nvjpg::boolean fill_input_buffer(nn::nvjpg::j_decompress_ptr cinfo)
{
    return true;
}

void skip_input_data(nn::nvjpg::j_decompress_ptr cinfo, long num_bytes)
{
    cinfo->src->next_input_byte += num_bytes;
    ASSERT_LE(num_bytes, cinfo->src->bytes_in_buffer);
    cinfo->src->bytes_in_buffer -= num_bytes;
}

void term_source(nn::nvjpg::j_decompress_ptr)
{
    // No-op
}

//~ jpeg_start_output
//~ jpeg_finish_output
//~ jpeg_input_complete

struct decoder
{
    nn::nvjpg::jpeg_decompress_struct cinfo{};
    nn::nvjpg::jpeg_workbuf workbuf{};
    nn::nvjpg::jpeg_error_mgr jerr{};
    nn::nvjpg::jpeg_source_mgr sourcemgr{};
    explicit decoder(std::vector<char>& workvec);
    ~decoder();
    void output(std::vector<nn::nvjpg::JSAMPLE> *out);
};

decoder::decoder(std::vector<char>& workvec)
{
    // Allocate and initialize a JPEG decompression object
    workbuf.ptr = &workvec[0];
    workbuf.total = workvec.size();
    cinfo.workbuf = &workbuf;

    cinfo.err = nn::nvjpg::jpeg_std_error(&jerr);
    nn::nvjpg::jpeg_create_decompress(&cinfo);
#if !defined(SOFTWARE)
    EXPECT_TRUE(cinfo.tegra_acceleration);
#endif

    // Specify the source of the compressed data
    sourcemgr.init_source = &init_source;
    sourcemgr.fill_input_buffer = &fill_input_buffer;
    sourcemgr.skip_input_data = &skip_input_data;
    sourcemgr.resync_to_restart = &nn::nvjpg::jpeg_resync_to_restart;
    sourcemgr.term_source = &term_source;
    cinfo.src = &sourcemgr;

    // Read the header to obtain image info
    nn::nvjpg::jpeg_save_markers(&cinfo, JPEG_COM, 4);
    const int marker_app_max = 15;
    for(int marker = 0; marker < marker_app_max; ++marker)
    {
        nn::nvjpg::jpeg_save_markers(&cinfo, JPEG_APP0 + marker, 4);
    }
    nn::nvjpg::jpeg_read_header(&cinfo, false);
    NN_SDK_LOG("jpeg_consume_input = %d\n", nn::nvjpg::jpeg_consume_input(&cinfo));
    const auto multiscan = nn::nvjpg::jpeg_has_multiple_scans(&cinfo);
    NN_UNUSED(multiscan);
    NN_SDK_LOG("jpeg_has_multiple_scans = %s\n", multiscan ? "yes" : "no");
    for(nn::nvjpg::jpeg_saved_marker_ptr marker = cinfo.marker_list; marker; marker = marker->next)
    {
        const char *marker_name = "<unknown-marker>";
        char marker_num[] = "XX";
        if(marker->marker == JPEG_COM)
        {
            marker_name = "JPEG_COM";
            marker_num[0] = 0;
        }
        else if((JPEG_APP0 <= marker->marker) && (marker->marker <= JPEG_APP0 + marker_app_max))
        {
            marker_name = "JPEG_APP";
            std::snprintf(marker_num, sizeof(marker_num), "%d", marker->marker - JPEG_APP0);
        }
        NN_SDK_LOG("marker = %s%s (%d), original_length = %d, data_length = %d\n", marker_name, marker_num, marker->marker, marker->original_length, marker->data_length);
    }
    nn::nvjpg::jpeg_calc_output_dimensions(&cinfo);
    NN_SDK_LOG("output_width = %d, output_height = %d\n", cinfo.output_width, cinfo.output_height);
}

decoder::~decoder()
{
    // Finish decompression
    //~ nn::nvjpg::jpeg_finish_output(&cinfo);
    nn::nvjpg::jpeg_finish_decompress(&cinfo);

    // Release the JPEG decompression object
    nn::nvjpg::jpeg_destroy_decompress(&cinfo);
}

void decoder::output(std::vector<nn::nvjpg::JSAMPLE> *out)
{
    // Set parameters for decompression
#if !defined(SOFTWARE)
    if(cinfo.out_color_space == nn::nvjpg::JCS_RGB_565)
    {
        cinfo.rgb565 = true;
    }
#endif
    // Start decompression
    nn::nvjpg::jpeg_start_decompress(&cinfo);
    //~ nn::nvjpg::jpeg_start_output(&cinfo, cinfo.input_scan_number);
    // Read the scanlines
    out->resize(cinfo.output_height*cinfo.output_width*cinfo.output_components);
    while(cinfo.output_scanline < cinfo.output_height)
    {
        nn::nvjpg::JSAMPLE *scanlines = &(*out)[cinfo.output_scanline * cinfo.output_width * cinfo.output_components];
        const nn::nvjpg::JDIMENSION lines_read = nn::nvjpg::jpeg_read_scanlines(&cinfo, &scanlines, 1);
        NN_UNUSED(lines_read);
        NN_SDK_LOG("lines_read = %d, output_scanline = %d, output.size() = %d\n", lines_read, cinfo.output_scanline, out->size());
    }
    NN_SDK_LOG("out_color_space = %s\n", color_space_to_string(cinfo.out_color_space));
    EXPECT_EQ(color_space_to_component_count(cinfo.out_color_space), cinfo.output_components);
}

TEST_F(NvJpgSanityTest, Decode)
{
    ASSERT_FALSE(data.empty());
    NN_SDK_LOG("data.size() = %d\n", data.size());
    const nn::nvjpg::J_COLOR_SPACE color_spaces[] =
    {
        nn::nvjpg::JCS_GRAYSCALE,
        nn::nvjpg::JCS_RGB,
#if !defined(SOFTWARE)
        nn::nvjpg::JCS_YCbCr,
        nn::nvjpg::JCS_CMYK,
        nn::nvjpg::JCS_YCCK,
        // Appears unsupported
        //~ nn::nvjpg::JCS_RGBA_8888,
        nn::nvjpg::JCS_RGB_565,
#endif
    };
    std::vector<char> workvec((data.size() * sizeof(std::int32_t) * 3) / 2);
    for (int c = 0; c < sizeof(color_spaces) / sizeof(*color_spaces); ++c)
    {
        decoder dec(workvec);
        std::vector<nn::nvjpg::JSAMPLE> output(dec.cinfo.output_height * dec.cinfo.output_width * dec.cinfo.output_components);
        dec.cinfo.out_color_space = color_spaces[c];
        dec.output(&output);
    }
}

std::vector<nn::nvjpg::JSAMPLE> create_image_data(nn::nvjpg::JDIMENSION width, nn::nvjpg::JDIMENSION height, nn::nvjpg::J_COLOR_SPACE colorspace)
{
    std::vector<nn::nvjpg::JSAMPLE> ret;
    const int colors = color_space_to_component_count(colorspace);
    switch(colorspace)
    {
        default:
        case nn::nvjpg::JCS_GRAYSCALE:
        case nn::nvjpg::JCS_RGB:
        case nn::nvjpg::JCS_CMYK: // No idea what this looks like at the binary level
        case nn::nvjpg::JCS_YCCK: // …not for this one either…
        {
            EXPECT_EQ(colors, 3);
            ret.resize(width * height * colors);
            for(nn::nvjpg::JDIMENSION y = 0; y < height; ++y)
            {
                for(nn::nvjpg::JDIMENSION x = 0; x < width; ++x)
                {
                    const int rand = std::rand() % 85;
                    ret[(y * width + x) * colors + 0] = static_cast<nn::nvjpg::JSAMPLE>(rand);
                    ret[(y * width + x) * colors + 1] = static_cast<nn::nvjpg::JSAMPLE>(rand + 85);
                    ret[(y * width + x) * colors + 2] = static_cast<nn::nvjpg::JSAMPLE>(rand + 170);
                }
            }
        }
        break;
        case nn::nvjpg::JCS_YCbCr:
        {
            ret.resize((width * height * colors) / 2);
            for(nn::nvjpg::JDIMENSION y = 0; y < height; ++y)
            {
                for(nn::nvjpg::JDIMENSION x = 0; x < width; ++x)
                {
                    const int rand = std::rand() % 85;
                    ret[y * width + x] = static_cast<nn::nvjpg::JSAMPLE>(rand);
                }
            }
            for(nn::nvjpg::JDIMENSION p = 0; p < width * height; p += 4)
            {
                const int rand = ret[p];
                ret[width * height + p / 2 + 0] = static_cast<nn::nvjpg::JSAMPLE>(rand + 85);
                ret[width * height + p / 2 + 1] = static_cast<nn::nvjpg::JSAMPLE>(rand + 170);
            }
        }
        break;
    }
    return ret;
}

void init_destination(nn::nvjpg::j_compress_ptr cinfo)
{
    NN_SDK_LOG("init_destination\n");
    NvJpgSanityTest::data.resize(10000);
    //~ NvJpgSanityTest::data.resize(1000);
    cinfo->dest->next_output_byte = &NvJpgSanityTest::data[0];
    cinfo->dest->free_in_buffer = NvJpgSanityTest::data.size();
}

nn::nvjpg::boolean empty_output_buffer(nn::nvjpg::j_compress_ptr cinfo)
{
    const auto size = NvJpgSanityTest::data.size();
    NvJpgSanityTest::data.resize(size + 1000);
    //~ NN_SDK_LOG("empty_output_buffer %d -> %d\n", size, NvJpgSanityTest::data.size());
    cinfo->dest->next_output_byte = &NvJpgSanityTest::data[0] + size;
    cinfo->dest->free_in_buffer = NvJpgSanityTest::data.size() - size;
    return true;
}

void term_destination(nn::nvjpg::j_compress_ptr cinfo)
{
    const auto size = NvJpgSanityTest::data.size();
    const auto end = &NvJpgSanityTest::data[0] + size;
    const std::ptrdiff_t used = end - cinfo->dest->next_output_byte;
    NN_SDK_LOG("term_destination: used = %d\n", used);
    NvJpgSanityTest::data.resize(used);
}

struct encoder
{
    nn::nvjpg::jpeg_compress_struct cinfo{};
    nn::nvjpg::jpeg_workbuf workbuf{};
    nn::nvjpg::jpeg_error_mgr jerr{};
    nn::nvjpg::jpeg_destination_mgr destmgr{};
    explicit encoder(std::vector<char>& workvec);
    ~encoder();
    void input(std::vector<nn::nvjpg::JSAMPLE> *in, int quality);
};

encoder::encoder(std::vector<char>& workvec)
{
    // Allocate and initialize a JPEG compression object
    cinfo.err = nn::nvjpg::jpeg_std_error(&jerr);
    workbuf.ptr = &workvec[0];
    workbuf.total = workvec.size() - 1;
    cinfo.workbuf = &workbuf;
    nn::nvjpg::jpeg_create_compress(&cinfo);
#if !defined(SOFTWARE)
    EXPECT_TRUE(cinfo.tegra_acceleration);
#endif

    destmgr.init_destination = &init_destination;
    destmgr.empty_output_buffer = &empty_output_buffer;
    destmgr.term_destination = &term_destination;
    cinfo.dest = &destmgr;
}

encoder::~encoder()
{
    nn::nvjpg::jpeg_finish_compress(&cinfo);
    NN_SDK_LOG("jpeg_destroy_compress\n");
    nn::nvjpg::jpeg_destroy_compress(&cinfo);
}

void encoder::input(std::vector<nn::nvjpg::JSAMPLE> *in, int quality)
{
    NN_SDK_LOG("jpeg_set_defaults\n");
    nn::nvjpg::jpeg_set_defaults(&cinfo);
    nn::nvjpg::jpeg_set_quality(&cinfo, quality, true);
    NN_SDK_LOG("jpeg_start_compress\n");
    nn::nvjpg::jpeg_start_compress(&cinfo, true);

    NN_SDK_LOG("cinfo.jpegTegraMgr = %p\n", cinfo.jpegTegraMgr);
    if(cinfo.jpegTegraMgr)
    {
        NN_SDK_LOG("cinfo.jpegTegraMgr->buff[0] = %p, cinfo.jpegTegraMgr->pitch[0] = %u\n", cinfo.jpegTegraMgr->buff[0], cinfo.jpegTegraMgr->pitch[0]);
        NN_SDK_LOG("cinfo.jpegTegraMgr->buff[1] = %p, cinfo.jpegTegraMgr->pitch[1] = %u\n", cinfo.jpegTegraMgr->buff[1], cinfo.jpegTegraMgr->pitch[1]);
        NN_SDK_LOG("cinfo.jpegTegraMgr->buff[2] = %p, cinfo.jpegTegraMgr->pitch[2] = %u\n", cinfo.jpegTegraMgr->buff[2], cinfo.jpegTegraMgr->pitch[2]);
    }
    NN_SDK_LOG("jpeg_write_scanlines\n");
    for(nn::nvjpg::JDIMENSION r = 0; r < cinfo.image_height; ++r)
    {
        nn::nvjpg::JSAMPLE *scanlines = &(*in)[r * cinfo.image_width * cinfo.input_components];
        nn::nvjpg::jpeg_write_scanlines(&cinfo, &scanlines, 1);
    }
}

TEST_F(NvJpgSanityTest, Encode)
{
    const nn::nvjpg::JDIMENSION width = 64;
    const nn::nvjpg::JDIMENSION height = 64;
    const nn::nvjpg::J_COLOR_SPACE colorspace = nn::nvjpg::JCS_YCbCr;
    std::vector<nn::nvjpg::JSAMPLE> image_data{create_image_data(width, height, colorspace)};
    NN_SDK_LOG("image_data = %p\n", &image_data[0]);
    std::vector<char> workvec((width * height * sizeof(std::int32_t) * 3) / 2);
    encoder enc(workvec);
    enc.cinfo.image_width = width;
    enc.cinfo.image_height = height;
    enc.cinfo.input_components = color_space_to_component_count(colorspace);
    enc.cinfo.in_color_space = colorspace;
    const int quality = 100;
    enc.input(&image_data, quality);
    unsigned long srcsize = enc.cinfo.dest->next_output_byte - &data[0];
    NN_UNUSED(srcsize);
    NN_SDK_LOG("quality = %d, original = %d, compressed = %u\n\tratio =%f\n", quality, image_data.size(), srcsize, double(srcsize) / image_data.size());
}

TEST_F(NvJpgSanityTest, Compare)
{
    const nn::nvjpg::JDIMENSION width = 64;
    const nn::nvjpg::JDIMENSION height = 64;
    const nn::nvjpg::J_COLOR_SPACE colorspace = nn::nvjpg::JCS_YCbCr;
    std::vector<nn::nvjpg::JSAMPLE> image_data{create_image_data(width, height, colorspace)};
    const int quality = 100;
    NN_SDK_LOG("image_data = %p\n", &image_data[0]);
    {
        //const nn::nvjpg::JSAMPLE *pixels = &image_data[0];
        for(nn::nvjpg::JDIMENSION y = 0; y < height; ++y)
        {
            for(nn::nvjpg::JDIMENSION x = 0; x < width; ++x)
            {
                //const nn::nvjpg::JSAMPLE luma = *pixels++;
                //NN_UNUSED(luma);
                //NN_SDK_LOG("%03d ", luma);
            }
            //NN_SDK_LOG("\n");
        }
        for(nn::nvjpg::JDIMENSION y = 0; y < height / 2; ++y)
        {
            for(nn::nvjpg::JDIMENSION x = 0; x < width / 2; ++x)
            {
                //const nn::nvjpg::JSAMPLE chroma_u = *pixels++;
                //NN_UNUSED(chroma_u);
                //NN_SDK_LOG("%03d ", chroma_u);
            }
            //NN_SDK_LOG("\n");
        }
        for(nn::nvjpg::JDIMENSION y = 0; y < height / 2; ++y)
        {
            for(nn::nvjpg::JDIMENSION x = 0; x < width / 2; ++x)
            {
                //const nn::nvjpg::JSAMPLE chroma_v = *pixels++;
                //NN_UNUSED(chroma_v);
                //NN_SDK_LOG("%03d ", chroma_v);
            }
            //NN_SDK_LOG("\n");
        }
        //NN_SDK_LOG("\n");
        //~ ASSERT_EQ(&image_data[0] + image_data.size(), pixels);
    }
    std::vector<char> workvec((width * height * sizeof(std::int32_t) * 3));
    {
        encoder enc(workvec);
        enc.cinfo.image_width = width;
        enc.cinfo.image_height = height;
        enc.cinfo.input_components = color_space_to_component_count(colorspace);
        enc.cinfo.in_color_space = colorspace;
        enc.input(&image_data, quality);
    }
    {
        decoder dec(workvec);
        std::vector<nn::nvjpg::JSAMPLE> output(dec.cinfo.output_height * dec.cinfo.output_width * dec.cinfo.output_components);
        dec.cinfo.out_color_space = colorspace;
        dec.output(&output);
#if !defined(SOFTWARE)
        EXPECT_EQ(dec.cinfo.jpegTegraMgr->mcu_type, 0);

        for(nn::nvjpg::JDIMENSION y = 0; y < height; ++y)
        {
            //const nn::nvjpg::JSAMPLE * const buff = dec.cinfo.jpegTegraMgr->buff[0] + (y * dec.cinfo.jpegTegraMgr->pitch[0]);
            for(nn::nvjpg::JDIMENSION x = 0; x < width; ++x)
            {
                //const nn::nvjpg::JSAMPLE out = buff[x];
                //~ const nn::nvjpg::JSAMPLE in = image_data[y * width + x];
                //~ const auto mm = std::minmax(in, out);
                //~ const nn::nvjpg::JSAMPLE diff = mm.second - mm.first;
                //NN_UNUSED(out);
               // NN_SDK_LOG("%03d ", out);
                ///@todo Input format and output format presently incomprehensible
                //~ EXPECT_EQ(diff & ~0b11, 0); // jpeglib on Cygwin runs this test with no failures
            }
            //NN_SDK_LOG("\n");
        }
        for(nn::nvjpg::JDIMENSION y = 0; y < height / 2; ++y)
        {
            //const nn::nvjpg::JSAMPLE * const buff = dec.cinfo.jpegTegraMgr->buff[1] + (y * dec.cinfo.jpegTegraMgr->pitch[1]);
            for(nn::nvjpg::JDIMENSION x = 0; x < width / 2; ++x)
            {
                //const nn::nvjpg::JSAMPLE out = buff[x];
                //~ const nn::nvjpg::JSAMPLE in = image_data[height * width + 2 * x];
                //~ const auto mm = std::minmax(in, out);
                //~ const nn::nvjpg::JSAMPLE diff = mm.second - mm.first;
                //NN_UNUSED(out);
                //NN_SDK_LOG("%03d ", out);
                ///@todo Input format and output format presently incomprehensible
                //~ EXPECT_EQ(diff & ~0b11, 0); // jpeglib on Cygwin runs this test with no failures
            }
            //NN_SDK_LOG("\n");
        }
        for(nn::nvjpg::JDIMENSION y = 0; y < height / 2; ++y)
        {
            //const nn::nvjpg::JSAMPLE * const buff = dec.cinfo.jpegTegraMgr->buff[2] + (y * dec.cinfo.jpegTegraMgr->pitch[2]);
            for(nn::nvjpg::JDIMENSION x = 0; x < width / 2; ++x)
            {
                //const nn::nvjpg::JSAMPLE out = buff[x];
                //~ const nn::nvjpg::JSAMPLE in = image_data[height * width + 2 * x + 1];
                //~ const auto mm = std::minmax(in, out);
                //~ const nn::nvjpg::JSAMPLE diff = mm.second - mm.first;
               // NN_UNUSED(out);
               // NN_SDK_LOG("%03d ", out);
                ///@todo Input format and output format presently incomprehensible
                //~ EXPECT_EQ(diff & ~0b11, 0); // jpeglib on Cygwin runs this test with no failures
            }
            //NN_SDK_LOG("\n");
        }
#else
        const nn::nvjpg::JSAMPLE * const buff = &output[0];
        for(nn::nvjpg::JDIMENSION y = 0; y < height; ++y)
        {
            for(nn::nvjpg::JDIMENSION x = 0; x < width; ++x)
            {
                //~ for(nn::nvjpg::JDIMENSION c = 0; c < dec.cinfo.output_components; ++c)
                for(nn::nvjpg::JDIMENSION c = 0; c < 1; ++c)
                {
                    const int output_components = dec.cinfo.output_components;
                    const nn::nvjpg::JSAMPLE out = buff[(y * width + x) * output_components + c];
                    const nn::nvjpg::JSAMPLE in = image_data[(y * width + x) * colors + c];
                    const auto mm = std::minmax(in, out);
                    const nn::nvjpg::JSAMPLE diff = mm.second - mm.first;
                    //~ NN_SDK_LOG("%03d ", diff);
                    EXPECT_EQ(diff & ~0b11, 0); // jpeglib on Cygwin runs this test with no failures
                }
            }
            NN_SDK_LOG("\n");
        }
#endif
        NN_SDK_LOG("\n");
        NN_SDK_LOG("quality = %d, original = %d, compressed = %u\n\tratio =%f\n", quality, image_data.size(), data.size(), double(data.size()) / image_data.size());
    }
}//NOLINT(impl/function_size)
