﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "test_MediaDecoderTest.h"
#include <nnt/nntest.h>
#include <movie/Utils.h>
#include <movie/Common.h>
#include <movie/Status.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <string>
#include <cinttypes>
#include <chrono>

#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

#include "ByteStreamReader.h"
#include "HeapTracker.h"

namespace // 'unnamed'
{
    const int FsHeapSize = 512 * 1024;
    const int mmHeapSize = 512 << 20;

    uint8_t                     g_FsHeapBuffer[FsHeapSize];
    nn::lmem::HeapHandle        g_FsHeap;
    char                        g_mmHeapBuffer[mmHeapSize];

    const size_t  g_GraphicsSystemMemorySize = 8 * 1024 * 1024;
    uint8_t g_GraphicsHeap[g_GraphicsSystemMemorySize] __attribute__((aligned(4096)));

    bool USE_HEAP_TRACKING = false;
}

void FsInitHeap()
{
    g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
}

void* FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

static nn::mem::StandardAllocator &GlobalAllocator()
{
    static nn::mem::StandardAllocator g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));
    return g_MultimediaAllocator;
}

static HeapTracker &MMHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

static HeapTracker &CoreHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

static HeapTracker &MallocHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

static HeapTracker &NewHeap()
{
    static HeapTracker g_heap;
    return g_heap;
}

void* MultimediaAllocate(size_t size, size_t alignment, void *userPtr)
{
    if(size == 0)
        return NULL;
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    void *address = GlobalAllocator().Allocate(size, alignment);
    static size_t totalAllocatedSize = 0;

    totalAllocatedSize += size;
    if (USE_HEAP_TRACKING)
    {
        static_cast<HeapTracker*>(userPtr)->Track(address, size);
    }
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
    return address;
}

void MultimediaFree(void *addr, void *userPtr)
{
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    GlobalAllocator().Free(addr);
    if (USE_HEAP_TRACKING)
    {
        static_cast<HeapTracker*>(userPtr)->Untrack(addr);
    }
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
}

void *MultimediaReallocate(void* addr, size_t newSize, void *userPtr)
{
    static_cast<HeapTracker*>(userPtr)->m_mutex.Lock();
    if (USE_HEAP_TRACKING)
    {
        if (addr)
            static_cast<HeapTracker*>(userPtr)->Untrack(addr);
    }
    void *memory = GlobalAllocator().Reallocate(addr, newSize);
    if (USE_HEAP_TRACKING)
    {
        static_cast<HeapTracker*>(userPtr)->Track(memory, newSize);
    }
    static_cast<HeapTracker*>(userPtr)->m_mutex.Unlock();
    return memory;
}

//------------------------------------------------------------------------------
extern "C" void* malloc(size_t aSize)
{
    return MultimediaAllocate(aSize, 8, &MallocHeap());
}

//------------------------------------------------------------------------------
extern "C" void free(void* aPtr)
{
    if (aPtr) {
        MultimediaFree(aPtr, &MallocHeap());
    }
}

//------------------------------------------------------------------------------
extern "C" void* calloc(size_t aNum, size_t aSize)
{
    const size_t sum = aNum * aSize;
    void* p = malloc(sum);
    if (p) {
        ::std::memset(p, 0, sum);
    }
    return p;
}

//------------------------------------------------------------------------------
extern "C" void* realloc(void* aPtr, size_t aNewSize)
{
    return MultimediaReallocate(aPtr, aNewSize, &MallocHeap());
}

//------------------------------------------------------------------------------
extern "C" void* aligned_alloc(size_t aAlignment, size_t aSize)
{
    return MultimediaAllocate(aSize, aAlignment, &MallocHeap());
}

//------------------------------------------------------------------------------
void* operator new(::std::size_t aSize, const ::std::nothrow_t&) throw()
{
    if(aSize == 0)
        aSize = 1;
    void *memory = MultimediaAllocate(static_cast<int>(aSize), 8, &NewHeap());
    return memory;
}

//------------------------------------------------------------------------------
void* operator new[](::std::size_t aSize, const ::std::nothrow_t&) throw()
{
    return operator new(aSize, ::std::nothrow_t());
}

//------------------------------------------------------------------------------
void* operator new(::std::size_t aSize) throw(::std::bad_alloc)
{
    return operator new(aSize, ::std::nothrow_t());
}

//------------------------------------------------------------------------------
void* operator new[](::std::size_t aSize) throw(::std::bad_alloc)
{
    return operator new(aSize, ::std::nothrow_t());
}

//------------------------------------------------------------------------------
void operator delete(void* aPtr) throw()
{
    if (aPtr) {
        MultimediaFree(aPtr, &NewHeap());
    }
}

//------------------------------------------------------------------------------
void operator delete[](void* aPtr) throw()
{
    operator delete(aPtr);
}

/* Process startup setup */
extern "C" void nninitStartup()
{
    const size_t heapSize = 256 << 20;
    const size_t mallocSize = 128 << 20;
    uintptr_t address;
    nn::Result result;

    result = nn::os::SetMemoryHeapSize(heapSize);
    if (!result.IsSuccess())
    {
        NN_SDK_LOG("Failed to SetMemoryHeapSize \n");
        return;
    }

    result = nn::os::AllocateMemoryBlock(&address, mallocSize);
    if (!result.IsSuccess())
    {
        NN_SDK_LOG("Failed to AllocateMemoryBlock \n");
        return;
    }
    // nn::init::InitializeAllocator(reinterpret_cast<void*>(address), mallocSize);
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);

}


/* MovieDecoderPlayer Player Usage */
static void Usage()
{
    NN_SDK_LOG("\n MovieDecoderPlayer ::Invalid options, refer Readme.MovieDecoderPlayer.txt \n");
}

MovieDecoderPlayer::MovieDecoderPlayer()
{
    m_Extractor = NULL;
    m_VideoDecoder = NULL;
    m_AudioDecoder = NULL;
    m_AudioOutRenderer = NULL;
    m_VideoRenderer = NULL;
    m_ByteStreamReader = NULL;
    m_AudioDecoderType = movie::DecoderType::DecoderType_Unknown;
    m_VideoDecoderType = movie::DecoderType::DecoderType_Unknown;
    m_ThreadStackSize = 1024 * 256;
    m_Tracks = 0;
    m_Width = -1;
    m_Height = -1;
    m_NumChannels = -1;
    m_SampleRate = -1;
    m_AudioTrackIndex = -1;
    m_VideoTrackIndex = -2;

#ifdef ENABLE_METADATA_READER
    m_MetaTrackIndex = -3;
#endif

    m_VideoFrameRate = -1.0;
    m_TrackDurationUs = -1;
    SetAudioPlaybackComplete();
    SetVideoPlaybackComplete();
    m_AudioDecoderCreated = false;
    m_VideoDecoderCreated = false;
    m_suppressRendering = false;

    nn::os::InitializeMutex(&m_VideoListMutex, false, 0);
    nn::os::InitializeMutex(&m_AudioListMutex, false, 0);
}

movie::Status MovieDecoderPlayer::Initialize(uint64_t coreMask)
{
    movie::Status movieStatus = movie::Status_Success;

    m_EndOfStream = false;
    m_BufferSize = 32;
    m_LoopCount = 0;

    if(coreMask)//set only valid core mask
    {
        movie::SetCoreMask(coreMask);
    }

    // Initialize Event for audio video worker thread
    nn::os::InitializeEvent(&m_AudioVideoWorkerThreadExitEvent, false, nn::os::EventClearMode_ManualClear);

    // Initialize MultiWaitHolder for AudioVideoWorkerThread
    nn::os::InitializeMultiWaitHolder(&m_AudioVideoWorkerThreadExitEventHolder, &m_AudioVideoWorkerThreadExitEvent);

    // Initialize MultiWait for AudioVideoWorkerThread
    nn::os::InitializeMultiWait(&m_AudioVideoWorkerMultiWait);

    // Initialize Events to register with audio decoder
    nn::os::InitializeEvent(&m_AudioDecoderEventTypes.errorEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_AudioDecoderEventTypes.formatChangedEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_AudioDecoderEventTypes.inputBufferAvailableEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_AudioDecoderEventTypes.outputBufferAvailableEvent, false, nn::os::EventClearMode_ManualClear);

    // Fill movie::DecoderEvents structure with audio events (for audio decoder)
    m_AudioDecoderEvents.errorEvent = &m_AudioDecoderEventTypes.errorEvent;
    m_AudioDecoderEvents.formatChangedEvent = &m_AudioDecoderEventTypes.formatChangedEvent;
    m_AudioDecoderEvents.inputBufferAvailableEvent = &m_AudioDecoderEventTypes.inputBufferAvailableEvent;
    m_AudioDecoderEvents.outputBufferAvailableEvent = &m_AudioDecoderEventTypes.outputBufferAvailableEvent;

    // Initialize AudioDecoderMultiWaitHolder with audio events (Used by movie decoder player thread to listen for audio events)
    nn::os::InitializeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.errorEventHolder, &m_AudioDecoderEventTypes.errorEvent);
    nn::os::InitializeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder, &m_AudioDecoderEventTypes.formatChangedEvent);
    nn::os::InitializeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder, &m_AudioDecoderEventTypes.inputBufferAvailableEvent);
    nn::os::InitializeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder, &m_AudioDecoderEventTypes.outputBufferAvailableEvent);

    // Initialize Events to register with video decoder
    nn::os::InitializeEvent(&m_VideoDecoderEventTypes.errorEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_VideoDecoderEventTypes.formatChangedEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_VideoDecoderEventTypes.inputBufferAvailableEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_VideoDecoderEventTypes.outputBufferAvailableEvent, false, nn::os::EventClearMode_ManualClear);

    // Fill movie::DecoderEvents structure with video events (for video decoder)
    m_VideoDecoderEvents.errorEvent = &m_VideoDecoderEventTypes.errorEvent;
    m_VideoDecoderEvents.formatChangedEvent = &m_VideoDecoderEventTypes.formatChangedEvent;
    m_VideoDecoderEvents.inputBufferAvailableEvent = &m_VideoDecoderEventTypes.inputBufferAvailableEvent;
    m_VideoDecoderEvents.outputBufferAvailableEvent = &m_VideoDecoderEventTypes.outputBufferAvailableEvent;

    // Initialize VideoDecoderMultiWaitHolderTypes with video events (Used by movie decoder player thread to listen for video events)
    nn::os::InitializeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.errorEventHolder, &m_VideoDecoderEventTypes.errorEvent);
    nn::os::InitializeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder, &m_VideoDecoderEventTypes.formatChangedEvent);
    nn::os::InitializeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder, &m_VideoDecoderEventTypes.inputBufferAvailableEvent);
    nn::os::InitializeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder, &m_VideoDecoderEventTypes.outputBufferAvailableEvent);

    m_MessageBuffer = new uintptr_t[m_BufferSize];
    if (m_MessageBuffer == NULL)
    {
        return movie::Status_OutOfMemory;
    }

    // Initialize MessageQueue used by movie decoder player
    nn::os::InitializeMessageQueue(&m_MessageQueue, m_MessageBuffer, m_BufferSize);

    // Create AudioAndVideoInputReader thread
    movieStatus = CreateThreadForAudioAndVideoInputReader();
    if (movieStatus == movie::Status_Success)
    {
        // Create AudioVideoWorker thread
        movieStatus = CreateThreadForAudioVideoWorker();
    }
    return movieStatus;
}

void MovieDecoderPlayer::Finalize()
{
    StopAndDestroyAudioVideoWorkerThread();
    if (m_AudioVideoWorkerThreadStack != NULL)
    {
        delete[] m_AudioVideoWorkerThreadStack;
    }

    if (m_AudioVideoInputReadThreadStack != NULL)
    {
        delete[] m_AudioVideoInputReadThreadStack;
    }
    // Finalize AudioVideoWorkerThreadExit event
    nn::os::FinalizeEvent(&m_AudioVideoWorkerThreadExitEvent);

    // Finalize VideoDecoder events
    nn::os::FinalizeEvent(&m_VideoDecoderEventTypes.errorEvent);
    nn::os::FinalizeEvent(&m_VideoDecoderEventTypes.formatChangedEvent);
    nn::os::FinalizeEvent(&m_VideoDecoderEventTypes.inputBufferAvailableEvent);
    nn::os::FinalizeEvent(&m_VideoDecoderEventTypes.outputBufferAvailableEvent);

    // Finalize VideoDecoder MultiWaitHolder
    nn::os::FinalizeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.errorEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);

    // Finalize AudioDecoder events
    nn::os::FinalizeEvent(&m_AudioDecoderEventTypes.errorEvent);
    nn::os::FinalizeEvent(&m_AudioDecoderEventTypes.formatChangedEvent);
    nn::os::FinalizeEvent(&m_AudioDecoderEventTypes.inputBufferAvailableEvent);
    nn::os::FinalizeEvent(&m_AudioDecoderEventTypes.outputBufferAvailableEvent);

    // Finalize AudioDecoder MultiWaitHolder
    nn::os::FinalizeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.errorEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);

    // Finalize AudioVideoWorkerThread MultiWait
    nn::os::FinalizeMultiWait(&m_AudioVideoWorkerMultiWait);

    // Finalize Mutex used by Audio and video list
    nn::os::FinalizeMutex(&m_VideoListMutex);
    nn::os::FinalizeMutex(&m_AudioListMutex);

    // Finalize MessageQueue used by movie decoder player
    nn::os::FinalizeMessageQueue(&m_MessageQueue);
    if (m_MessageBuffer != NULL)
    {
        delete[] m_MessageBuffer;
    }
}

void MovieDecoderPlayer::ResetPlaybackComplete()
{
    ResetAudioPlaybackComplete();
    ResetVideoPlaybackComplete();
}

bool MovieDecoderPlayer::IsPlaybackComplete()
{
    return IsAudioPlaybackComplete() && IsVideoPlaybackComplete();
}

movie::Status MovieDecoderPlayer::CreateThreadForAudioAndVideoInputReader()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();

    m_AudioVideoInputReadThreadStack = new char[m_ThreadStackSize * 2];
    if (m_AudioVideoInputReadThreadStack == NULL)
    {
        movieStatus = movie::Status_OutOfMemory;
    }
    else
    {
        void *mem = (void*)m_AudioVideoInputReadThreadStack;
        void *alignedStackPointer = (void*)(((uintptr_t)mem + 4095) & ~(uintptr_t)0x0FFF);
        m_AudioVideoInputReadThreadDone = false;
        nnResult = nn::os::CreateThread(&m_AudioVideoInputReadThreadType,
            &AudioAndVideoInputReadThreadFunction,
            (void*)this,
            alignedStackPointer,
            m_ThreadStackSize,
            nn::os::DefaultThreadPriority);
        if (nnResult.IsSuccess())
        {
            nn::os::SetThreadName(&m_AudioVideoInputReadThreadType, "AudioVideoInputReadThread");
            m_AudioVideoInputReadThreadCreated = true;
        }
        else
        {
            movieStatus = movie::Status_FailedToCreateThread;
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::CreateThreadForAudioVideoWorker()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();
    m_AudioVideoWorkerThreadStack = new char[m_ThreadStackSize * 2];
    if (m_AudioVideoWorkerThreadStack == NULL)
    {
        movieStatus = movie::Status_OutOfMemory;
    }
    else
    {
        void *mem = (void*)m_AudioVideoWorkerThreadStack;
        void *alignedStackPointer = (void*)(((uintptr_t)mem + 4095) & ~(uintptr_t)0x0FFF);
        m_AudioVideoWorkerThreadDone = false;
        nnResult = nn::os::CreateThread(&m_AudioVideoWorkerThreadType,
            &AudioVideoWorkerThreadFunction,
            (void*)this,
            alignedStackPointer,
            m_ThreadStackSize,
            nn::os::DefaultThreadPriority);
        if (nnResult.IsSuccess())
        {
            nn::os::SetThreadName(&m_AudioVideoWorkerThreadType, "AudioVideoWorkerThread");
            m_AudioVideoWorkerThreadCreated = true;
        }
        else
        {
            movieStatus = movie::Status_FailedToCreateThread;
        }
    }
    return movieStatus;
}

void MovieDecoderPlayer::StartAudioAndVideoInputReaderThread()
{
    nn::os::StartThread(&m_AudioVideoInputReadThreadType);
}

void MovieDecoderPlayer::StartAudioVideoWorkerThread()
{
    nn::os::StartThread(&m_AudioVideoWorkerThreadType);
}

void MovieDecoderPlayer::StopAndDestroyAudioAndVideoInputReaderThread()
{
    if (m_AudioVideoInputReadThreadCreated == true)
    {
        m_AudioVideoInputReadThreadDone = true;
        SignalAudioAndVideoInputReadThreadExit();
        nn::os::WaitThread(&m_AudioVideoInputReadThreadType);
        nn::os::DestroyThread(&m_AudioVideoInputReadThreadType);
        m_AudioVideoInputReadThreadCreated = false;
    }
}

void MovieDecoderPlayer::StopAndDestroyAudioVideoWorkerThread()
{
    if (m_AudioVideoWorkerThreadCreated == true)
    {
        m_AudioVideoWorkerThreadDone = true;
        SignalAudioVideoWorkerThreadExit();
        nn::os::WaitThread(&m_AudioVideoWorkerThreadType);
        nn::os::DestroyThread(&m_AudioVideoWorkerThreadType);
        m_AudioVideoWorkerThreadCreated = false;
    }
}

movie::Status MovieDecoderPlayer::CreateAudioVideoRenderer()
{
    movie::Status movieStatus = movie::Status_Success;
    if (IsAudioDecoderCreated())
    {
        m_AudioOutRenderer = new AudioOutRenderer();
        if (m_AudioOutRenderer == NULL)
        {
            movieStatus = movie::Status_OutOfMemory;
            return movieStatus;
        }
    }
    if (IsVideoDecoderCreated())
    {
        m_VideoRenderer = new VideoRenderer();
        if (m_VideoRenderer == NULL)
        {
            movieStatus = movie::Status_OutOfMemory;
        }
    }
    return movieStatus;
}

void MovieDecoderPlayer::StopAudioVideoRenderer()
{
    if (m_AudioOutRenderer != NULL)
    {
        m_AudioOutRenderer->Stop();
        m_AudioOutRenderer->Close();
    }

    if (m_VideoRenderer != NULL)
    {
        m_VideoRenderer->Stop();
        m_VideoRenderer->Close();
    }
}

void MovieDecoderPlayer::DestroypAudioVideoRenderer()
{
    if (m_AudioOutRenderer != NULL)
    {
        delete m_AudioOutRenderer;
    }

    if (m_VideoRenderer != NULL)
    {
        delete m_VideoRenderer;
    }
}

movie::Status MovieDecoderPlayer::CreateExtractor(const char* uri, movie::ContainerType container, movie::Extractor** extractor, bool use_buffer_as_datasource)
{
    movie::Status movieStatus = movie::Status_Success;
    std::string path = uri;
    // Creates Extractor with an internal data cache of default size (movie::CacheSize_20MB)
    // Data cache size can be set by creating extractor using a constructor which takes cache size as input parameter
    // Example: Create an extractor for mp4 container type with 10MB of cache
    // movie::Extractor* movieExtractor = new movie::Extractor(movie::ContainerType_Mpeg4, movie::CacheSize_10MB);
    movie::Extractor* movieExtractor = new movie::Extractor(container);
    if (movieExtractor == NULL)
    {
        *extractor = NULL;
        movieStatus = movie::Status_OutOfMemory;
    }
    else
    {
        nn::fs::FileHandle fileHandle;
        fileHandle.handle = NULL;
        nn::Result result = nn::fs::OpenFile(&fileHandle, uri, nn::fs::OpenMode_Read);
        if (result.IsFailure())
        {
            NN_SDK_LOG("\n Failed to open %s \n \n Terminating PlayerSample MediaDecoderPlayer\n", uri);
            movieStatus = movie::Status_NotFound;
            return movieStatus;
        }

        if (use_buffer_as_datasource) {
            NN_SDK_LOG("\nMovieDecoderPlayer::CreateExtractor:: Using SetDataSource's direct buffer API. \n");
            int64_t filesize;
            nn::fs::GetFileSize(&filesize, fileHandle);
            //void* buffer = new char[filesize];
            m_SetDataSourceBuffer.clear();
            m_SetDataSourceBuffer.resize(filesize);
            nn::fs::ReadFile(fileHandle, 0, m_SetDataSourceBuffer.data(), filesize);
            movieStatus = movieExtractor->SetDataSource(m_SetDataSourceBuffer.data(), filesize);
            nn::fs::CloseFile(fileHandle);
        }
        else {
            NN_SDK_LOG("\nMovieDecoderPlayer::CreateExtractor:: Using SetDataSource's uri API. \n");
            nn::fs::CloseFile(fileHandle);
            movieStatus = movieExtractor->SetDataSource(uri);
        }

        if (movieStatus == movie::Status_Success)
        {
            m_Tracks = 0;
            movieStatus = movieExtractor->GetTrackCount(&m_Tracks);
            if (movieStatus == movie::Status_Success)
            {
                if (m_Tracks <= 0)
                {
                    movieStatus = movie::Status_TrackNotFound;
                }
            }
        }
        *extractor = movieExtractor;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::CreateDecoder(movie::Decoder** decoder, movie::DecoderType decoderType, movie::DecoderOutputFormat decoderOutputFormat)
{
    if (decoder == NULL)
    {
        return movie::Status_ErrorBadValue;
    }
    else
    {
        movie::Decoder* movieDecoder = new movie::Decoder(decoderType, decoderOutputFormat);
        if (movieDecoder == NULL)
        {
            *decoder = NULL;
            return movie::Status_OutOfMemory;
        }
        else
        {
            *decoder = movieDecoder;
            return movie::Status_Success;
        }
    }
}

movie::Status MovieDecoderPlayer::ConfigureDecoder(movie::Decoder* decoder, movie::DecoderEvents* decoderEvents, movie::MediaData* configuration)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if (decoder == NULL)
    {
        return movie::Status_ErrorBadValue;
    }
    movieStatus = decoder->RegisterClientEvents(decoderEvents);
    if (movieStatus == movie::Status_Success)
    {
        movieStatus = decoder->Configure(configuration);
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::StartDecoder(movie::Decoder* decoder)
{
    if (decoder == NULL)
    {
        return movie::Status_NullDecoder;
    }
    else
    {
        return decoder->Start();
    }
}

movie::Status MovieDecoderPlayer::StopDecoder(movie::Decoder* decoder)
{
    if (decoder == NULL)
    {
        return movie::Status_NullDecoder;
    }
    else
    {
        return decoder->Stop();
    }
}

movie::Status MovieDecoderPlayer::FlushDecoder(movie::Decoder* decoder)
{
    if (decoder == NULL)
    {
        return movie::Status_NullDecoder;
    }
    else
    {
        return decoder->Flush();
    }
}

void MovieDecoderPlayer::DestroyDecoder(movie::Decoder* decoder)
{
    if (decoder != NULL)
    {
        delete decoder;
    }
}

movie::Status MovieDecoderPlayer::CreateDecoders(movie::Extractor*  extractor)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::MediaData trackFormat;
    if (extractor == NULL)
    {
        movieStatus = movie::Status_NullExtractor;
    }
    else
    {
        for (int i = 0; i < m_Tracks; i++)
        {
            movie::DecoderType decoderType = movie::DecoderType::DecoderType_Unknown;
            extractor->GetTrackConfiguration(i, &trackFormat);
            const char *mime;
            if (trackFormat.FindString("mime", &mime))
            {
                if ((!strncasecmp("video/", mime, 6)) && (m_VideoDecoder == NULL))
                {
                    if (!strncasecmp("video/avc", mime, 9))
                    {
                        decoderType = movie::DecoderType_VideoAvc;
                    }
                    else if (!strncasecmp("video/x-vnd.on2.vp8", mime, 19))
                    {
                        decoderType = movie::DecoderType_VideoVp8;
                    }
                    else if (!strncasecmp("video/x-vnd.on2.vp9", mime, 19))
                    {
                        decoderType = movie::DecoderType_VideoVp9;
                    }

                    movieStatus = CreateDecoder(&m_VideoDecoder, decoderType, movie::DecoderOutputFormat_VideoColorNv12);
                    if (movieStatus == movie::Status_Success)
                    {
                        movieStatus = ConfigureDecoder(m_VideoDecoder, &m_VideoDecoderEvents, &trackFormat);
                        if (movieStatus == movie::Status_Success)
                        {
                            m_VideoTrackIndex = i;
                            int32_t width = 0;
                            int32_t height = 0;
                            trackFormat.FindInt32("width", &width);
                            m_Width = width;
                            trackFormat.FindInt32("height", &height);
                            m_Height = height;
                            int32_t outSize = 0;
                            int32_t outalign = 0;
                            int32_t videoFrameRate = 0;
                            trackFormat.FindInt32("frame-rate", &videoFrameRate);
                            if (videoFrameRate > 0)
                            {
                                m_VideoFrameRate = videoFrameRate;
                            }
                            else
                            {
                                m_VideoFrameRate = 60;
                            }
                            m_VideoDecoder->GetOutputBufferSize(&outSize, &outalign);
                            movieStatus = extractor->SelectTrack(i);
                            m_VideoDecoderCreated = true;
                            m_VideoDecoderType = decoderType;
                            ResetVideoPlaybackComplete();
                            NN_SDK_LOG("\n Video decoder created \n");
                        }
                        else
                        {
                            delete m_VideoDecoder;
                            m_VideoDecoder = NULL;
                        }
                    }
                }

                if ((!strncasecmp("audio/", mime, 6)) && (m_AudioDecoder == NULL))
                {
                    if (!strncasecmp("audio/mp4a-latm", mime, 15))
                    {
                        decoderType = movie::DecoderType_AudioAac;
                    }
                    else if (!strncasecmp("audio/vorbis", mime, 12))
                    {
                        decoderType = movie::DecoderType_AudioVorbis;
                    }
                    int32_t sampleRate = 0;
                    trackFormat.FindInt32("sample-rate", &sampleRate);
                    int32_t numChannels = 0;
                    trackFormat.FindInt32("channel-count", &numChannels);

                    if ((sampleRate == 48000) && (numChannels == 2))
                    {
                        movieStatus = CreateDecoder(&m_AudioDecoder, decoderType, movie::DecoderOutputFormat_AudioPcm16);
                        if (movieStatus == movie::Status_Success)
                        {
                            movieStatus = ConfigureDecoder(m_AudioDecoder, &m_AudioDecoderEvents, &trackFormat);
                            if (movieStatus == movie::Status_Success)
                            {
                                m_AudioTrackIndex = i;
                                int32_t outSize = 0;
                                int32_t outalign = 0;
                                m_AudioDecoder->GetOutputBufferSize(&outSize, &outalign);
                                movieStatus = extractor->SelectTrack(i);
                                m_AudioDecoderCreated = true;
                                m_AudioDecoderType = decoderType;
                                ResetAudioPlaybackComplete();
                                NN_SDK_LOG("\n Audio decoder created \n");
                            }
                        }
                    }
                    else
                    {
                        NN_SDK_LOG("\n Un-supported audio sample rate or number of channels \n");
                        movieStatus = movie::Status_UnknownError;
                    }

                    if (movieStatus != movie::Status_Success)
                    {
                        m_AudioDecoderType = movie::DecoderType::DecoderType_Unknown;
                        m_AudioDecoderCreated = false;
                        delete m_AudioDecoder;
                        m_AudioDecoder = NULL;
                    }
                }
#ifdef ENABLE_METADATA_READER
                if (!strncasecmp("application/", mime, 12)) {
                    NN_SDK_LOG("\n Found metadata track! \n");
                    m_MetaTrackIndex = i;

                    movieStatus = extractor->SelectTrack(i);
                    if (movieStatus == movie::Status_Success) {
                        NN_SDK_LOG("Selected metadata track!\n");
                    }
                }
#endif
            }
            m_TrackFormat[i] = trackFormat;
            trackFormat.Clear();
        }
    }
    if ((m_AudioDecoder == NULL) && (m_VideoDecoder == NULL))
    {
        movieStatus = movie::Status_UnknownError;
    }

    return movieStatus;
}//NOLINT(impl/function_size)

 /**
 * Destroy extractor and release resources
 */
void MovieDecoderPlayer::DestroyExtractor(movie::Extractor* extractor)
{
    if (extractor != NULL)
    {
        delete extractor;
    }
}

/**
* Empty buffers (input) to be filled with compressed audio data
*/
movie::Status MovieDecoderPlayer::AddBufferToAudioIndexList(int32_t bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_AudioListMutex);
    m_AudioBufferIndexList.push_back(bufferIndex);
    nn::os::UnlockMutex(&m_AudioListMutex);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::RemoveBufferFromAudioIndexList(int32_t* bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_AudioListMutex);
    int32_t index = -1;
    int audioListSize = m_AudioBufferIndexList.size();
    if (audioListSize > 0)
    {
        index = m_AudioBufferIndexList.front();
        m_AudioBufferIndexList.erase(m_AudioBufferIndexList.begin());
    }
    *bufferIndex = index;
    nn::os::UnlockMutex(&m_AudioListMutex);
    return movieStatus;
}

/**
* Empty buffers (input) to be filled with compressed video data
*/
movie::Status MovieDecoderPlayer::AddBufferToVideoIndexList(int32_t bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_VideoListMutex);
    m_VideoBufferIndexList.push_back(bufferIndex);
    nn::os::UnlockMutex(&m_VideoListMutex);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::RemoveBufferFromVideoIndexList(int32_t* bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_VideoListMutex);
    int32_t index = -1;
    int videoListSize = m_VideoBufferIndexList.size();
    if (videoListSize > 0)
    {
        index = m_VideoBufferIndexList.front();
        m_VideoBufferIndexList.erase(m_VideoBufferIndexList.begin());
    }
    *bufferIndex = index;
    nn::os::UnlockMutex(&m_VideoListMutex);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::ReadInputDataFromExtractorSendTodecoder(ByteStreamReader* byteStreamReader, int index, movie::Decoder* decoder)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::Buffer buffer;
    uint32_t flags = movie::BufferFlags_None;
    int64_t presentationTimeUs = 0;
    decoder->GetInputBuffer(index, &buffer);
    int32_t frameDataSize = 0;

    movieStatus = byteStreamReader->ReadNextFrame((char*)buffer.Base(), buffer.Capacity(), &frameDataSize, &presentationTimeUs);
    if ((movieStatus != movie::Status_Success) || (frameDataSize <= 0) || (movieStatus == movie::Status_EndOfStream))
    {
        if (movieStatus == movie::Status_NotEnoughData)
        {
            movieStatus = byteStreamReader->ReadNextFrame((char*)buffer.Base(), buffer.Capacity(), &frameDataSize, &presentationTimeUs);
            if ((movieStatus != movie::Status_Success) || (frameDataSize <= 0) || (movieStatus == movie::Status_EndOfStream))
            {
                NN_SDK_LOG("\n BufferFlags_EndOfStream\n");
                flags = movie::BufferFlags_EndOfStream;
                // End of stream reached(Second read also returned "Status_NotEnoughData")
                movieStatus = movie::Status_EndOfStream;
                m_EndOfStream = true;
            }
        }
        else
        {
            NN_SDK_LOG("\n BufferFlags_EndOfStream\n");
            flags = movie::BufferFlags_EndOfStream;
            m_EndOfStream = true;
        }
    }
    if ((movieStatus == movie::Status_EndOfStream) && (m_LoopPlayback == true))
    {
        byteStreamReader->PrepareForLooping();
        m_EndOfStream = false;
        frameDataSize = 0;
        flags = movie::BufferFlags_None;
        presentationTimeUs = 0;
        m_LoopCount++;
        NN_SDK_LOG("\n Loop Count : %d\n", m_LoopCount);
        movieStatus = byteStreamReader->ReadNextFrame((char*)buffer.Base(), buffer.Capacity(), &frameDataSize, &presentationTimeUs);
        if ((movieStatus != movie::Status_Success) || (frameDataSize <= 0) || (movieStatus == movie::Status_EndOfStream))
        {
            NN_SDK_LOG("\n Looping failed, Setting EndOfStream\n");
            flags = movie::BufferFlags_EndOfStream;
            m_EndOfStream = true;
        }
    }
    decoder->SendInputBufferForDecode(index, 0, frameDataSize, presentationTimeUs, flags);
    if (m_EndOfStream == true)
    {
        SendEosToDecoder();
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::ReadInputDataFromExtractorSendTodecoder(int index, movie::Decoder* decoder)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::Buffer buffer;
    uint32_t flags = movie::BufferFlags_None;
    int64_t presentationTimeUs = 0;
    decoder->GetInputBuffer(index, &buffer);
    movieStatus = m_Extractor->Read(&buffer);
    if ((movieStatus != movie::Status_Success) || (buffer.Size() <= 0) || (movieStatus == movie::Status_EndOfStream))
    {
        NN_SDK_LOG("\n BufferFlags_EndOfStream\n");
        flags = movie::BufferFlags_EndOfStream;
        m_EndOfStream = true;
    }
    if ((movieStatus == movie::Status_EndOfStream) && (m_LoopPlayback == true))
    {
        m_EndOfStream = false;
        flags = movie::BufferFlags_None;
        m_LoopCount++;
        NN_SDK_LOG("\n Loop Count : %d\n", m_LoopCount);
        movieStatus = m_Extractor->SeekTo(0ll);
        if (movieStatus == movie::Status_Success)
        {
            movieStatus = m_Extractor->Read(&buffer);
            if ((movieStatus != movie::Status_Success) || (buffer.Size() <= 0) || (movieStatus == movie::Status_EndOfStream))
            {
                NN_SDK_LOG("\n Looping failed, Setting EndOfStream\n");
                flags = movie::BufferFlags_EndOfStream;
                m_EndOfStream = true;
            }
        }
    }
    m_Extractor->GetSampleTime(&presentationTimeUs);
    decoder->SendInputBufferForDecode(index, buffer.Offset(), buffer.Size(), presentationTimeUs, flags);
    m_Extractor->Advance();
    if (m_EndOfStream == true)
    {
        SendEosToDecoder();
    }
    return movieStatus;
}

#ifdef ENABLE_METADATA_READER
movie::Status MovieDecoderPlayer::ReadMetaDataFromExtractor()
{
    movie::Status movieStatus = movie::Status_OutOfMemory;
    movie::Buffer* buffer = new movie::Buffer(sizeof(uint64_t));
    uint32_t flags = movie::BufferFlags_None;
    int64_t presentationTimeUs = 0;

    if(buffer != nullptr)
    {
        movieStatus = m_Extractor->Read(buffer);
        if ((movieStatus != movie::Status_Success) || (buffer->Size() <= 0) || (movieStatus == movie::Status_EndOfStream)) {
            flags = movie::BufferFlags_EndOfStream;
            NN_SDK_LOG("METADATA EOS REACHED!\n");
        }
        movieStatus = m_Extractor->GetSampleTime(&presentationTimeUs);
        NN_SDK_LOG("Read metadata frame #%llu @presentation time = %lldus!\n", *((uint64_t*)buffer->Data()), presentationTimeUs);
        movieStatus = m_Extractor->Advance();
        if (movieStatus != movie::Status_Success) {
            NN_SDK_LOG("Failed to advance extractor!\n");
        }
        delete buffer;
        buffer = nullptr;
    }

    return movieStatus;
}
#endif

movie::Status MovieDecoderPlayer::SendEosToDecoder()
{
    movie::Status movieStatus = movie::Status_Success;
    movie::Buffer buffer;
    uint32_t flags = movie::BufferFlags_EndOfStream;
    nn::os::LockMutex(&m_AudioListMutex);
    int audioInputListSize = m_AudioBufferIndexList.size();
    nn::os::UnlockMutex(&m_AudioListMutex);

    for (int i = 0; i < audioInputListSize; i++)
    {
        int index = -1;
        RemoveBufferFromAudioIndexList(&index);
        if (index != -1)
        {
            m_AudioDecoder->GetInputBuffer(index, &buffer);
            m_AudioDecoder->SendInputBufferForDecode(index, 0, 0, 0, flags);
        }
    }

    nn::os::LockMutex(&m_VideoListMutex);
    int videoInputListSize = m_VideoBufferIndexList.size();
    nn::os::UnlockMutex(&m_VideoListMutex);
    for (int i = 0; i < videoInputListSize; i++)
    {
        int index = -1;
        RemoveBufferFromVideoIndexList(&index);
        if (index != -1)
        {
            m_VideoDecoder->GetInputBuffer(index, &buffer);
            m_VideoDecoder->SendInputBufferForDecode(index, 0, 0, 0, flags);
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::CheckForInputBuffersAndReadInputdataDecoder()
{
    movie::Status movieStatus = movie::Status_Success;
    size_t currentTrackIndex = -1;
    int index = -1;
    if (m_ByteStreamReader == NULL)
    {
        movieStatus = m_Extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
        if ((movieStatus != movie::Status_Success) || (movieStatus == movie::Status_EndOfStream))
        {
            m_EndOfStream = true;
        }

        if ((movieStatus == movie::Status_EndOfStream) && (m_LoopPlayback == true))
        {
            m_EndOfStream = false;
            movieStatus = m_Extractor->SeekTo(0ll);
            if (movieStatus == movie::Status_Success)
            {
                m_LoopCount++;
                NN_SDK_LOG("\n Loop Count : %d\n", m_LoopCount);
                movieStatus = m_Extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
                if ((movieStatus != movie::Status_Success) || (movieStatus == movie::Status_EndOfStream))
                {
                    m_EndOfStream = true;
                }
            }
        }

        if (m_EndOfStream == true)
        {
            SendEosToDecoder();
            return movieStatus;
        }
    }
    else
    {
        if ((m_VideoDecoderType == movie::DecoderType_VideoVp8) || (m_VideoDecoderType == movie::DecoderType_VideoVp9)
            || (m_VideoDecoderType == movie::DecoderType_VideoAvc))
        {
            currentTrackIndex = m_VideoTrackIndex;
        }
    }

    if (currentTrackIndex == m_AudioTrackIndex)
    {
        RemoveBufferFromAudioIndexList(&index);
        if (index != -1)
        {
            if (m_ByteStreamReader != NULL)
            {
                ReadInputDataFromExtractorSendTodecoder(m_ByteStreamReader, index, m_AudioDecoder);
            }
            else
            {
                ReadInputDataFromExtractorSendTodecoder(index, m_AudioDecoder);
            }
        }
    }

    if (currentTrackIndex == m_VideoTrackIndex)
    {
        RemoveBufferFromVideoIndexList(&index);
        if (index != -1)
        {
            if (m_ByteStreamReader != NULL)
            {
                ReadInputDataFromExtractorSendTodecoder(m_ByteStreamReader, index, m_VideoDecoder);
            }
            else
            {
                ReadInputDataFromExtractorSendTodecoder(index, m_VideoDecoder);
            }
        }
    }
#ifdef ENABLE_METADATA_READER
    if ( currentTrackIndex == m_MetaTrackIndex )
    {
        // Metadata
        ReadMetaDataFromExtractor();
        SignalInputBufferAvailable();
    }
#endif
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SignalInputBufferAvailable()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SendMessageQueue(&m_MessageQueue, MovieDecoderPlayer::InputBufferAvailable);
    return movieStatus;
}


movie::Status MovieDecoderPlayer::SignalAudioAndVideoInputReadThreadExit()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SendMessageQueue(&m_MessageQueue, MovieDecoderPlayer::InputReadThreadDone);
    return movieStatus;
}


movie::Status MovieDecoderPlayer::SignalAudioVideoWorkerThreadExit()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SignalEvent(&m_AudioVideoWorkerThreadExitEvent);
    return movieStatus;
}

void AudioAndVideoInputReadThreadFunction(void *arg)
{
    MovieDecoderPlayer *movieDecoderPlayer = (MovieDecoderPlayer*)arg;
    for (;;)
    {
        uintptr_t data;
        nn::os::ReceiveMessageQueue(&data, &movieDecoderPlayer->m_MessageQueue);
        if (data == MovieDecoderPlayer::InputBufferAvailable)
        {
            movieDecoderPlayer->CheckForInputBuffersAndReadInputdataDecoder();
        }
        else if (data == MovieDecoderPlayer::InputReadThreadDone)
        {
            movieDecoderPlayer->m_AudioVideoInputReadThreadDone = true;
            break;
        }
    }
}

movie::Status MovieDecoderPlayer::VideoInputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    movie::Buffer buffer;
    int32_t numberOfBuffersAvailable = 0;;
    movieStatus = m_VideoDecoder->AcquireInputBufferIndex(&index, &numberOfBuffersAvailable);
    if (movieStatus == movie::Status_Success)
    {
        AddBufferToVideoIndexList(index);
        SignalInputBufferAvailable();
        for (int32_t i = 0; i < numberOfBuffersAvailable; i++)
        {
            int32_t remainingBuffers = 0;
            movieStatus = m_VideoDecoder->AcquireInputBufferIndex(&index, &remainingBuffers);
            if (movieStatus == movie::Status_Success)
            {
                AddBufferToVideoIndexList(index);
                SignalInputBufferAvailable();
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::VideoOutputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    int64_t presentationTimeUs = 0ll;
    uint32_t flags = 0;
    int32_t numberOfBuffersAvailable = 0;

    movie::Buffer buffer;

    movieStatus = m_VideoDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &numberOfBuffersAvailable);

    if (movieStatus == movie::Status_Success)
    {
        if (flags == movie::BufferFlags_EndOfStream)
        {
            NN_SDK_LOG("\n BufferFlags_EndOfStream \n");
            m_VideoDecoder->ReleaseOutputBufferIndex(index);
            SetVideoPlaybackComplete();
        }
        else
        {
            if (this->m_suppressRendering) {
                m_VideoDecoder->ReleaseOutputBufferIndex(index);
            }
            else {
                m_VideoRenderer->OnOutputAvailable(index, presentationTimeUs, flags);
            }
        }

        for (int32_t i = 0; i < numberOfBuffersAvailable; i++)
        {
            int32_t remainingBuffers = 0;
            movieStatus = m_VideoDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &remainingBuffers);
            if (movieStatus == movie::Status_Success)
            {
                if (flags == movie::BufferFlags_EndOfStream)
                {
                    NN_SDK_LOG("\n BufferFlags_EndOfStream \n");
                    m_VideoDecoder->ReleaseOutputBufferIndex(index);
                    SetVideoPlaybackComplete();
                }
                else
                {
                    if (this->m_suppressRendering) {
                        m_VideoDecoder->ReleaseOutputBufferIndex(index);
                    }
                    else {
                        m_VideoRenderer->OnOutputAvailable(index, presentationTimeUs, flags);
                    }
                }
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::VideoFormatChangedEvent()
{
    NN_SDK_LOG("\n VideoFormatChangedEvent \n");
    movie::Status movieStatus = movie::Status_Success;
    movie::MediaData format;
    m_VideoDecoder->GetOutputFormat(&format);

    static bool videoRenderStarted = false;
    int32_t width = 0;
    int32_t height = 0;
    format.FindInt32("width", &width);
    m_Width = width;
    format.FindInt32("height", &height);
    m_Height = height;
    int32_t cropLeft = 0;
    int32_t cropTop = 0;
    int32_t cropRight = 0;
    int32_t cropBottom = 0;
    int32_t stride = 0;
    format.FindInt32("stride", &stride);
    format.FindRect("crop", &cropLeft, &cropTop, &cropRight, &cropBottom);
    m_Width = cropRight - cropLeft + 1;
    m_Height = cropBottom - cropTop + 1;

    m_VideoRenderer->UpdateVideoInformation(m_Width, m_Height);
    if( videoRenderStarted == false )
    {
        m_VideoRenderer->Start(m_Width, m_Height);
        videoRenderStarted = true;
    }
    m_VideoRenderer->ResizeTextures(m_Width, m_Height);
    NN_SDK_LOG("VideoFormatChangedEvent width %d, height %d, stride %d, cropLeft %d, cropTop %d, cropRight %d, cropBottom %d\n",
        width, height, stride, cropLeft, cropTop, cropRight, cropBottom);
    format.Clear();
    return movieStatus;
}

movie::Status MovieDecoderPlayer::AudioInputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    movie::Buffer buffer;
    int32_t numberOfBuffersAvailable = 0;;
    movieStatus = m_AudioDecoder->AcquireInputBufferIndex(&index, &numberOfBuffersAvailable);
    if (movieStatus == movie::Status_Success)
    {
        AddBufferToAudioIndexList(index);
        SignalInputBufferAvailable();
        for (int32_t i = 0; i < numberOfBuffersAvailable; i++)
        {
            int32_t remainingBuffers = 0;
            movieStatus = m_AudioDecoder->AcquireInputBufferIndex(&index, &remainingBuffers);
            if (movieStatus == movie::Status_Success)
            {
                AddBufferToAudioIndexList(index);
                SignalInputBufferAvailable();
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::AudioOutputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    int64_t presentationTimeUs = 0ll;
    uint32_t flags = 0;
    int32_t numberOfBuffersAvailable = 0;

    movieStatus = m_AudioDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &numberOfBuffersAvailable);
    if (movieStatus == movie::Status_Success)
    {
        if (m_suppressRendering) {
            m_AudioDecoder->ReleaseOutputBufferIndex(index);
        }
        else {
            m_AudioOutRenderer->OnOutputAvailable(index, presentationTimeUs, flags);
        }
        if (flags == movie::BufferFlags_EndOfStream)
        {
            NN_SDK_LOG("\n BufferFlags_EndOfStream \n");
            SetAudioPlaybackComplete();
        }

        for (int32_t i = 0; i < numberOfBuffersAvailable; i++)
        {
            int32_t remainingBuffers = 0;
            movieStatus = m_AudioDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &remainingBuffers);
            if (movieStatus == movie::Status_Success)
            {
                if (m_suppressRendering) {
                    m_AudioDecoder->ReleaseOutputBufferIndex(index);
                }
                else {
                    m_AudioOutRenderer->OnOutputAvailable(index, presentationTimeUs, flags);
                }
                //m_AudioOutRenderer->OnOutputAvailable(index, presentationTimeUs, flags);
            }
            if (flags == movie::BufferFlags_EndOfStream)
            {
                NN_SDK_LOG("\n BufferFlags_EndOfStream \n");
                SetAudioPlaybackComplete();
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::AudioFormatChangedEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    NN_SDK_LOG("\n AudioFormatChangedEvent \n");
    return movieStatus;
}

void AudioVideoWorkerThreadFunction(void *arg)
{
    MovieDecoderPlayer *movieDecoderPlayer = (MovieDecoderPlayer*)arg;

    // Thread is starting, link all MultiWait holders
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioVideoWorkerThreadExitEventHolder);

    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.errorEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);

    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.errorEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);

    // Wait for events and process them.

    unsigned frame_counter{};
    auto begin_time = std::chrono::high_resolution_clock::now();

    for (;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait);

        if (holder == &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder)
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_VideoDecoderEvents.inputBufferAvailableEvent);
            movieDecoderPlayer->VideoInputBufferAvailableEvent();
        }
        else if (holder == &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder)
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_VideoDecoderEvents.outputBufferAvailableEvent);
            movieDecoderPlayer->VideoOutputBufferAvailableEvent();
            ++frame_counter;
        }
        else if (holder == &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder)
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_VideoDecoderEvents.formatChangedEvent);
            movieDecoderPlayer->VideoFormatChangedEvent();
        }
        if (holder == &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder)
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_AudioDecoderEvents.inputBufferAvailableEvent);
            movieDecoderPlayer->AudioInputBufferAvailableEvent();
        }
        else if (holder == &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder)
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_AudioDecoderEvents.outputBufferAvailableEvent);
            movieDecoderPlayer->AudioOutputBufferAvailableEvent();
        }
        else if (holder == &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder)
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_AudioDecoderEvents.formatChangedEvent);
            movieDecoderPlayer->AudioFormatChangedEvent();
        }
        else if (holder == &movieDecoderPlayer->m_AudioVideoWorkerThreadExitEventHolder)
        {
            nn::os::ClearEvent(&movieDecoderPlayer->m_AudioVideoWorkerThreadExitEvent);
            movieDecoderPlayer->m_AudioVideoWorkerThreadDone = true;

            // Thread is exiting, unlink VideoDecoder MultiWait holders
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.errorEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);

            // Thread is exiting, unlink AudioDecoder MultiWait holders
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.errorEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);

            // Thread is exiting, unlink AudioVideoWorker MultiWait holder
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerThreadExitEventHolder);

            break;
        }
    }

    // calculate fps and report to teamcity..
    if (movieDecoderPlayer->m_LoopPlayback == false) {
        auto execution_time_milliseconds = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::high_resolution_clock::now() - begin_time).count();
        double fps = (static_cast<double>(frame_counter) / (execution_time_milliseconds > 0 ? execution_time_milliseconds : 1.0)) * 1000.0;
        NN_LOG("##teamcity[buildStatisticValue key='decoder_video_output_fps' value='%.0f']\n", fps);
    }
    else {
        // probably should not report if looping..
    }

}//NOLINT(impl/function_size)

void MovieDecoderPlayer::Reset()
{
    movie::Status movieStatus = movie::Status_Success;
    if( m_AudioDecoder != NULL )
    {
        m_AudioDecoder->Reset();
        nn::os::LockMutex(&m_AudioListMutex);
        m_AudioBufferIndexList.clear();
        m_AudioOutRenderer->Stop();
        nn::os::UnlockMutex(&m_AudioListMutex);
    }
    if( m_VideoDecoder != NULL )
    {
        m_VideoDecoder->Reset();
        nn::os::LockMutex(&m_VideoListMutex);
        m_VideoBufferIndexList.clear();
        m_VideoRenderer->Stop();
        nn::os::UnlockMutex(&m_VideoListMutex);
    }

    if(m_VideoTrackIndex >= 0)
    {
        movieStatus = ConfigureDecoder(m_VideoDecoder, &m_VideoDecoderEvents, m_TrackFormat + m_VideoTrackIndex);
        if( movieStatus == movie::Status_Success )
        {
            StartDecoder(m_VideoDecoder);
            NN_SDK_LOG( "\n Failed to reconfigure video decoder \n" );
        }
    }

    if(m_AudioTrackIndex >= 0)
    {
        movieStatus = ConfigureDecoder(m_AudioDecoder, &m_AudioDecoderEvents, m_TrackFormat + m_AudioTrackIndex);
        if( movieStatus == movie::Status_Success )
        {
            StartDecoder(m_AudioDecoder);
            NN_SDK_LOG("\n Failed to reconfigure audio decoder \n");
        }
    }
}

struct MediaPlayerTest : public ::testing::Test
{
    static void SetUpTestCase();
    static void TearDownTestCase();

    virtual void SetUp() {}
    virtual void TearDown() {}
};

void MediaPlayerTest::SetUpTestCase()
{
}

void MediaPlayerTest::TearDownTestCase()
{
}

TEST_F(MediaPlayerTest, NormalPlayback)
{
    NN_SDK_LOG("\n MediaCodecTest:: NormalPlayback\n");

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();

    /* Set allocator callback functions */
    movie::SetAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, &MMHeap());
    // Donate memory for graphics driver to work in
    nv::SetGraphicsServiceName("nvdrv:t");
    nv::InitializeGraphics(g_GraphicsHeap, g_GraphicsSystemMemorySize);
    nv::SetGraphicsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, &CoreHeap());
    nv::SetGraphicsDevtoolsAllocator(MultimediaAllocate, MultimediaFree, MultimediaReallocate, &CoreHeap());

    if (argc <= 1)
    {
        Usage();
        return;
    }
    int32_t playbackAgingRepeatCount = 0;
    bool runPlaybackAgingTest = false;
    uint64_t coreMask = 0;
    int32_t resetCount = 0;
    bool useSetDataSourceWithBuffers = false;

    do
    {
        bool invalidOptions = false;
        bool loopPlayback = false;
        bool suppressRendering = false; //  default mode has rendering (audio and video) enabled

        StreamFormat streamFormat = StreamFormat::StreamFormat_Unknown;
        movie::ContainerType container = movie::ContainerType::ContainerType_Unknown;
        for (int options = 0; options < argc; options++)
        {
            if (!strcmp(argv[options], "-me"))
            {
                options++;
                invalidOptions = true;
                if (options < argc)
                {
                    if ((!strcmp(argv[options], "mp4")) || (!strcmp(argv[options], "m4a")))
                    {
                        container = movie::ContainerType_Mpeg4;
                        invalidOptions = false;
                    }
                    else if ((!strcmp(argv[options], "webm")))
                    {
                        container = movie::ContainerType_WebM;
                        invalidOptions = false;
                    }
                    else if ((!strcmp(argv[options], "mkv")))
                    {
                        container = movie::ContainerType_Matroska;
                        invalidOptions = false;
                    }
                    else if ((!strcmp(argv[options], "ogg")))
                    {
                        container = movie::ContainerType_Ogg;
                        invalidOptions = false;
                    }
                    else if ((!strcmp(argv[options], "ivf")))
                    {
                        streamFormat = StreamFormat_ivf;
                        invalidOptions = false;
                    }
                    else if ((!strcmp(argv[options], "h264")))
                    {
                        streamFormat = StreamFormat_h264;
                        invalidOptions = false;
                    }
                    else
                    {
                        break;
                    }
                }
            }
            else if (!strcmp(argv[options], "-l"))
            {
                options++;
                invalidOptions = true;
                if (options < argc)
                {
                    if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                    {
                        loopPlayback = atoi(argv[options]);
                        invalidOptions = false;
                    }
                    else
                    {
                        break;
                    }
                }
            }
            else if (!strcmp(argv[options], "-aging"))
            {
                options++;
                invalidOptions = true;
                if (options < argc)
                {
                    if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                    {
                        runPlaybackAgingTest = atoi(argv[options]);
                        invalidOptions = false;
                    }
                    else
                    {
                        break;
                    }
                }
            }
            else if( !strcmp(argv[ options ], "-core") )
            {
                options++;
                invalidOptions = true;
                if( options < argc )
                {
                    if( ( !strcmp(argv[ options ], "1") ) || ( !strcmp(argv[ options ], "2") )||( !strcmp(argv[ options ], "4") ))
                    {
                        coreMask = atoi(argv[ options ]);
                        invalidOptions = false;
                    }
                    else
                    {
                        break;
                    }
                }
            }
            else if( !strcmp(argv[ options ], "-reset") )
            {
                options++;
                invalidOptions = true;
                if( options < argc )
                {
                    resetCount = atoi(argv[ options ]);
                    invalidOptions = false;
                }
            }
            else if (!strcmp(argv[options], "-norender"))
            {
                suppressRendering = true;
                invalidOptions = false;
            }
            else if (!strcmp(argv[options], "-use-buffer-api"))
            {
                useSetDataSourceWithBuffers = true;
                invalidOptions = false;
            }
            else if (!strcmp(argv[options], "-heap"))
            {
                options++;
                invalidOptions = true;
                if (options < argc)
                {
                    if ((!strcmp(argv[options], "0")) || (!strcmp(argv[options], "1")))
                    {
                        USE_HEAP_TRACKING = atoi(argv[options]);
                        invalidOptions = false;
                    }
                }
            }
        }

        if ((invalidOptions == true)
            || ((streamFormat == StreamFormat::StreamFormat_Unknown) && (container == movie::ContainerType::ContainerType_Unknown)))
        {
            Usage();
            return;
        }

        if (loopPlayback == true)
        {
            runPlaybackAgingTest = false;
        }

        char* inputFileName = inputFileName = argv[1];
        std::string path = inputFileName;
        std::string delimiter = ":";
        std::string token = path.substr(0, path.find(delimiter));
        bool sdcardMounted = false;
        if (token == "sdcard")
        {
            nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
            if (resultSdcardMount.IsFailure())
            {
                NN_SDK_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
                    resultSdcardMount.GetModule(),
                    resultSdcardMount.GetDescription());
                return;
            }
            sdcardMounted = true;
        }
        else
        {
            nn::Result resultHostMount = nn::fs::MountHostRoot();
            if (resultHostMount.IsFailure())
            {
                NN_SDK_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
                    resultHostMount.GetModule(),
                    resultHostMount.GetDescription());
                return;
            }
            }

        NN_SDK_LOG("\n MovieDecoderPlayer:: Input File Name : %s\n", inputFileName);
        movie::Status movieStatus = movie::Status_Success;
        MovieDecoderPlayer* movieDecoderPlayer = new MovieDecoderPlayer;
        if (movieDecoderPlayer != NULL)
        {
            movieStatus = movieDecoderPlayer->Initialize(coreMask);
        }
        movieDecoderPlayer->m_LoopPlayback = loopPlayback;
        movieDecoderPlayer->m_suppressRendering = suppressRendering;

        if ((movieStatus != movie::Status_Success) || (movieDecoderPlayer == NULL))
        {
            NN_SDK_LOG("\n Failed to create/initialize  MovieDecoderPlayer, Error : 0x%x - Terminating Application ! \n", movieStatus);
            if (movieDecoderPlayer != NULL)
            {
                delete movieDecoderPlayer;
                movieDecoderPlayer = NULL;
            }
            return;
        }

        // Check for raw streams
        if ((streamFormat == StreamFormat_ivf) || (streamFormat == StreamFormat_h264))
        {
            movieDecoderPlayer->m_ByteStreamReader = new ByteStreamReader();
            if (movieDecoderPlayer->m_ByteStreamReader == NULL)
            {
                NN_SDK_LOG("\n Failed to create IVF stream reader - Terminating Application ! \n");
                delete movieDecoderPlayer;
                movieDecoderPlayer = NULL;
                return;
            }
            else
            {
                movieStatus = movieDecoderPlayer->m_ByteStreamReader->Open(inputFileName, streamFormat);
                if (movieStatus == movie::Status_Success)
                {
                    movieStatus = movieDecoderPlayer->m_ByteStreamReader->Prepare();
                    if (movieStatus == movie::Status_Success)
                    {
                        movieStatus = movieDecoderPlayer->m_ByteStreamReader->GetStreamProperties(&movieDecoderPlayer->m_VideoDecoderType, &movieDecoderPlayer->m_Width, &movieDecoderPlayer->m_Height, &movieDecoderPlayer->m_VideoFrameRate);
                        if (movieStatus == movie::Status_Success)
                        {
                            movieStatus = movieDecoderPlayer->CreateDecoder(&movieDecoderPlayer->m_VideoDecoder, movieDecoderPlayer->m_VideoDecoderType, movie::DecoderOutputFormat_VideoColorNv12);
                            if (movieStatus == movie::Status_Success)
                            {
                                movie::MediaData config;
                                movieDecoderPlayer->m_VideoDecoderCreated = true;
                                if (movieDecoderPlayer->m_VideoDecoderType == movie::DecoderType_VideoVp8)
                                {
                                    CreateVp8DecoderConfigurationData(movieDecoderPlayer->m_Width, movieDecoderPlayer->m_Height, movieDecoderPlayer->m_VideoFrameRate, &config);
                                }
                                else if (movieDecoderPlayer->m_VideoDecoderType == movie::DecoderType_VideoVp9)
                                {
                                    CreateVp9DecoderConfigurationData(movieDecoderPlayer->m_Width, movieDecoderPlayer->m_Height, movieDecoderPlayer->m_VideoFrameRate, &config);
                                }
                                else if (movieDecoderPlayer->m_VideoDecoderType == movie::DecoderType_VideoAvc)
                                {
                                    CreateH264DecoderConfigurationData(movieDecoderPlayer->m_ByteStreamReader->m_H264Sps, movieDecoderPlayer->m_ByteStreamReader->m_H264SpsSize, movieDecoderPlayer->m_ByteStreamReader->m_H264Pps, movieDecoderPlayer->m_ByteStreamReader->m_H264PpsSize, &config);
                                }
                                else
                                {
                                    movieStatus = movie::Status_UnknownError;
                                }
                                if (movieStatus == movie::Status_Success)
                                {
                                    movieStatus = movieDecoderPlayer->ConfigureDecoder(movieDecoderPlayer->m_VideoDecoder, &movieDecoderPlayer->m_VideoDecoderEvents, &config);
                                }
                                movieDecoderPlayer->ResetVideoPlaybackComplete();
                                movieDecoderPlayer->m_VideoTrackIndex = 0;
                                movieDecoderPlayer->m_TrackFormat[0] = config;
                            }
                        }
                    }
                }
            }
        }
        else if (container != movie::ContainerType_Unknown)
        {
            // Create extractor to parse media data
            movieStatus = movieDecoderPlayer->CreateExtractor(inputFileName, container, &movieDecoderPlayer->m_Extractor, useSetDataSourceWithBuffers);
            if (movieStatus == movie::Status_Success)
            {
                // Create audio and video decoder for decoding media data
                movieStatus = movieDecoderPlayer->CreateDecoders(movieDecoderPlayer->m_Extractor);
            }
        }
        else
        {
            movieStatus = movie::Status_UnknownError;
        }

        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n Failed to createDecoders and Extractor, Error : 0x%x - Terminating Application ! \n", movieStatus);
            if (movieDecoderPlayer->m_ByteStreamReader != NULL)
            {
                movieDecoderPlayer->m_ByteStreamReader->Close();
                delete movieDecoderPlayer->m_ByteStreamReader;
                movieDecoderPlayer->m_ByteStreamReader = NULL;
            }
            delete movieDecoderPlayer;
            movieDecoderPlayer = NULL;
            return;
        }

        // Create audio and video renderer for rendering decoded audio and video data
        movieStatus = movieDecoderPlayer->CreateAudioVideoRenderer();
        if (movieStatus == movie::Status_Success)
        {
            // Setup AudioOut audio renderer
            if (movieDecoderPlayer->IsAudioDecoderCreated())
            {
                NN_SDK_LOG("\n MovieDecoderPlayer:: Create AudioOut renderer \n");
                movieStatus = movieDecoderPlayer->m_AudioOutRenderer->Initialize();
                if (movieStatus == movie::Status_Success)
                {
                    movieStatus = movieDecoderPlayer->m_AudioOutRenderer->Open(movieDecoderPlayer->m_AudioDecoder, 48000, 2);
                    if (movieStatus == movie::Status_Success)
                    {
                        movieStatus = movieDecoderPlayer->m_AudioOutRenderer->Start();
                    }
                }
            }
            if (movieStatus == movie::Status_Success)
            {
                if (movieDecoderPlayer->IsVideoDecoderCreated())
                {
                    NN_SDK_LOG("\n MovieDecoderPlayer:: Create Video renderer \n");
                    movieStatus = movieDecoderPlayer->m_VideoRenderer->Initialize();
                    if (movieStatus == movie::Status_Success)
                    {
                        if (movieDecoderPlayer->IsAudioDecoderCreated())
                        {
                            movieDecoderPlayer->m_VideoFrameRate = -1;
                        }
                        movieStatus = movieDecoderPlayer->m_VideoRenderer->Open(movieDecoderPlayer->m_VideoDecoder, movieDecoderPlayer->m_Width, movieDecoderPlayer->m_Height, movieDecoderPlayer->m_VideoFrameRate);
                    }
                }
            }
        }

        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n Failed to create AudioVideoRenderer, Error : 0x%x - Terminating Application ! \n", movieStatus);
            if (movieDecoderPlayer->m_ByteStreamReader != NULL)
            {
                movieDecoderPlayer->m_ByteStreamReader->Close();
                delete movieDecoderPlayer->m_ByteStreamReader;
                movieDecoderPlayer->m_ByteStreamReader = NULL;
            }
            delete movieDecoderPlayer;
            movieDecoderPlayer = NULL;
            return;
        }

        // Start input reader thread
        NN_SDK_LOG("\n MovieDecoderPlayer:: Start Input Reader Thread \n");
        movieDecoderPlayer->StartAudioAndVideoInputReaderThread();

        // Start worker thread to receive decoder events
        movieDecoderPlayer->StartAudioVideoWorkerThread();
        if( movieDecoderPlayer->m_Extractor != NULL )
        {
            NN_SDK_LOG("\n MovieDecoderPlayer:: Wait for cache fill \n");
            movieStatus = movieDecoderPlayer->m_Extractor->SeekTo(0ll);
            if( movieStatus == movie::Status_Success )
            {

                int64_t startTimeUs = 0ll;
                int64_t endTimeUs = 0ll;
                // Cache 5 seconds worth of data
                int64_t cacheTimeUs = 5000000ll;

                // Wait maximum of 5 Seconds
                int sleepTimeMs = 10;
                int waitCount = 500;
                while( waitCount > 0 )
                {
                    movieStatus = movieDecoderPlayer->m_Extractor->GetBufferedRange(&startTimeUs, &endTimeUs);
                    if( ( movieStatus != movie::Status_Success ) || ( endTimeUs >= cacheTimeUs ) )
                    {
                        break;
                    }
                    waitCount--;
                    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleepTimeMs));
                }
            }
        }

        NN_SDK_LOG("\n MovieDecoderPlayer:: start Codecs.\n");

        // Start decoders
        if (movieDecoderPlayer->m_AudioDecoder != NULL)
        {
            movieStatus = movieDecoderPlayer->StartDecoder(movieDecoderPlayer->m_AudioDecoder);
        }

        if (movieDecoderPlayer->m_VideoDecoder != NULL)
        {
            movieStatus = movieDecoderPlayer->StartDecoder(movieDecoderPlayer->m_VideoDecoder);
        }

        if (movieStatus != movie::Status_Success)
        {
            NN_SDK_LOG("\n Failed to  StartDecoders , Error : 0x%x - Terminating Application ! \n", movieStatus);
            if (movieDecoderPlayer->m_ByteStreamReader != NULL)
            {
                movieDecoderPlayer->m_ByteStreamReader->Close();
                delete movieDecoderPlayer->m_ByteStreamReader;
                movieDecoderPlayer->m_ByteStreamReader = NULL;
            }
            delete movieDecoderPlayer;
            movieDecoderPlayer = NULL;
            return;
        }

        NN_SDK_LOG("\n MovieDecoderPlayer:: Playback is Started \n");
        // Wait till playback is complete
        int logCount = 0;
        NN_SDK_LOG("\n ******* Waiting for Playback to complete ******* \n");
        while (!movieDecoderPlayer->IsPlaybackComplete())
        {
            logCount++;
            // Print once in 5 seconds
            if (logCount == 50)
            {
                NN_SDK_LOG("\n ******* Waiting for Playback to complete ******* \n");
                logCount = 0;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

            if(resetCount > 0)
            {
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
                movieDecoderPlayer->Reset();
                --resetCount;
            }
        }

        EXPECT_TRUE(movieDecoderPlayer->IsPlaybackComplete());

        // Stop decoder and finalize resources
        if (movieDecoderPlayer->m_AudioDecoder != NULL)
        {
            movieDecoderPlayer->StopDecoder(movieDecoderPlayer->m_AudioDecoder);
        }

        if (movieDecoderPlayer->m_VideoDecoder != NULL)
        {
            movieDecoderPlayer->StopDecoder(movieDecoderPlayer->m_VideoDecoder);
        }

        movieDecoderPlayer->StopAndDestroyAudioAndVideoInputReaderThread();
        movieDecoderPlayer->StopAudioVideoRenderer();
        NN_SDK_LOG("\n MovieDecoderPlayer:: Playback is Stopped \n");
        NN_SDK_LOG("\n MovieDecoderPlayer:: Finalize resources\n");
        movieDecoderPlayer->DestroypAudioVideoRenderer();

        if (movieDecoderPlayer->m_AudioDecoder != NULL)
        {
            movieDecoderPlayer->DestroyDecoder(movieDecoderPlayer->m_AudioDecoder);
            movieDecoderPlayer->m_AudioDecoder = NULL;
        }

        if (movieDecoderPlayer->m_VideoDecoder != NULL)
        {
            movieDecoderPlayer->DestroyDecoder(movieDecoderPlayer->m_VideoDecoder);
            movieDecoderPlayer->m_VideoDecoder = NULL;
        }

        if (movieDecoderPlayer->m_Extractor != NULL)
        {
            movieDecoderPlayer->DestroyExtractor(movieDecoderPlayer->m_Extractor);
            movieDecoderPlayer->m_Extractor = NULL;
        }

        if (movieDecoderPlayer->m_ByteStreamReader != NULL)
        {
            movieDecoderPlayer->m_ByteStreamReader->Close();
            delete movieDecoderPlayer->m_ByteStreamReader;
            movieDecoderPlayer->m_ByteStreamReader = NULL;
        }

        movieDecoderPlayer->Finalize();

        if (sdcardMounted == true)
        {
            nn::fs::Unmount("sdcard");
        }
        else
        {
            nn::fs::UnmountHostRoot();
        }
        delete movieDecoderPlayer;

        if ((runPlaybackAgingTest == true) && (playbackAgingRepeatCount > 0))
        {
            NN_SDK_LOG("\n MovieDecoderPlayer:: Aging test PlayBack repeat count : %d \n", ++playbackAgingRepeatCount);
        }
    } while (runPlaybackAgingTest == true);
    nv::FinalizeGraphics();

    if (USE_HEAP_TRACKING)
    {
        NN_LOG("mm heap\n");
        MMHeap().OutputUsage();
        NN_LOG("core heap\n");
        CoreHeap().OutputUsage();
        NN_LOG("malloc heap\n");
        MallocHeap().OutputUsage();
        NN_LOG("new heap\n");
        NewHeap().OutputUsage();
    }
}//NOLINT(impl/function_size)
