﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>

#include <movie/Common.h>
#include "ByteStreamReader.h"

#define MAKE_FOURCC(a,b,c,d) ((uint32_t)((a)|(b)<<8|(c)<<16|(d)<<24))

ByteStreamReader::ByteStreamReader()
{
    m_VideoWidth = 0;
    m_VideoHeight = 0;
    m_VideoFrameRate = 0.0;
    m_TrackDurationUs = 0ll;
    m_FileReadOffset = 0ll;
    m_H264PpsSize = 0;
    m_H264SpsSize = 0;
    m_H264Pps = nullptr;
    m_H264Sps = nullptr;
    m_AvailableInputData = 0ll;
    m_DataBufferPointer = nullptr;
    m_PresentationTimeIncrementUs = 0ll;
    m_PresentationTimeUs = 0ll;
    m_StreamFormat = StreamFormat_Unknown;
    m_DecoderType = movie::DecoderType::DecoderType_Unknown;
}

ByteStreamReader::~ByteStreamReader()
{
    if( m_H264Pps != nullptr )
    {
        delete[] m_H264Pps;
    }
    if( m_H264Sps != nullptr )
    {
        delete[] m_H264Sps;
    }
}

movie::Status ByteStreamReader::Open(const char* fileName, StreamFormat streamFormat)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( fileName != NULL )
    {
        nn::Result result = nn::fs::OpenFile(&m_FileHandle, fileName, nn::fs::OpenMode_Read);
        if( result.IsSuccess() )
        {
            movieStatus = movie::Status_Success;
            m_StreamFormat = streamFormat;
        }
    }
    return movieStatus;
}

movie::Status ByteStreamReader::Prepare()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    switch( m_StreamFormat )
    {
        case StreamFormat_ivf :
            movieStatus = PrepareIVFStream();
            break;

        case StreamFormat_h264:
            movieStatus = PrepareH264Stream();
            break;

        default:
            break;
    }

    return movieStatus;
}

movie::Status ByteStreamReader::Close()
{
    nn::fs::CloseFile(m_FileHandle);
    return movie::Status_Success;
}

movie::Status ByteStreamReader::ReadNextFrame(char* frameBuffer, int32_t frameBufferSize, int32_t* frameDataSize, int64_t* presentationTimeUs)
{
    movie::Status movieStatus = movie::Status_UnknownError;

    switch( m_StreamFormat )
    {
        case StreamFormat_ivf:
            movieStatus = ReadIVFNextFrame(frameBuffer, frameBufferSize, frameDataSize, presentationTimeUs);
            break;

        case StreamFormat_h264:
            movieStatus = ReadH264NextFrame(frameBuffer, frameBufferSize, frameDataSize, presentationTimeUs);
            break;

        default:
            break;
    }
    return movieStatus;
}

movie::Status ByteStreamReader::PrepareIVFStream()
{
    // IVF Stream Header (32-byte):

    //   bytes 0-3    Signature: 'DKIF'
    //   bytes 4-5    Version (should be 0)
    //   bytes 6-7    Length of header in bytes
    //   bytes 8-11   Codec FourCC (e.g., 'VP80')
    //   bytes 12-13  Width in pixels
    //   bytes 14-15  Height in pixels
    //   bytes 16-19  Frame rate
    //   bytes 20-23  Time scale
    //   bytes 24-27  Number of frames in file
    //   bytes 28-31  Unused

    movie::Status movieStatus = movie::Status_UnknownError;
    size_t dataRead = 0;
    uint32_t signature = 0;
    uint32_t fourcc = 0;
    uint32_t rateNumerator = 0;
    uint32_t rateDenominator = 0;
    uint8_t *dataPointer = nullptr;
    uint8_t* data = &m_IvfHeader[ 0 ];
    nn::Result result = nn::fs::ReadFile(&dataRead, m_FileHandle, m_FileReadOffset, ( void* )m_IvfHeader, m_IvfHeaderSize);
    if( result.IsSuccess() )
    {
        m_FileReadOffset += dataRead;
        dataPointer = data;
        signature = dataPointer[ 3 ] << 24 | dataPointer[ 2 ] << 16 | dataPointer[ 1 ] << 8 | dataPointer[ 0 ];
        dataPointer = data + 8;
        fourcc = dataPointer[ 3 ] << 24 | dataPointer[ 2 ] << 16 | dataPointer[ 1 ] << 8 | dataPointer[ 0 ];

        dataPointer = data + 12;
        m_VideoWidth = dataPointer[ 0 ] | ( dataPointer[ 1 ] << 8 );

        dataPointer = data + 14;
        m_VideoHeight = dataPointer[ 0 ] | ( dataPointer[ 1 ] << 8 );

        dataPointer = data + 16;
        rateNumerator = dataPointer[ 3 ] << 24 | dataPointer[ 2 ] << 16 | dataPointer[ 1 ] << 8 | dataPointer[ 0 ];

        dataPointer = data + 20;
        rateDenominator = dataPointer[ 3 ] << 24 | dataPointer[ 2 ] << 16 | dataPointer[ 1 ] << 8 | dataPointer[ 0 ];
    }

    if( signature == MAKE_FOURCC('D', 'K', 'I', 'F') )
    {
        movieStatus = movie::Status_Success;
    }

    if( fourcc == MAKE_FOURCC('V', 'P', '8', '0') )
    {
        m_DecoderType = movie::DecoderType_VideoVp8;
    }

    if( fourcc == MAKE_FOURCC('V', 'P', '9', '0') )
    {
        m_DecoderType = movie::DecoderType_VideoVp9;
    }
    if( rateDenominator > 0 )
    {
        m_VideoFrameRate = (double)rateNumerator / (double)rateDenominator;
    }

    if( m_VideoFrameRate > 0 )
    {
        m_PresentationTimeIncrementUs = int64_t((1000.0 / m_VideoFrameRate) * 1000.0);
    }
    else
    {
        m_PresentationTimeIncrementUs = int64_t(( 1000.0 / 60.0 ) * 1000.0);
    }
    return movieStatus;
}

movie::Status ByteStreamReader::ReadIVFNextFrame(char* frameBuffer, int32_t frameBufferSize, int32_t* frameDataSize, int64_t* presentationTimeUs)
{
    // IVF Frame Header (12-byte):
    //   bytes 0-3    Size of frame in bytes (12-byte header not included)
    //   bytes 4-11   64-bit Presentation timestamp
    //   bytes 12..   Frame data
    movie::Status movieStatus = movie::Status_EndOfStream;
    *frameDataSize = 0;
    *presentationTimeUs = 0ll;
    uint8_t *dataPointer = nullptr;
    size_t dataRead = 0;
    nn::Result result = nn::fs::ReadFile(&dataRead, m_FileHandle, m_FileReadOffset, ( void* ) m_IvfFrameHeader, m_IvfFrameHeaderSize);
    if( result.IsSuccess() )
    {
        m_FileReadOffset += dataRead;
        uint8_t* data = &m_IvfFrameHeader[ 0 ];
        dataPointer = data;
        uint32_t frameSize = dataPointer[ 3 ] << 24 | dataPointer[ 2 ] << 16 | dataPointer[ 1 ] << 8 | dataPointer[ 0 ];

        dataPointer = data + 4;
        uint32_t partOne  = dataPointer[ 3 ] << 24 | dataPointer[ 2 ] << 16 | dataPointer[ 1 ] << 8 | dataPointer[ 0 ];
        dataPointer = dataPointer + 4;
        uint32_t partTwo  = dataPointer[ 3 ] << 24 | dataPointer[ 2 ] << 16 | dataPointer[ 1 ] << 8 | dataPointer[ 0 ];
        uint64_t framePts = ( ( uint64_t ) partTwo ) << 32 | partOne;

        if( frameBufferSize >= frameSize )
        {
            size_t dataRead = 0;
            nn::Result result = nn::fs::ReadFile(&dataRead, m_FileHandle, m_FileReadOffset, ( void* ) frameBuffer, frameSize);
            if( result.IsSuccess() )
            {
                m_FileReadOffset += dataRead;
                *frameDataSize = dataRead;
                *presentationTimeUs = framePts;
                movieStatus = movie::Status_Success;
            }
        }
    }
    return movieStatus;
}

movie::Status ByteStreamReader::GetStreamProperties(movie::DecoderType* decoderType, int32_t* videoWidth, int32_t* videoHeight, double* videoFrameRate)
{
    movie::Status movieStatus = movie::Status_UnknownError;

    if( m_DecoderType != movie::DecoderType::DecoderType_Unknown )
    {
        *decoderType    = m_DecoderType;
        *videoWidth     = m_VideoWidth;
        *videoHeight    = m_VideoHeight;
        *videoFrameRate = m_VideoFrameRate;
        movieStatus = movie::Status_Success;
    }
    return movieStatus;
}

movie::Status ByteStreamReader::PrepareH264Stream()
{
    movie::Status movieStatus = movie::Status_UnknownError;

    size_t dataRead = 0;
    size_t dataToRead = m_DataBufferSize;
    nn::Result result = nn::fs::ReadFile(&dataRead, m_FileHandle, m_FileReadOffset, ( void* ) m_dataBuffer, dataToRead);
    if( result.IsSuccess() )
    {
        uint8_t* picParam = NULL;
        uint8_t* seqParam = NULL;
        uint8_t* dataBuffer = ( uint8_t* ) m_dataBuffer;

        uint8_t* nalStart = NULL;
        size_t size = 0;
        uint8_t* inData = dataBuffer;
        size_t inDataSize = dataRead;
        unsigned spsNalType = 7;
        unsigned ppsNalType = 8;
        // Find h264 SPS data
        while( movie::GetH264NalUnitStartPositionAndNalSize(&inData, &inDataSize, &nalStart, &size) == true )
        {
            unsigned nalType = ( nalStart[ 0 ] & 0x1f );
            if( nalType == spsNalType )
            {
                seqParam = &nalStart[ 0 ];
                m_H264SpsSize = size;
                break;
            }
            if( inDataSize <= 3 )
            {
                return movieStatus;
            }
        }
        size = 0;
        inData = dataBuffer;
        inDataSize = dataRead;

        // Find h264 PPS data
        while( movie::GetH264NalUnitStartPositionAndNalSize(&inData, &inDataSize, &nalStart, &size) == true )
        {
            unsigned nalType = ( nalStart[ 0 ] & 0x1f );
            if( nalType == ppsNalType )
            {
                picParam = &nalStart[ 0 ];
                m_H264PpsSize = size;
                break;
            }
            if( inDataSize <= 3 )
            {
                return movieStatus;
            }
        }

        if( ( m_H264PpsSize == 0 ) || ( m_H264SpsSize == 0 ) )
        {
            return movieStatus;
        }
        else
        {

            m_H264Pps = new uint8_t[m_H264PpsSize];
            if( m_H264Pps == nullptr )
            {
                return movieStatus;
            }
            else
            {
                memcpy(( void* ) m_H264Pps, ( void* )picParam, m_H264PpsSize * sizeof(uint8_t));
            }

            m_H264Sps = new uint8_t[ m_H264SpsSize ];
            if( m_H264Sps == nullptr )
            {
                return movieStatus;
            }
            else
            {
                memcpy(( void* ) m_H264Sps, ( void* ) seqParam, m_H264SpsSize * sizeof(uint8_t));
            }
            // Get stream properties from SPS data
            movie::GetH264PropertiresFromSeqParamSetData(seqParam, m_H264SpsSize, &m_VideoWidth, &m_VideoHeight, &m_VideoFrameRate);
            if( m_VideoFrameRate > 0 )
            {
                m_PresentationTimeIncrementUs = int64_t(( 1000.0 / m_VideoFrameRate ) * 1000.0);
            }
            else
            {
                m_PresentationTimeIncrementUs = int64_t(( 1000.0 / 60.0 ) * 1000.0);
            }

            movieStatus = movie::Status_Success;
            m_DecoderType = movie::DecoderType_VideoAvc;
        }
    }
    return movieStatus;
}

movie::Status ByteStreamReader::ReadH264NextFrame(char* frameBuffer, int32_t frameBufferSize, int32_t* frameDataSize, int64_t* presentationTimeUs)
{
    movie::Status movieStatus = movie::Status_EndOfStream;
    if( m_AvailableInputData <= 0 )
    {
        size_t dataRead = 0;
        nn::Result result = nn::fs::ReadFile(&dataRead, m_FileHandle, m_FileReadOffset, m_dataBuffer, m_DataBufferSize);
        if( result.IsSuccess() )
        {
            m_FileReadOffset += dataRead;
            m_AvailableInputData = dataRead;
            m_DataBufferPointer = m_dataBuffer;
        }
        else
        {
            return movieStatus;
        }
    }
    uint8_t* nalStart = NULL;
    size_t nalSize = 0;
    size_t availableData = m_AvailableInputData;
    bool nalFound = false;
    while( 1 )
    {
        nalFound = movie::GetH264NalUnitStartPositionAndNalSize(&m_DataBufferPointer, &availableData, &nalStart, &nalSize);
        if( nalFound == true )
        {
            // nal unit found, entire nal data not found, rewind file back and do a new file read
            if( ( availableData <= 0) && (m_DataBufferPointer == NULL ) )
            {
                // Note: If a complete nal size is greater than file read buffer(m_dataBuffer) size, increase file read buffer size
                if( ( nalSize + 3 ) >= m_DataBufferSize )
                {
                    NN_SDK_LOG("\n Error, file read buffer size is not enough \n");
                    break;
                }
                movieStatus = movie::Status_NotEnoughData;
                m_AvailableInputData = 0;
                m_FileReadOffset -= (nalSize + 3);
                break;
            }
            // complete nal unit found
            // "nalSize" does not include start code size
            // "nalStart" does not include start code
            if( frameBufferSize >= nalSize + 3 )
            {
                memcpy(frameBuffer, nalStart - 3, nalSize + 3);
                *frameDataSize = nalSize + 3;
                m_AvailableInputData = availableData;
                *presentationTimeUs = m_PresentationTimeUs;
                m_PresentationTimeUs += m_PresentationTimeIncrementUs;
                movieStatus = movie::Status_Success;
            }
            break;
        }
        if( availableData <= 0  )
        {
            // All data searched, no nal unit found.
            movieStatus = movie::Status_NotEnoughData;
            NN_SDK_LOG("\n Status_NotEnoughDatan");
            break;
        }
    }
    return movieStatus;
}

movie::Status ByteStreamReader::PrepareForLooping()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    switch( m_StreamFormat )
    {
        case StreamFormat_ivf:
            m_AvailableInputData = 0ll;
            m_FileReadOffset = m_IvfHeaderSize;
            movieStatus = movie::Status_Success;
            break;

        case StreamFormat_h264:
            movieStatus = movie::Status_Success;
            m_AvailableInputData = 0ll;
            m_FileReadOffset = 0ll;
            break;

        default:
            break;
    }
    return movieStatus;
}
