﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/mii/mii_PrivateCommon.h>
#include <nn/mii/detail/mii_Debug.h>
#include <nn/mii/detail/mii_UtilityImpl.h>
#include <nn/mii/mii_Database.h>
#include <nn/mii/detail/mii_DetailCommon.h>
#include <nn/mii/detail/mii_StoreDataRaw.h>
#include <nn/mii/detail/mii_DatabaseFileManager.h>
#include <nn/mii/mii_StoreDataAccessor.h>
#include <nn/mii/mii_StoreDataContext.h>
#include <nn/mii/mii_Result.h>
#include <nn/mii/mii_PrivateResult.h>
//#include "Utility.h"
#include <nn/fs.h>
#include <nn/nn_SdkLog.h>
#include "utility/utility_ScopedPrivateDatabase.h"
#include "utility/utility_ScopedDatabaseImpl.h"
#include "utility/utility_DatabaseUtility.h"
#include "utility/utility_FileUtility.h"

#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

namespace
{
    void SetMiiDatabaseTestModeEnabled(bool isEnabled) NN_NOEXCEPT
    {
#if defined (NN_BUILD_CONFIG_OS_HORIZON)
        nn::settings::fwdbg::SetSettingsItemValue("mii", "is_db_test_mode_enabled", &isEnabled, sizeof(isEnabled));
#else
        // do nothing
        NN_UNUSED(isEnabled);
#endif
    }

#if defined (NN_BUILD_CONFIG_OS_HORIZON)
    class DatabaseTestingModeDisabledTest : public ::testing::Test
#else
    // settings::fwdbg が HORIZON のみ機能し、
    // HORIZON 以外ではテストモード無効にできないので DISABLED
    class DISABLED_DatabaseTestingModeDisabledTest : public ::testing::Test
#endif
    {
    protected:
        static void SetUpTestCase()
        {
            SetMiiDatabaseTestModeEnabled(false);
        }
        static void TearDownTestCase()
        {

        }
    };

    class DatabaseTestingModeEnabledTest : public ::testing::Test
    {
    protected:
        static void SetUpTestCase()
        {
            SetMiiDatabaseTestModeEnabled(true);
        }
        static void TearDownTestCase()
        {

        }
    };
}


#if defined (NN_BUILD_CONFIG_OS_HORIZON)
#define NNT_MII_DATABASE_TESTING_MODE_DISABLED_FIXTURE_NAME DatabaseTestingModeDisabledTest
#else
#define NNT_MII_DATABASE_TESTING_MODE_DISABLED_FIXTURE_NAME DISABLED_DatabaseTestingModeDisabledTest
#endif

#define NNT_MII_DATABASE_TESTING_MODE_ENABLED_FIXTURE_NAME DatabaseTestingModeEnabledTest

TEST_F(NNT_MII_DATABASE_TESTING_MODE_DISABLED_FIXTURE_NAME, DestroyFile)
{
    utility::ScopedPrivateDatabase database;
    NNT_ASSERT_RESULT_FAILURE(nn::mii::ResultDevelopmentOnly, database.Get().DestroyFile());
}
TEST_F(NNT_MII_DATABASE_TESTING_MODE_ENABLED_FIXTURE_NAME, DestroyFile)
{
    utility::ScopedPrivateDatabase database;
    NNT_ASSERT_RESULT_SUCCESS(database.Get().DestroyFile() );
}

TEST_F(NNT_MII_DATABASE_TESTING_MODE_DISABLED_FIXTURE_NAME, DeleteFile)
{
    utility::ScopedPrivateDatabase database;
    NNT_ASSERT_RESULT_FAILURE(nn::mii::ResultDevelopmentOnly, database.Get().DeleteFile());
}
TEST_F(NNT_MII_DATABASE_TESTING_MODE_ENABLED_FIXTURE_NAME, DeleteFile)
{
    utility::ScopedPrivateDatabase database;
    NNT_ASSERT_RESULT_SUCCESS(database.Get().DeleteFile() );
}

TEST_F(NNT_MII_DATABASE_TESTING_MODE_DISABLED_FIXTURE_NAME, Format)
{
    utility::ScopedPrivateDatabase database1;
    EXPECT_FALSE(database1.Get().IsUpdated(nn::mii::SourceFlag_All));

    utility::ScopedPrivateDatabase database2;
    database2.Get().Format(); // void

    // ↑の Format は失敗しているはずなので、 IsUpdated() == false のままのはず
    EXPECT_FALSE(database1.Get().IsUpdated(nn::mii::SourceFlag_All));
}
TEST_F(NNT_MII_DATABASE_TESTING_MODE_ENABLED_FIXTURE_NAME, Format)
{
    utility::ScopedPrivateDatabase database1;
    EXPECT_FALSE(database1.Get().IsUpdated(nn::mii::SourceFlag_All));

    utility::ScopedPrivateDatabase database2;
    database2.Get().Format(); // void

    // ↑の Format は成功しているはずなので、 IsUpdated() == true になるはず
    EXPECT_TRUE(database1.Get().IsUpdated(nn::mii::SourceFlag_All));
}

TEST_F(NNT_MII_DATABASE_TESTING_MODE_DISABLED_FIXTURE_NAME, Import)
{
    utility::ScopedPrivateDatabase database;

    nn::mii::detail::ImportFile file;
    nn::mii::detail::DatabaseFile databaseFile;
    databaseFile.Initialize();
    {
        nn::mii::StoreData storeData;
        nn::mii::StoreDataContext context;
        nn::mii::StoreDataAccessor(&storeData).BuildDefault(&context, 0);
        databaseFile.AddOrReplace(nn::mii::detail::StoreDataRaw::Cast(storeData));
    }
    databaseFile.UpdateCrc();
    databaseFile.Export(&file);
    NNT_ASSERT_RESULT_FAILURE(nn::mii::ResultDevelopmentOnly, database.Get().Import(&file, sizeof(file)));
}
TEST_F(NNT_MII_DATABASE_TESTING_MODE_ENABLED_FIXTURE_NAME, Import)
{
    utility::ScopedPrivateDatabase database;

    nn::mii::detail::ImportFile file;
    nn::mii::detail::DatabaseFile databaseFile;
    databaseFile.Initialize();
    {
        nn::mii::StoreData storeData;
        nn::mii::StoreDataContext context;
        nn::mii::StoreDataAccessor(&storeData).BuildDefault(&context, 0);
        databaseFile.AddOrReplace(nn::mii::detail::StoreDataRaw::Cast(storeData));
    }
    databaseFile.UpdateCrc();
    databaseFile.Export(&file);
    NNT_ASSERT_RESULT_SUCCESS(database.Get().Import(&file, sizeof(file)));
}

TEST_F(NNT_MII_DATABASE_TESTING_MODE_DISABLED_FIXTURE_NAME, Export)
{
    utility::ScopedPrivateDatabase database;
    nn::mii::detail::ImportFile file;
    database.Get().Export(&file, sizeof(file)); // void

    // Export は void だが、内部で正常に Export がされていない場合は、バッファが適当に埋められているはず
    EXPECT_FALSE(file.IsValid());
}
TEST_F(NNT_MII_DATABASE_TESTING_MODE_ENABLED_FIXTURE_NAME, Export)
{
    utility::ScopedPrivateDatabase database;
    nn::mii::detail::ImportFile file;
    file.Initialize();
    database.Get().Export(&file, sizeof(file));

    EXPECT_TRUE(file.IsValid());
}
