﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testUserMigrationDataTransfer_TestBasic.h"
#include "testUserMigrationDataTransfer_Util.h"

#include <curl/curl.h>

#include <nnt.h>
#include <nnt/nnt_Argument.h>
#include <nnt/teamcity/testTeamcity_Logger.h>

#include <nn/nn_Abort.h>
#include <nn/account/account_ApiForAdministrators.h>
#include <nn/friends/friends_Api.h>
#include <nn/migration/detail/migration_LoginSession.h>
#include <nn/migration/detail/migration_SubsystemInitializer.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/os/os_Thread.h>
#include <nn/ovln/ovln_ForDevelop.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Config.h>
#include <nn/socket/socket_SystemConfig.h>
#include <nn/spl/spl_Api.h>
#include <nn/time/time_Api.h>

#include "nntMigration_Account.h"
#include "nntMigration_ConnectionPolicy.h"
#include "nntMigration_EncryptionPolicy.h"
#include "nntMigration_SaveData.h"

namespace {
bool g_ServerResult;

struct SaveDataCreationInfo
{
    nn::ncm::ApplicationId appId;
    size_t size;
    size_t journalSize;
};

struct TestConfiguration
{
    nn::account::Uid srcUser;
    nn::account::Uid dstUser;
    nn::migration::detail::LocalUserLoginSession login;
    nn::migration::detail::ThreadResourceAllocator* pThreadResourceAllocator;
};

template <typename ConnectionPolicy, typename EncryptionPolicy>
void ThreadFunction(void* data) NN_NOEXCEPT
{
    auto& config = *reinterpret_cast<TestConfiguration*>(data);
    g_ServerResult = RunServer<ConnectionPolicy, EncryptionPolicy>(config.srcUser, config.login, config.pThreadResourceAllocator->Allocate());
}

template <typename ConnectionPolicy, typename EncryptionPolicy>
void RunBasicTest(const SaveDataCreationInfo* saveDataCreationInfoList, size_t count) NN_NOEXCEPT
{
    // バックグラウンドデーモンの停止など
    nn::migration::detail::SubsystemInitializer subInit0;
    nnt::migration::IdcSubsystemInitializer subInit1;

    // クリーンアップ
    nnt::migration::CleanupUsers();
    server::CleanupStorage();
    client::CleanupStorage();

    // ユーザーアカウントの作成
    auto uid0 = nnt::migration::SetupUser("moriyotd1", "N1ntend0");
    auto uid1 = nnt::migration::SetupUser();
    for (size_t i = 0; i < count; ++ i)
    {
        const auto& info = saveDataCreationInfoList[i];
        nnt::migration::CreateSaveData(uid0, info.appId, info.size, info.journalSize);
    }

    static NN_ALIGNAS(4096) nn::Bit8 s_ThreadBuffer[32 * 4096][2];
    nn::migration::detail::ThreadResourceAllocator threadResourceAllocator(nn::os::DefaultThreadPriority, "TestDataTransfer");
    threadResourceAllocator.Initialize<32 * 1024>(s_ThreadBuffer, sizeof(s_ThreadBuffer));

    // テストオブジェクトの作成
    TestConfiguration config;
    config.srcUser = uid0;
    config.dstUser = uid1;
    config.login.Initialize(uid0);
    config.pThreadResourceAllocator = &threadResourceAllocator;

    // テストの実施
    {
        PAGE_ALIGN static char g_ThreadStack[128 * 1024];

        // サーバーの起動
        nn::os::ThreadType thread;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &thread,
            ThreadFunction<ConnectionPolicy, EncryptionPolicy>, &config,
            g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority));

        nn::os::StartThread(&thread);
        NN_UTIL_SCOPE_EXIT
        {
            nn::os::WaitThread(&thread);
            nn::os::DestroyThread(&thread);
        };
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));

        // クライアントの実行
        ASSERT_TRUE((RunClient<ConnectionPolicy, EncryptionPolicy>(config.dstUser, config.login, config.pThreadResourceAllocator->Allocate())));
    }
    ASSERT_TRUE(g_ServerResult);

    // ユーザーアカウントの削除
    NNT_MIGRATION_ASSERT_RESULT_SUCCESS(config.login.DeleteUser(nullptr));

}

const SaveDataCreationInfo saveDataCreationInfo[] = {
    {{0x010000000000B123ull}, 64 * 1024 * 1024, 256 * 1024},
    {{0x010000000000B124ull}, 64 * 1024 * 1024, 256 * 1024},
    {{0x010000000000B125ull}, 1024 * 1024 * 1024, 256 * 1024},
};

} // ~namespace <anonoymous>

TEST(Delayed, Basic_NoEncryption_1M_24Mbps)
{
    RunBasicTest<nnt::migration::DelayedInprocessConnectionPolicy<24>, nnt::migration::NoEncryptionPolicy<1024 * 1024>>(
        saveDataCreationInfo, std::extent<decltype(saveDataCreationInfo)>::value);
}

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
TEST(Delayed, Basic_MessageEncryption_1M_24Mbps)
{
    RunBasicTest<nnt::migration::DelayedInprocessConnectionPolicy<24>, nnt::migration::MessageOnlyEncryptionPolicy<1024 * 1024>>(
        saveDataCreationInfo, std::extent<decltype(saveDataCreationInfo)>::value);
}
#endif

TEST(Buffer, Basic_NoEncryption_1M)
{
    RunBasicTest<nnt::migration::InprocessConnectionPolicy, nnt::migration::NoEncryptionPolicy<1024 * 1024>>(
        saveDataCreationInfo, std::extent<decltype(saveDataCreationInfo)>::value);
}

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
TEST(Buffer, Basic_MessageEncryption_1M)
{
    RunBasicTest<nnt::migration::InprocessConnectionPolicy, nnt::migration::MessageOnlyEncryptionPolicy<1024 * 1024>>(
        saveDataCreationInfo, std::extent<decltype(saveDataCreationInfo)>::value);
}
#endif

TEST(Localhost, Basic_NoEncryption_1M)
{
    RunBasicTest<nnt::migration::LocalhostConnectionPolicy, nnt::migration::NoEncryptionPolicy<1024 * 1024>>(
        saveDataCreationInfo, std::extent<decltype(saveDataCreationInfo)>::value);
}

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
TEST(Localhost, Basic_MessageEncryption_1M)
{
    RunBasicTest<nnt::migration::LocalhostConnectionPolicy, nnt::migration::MessageOnlyEncryptionPolicy<1024 * 1024>>(
        saveDataCreationInfo, std::extent<decltype(saveDataCreationInfo)>::value);
}
#endif

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    testing::InitGoogleTest(&argc, argv);
    testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    static nn::socket::SystemConfigDefaultWithMemory<2, 0, 2> s_SocketConfig(2);

    // サブシステムの初期化
    nn::account::InitializeForAdministrator();
    nn::friends::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(s_SocketConfig));
    NN_ABORT_UNLESS(curl_global_init(CURL_GLOBAL_ALL) == CURLE_OK);

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::InitializeForSystem());
    nn::pdm::InitializeForNotification();
    nn::spl::InitializeForCrypto();
#else
    nn::ncm::Initialize();
    nn::ovln::PrepareSenderAndReceiverForDevelop();
    nn::ovln::InitializeSenderLibraryForOverlay();
    nn::nim::InitializeForNetworkInstallManager();
#endif // defined(NN_BUILD_CONFIG_OS_HORIZON)
    nn::ns::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    auto result = RUN_ALL_TESTS();
    nnt::Exit(result);
}
