﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <map>
#include <string>

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/migration/detail/migration_AbstractStorage.h>

namespace nnt { namespace migration {

template <typename Tag>
class RamStorage
    : public nn::migration::detail::AbstractStorage
{
private:
    typedef std::pair<void*, size_t> Data;
    static std::map<std::string, Data> s_Storage;
    static nn::migration::detail::Mutex s_Lock;

    char m_VolumeName[32];

    std::string GetPathString(const char* path) const NN_NOEXCEPT;

protected:
    virtual void CleanupImpl() const NN_NOEXCEPT final NN_OVERRIDE;

    virtual nn::Result CreateImpl(const char* path, size_t sizeToReserve) const NN_NOEXCEPT final NN_OVERRIDE;
    virtual nn::Result DeleteImpl(const char* path) const NN_NOEXCEPT final NN_OVERRIDE;
    virtual nn::Result MoveImpl(const char* to, const char* from) const NN_NOEXCEPT final NN_OVERRIDE;
    virtual nn::Result GetSizeImpl(size_t* pOutSize, const char* path) const NN_NOEXCEPT final NN_OVERRIDE;

    virtual nn::Result CommitImpl() const NN_NOEXCEPT final NN_OVERRIDE;
    virtual nn::Result WriteImpl(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT final NN_OVERRIDE;
    virtual nn::Result AppendImpl(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT final NN_OVERRIDE;
    virtual nn::Result ReadImpl(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT final NN_OVERRIDE;

public:
    explicit RamStorage() NN_NOEXCEPT;
};

template <typename Tag>
class RamStoragePolicy
{
public:
    typedef RamStorage<Tag> Storage;
};

}} // ~namespace nnt::migration

#include <mutex>

#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace migration {

template <typename Tag>
nn::migration::detail::Mutex  RamStorage<Tag>::s_Lock = NN_MIGRATION_DETAIL_MUTEX_INITIALIZER(false);

template <typename Tag>
std::map<std::string, typename RamStorage<Tag>::Data>  RamStorage<Tag>::s_Storage;

template <typename Tag>
inline RamStorage<Tag>::RamStorage() NN_NOEXCEPT
{
    std::strncpy(m_VolumeName, Tag::VolumeName, sizeof(m_VolumeName));
    NN_SDK_ASSERT_LESS(strnlen(m_VolumeName, sizeof(m_VolumeName)), sizeof(m_VolumeName));
}
template <typename Tag>
inline std::string RamStorage<Tag>::GetPathString(const char* path) const NN_NOEXCEPT
{
    std::string str(m_VolumeName);
    str += ":/";
    str += path;
    NN_SDK_LOG("[Migration > RamStorage] %s\n", str.c_str());
    return str;
}

template <typename Tag>
inline void RamStorage<Tag>::CleanupImpl() const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);
    for (auto& e : s_Storage)
    {
        std::free(e.second.first);
    }
    s_Storage.clear();
}

template <typename Tag>
inline nn::Result RamStorage<Tag>::CreateImpl(const char* path, size_t sizeToReserve) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

    auto str = GetPathString(path);
    NN_RESULT_THROW_UNLESS(s_Storage.count(str) == 0, nn::fs::ResultPathAlreadyExists());

    s_Storage[str] = Data(std::malloc(sizeToReserve), sizeToReserve);
    NN_RESULT_SUCCESS;
}
template <typename Tag>
inline nn::Result RamStorage<Tag>::DeleteImpl(const char* path) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

    auto str = GetPathString(path);
    NN_RESULT_THROW_UNLESS(s_Storage.count(str) != 0, nn::fs::ResultPathNotFound());

    const auto& d = s_Storage[str];
    std::free(d.first);
    s_Storage.erase(str);
    NN_RESULT_SUCCESS;
}
template <typename Tag>
inline nn::Result RamStorage<Tag>::MoveImpl(const char* to, const char* from) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

    auto strFrom = GetPathString(from);
    NN_RESULT_THROW_UNLESS(s_Storage.count(strFrom) != 0, nn::fs::ResultPathNotFound());
    auto strTo = GetPathString(to);
    NN_RESULT_THROW_UNLESS(s_Storage.count(strTo) == 0, nn::fs::ResultPathAlreadyExists());

    auto d = s_Storage[strFrom];
    s_Storage.erase(strFrom);
    s_Storage[strTo] = std::move(d);
    NN_RESULT_SUCCESS;
}
template <typename Tag>
inline nn::Result RamStorage<Tag>::GetSizeImpl(size_t* pOutSize, const char* path) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

    auto str = GetPathString(path);
    NN_RESULT_THROW_UNLESS(s_Storage.count(str) != 0, nn::fs::ResultPathNotFound());

    const auto& d = s_Storage[str];
    *pOutSize = d.second;
    NN_RESULT_SUCCESS;
}

template <typename Tag>
inline nn::Result RamStorage<Tag>::CommitImpl() const NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}
template <typename Tag>
inline nn::Result RamStorage<Tag>::WriteImpl(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

    auto str = GetPathString(path);
    NN_RESULT_THROW_UNLESS(s_Storage.count(str) != 0, nn::fs::ResultPathNotFound());

    auto& d = s_Storage[str];
    std::free(d.first);
    d = Data(std::malloc(bufferSize), bufferSize);
    std::memcpy(d.first, buffer, bufferSize);
    NN_RESULT_SUCCESS;
}
template <typename Tag>
inline nn::Result RamStorage<Tag>::AppendImpl(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

    auto str = GetPathString(path);
    NN_RESULT_THROW_UNLESS(s_Storage.count(str) != 0, nn::fs::ResultPathNotFound());

    auto& d = s_Storage[str];
    if (d.second < offset + bufferSize)
    {
        auto* p = std::malloc(offset + bufferSize);
        std::memcpy(p, d.first, d.second);
        free(d.first);
        d.first = p;
        d.second = offset + bufferSize;
    }
    std::memcpy(reinterpret_cast<char*>(d.first) + offset, buffer, bufferSize);
    NN_RESULT_SUCCESS;
}
template <typename Tag>
inline nn::Result RamStorage<Tag>::ReadImpl(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT
{
    std::lock_guard<decltype(s_Lock)> lock(s_Lock);

    auto str = GetPathString(path);
    NN_RESULT_THROW_UNLESS(s_Storage.count(str) != 0, nn::fs::ResultPathNotFound());

    const auto& d = s_Storage[str];
    auto readSize = std::min(bufferSize, d.second - offset);
    std::memcpy(buffer, reinterpret_cast<char*>(d.first) + offset, readSize);
    *pOutActualSize = readSize;
    NN_RESULT_SUCCESS;
}

}} // ~namespace nnt::migration
