﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <utility>
#include <random>

#include <nnt/nntest.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/init.h>

#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

extern "C" void nninitStartup()
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // Horizon でテストする場合 fs アロケータ用の malloc 領域が必要
    nn::init::InitializeAllocator(nullptr, 2 * 1024 * 1024);
#endif

}

namespace {
/**
 * @brief   テストに与えるパラメータです。
 */
enum StandardAllocatorTestParam
{
    StandardAllocatorTestParam_DisableThreadCache = 0,  //!< スレッドキャッシュを利用しない
    StandardAllocatorTestParam_EnableThreadCache        //!< スレッドキャッシュを利用する
};

/**
* @brief   テストで利用するテストフィクスチャです。
*/
class StandardAllocatorVammTest : public ::testing::TestWithParam< std::pair<StandardAllocatorTestParam, size_t> >
{
protected:

    /**
    * @brief   テスト開始時に毎回呼び出される関数です。
    */
    virtual void SetUp()
    {
        ASSERT_TRUE(nn::os::IsVirtualAddressMemoryEnabled());
        std::pair<StandardAllocatorTestParam, size_t> pattern = GetParam();
        size_t virtualSize = pattern.second * 1024 * 1024 * 1024;
        ASSERT_LE(virtualSize, 63ull * 1024 * 1024 * 1024); // NX での最大値
        m_Param = pattern.first;

        // ヒープを初期化
        if (pattern.first == StandardAllocatorTestParam_DisableThreadCache)
        {
            NN_LOG("ThreadCache: disabled heapSize: %zu GiB\n", pattern.second);
            m_Allocator.Initialize(nullptr, virtualSize);
        }
        else
        {
            NN_LOG("ThreadCache: enabled heapSize: %zu GiB\n", pattern.second);
            m_Allocator.Initialize(nullptr, virtualSize, true);
        }
    }

    /**
    * @brief   テスト終了時に毎回呼び出される関数です。
    */
    virtual void TearDown()
    {
        m_Allocator.Finalize();
    }

protected:
    nn::mem::StandardAllocator m_Allocator;
    StandardAllocatorTestParam m_Param;
};

}   // namespace

/**
 * @brief   サイズ取得関数で物理的に確保可能なサイズか帰ってくることを確認します。
 */
TEST_P(StandardAllocatorVammTest, GetSize)
{
    size_t allocatableSize = m_Allocator.GetAllocatableSize();
    size_t totalFreeSize = m_Allocator.GetTotalFreeSize();

    NN_LOG("allocatableSize: %zu bytes\n", allocatableSize);
    NN_LOG("totalFreeSize:   %zu bytes\n", totalFreeSize);
    // Windows の場合は内部で呼んでいる GlobalMemoryStatusEx() の情報が揮発性なので、
    // 最低限の確認しかできない
    EXPECT_GT(allocatableSize, 0);
    EXPECT_GT(totalFreeSize, 0);
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    EXPECT_LE(allocatableSize, totalFreeSize);  // Windows ではグローバルヒープなので保証できない
    // NX では現状 6 GB より大きくなることはない
    EXPECT_LE(allocatableSize, 6ull * 1024 * 1024 * 1024);
    void* ptr = m_Allocator.Allocate(allocatableSize);
    ASSERT_TRUE(ptr != nullptr);
#endif
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/**
 * @brief   サイズ取得関数で物理割り当てが行われている場所といない場所が
 *          混在していても、適切なサイズが返ってくることを確認します。
 */
TEST_P(StandardAllocatorVammTest, GetSize2)
{
    m_Allocator.Allocate(4096); // SpanPage 作成のための alloc

    const size_t allocSize = 1024 * 1024;
    const size_t physicalPageSize = 4 * 1024 * 1024;    // nlib に合わせる

    size_t freeSize = m_Allocator.GetTotalFreeSize();

    // 物理ページマップ境界からずれた位置を記憶
    uintptr_t position1 = 0;
    for (int i = 0; i < 10; ++i)
    {
        uintptr_t addr = reinterpret_cast<uintptr_t>(m_Allocator.Allocate(allocSize));
        ASSERT_TRUE(addr != 0);
        if (addr != (addr & ~(physicalPageSize - 1)))
        {
            position1 = addr;
            // この時点でのヒープ空きサイズの合計
            size_t remainSize = freeSize - allocSize * (i + 1);
            size_t allocatableSize = m_Allocator.GetAllocatableSize();
            EXPECT_EQ(remainSize, m_Allocator.GetTotalFreeSize());
            // GetAllocatableSize() は 4MB アラインの端数ぶん remainsize より小さくなっている
            EXPECT_GE(remainSize, allocatableSize);
            EXPECT_LE(remainSize - physicalPageSize * 2, allocatableSize);

            void* p = m_Allocator.Allocate(allocatableSize);
            ASSERT_TRUE(p != nullptr);
            m_Allocator.Free(p);
            break;
        }
    }
    ASSERT_TRUE(position1 != 0);

    // この時点で確保済み領域の先頭は physicalPageSize アラインを外れている
    // 図にすると以下のような感じ
    //
    // [記号]
    // # :確保済み - :空き(物理未マップ) _ :空き(物理マップ済み) | :4MB 境界を表す区切り
    //
    // 1 文字 1MB とする
    //
    // ####|#___|--- (略) ----|----|----|----|----|___#|#

    const size_t allocNum = (physicalPageSize / allocSize) * 3;
    void* ptrs[allocNum];
    for (int i = 0; i < allocNum; ++i)
    {
        ptrs[i] = m_Allocator.Allocate(allocSize);
        ASSERT_TRUE(ptrs[i] != nullptr);
    }

    // この時点で、 ptrs の持つ領域は連続かつ物理ページマップ境界からずれた
    // アドレスを両端にもつことになる
    //
    // ####|#___|--- (略) ----|----|___#|####|####|####|#

    uintptr_t position2 = reinterpret_cast<uintptr_t>(ptrs[allocNum - 1]);
    ASSERT_TRUE(position2 != (position2 & ~(physicalPageSize - 1)));
    // ギリギリまで確保
    EXPECT_TRUE(m_Allocator.Allocate(m_Allocator.GetAllocatableSize()) != nullptr);
    for (int i = 0; i < allocNum; ++i)
    {
        m_Allocator.Free(ptrs[i]);
    }

    // ここで空き領域で一番大きいのは、 ptrs が確保していた領域
    // この領域には物理マップ済みの部分と未マップの部分が混在する
    //
    // ####|#___|--- (略) ####|####|###_|----|----|___#|#
    //                                   ↑ ptrs が確保していた領域

    {
        size_t allocatableSize = m_Allocator.GetAllocatableSize();
        // 物理マップ済みの部分と未マップの部分が混在する領域であっても
        // 正しく空き領域サイズを取得できるか
        EXPECT_EQ(allocNum * allocSize, allocatableSize);
        EXPECT_LE(allocNum * allocSize, m_Allocator.GetTotalFreeSize());

        EXPECT_TRUE(m_Allocator.Allocate(allocatableSize) != nullptr);
    }
}

/**
 * @brief   NINTENDOSDK-6642 の対処確認
 * @details SDEV を 4GB モードにした場合にのみ不具合が起きた
 */
TEST_P(StandardAllocatorVammTest, NINTENDOSDK6642)
{
    // 1MB ずつ確保
    void* Ptr[1024];
    for (int cnt = 0; cnt < 1024; ++cnt)
    {
        Ptr[cnt] = m_Allocator.Allocate(1024 * 1024);
    }

    size_t freeSize = m_Allocator.GetTotalFreeSize();

    // 4MB ごとに 1MB 確保済み領域を残しつつ、残りは解放
    for (int cnt = 0; cnt<1024; ++cnt)
    {
        if (cnt % 4 != 0)
        {
            m_Allocator.Free(Ptr[cnt]);
            // 解放しているので、空き領域の合計は増えるはず
            EXPECT_LT(freeSize, m_Allocator.GetTotalFreeSize());
        }
    }

}
#endif

/**
 * @brief   Dump() でとまりやはまりがないか最低限の確認をします。
 */
TEST_P(StandardAllocatorVammTest, Dump)
{
    m_Allocator.Dump();

    void* ptr = m_Allocator.Allocate(4096);

    m_Allocator.Dump();

    m_Allocator.Free(ptr);

    m_Allocator.Dump();
}

INSTANTIATE_TEST_CASE_P(VirtualSpaceSize,
                        StandardAllocatorVammTest,
                        testing::Values(std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 4),
                                        std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 8),
                                        std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 16),
                                        std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 32),
                                        std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 60),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 4),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 8),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 16),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 32),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 60)));

