﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <new>
#include <map>
#include <vector>
#include <random>

#include <nnt/nntest.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/init.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/svc/svc_Base.h>
#endif

#define ASSERT_NULL(p) ASSERT_TRUE(p == nullptr)
#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

// マクロを有効にすると標準の malloc と実行速度の比較ができます( Windows のみ)
//#define NN_TEST_COMPARE_TO_MALLOC

size_t g_AddressSpaceSize = 0;                      // 使用するアドレス空間のサイズ
nn::mem::StandardAllocator* g_pAllocator = nullptr; // STL 用アロケータ


namespace {

/**
 * @brief   テストに与えるパラメータです。
 */
enum StandardAllocatorTestParam
{
    StandardAllocatorTestParam_DisableThreadCache = 0,  //!< スレッドキャッシュを利用しない
    StandardAllocatorTestParam_EnableThreadCache        //!< スレッドキャッシュを利用する
};

/**
 * @brief   STL で利用するためのアロケータです。
 * @details グローバルの Standard Allocator インスタンスが持つメモリプールを使って、
 *          STL 用の領域を確保／解放します。
 */
template<class T>
class StlAllocatorForTest
{
public:
    typedef size_t size_type;
    typedef T value_type;
    typedef T* pointer;
    typedef const T* const_pointer;
    typedef T& reference;
    typedef const T& const_reference;
    typedef ptrdiff_t difference_type;
    template<class U>
    struct rebind
    {
        typedef StlAllocatorForTest<U> other;
    };

    // コンストラクタ
    StlAllocatorForTest() { }
    explicit StlAllocatorForTest(const StlAllocatorForTest&) { }
    template<class U>
    StlAllocatorForTest(const StlAllocatorForTest<U>&) { }
    // デストラクタ
    ~StlAllocatorForTest() { }

    /**
     * @brief   メモリを割り当てます。
     */
    pointer allocate(size_type n, const_pointer hint = 0)
    {
        NN_UNUSED(hint);
        NN_ABORT_UNLESS(g_pAllocator != nullptr);
        NN_ABORT_UNLESS(g_AddressSpaceSize != 0);
        return reinterpret_cast<pointer>(g_pAllocator->Allocate(n * sizeof(T)));
    }

    /**
     * @brief   メモリ解放します。
     */
    void deallocate(pointer ptr, size_type n)
    {
        NN_UNUSED(n);
        NN_ABORT_UNLESS(g_pAllocator != nullptr);
        NN_ABORT_UNLESS(g_AddressSpaceSize != 0);
        g_pAllocator->Free(ptr);
    }

    /**
     * @brief   割り当て済みの領域を初期化します。
     */
    void construct(pointer p, const T& t)
    {
        new ((void*)p) T(t);
    }
    void construct(pointer p)
    {
        new ((void*)p) T();
    }

    /**
     * @brief   初期化済みの領域を削除します。
     */
    void destroy(pointer ptr)
    {
        NN_UNUSED(ptr);
        (*ptr).~T();
    }
    template<class X>
    void destroy(X* p)
    {
        NN_UNUSED(p);
        p->~X();
    }

    /**
     * @brief   アドレスを返します。
     */
    pointer address(reference value) const
    {
        return &value;
    }
    const_pointer address(const_reference value) const
    {
        return &value;
    }

    /**
     * @brief   割り当てられる最大の要素数を返します。
     */
    size_t max_size() const
    {
        return std::numeric_limits<size_t>::max() / sizeof(T);
    }
};

/**
 * @brief   テストで利用するテストフィクスチャです。
 */
class StandardAllocatorTest : public ::testing::TestWithParam< std::pair<StandardAllocatorTestParam, size_t> >
{
protected:

    /**
     * @brief   テスト開始時に毎回呼び出される関数です。
     */
    virtual void SetUp()
    {
        ASSERT_TRUE(nn::os::IsVirtualAddressMemoryEnabled());
        std::pair<StandardAllocatorTestParam, size_t> pattern = GetParam();
        m_VirtualSize = pattern.second * 1024 * 1024 * 1024;

        ASSERT_LE(m_VirtualSize, 63ull * 1024 * 1024 * 1024); // NX での最大値

        // ヒープを初期化
        if (pattern.first == StandardAllocatorTestParam_DisableThreadCache)
        {
            NN_LOG("ThreadCache: disabled heapSize: %zu GiB\n", pattern.second);
            m_Allocator.Initialize(nullptr, m_VirtualSize);
        }
        else
        {
            //NN_LOG("ThreadCache: enabled heapSize: %zu GiB\n", pattern.second);
            m_Allocator.Initialize(nullptr, m_VirtualSize, true);
        }

        g_pAllocator = &m_Allocator;
        g_AddressSpaceSize = m_VirtualSize;
    }

    /**
     * @brief   テスト終了時に毎回呼び出される関数です。
     */
    virtual void TearDown()
    {
        m_Allocator.Finalize();
    }

protected:
    nn::mem::StandardAllocator m_Allocator;
    size_t m_VirtualSize;
};

}   // unnamed namespace

// ======
// テスト
// ======

/**
 * @brief   小さなサイズの確保→解放を繰り返します。
 */
TEST_P(StandardAllocatorTest, Simple)
{

    for (int i = 0; i < 10; ++i)
    {
        void* ptr = m_Allocator.Allocate(8, 0); // 不正アライメント
        ASSERT_NULL(ptr);
        ptr = m_Allocator.Allocate(8, 5);       // 2^n でないアライメント
        ASSERT_NULL(ptr);
    }

    for (int i = 0; i < 1000000; ++i)
    {
        void* addr = m_Allocator.Allocate(8);
        m_Allocator.Free(addr);
    }
    void* ptr = m_Allocator.Allocate(8);
    ASSERT_EQ(NN_ALIGNOF(std::max_align_t), m_Allocator.GetSizeOf(ptr));
    m_Allocator.Free(ptr);
}

/**
 * @brief   1Byte から 10KB まで順に確保します。
 */
TEST_P(StandardAllocatorTest, Size)
{
    for (size_t i = 1; i < 10000; ++i)
    {
        void* ptr = m_Allocator.Allocate(i);
        ASSERT_TRUE(ptr != NULL);
        ASSERT_LE(i, m_Allocator.GetSizeOf(ptr));
        m_Allocator.Free(ptr);
    }
}

/**
 * @brief   小さなサイズの確保を連続で行い、その後確保した領域の解放を連続で行います。
 */
TEST_P(StandardAllocatorTest, Simple2)
{
    void** p = reinterpret_cast<void**>(m_Allocator.Allocate(100000 * sizeof(void*)));
    ASSERT_TRUE(p != NULL);
    for (int i = 0; i < 100000; ++i)
    {
        p[i] = m_Allocator.Allocate(8);
        ASSERT_TRUE(p[i] != NULL);
    }
    for (int i = 0; i < 100000; ++i)
    {
        m_Allocator.Free(p[i]);
    }
    m_Allocator.Free(p);
}

/**
 * @brief   あるまとまった単位での連続確保→連続解放を繰り返します。
 */
TEST_P(StandardAllocatorTest, Simple3)
{
    void* p[64];
    for (int j = 0; j < 2000; ++j)
    {
        for (int i = 0; i < 64; ++i)
        {
            p[i] = m_Allocator.Allocate(8);
        }
        for (int i = 0; i < 64; ++i)
        {
            m_Allocator.Free(p[i]);
        }
    }
}

/**
 * @brief   std::vector のアロケータを Standard Allocator に置き換え、
 *          要素の追加を繰り返します。
 */
TEST_P(StandardAllocatorTest, Vector)
{
    std::vector<int, StlAllocatorForTest<int> > vec;
    for (int i = 0; i < 100000; ++i)
    {
        vec.push_back(i);
    }
    for (int i = 0; i < 100000; ++i)
    {
        ASSERT_EQ(i, vec[i]);
    }

}

/**
 * @brief   要素を一定数追加した後、 std::vector が確保している領域を変化させても、
 *          メモリ内の状況が変化しないことを確かめます。
 */
TEST_P(StandardAllocatorTest, Vector2)
{
    typedef std::vector<int, StlAllocatorForTest<int> > MyVec;
    MyVec vec;
    for (int i = 0; i < 1000; ++i)
    {
        vec.push_back(i);
    }
    for (int i = 0; i < 1000; ++i)
    {
        for (int j = 0; j < 1000 - i; ++j)
        {
            ASSERT_EQ(j, vec[j]) << j;
        }
        vec.pop_back();
        vec.shrink_to_fit();   // vector の領域を確保済みのサイズに縮小する
    }
}

/**
 * @brief   複数の std::vector を定義して一斉確保、一斉解放を行います。
 */
TEST_P(StandardAllocatorTest, Vector3)
{
    typedef std::vector<int, StlAllocatorForTest<int> > MyVec;
    MyVec vec0, vec1, vec2;
    for (int i = 0; i < 100000; ++i)
    {
        vec0.push_back(i);
        vec1.push_back(i * 2);
        vec2.push_back(i * 3);
    }
    for (int i = 0; i < 100000; ++i)
    {
        ASSERT_EQ(i, vec0[i]) << i;
        ASSERT_EQ(i * 2, vec1[i]) << i;
        ASSERT_EQ(i * 3, vec2[i]) << i;
    }
}

/**
 * @brief   std::map に要素を追加し、正しく追加できているか確認します。
 */
TEST_P(StandardAllocatorTest, Map)
{
    // キーが昇順に並ぶ map を作成
    std::map<int, int, std::less<int>,
                StlAllocatorForTest<std::pair<const int, int> > > m;
    for (int i = 0; i < 10000; ++i)
    {
        m[i] = i;
    }
    for (int i = 0; i < 10000; ++i)
    {
        ASSERT_EQ(i, m[i]);
    }
}

/**
 * @brief   std::map で要素の削除が行えるか確認します。
 */
TEST_P(StandardAllocatorTest, Map2)
{

    typedef std::map<int, int, std::less<int>,
         StlAllocatorForTest<std::pair<const int, int> > > MyMap0;
    MyMap0 m;
    for (int i = 0; i < 10000; ++i)
    {
        m[i] = i;
    }
    for (int i = 0; i < 5000; ++i)
    {
        m.erase(m.find(i));
    }
    ASSERT_EQ(5000UL, m.size());
    for (int i = 5000; i < 10000; ++i)
    {
        ASSERT_EQ(i, m[i]);
    }
}

namespace {

// スレッドプール
const int ThreadNum = 50;
const int TestParallelThreadStackSize = 16384;               // スタックサイズ
nn::os::ThreadType    g_ParallelThreadPool[ThreadNum];
NN_ALIGNAS(4096) char g_ParallelThreadStack[ThreadNum][TestParallelThreadStackSize];

/**
 * @brief   マルチスレッド実行する関数です。
 */
void ParallelFunc(void* arg)
{
    nn::mem::StandardAllocator* heap = reinterpret_cast<nn::mem::StandardAllocator*>(arg);
    const int NUM = 1000;
    double* p[NUM];  // スタックを 4KB 使うのでスタックサイズが小さいと死ぬ
    for (int i = 0; i < NUM; ++i)
    {
        p[i] = reinterpret_cast<double*>(heap->Allocate(8));
        ASSERT_TRUE(p[i] != NULL);
        *p[i] = 1.0;  // 偽共有を起こそうとする。
    }
    for (int i = 0; i < NUM; ++i)
    {
        heap->Free(p[i]);
    }
    heap->ClearThreadCache();
}

}   // unnamed namespace

/**
 * @brief   複数スレッドを動作させた状態でメモリ確保を行います。
 */
TEST_P(StandardAllocatorTest, Parallel)
{
    int repeatCount = ThreadNum;

    for (int i = 0; i < repeatCount; ++i)
    {
        nn::Result result = nn::os::CreateThread(&g_ParallelThreadPool[i], ParallelFunc, reinterpret_cast<void*>(&m_Allocator), g_ParallelThreadStack[i], TestParallelThreadStackSize, nn::os::GetThreadPriority(nn::os::GetCurrentThread()));
        ASSERT_TRUE(result.IsSuccess());
        nn::os::StartThread(&g_ParallelThreadPool[i]);
    }
    ParallelFunc(reinterpret_cast<void*>(&m_Allocator));
    for (int i = 0; i < repeatCount; ++i)
    {
        nn::os::WaitThread(&g_ParallelThreadPool[i]);
        nn::os::DestroyThread(&g_ParallelThreadPool[i]);
    }
}

/**
 * @brief   Dump 出力を確認します。
 */
TEST_P(StandardAllocatorTest, Dump)
{
    void* p = m_Allocator.Allocate(8);
    m_Allocator.ClearThreadCache();
    m_Allocator.Dump();
    m_Allocator.Free(p);
    m_Allocator.ClearThreadCache();
    m_Allocator.Dump();
}

/**
 * @brief   正しくハッシュが生成されているか確認します。
 */
TEST_P(StandardAllocatorTest, leakchecker)
{
    nn::mem::StandardAllocator::AllocatorHash hash, hash0, hash1, tmp;

    // AllocatorHash が一致しているか確認
    auto EqCheck = [](nn::mem::StandardAllocator::AllocatorHash h1, nn::mem::StandardAllocator::AllocatorHash h2) -> bool
    {
        if (h1.allocCount == h2.allocCount && h1.allocSize == h2.allocSize && h1.hash == h2.hash)
        {
            return true;
        }
        return false;
    };

    hash = m_Allocator.Hash();
    hash0 = m_Allocator.Hash();
    ASSERT_TRUE(EqCheck(hash, hash0));

    void* p0 = m_Allocator.Allocate(8);
    m_Allocator.ClearThreadCache();
    hash0 = m_Allocator.Hash();
    ASSERT_FALSE(EqCheck(hash, hash0));

    void* p1 = m_Allocator.Allocate(8);
    m_Allocator.ClearThreadCache();
    hash1 = m_Allocator.Hash();
    ASSERT_FALSE(EqCheck(hash0, hash1));
    ASSERT_FALSE(EqCheck(hash, hash1));

    m_Allocator.Free(p1);
    m_Allocator.ClearThreadCache();
    tmp = m_Allocator.Hash();
    ASSERT_TRUE(EqCheck(tmp, hash0));

    m_Allocator.Free(p0);
    m_Allocator.ClearThreadCache();
    tmp = m_Allocator.Hash();
    ASSERT_TRUE(EqCheck(tmp, hash));
}

/**
 * @brief   小さな領域確保をたくさん確保→解放を何度も繰り返します。
 */
TEST_P(StandardAllocatorTest, SmallAllocate)
{
    const int n = 1000;
    void* p[n];
    for (int j = 0; j < 1000; ++j)
    {
        for (int i = 0; i < n; ++i)
        {
            p[i] = m_Allocator.Allocate(8);
        }
        for (int i = 0; i < n; ++i)
        {
            m_Allocator.Free(p[i]);
        }
    }
}

/**
 * @brief   SmallAllocate に比べて大きな領域をたくさん確保→解放を何度も繰り返します。
 */
TEST_P(StandardAllocatorTest, LargeAllocate)
{
    const int n = 100;
    void* p[n];
    for (int j = 0; j < 1000; ++j)
    {
        for (int i = 0; i < n; ++i)
        {
            p[i] = m_Allocator.Allocate(65536);
        }
        for (int i = 0; i < n; ++i)
        {
            m_Allocator.Free(p[i]);
        }
    }
}

/**
 * @brief   確保した領域より大きな領域を何度も再確保するのを繰り返します。
 */
TEST_P(StandardAllocatorTest, ReallocateSmallGain)
{
    for (int j = 0; j < 100; ++j)
    {
        const int n = 1000;
        uint8_t* p;
        p = reinterpret_cast<uint8_t*>(m_Allocator.Allocate(1));
        if (!p)
        {
            ASSERT_NOT_NULL(p);
            return;
        }
        p[0] = 0;
        for (int i = 1; i < n; ++i)
        {
            p = reinterpret_cast<uint8_t*>(m_Allocator.Reallocate(p, i));
            if (!p)
            {
                ASSERT_NOT_NULL(p);
                return;
            }
            p[i - 1] = static_cast<uint8_t>((i - 1) % 256);     // uint8_t に収まるサイズを代入
            for (int k = 0; k < i; ++k)
            {
                // Reallocate 前に書き込まれた値もちゃんとコピーされているか確認
                ASSERT_TRUE(p[k] == (k % 256));
            }
        }
        m_Allocator.Free(p);
    }
}

/**
 * @brief   確保した領域より小さな領域を何度も再確保するのを繰り返します。
 */
TEST_P(StandardAllocatorTest, ReallocateSmallReduce)
{
    for (int j = 0; j < 100; ++j)
    {
        const int n = 1000;
        void* p;
        p = m_Allocator.Allocate(n);
        ASSERT_NOT_NULL(p);
        for (int i = n; i > 0; --i)
        {
            p = m_Allocator.Reallocate(p, i);
            ASSERT_NOT_NULL(p);
        }
        m_Allocator.Free(p);
    }
}

/**
 * @brief   ReallocateSmallGain より大きなサイズまで領域を再確保します。
 */
TEST_P(StandardAllocatorTest, ReallocateGain)
{
    const int n = 10000;
    uint8_t* p;
    p = reinterpret_cast<uint8_t*>(m_Allocator.Allocate(1));
    if (!p)
    {
        ASSERT_TRUE(p != 0);
        return;
    }
    p[0] = 0;
    for (int i = 1; i < n; ++i)
    {
        p = reinterpret_cast<uint8_t*>(m_Allocator.Reallocate(p, i));
        if (!p)
        {
            ASSERT_TRUE(p != 0);
            return;
        }
        p[i - 1] = static_cast<uint8_t>((i - 1) % 256);
        for (int k = 0; k < i; ++k)
        {
            ASSERT_TRUE(p[k] == (k % 256));
        }
    }
    m_Allocator.Free(p);
}

/**
 * @brief   ReallocateSmallReduce より大きなサイズから領域の再確保を行います。
 */
TEST_P(StandardAllocatorTest, ReallocateReduce)
{
    const int n = 100000;
    void* p;
    p = m_Allocator.Allocate(n);
    for (int i = n; i > 0; --i)
    {
        p = m_Allocator.Reallocate(p, i);
    }
    m_Allocator.Free(p);
}

/**
 * @brief   指定のアライメントでメモリ確保を行います。
 */
TEST_P(StandardAllocatorTest, Alignment)
{
    const int Seed = 12345;
    const int MaxRandomAllocateSize = 1024;
    const int AllocateCount = 1000;
    const int AlignCount = 16;
    const int MaxAlign = 131072;    // 確認する最大アライメント(128K)
    int pAlignment[AlignCount] = {4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 8192, 16384, 32768, 65536, 131072};

    std::mt19937 engine(Seed);
    std::uniform_int_distribution<uint32_t> distributionForAllocate(1, MaxRandomAllocateSize - 1);
    std::uniform_int_distribution<uint32_t> distributionForAlign(0, AlignCount - 1);

    NN_LOG("Seed : %d\n", Seed);

    // ランダムなサイズ、同じアライメントで繰り返し確保
    void* pAddr[AllocateCount];
    for(int alignment = 4; alignment <= MaxAlign; alignment *= 2)
    {
        for(int j = 0; j < AllocateCount; ++j)
        {
            size_t allocateSize = distributionForAllocate(engine);
            pAddr[j] = m_Allocator.Allocate(allocateSize, alignment);
            ASSERT_NOT_NULL(pAddr[j]) << "pAddr[" << j << "]: " << pAddr[j];
            ASSERT_EQ( (reinterpret_cast<uintptr_t>(pAddr[j]) & (alignment - 1)), static_cast<uintptr_t>(0) ) << "size: " << allocateSize << " alignment: " << alignment << "GetsizeOf: " << m_Allocator.GetSizeOf(pAddr[j]) << " pAddr[j]: 0x" << pAddr[j];
        }
        for(int j = 0; j < AllocateCount; ++j)
        {
            m_Allocator.Free(pAddr[j]);
        }
    }

    // ランダムなサイズ、ランダムなアライメントで繰り返し確保
    int count = 0;
    const int SetNum = 3;
    for(int i = 0; i < SetNum; ++i)
    {
        // アライメントの値をシャッフル
        for(int j = 0; j < AlignCount - 1; ++j)
        {
            int swapNum = distributionForAlign(engine);
            std::swap(pAlignment[j], pAlignment[swapNum]);
        }

        for(int j = 0; j < AlignCount; ++j)
        {
            size_t allocateSize = distributionForAllocate(engine);
            pAddr[count] = m_Allocator.Allocate(allocateSize, pAlignment[j]);
            ASSERT_NOT_NULL(pAddr[count]);
            ASSERT_EQ( (reinterpret_cast<uintptr_t>(pAddr[count]) & (pAlignment[j] - 1)), static_cast<uintptr_t>(0) ) << "size: " << allocateSize << " alignment: " << pAlignment[j];
            count++;
        }
    }

    for(int i = 0; i < count; ++i)
    {
        m_Allocator.Free(pAddr[i]);
    }
}

/**
 * @brief   Walk のテスト用コールバック関数です。
 * @details addr から size 分適当な値を書きます。
 */
int WalkAndWriteAllBlocks( void *addr, size_t size, void* userPtr)
{
    char* str = reinterpret_cast<char*>(userPtr);
    // 指定した文字列を受け取れているか
    EXPECT_STREQ("Standard Allocator", str);

    char* ptr = reinterpret_cast<char*>(addr);
    for(int i = 0; i < static_cast<int>(size); ++i)
    {
        ptr[i] = static_cast<char>(i);
    }
    return 1;   // 継続してメモリブロック探索する場合は 1 を指定する
}

/**
 * @brief   Walk のテスト用コールバック関数です。
 * @details addr から size 分 WalkAndWriteAllBlocks() で書かれた値が入っているか確認します。
 */
int WalkAndReadAllBlocks( void *addr, size_t size, void* userPtr)
{
    char* str = reinterpret_cast<char*>(userPtr);
    // 指定した文字列を受け取れているか
    EXPECT_STREQ("Standard Allocator", str);

    char* ptr = reinterpret_cast<char*>(addr);
    for(int i = 0; i < static_cast<int>(size); ++i)
    {
        EXPECT_EQ(ptr[i], static_cast<char>(i));
    }
    return 1;   // 継続してメモリブロック探索する場合は 1 を指定する
}

/**
 * @brief   Walk のテスト用コールバック関数です。
 * @details addr から size 分 0xFF を書きます。
 */
int WalkAndWriteSomeBlocks( void *addr, size_t size, void* userPtr)
{
    int* count = reinterpret_cast<int*>(userPtr);
    char* ptr = reinterpret_cast<char*>(addr);

    for(int i = 0; i < static_cast<int>(size); ++i)
    {
        ptr[i] = std::numeric_limits<char>::max();
    }

    *count += 1;

    if(*count < 20)
    {
        // 確保されたメモリのうち 20 個は値を書き込む
        return 1;
    }
    else
    {
        return 0;
    }
}

/**
 * @brief   Walk のテスト用コールバック関数です。
 * @details addr から size 分 WalkAndWriteSomeBlocks() で書かれた値が入っているか確認します。
 */
int WalkAndReadSomeBlocks( void *addr, size_t size, void* userPtr)
{
    int* count = reinterpret_cast<int*>(userPtr);
    char* ptr = reinterpret_cast<char*>(addr);
    bool isAllBitsAsserted = true;

    for(int i = 0; i < static_cast<int>(size); ++i)
    {
        if(ptr[i] != std::numeric_limits<char>::max())
        {
            isAllBitsAsserted = false;
        }
    }

    if(*count < 20)
    {
        EXPECT_TRUE(isAllBitsAsserted) << *count;
    }
    else
    {
        EXPECT_FALSE(isAllBitsAsserted) << *count;
    }

    *count += 1;

    return 1;
}

/**
 * @brief   確保されたメモリ領域に対して指定されたコールバック関数を呼び出します。
 * @details 全てのメモリブロックに対してコールバックを呼び出します。
 */
TEST_P(StandardAllocatorTest, WalkAllBlocks)
{
    const int AllocateCount = 1000;
    const size_t AllocateSize = 32;
    void* pAddr[AllocateCount];

    for(int i = 0; i < AllocateCount; ++i)
    {
        pAddr[i] = m_Allocator.Allocate(AllocateSize);
        ASSERT_NOT_NULL(pAddr[i]);
    }

    const char* Str = "Standard Allocator";
    // Write で書き込んだ値を Read で読む
    m_Allocator.WalkAllocatedBlocks(WalkAndWriteAllBlocks, reinterpret_cast<void*>(const_cast<char*>(Str)));
    m_Allocator.WalkAllocatedBlocks(WalkAndReadAllBlocks, reinterpret_cast<void*>(const_cast<char*>(Str)));

    for(int i = 0; i < AllocateCount; ++i)
    {
        m_Allocator.Free(pAddr[i]);
    }
}

/**
 * @brief   確保されたメモリ領域に対して指定されたコールバック関数を呼び出します。
 * @details 実行を途中で中断します。
 */
TEST_P(StandardAllocatorTest, WalkAndCancel)
{
    const int AllocateCount = 1000;
    const size_t AllocateSize = 32;
    void* pAddr[AllocateCount];

    for(int i = 0; i < AllocateCount; ++i)
    {
        pAddr[i] = m_Allocator.Allocate(AllocateSize);
        ASSERT_NOT_NULL(pAddr[i]);
    }

    int count = 0;
    // Write で書き込んだ値を Read で読む
    m_Allocator.WalkAllocatedBlocks(WalkAndWriteSomeBlocks, reinterpret_cast<void*>(&count));
    count = 0;
    m_Allocator.WalkAllocatedBlocks(WalkAndReadSomeBlocks, reinterpret_cast<void*>(&count));
    EXPECT_EQ(count, AllocateCount) << count;

    for(int i = 0; i < AllocateCount; ++i)
    {
        m_Allocator.Free(pAddr[i]);
    }
}

/**
 * @brief   正しいハッシュ値が生成されるか確認します。
 */
TEST_P(StandardAllocatorTest, Hash)
{
    const int AllocateCount = 100;
    const size_t AllocateSize = 32;
    void* pAddr[AllocateCount];
    nn::mem::StandardAllocator::AllocatorHash hash[AllocateCount * 2];

    pAddr[0] = m_Allocator.Allocate(AllocateSize);
    ASSERT_NOT_NULL(pAddr[0]);

    hash[0] = m_Allocator.Hash();
    hash[1] = m_Allocator.Hash();

    EXPECT_EQ(hash[0].allocCount, 1U);
    EXPECT_LE(AllocateSize, hash[0].allocSize);
    EXPECT_EQ(hash[0].allocCount, hash[0].allocCount);
    EXPECT_EQ(hash[1].allocSize, hash[0].allocSize);
    // 同じメモリ状態からは同じハッシュ値が作られる
    EXPECT_EQ(hash[0].hash, hash[1].hash);

    // 確保数を増やすと、ハッシュ値も変わる
    pAddr[1] = m_Allocator.Allocate(AllocateSize);
    ASSERT_NOT_NULL(pAddr[1]);
    hash[1] = m_Allocator.Hash();
    EXPECT_EQ(hash[1].allocCount, 2U);
    EXPECT_LE(AllocateSize * 2, hash[1].allocSize);
    EXPECT_NE(hash[0].hash, hash[1].hash);

    // 最初に確保した方を解放すると、確保数と確保サイズも元に戻る
    m_Allocator.Free(pAddr[0]);
    hash[1] = m_Allocator.Hash();
    EXPECT_EQ(hash[0].allocCount, 1U);
    EXPECT_LE(AllocateSize, hash[0].allocSize);
    // 確保された数とサイズが一緒でも確保場所が違うなら、ハッシュ値も変わる
    EXPECT_NE(hash[0].hash, hash[1].hash);
    m_Allocator.Free(pAddr[1]);

    // 確保数をひとつずつ増やしていき、想定通りの AllocatorHash が生成されるか確認
    for(int i = 0; i < AllocateCount; ++i)
    {
        pAddr[i] = m_Allocator.Allocate(AllocateSize);
        hash[i] = m_Allocator.Hash();
        EXPECT_EQ(hash[i].allocCount, static_cast<size_t>(i + 1));
        EXPECT_LE(AllocateSize * (i + 1), hash[i].allocSize);
        for(int j = 0; j < i; ++j)
        {
            EXPECT_NE(hash[i].hash, hash[j].hash) << "i: " \
                                                  << i \
                                                  << " j: " \
                                                  << j \
                                                  << " allocCount1: " \
                                                  << hash[i].allocCount \
                                                  << " allocSize1: " \
                                                  << hash[i].allocSize \
                                                  << " allocCount2: " \
                                                  << hash[j].allocCount \
                                                  << " allocSize2: " \
                                                  << hash[j].allocSize;
        }
    }
    // 確保したメモリをひとつずつ解放していき、想定通りの AllocatorHash が生成されるか確認
    for(int i = 0; i < AllocateCount; ++i)
    {
        m_Allocator.Free(pAddr[i]);
        hash[AllocateCount + i] = m_Allocator.Hash();
        EXPECT_EQ(hash[AllocateCount + i].allocCount, static_cast<size_t>(AllocateCount - (i + 1)));
        EXPECT_LE(AllocateSize * (AllocateCount - (i + 1)), hash[AllocateCount + i].allocSize);
        for(int j = 0; j < AllocateCount + i; ++j)
        {
            EXPECT_NE(hash[AllocateCount + i].hash, hash[j].hash) << "i: " \
                                                                  << i \
                                                                  << " j: " \
                                                                  << j \
                                                                  << " allocCount1: " \
                                                                  << hash[AllocateCount + i].allocCount \
                                                                  << " allocSize1: " \
                                                                  << hash[AllocateCount + i].allocSize \
                                                                  << " allocCount2: " \
                                                                  << hash[j].allocCount \
                                                                  << " allocSize2: " \
                                                                  << hash[j].allocSize;
        }
    }
}

/**
 * @brief   確保できない巨大な領域を指定した際に NULL が返るか確認します。
 */
TEST_P(StandardAllocatorTest, HugeSize)
{
    ASSERT_TRUE(NULL == m_Allocator.Allocate(static_cast<size_t>(-1)));
    void* ptr = m_Allocator.Allocate(8);
    ASSERT_TRUE(NULL == m_Allocator.Reallocate(ptr, static_cast<size_t>(-1)));
    m_Allocator.Free(ptr);
}

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
/**
 * @brief   確保を繰り返した際にサイズ取得関数で正常な値が取得できるか確認します。
 */
TEST_P(StandardAllocatorTest, GetSizeAllocate)
{
    // 1 度に確保可能な大きなサイズをアロケート
    void* ptr = m_Allocator.Allocate(m_Allocator.GetAllocatableSize());
    EXPECT_TRUE(ptr != NULL);

    // 実際の物理メモリの残りは 4MB(=TlsHeapStatic::kPhyPageSize) 以下になる
    nn::os::MemoryInfo info;
    nn::os::QueryMemoryInfo(&info);
    size_t phyFreeSize = static_cast<size_t>(info.totalAvailableMemorySize - info.totalUsedMemorySize);
    if (m_VirtualSize > info.totalAvailableMemorySize)
    {
        EXPECT_LE(phyFreeSize, 4 * 1024 * 1024);
    }
}

/**
 * @brief   確保・解放でサイズ取得関数で得られるサイズが正しく変化することを確認します。
 */
TEST_P(StandardAllocatorTest, GetSizeAllocateFree)
{
    const size_t allocCount = 100;
    void* ptr[allocCount];
    size_t prevGetAllocatableSize = m_Allocator.GetAllocatableSize();
    size_t prevFreeSize = m_Allocator.GetTotalFreeSize();

    for(int i = 0; i < allocCount; ++i)
    {
        ptr[i] = m_Allocator.Allocate(4 * 1024 * 1024);
        size_t allocatableSize = m_Allocator.GetAllocatableSize();
        size_t freeSize = m_Allocator.GetTotalFreeSize();
        EXPECT_GT(prevFreeSize, freeSize);
        EXPECT_GT(prevGetAllocatableSize, allocatableSize);
        prevGetAllocatableSize = allocatableSize;
        prevFreeSize = freeSize;
    }
    for(int i = allocCount - 1; i >= 0; --i)
    {
        m_Allocator.Free(ptr[i]);
        size_t allocatableSize = m_Allocator.GetAllocatableSize();
        size_t freeSize = m_Allocator.GetTotalFreeSize();
        EXPECT_LT(prevFreeSize, freeSize);
        EXPECT_LE(prevGetAllocatableSize, allocatableSize);
        prevGetAllocatableSize = allocatableSize;
        prevFreeSize = freeSize;
    }
}
#endif  // defined(NN_BUILD_CONFIG_OS_HORIZON)

// =============
// DEATH テスト
// =============

// ASSERT で落ちるテストは、 Debug ビルド、 Develop ビルドでのみ有効
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

void DoTestInitialize(void* ptr, size_t size)
{
    nn::mem::StandardAllocator allocator;

    allocator.Initialize(ptr, size);    // 不正なサイズであればこの内部の ASSERT で落ちる
    allocator.Finalize();

    // 正常終了
    exit(0);
}

// 不正アドレスでの GetSizeOf をします
void DoTestGetSizeOf(nn::mem::StandardAllocator* pAllocator)
{
    pAllocator->GetSizeOf(NULL);
    exit(0);
}

/**
 * @brief   ヒープが指定より小さいサイズで作成されようとした場合は Abort します
 */
TEST(StandardAllocatorDeathTest, Initialize)
{
    // アロケータに最小限必要なサイズ（管理領域のサイズ）
    const size_t MinimumAllocatorSize = 16 * 1024;

    for(size_t size = 2; size <= 1 * 1024 * 1024; size *= 2)
    {
        if(size < MinimumAllocatorSize)
        {
            EXPECT_DEATH_IF_SUPPORTED(DoTestInitialize(nullptr, size), "");
        }
        else
        {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            EXPECT_EXIT(DoTestInitialize(nullptr, size), testing::ExitedWithCode(0), "");
#endif
        }
    }
}

/**
 * @brief   不正アドレスのサイズを取得しようとした場合は debug break します。
 */
TEST(StandardAllocatorDeathTest, GetSizeOfInvalidAddress)
{
    const size_t getSizeOfTestHeapSize = 32 * 1024;
    nn::mem::StandardAllocator allocator;
    allocator.Initialize(nullptr, getSizeOfTestHeapSize);

    EXPECT_DEATH_IF_SUPPORTED(DoTestGetSizeOf(&allocator), "");

    allocator.Finalize();
}

#endif  // defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

INSTANTIATE_TEST_CASE_P(SwitchThreadCache,
                        StandardAllocatorTest,
                        testing::Values(std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 4),
                                        std::make_pair(StandardAllocatorTestParam_DisableThreadCache, 60),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 4),
                                        std::make_pair(StandardAllocatorTestParam_EnableThreadCache, 60)));
