﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <random>
#include <nnt/nntest.h>

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/mem.h>

#define ASSERT_NULL(p) ASSERT_TRUE(p == nullptr)
#define ASSERT_NOT_NULL(p) ASSERT_TRUE(p != nullptr)

/**
 * @brief   管理領域(SpanPage)増加時に特定の確保パターンで問題ないか試します。
 * @details SpanPage が新たに Alloc されるときに残りの空き領域の NumPages が 256 だった場合
 */
TEST(StandardAllocatorCornerCaseTest, AllocSpanPage)
{
    size_t heapSize = 64 * 1024 * 1024;
    nn::mem::StandardAllocator allocator;

    void* heapStartAddr = std::malloc(heapSize + 4096);
    uintptr_t alignedStart = reinterpret_cast<uintptr_t>(heapStartAddr);
    alignedStart = (alignedStart + 4096 - 1) & ~(4096 - 1);
    void* alignedPtr = reinterpret_cast<void*>(alignedStart);

    allocator.Initialize(alignedPtr, heapSize);

    //void* ptr = allocator.Allocate(allocator.GetAllocatableSize());
    //allocator.Free(ptr);

    size_t spanSize = 120;  // Span 構造体のサイズ(64bit)
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    spanSize = 96;      // Span 構造体のサイズ(32bit)
#endif
    int runOut = static_cast<int>(4096 / spanSize);
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    runOut = static_cast<int>(4096 / spanSize) + 1;
#endif

    // 257 ページ の Span を一つ残し、残りの領域は全て Alloc()
    // SpanPage も残り 1 つまで使い切る
    int spanPageForSys = 2;         // システム側で初期化時に消費されている SpanPage の数
    for (int i = 0; i < runOut - spanPageForSys; ++i)
    {
        size_t allocSize = 4096;
        if (i == runOut - spanPageForSys - 1)
        {
            // 最後の 1 Alloc で帳尻を合わせる
            allocSize = allocator.GetAllocatableSize() - 257 * 4096;
        }
        ASSERT_NOT_NULL(allocator.Allocate(allocSize));
    }

    // ここで Allocate すると以下の状況になっている
    //   SpanPage が切れており、新たに SpanPage を Alloc() しなければならない
    //   freelist の先頭の num_pages は 256 になっており、 SpanPage 用の Span をとると
    //   余りは num_pages = 255 になる
    //   この num_pages = 255 の Span は freelist に返却される
    ASSERT_NOT_NULL(allocator.Allocate(4096));

    allocator.Finalize();
    std::free(heapStartAddr);
}

/**
 * @brief   Large Memory の空き領域発見失敗時の挙動を確認します。
 */
TEST(StandardAllocatorCornerCaseTest, SearchFreeSpan)
{
    size_t heapSize = 16 * 1024 * 1024;
    nn::mem::StandardAllocator allocator;

    void* heapStartAddr = std::malloc(heapSize + 4096);
    uintptr_t alignedStart = reinterpret_cast<uintptr_t>(heapStartAddr);
    alignedStart = (alignedStart + 4096 - 1) & ~(4096 - 1);
    void* alignedPtr = reinterpret_cast<void*>(alignedStart);

    allocator.Initialize(alignedPtr, heapSize);

    size_t spanSize = 120;  // Span 構造体のサイズ(64bit)
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    spanSize = 96;      // Span 構造体のサイズ(32bit)
#endif
    int pageSpanNum = static_cast<int>(4096 / spanSize);    // 1 ページに入れられる Span 構造体の数
    int spanPageForSys = 2;         // システム側で初期化時に消費されている SpanPage の数

    for (int i = 0; i < pageSpanNum - spanPageForSys; ++i)
    {
        allocator.Allocate(512 * 1024);
    }

    allocator.Finalize();
    std::free(heapStartAddr);
}

/**
 * @brief   メモリ枯渇時に異なるチャンクサイズのスパンからメモリを確保し安易な NULL 返しを回避していることを確認します。
 */
TEST(StandardAllocatorCornerCaseTest, AllocAnotherChunk)
{
    size_t heapSize = 28 * 1024;    // ヒープサイズが明らかに小さい場合に起こりやすい

    nn::mem::StandardAllocator allocator;

    void* heapStartAddr = std::malloc(heapSize + 4096);
    uintptr_t alignedStart = reinterpret_cast<uintptr_t>(heapStartAddr);
    alignedStart = (alignedStart + 4096 - 1) & ~(4096 - 1);
    void* alignedPtr = reinterpret_cast<void*>(alignedStart);

    allocator.Initialize(alignedPtr, heapSize);

    // 128Bytes
    // cls = 16(Windows)/8(Horizon)
    // numPages = 1(Windows)/1(Horizon)
    ASSERT_NOT_NULL(allocator.Allocate(128));
    // 48Bytes
    // cls = 6(Windows)/3(Horizon)
    // numPages = 3(Windows)/3(Horizon)
    void* ptr48 = allocator.Allocate(48);
    ASSERT_NOT_NULL(ptr48);

    // この時点で管理領域を含めすべてのページを使い切ってしまうため
    // 24Bytes 用チャンクのためのページを切り出せない

    size_t prevFreeSize = allocator.GetTotalFreeSize();

    // 24Bytes
    // cls = 3(Windows)/2(Horizon)
    // numPages = 3(Windows)/1(Horizon)
    void* ptr24 = allocator.Allocate(24);
    // 48Bytes 用のチャンクを借りて 24Bytes の確保成功となる
    ASSERT_NOT_NULL(ptr24);
    ASSERT_GT(reinterpret_cast<uintptr_t>(ptr24), reinterpret_cast<uintptr_t>(ptr48));
    ASSERT_LT(reinterpret_cast<uintptr_t>(ptr24), reinterpret_cast<uintptr_t>(ptr48) + 12 * 1024);
    ASSERT_EQ(prevFreeSize - 48, allocator.GetTotalFreeSize());

    allocator.Finalize();
    std::free(heapStartAddr);
}

/**
 * @brief   大きいサイズのメモリ確保をした後 Dump が正常に終了することを確認します。
 */
TEST(StandardAllocatorCornerCaseTest, DumpHugeMemory)
{
    const size_t heapSize = 32 * 1024 * 1024;
    void* pHeapMemory = std::malloc(heapSize);
    nn::mem::StandardAllocator allocator(pHeapMemory, heapSize);
    void * ptr = allocator.Allocate(4 * 1024 * 1024);
    ASSERT_NOT_NULL(ptr);
    allocator.Dump();
    allocator.Finalize();
    std::free(pHeapMemory);
}

/**
 * @brief   大きいサイズと小さいサイズのメモリ確保が混合した状態であっても Dump が正常に終了することを確認します。
 */
TEST(StandardAllocatorCornerCaseTest, DumpVariousMemory)
{
    const int seed = 12345;
    const size_t heapSize = 32 * 1024 * 1024;

    std::mt19937 engine(seed);
    std::uniform_int_distribution<int> distribution(1, 3);

    void* pHeapMemory = std::malloc(heapSize);
    nn::mem::StandardAllocator allocator(pHeapMemory, heapSize);

    // 適当な数の LargeMemory を確保
    for (int i = 0; i < 3; ++i)
    {
        void* ptr = allocator.Allocate(4 * 1024 * 1024);
        ASSERT_NOT_NULL(ptr);
    }

    // 適当な数の SmallMemory を確保
    for (int i = 0; i < 4096; ++i)
    {
        void* ptr;
        int num = distribution(engine);

        switch(num)
        {
        case 1:
            ptr = allocator.Allocate(8);
            break;
        case 2:
            ptr = allocator.Allocate(64);
            break;
        case 3:
            ptr = allocator.Allocate(512);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        ASSERT_NOT_NULL(ptr);
    }

    allocator.Dump();
    allocator.Finalize();
    std::free(pHeapMemory);
}
