﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/mbuf/mbuf_Mbuf.h>
#include <nn/mbuf/mbuf_MbufInit.h>
#include <nn/os/os_SystemEvent.h>

namespace
{
    nn::Bit8 g_MbufBuffer[8 * 1024 * 1024];
}

TEST(MultiplePools, SmallestOnePool)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMin, nn::mbuf::PoolCapacityMin, 0 };

    // 必要なバッファサイズを計算します。
    size_t requiredSize = nn::mbuf::GetRequiredBufferSize(&config, 1);
    ASSERT_LE(requiredSize, sizeof(g_MbufBuffer));

    // 最小の Unit Size で mbuf ライブラリを初期化できます。
    nn::mbuf::Initialize(&type, 1, &config, 1, g_MbufBuffer, requiredSize);

    // mbuf の確保を繰り返し実行できることを確認します。
    for (int i = 0; i < 256; ++i)
    {
        // 生成時に指定した数の mbuf クラスタを確保できます。
        nn::mbuf::Mbuf* mbufs[nn::mbuf::PoolCapacityMin];
        for (int j = 0; j < nn::mbuf::PoolCapacityMin; ++j)
        {
            mbufs[j] = nn::mbuf::MbufGetm(
                nullptr, nn::mbuf::MbufDataSizeMin, nn::mbuf::MbufAllocationMode_DontWait, 0);
            ASSERT_NE(nullptr, mbufs[j]);
            ASSERT_EQ(nullptr, mbufs[j]->_pNext);
            ASSERT_EQ(0, mbufs[j]->_pool);
        }

        // 空きがないため確保できません。
        nn::mbuf::Mbuf* pMbuf;
        pMbuf = nn::mbuf::MbufGetm(
            nullptr, nn::mbuf::MbufDataSizeMin, nn::mbuf::MbufAllocationMode_DontWait, 0);
        ASSERT_EQ(nullptr, pMbuf);

        // mbuf を全て解放します。
        for (int j = 0; j < nn::mbuf::PoolCapacityMin; ++j)
        {
            nn::mbuf::MbufFreem(mbufs[j]);
        }
    }
    nn::mbuf::Finalize();
}

TEST(MultiplePools, LargestOnePool)
{
    // mbuf プールの設定です。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };
    nn::mbuf::Config config = { nn::mbuf::MbufUnitSizeMax, nn::mbuf::PoolCapacityMin, 0 };

    // 必要なバッファサイズを計算します。
    size_t requiredSize = nn::mbuf::GetRequiredBufferSize(&config, 1);
    ASSERT_LE(requiredSize, sizeof(g_MbufBuffer));

    // 最大の Unit Size で mbuf ライブラリを初期化できます。
    nn::mbuf::Initialize(&type, 1, &config, 1, g_MbufBuffer, requiredSize);

    // mbuf の確保を繰り返し実行できることを確認します。
    for (int i = 0; i < 256; ++i)
    {
        // 生成時に指定した数の mbuf クラスタを確保できます。
        nn::mbuf::Mbuf* mbufs[nn::mbuf::PoolCapacityMin];
        for (int j = 0; j < nn::mbuf::PoolCapacityMin; ++j)
        {
            mbufs[j] = nn::mbuf::MbufGetm(
                nullptr, nn::mbuf::MbufDataSizeMax, nn::mbuf::MbufAllocationMode_DontWait, 0);
            ASSERT_NE(nullptr, mbufs[j]);
            ASSERT_EQ(nullptr, mbufs[j]->_pNext);
            ASSERT_EQ(0, mbufs[j]->_pool);
        }

        // 空きがないため確保できません。
        nn::mbuf::Mbuf* pMbuf;
        pMbuf = nn::mbuf::MbufGetm(
            nullptr, nn::mbuf::MbufDataSizeMax, nn::mbuf::MbufAllocationMode_DontWait, 0);
        ASSERT_EQ(nullptr, pMbuf);

        // mbuf を全て解放します。
        for (int j = 0; j < nn::mbuf::PoolCapacityMin; ++j)
        {
            nn::mbuf::MbufFreem(mbufs[j]);
        }
    }
    nn::mbuf::Finalize();
}

TEST(MultiplePools, AllTheSameTypes)
{
    // mbuf ライブラリが使用するイベントを初期化します。
    nn::os::SystemEvent systemEvent(nn::os::EventClearMode_AutoClear, false);
    nn::mbuf::Type type = { systemEvent.GetBase() };

    // 最後の mbuf プールだけユニットサイズが大きくなるように設定します。
    nn::mbuf::Config configs[nn::mbuf::PoolCountMax] = { };
    for (int i = 0; i < nn::mbuf::PoolCountMax; ++i)
    {
        configs[i].unitSize = nn::mbuf::MbufUnitSizeMin;
        configs[i].count = nn::mbuf::PoolCapacityMin;
        configs[i].type = 0;
    }
    configs[nn::mbuf::PoolCountMax - 1].unitSize = nn::mbuf::MbufUnitSizeMax;

    // 必要なバッファサイズを計算します。
    size_t requiredSize = nn::mbuf::GetRequiredBufferSize(configs, nn::mbuf::PoolCountMax);
    ASSERT_LE(requiredSize, sizeof(g_MbufBuffer));

    // mbuf ライブラリを初期化します。
    nn::mbuf::Initialize(
        &type, 1, configs, nn::mbuf::PoolCountMax, g_MbufBuffer, requiredSize);

    // mbuf の確保を繰り返し実行できることを確認します。
    for (int i = 0; i < 256; ++i)
    {
        // 末尾のプールから mbuf クラスタを確保します。
        nn::mbuf::Mbuf* mbufs[nn::mbuf::PoolCountMax][nn::mbuf::PoolCapacityMin];
        for (int k = 0; k < nn::mbuf::PoolCapacityMin; ++k)
        {
            int j = nn::mbuf::PoolCountMax - 1;
            mbufs[j][k] = nn::mbuf::MbufGetm(
                nullptr, nn::mbuf::MbufDataSizeMax, nn::mbuf::MbufAllocationMode_DontWait, 0);
            ASSERT_NE(nullptr, mbufs[j][k]);
            ASSERT_EQ(nullptr, mbufs[j][k]->_pNext);
            ASSERT_EQ(j, mbufs[j][k]->_pool);
        }

        // 前方のプールから mbuf クラスタを確保します。
        for (int j = 0; j < nn::mbuf::PoolCountMax - 1; ++j)
        {
            for (int k = 0; k < nn::mbuf::PoolCapacityMin; ++k)
            {
                mbufs[j][k] = nn::mbuf::MbufGetm(
                    nullptr, nn::mbuf::MbufDataSizeMin,
                    nn::mbuf::MbufAllocationMode_DontWait, 0);
                ASSERT_NE(nullptr, mbufs[j][k]);
                ASSERT_EQ(nullptr, mbufs[j][k]->_pNext);
                ASSERT_EQ(j, mbufs[j][k]->_pool);
            }
        }

        // 空きがないため確保できません。
        nn::mbuf::Mbuf* pMbuf;
        pMbuf = nn::mbuf::MbufGetm(
            nullptr, nn::mbuf::MbufDataSizeMin, nn::mbuf::MbufAllocationMode_DontWait, 0);
        ASSERT_EQ(nullptr, pMbuf);

        // mbuf を全て解放します。
        for (int j = 0; j < nn::mbuf::PoolCountMax; ++j)
        {
            for (int k = 0; k < nn::mbuf::PoolCapacityMin; ++k)
            {
                nn::mbuf::MbufFreem(mbufs[j][k]);
            }
        }
    }
    nn::mbuf::Finalize();
}

TEST(MultiplePools, AllDifferentTypes)
{
    // mbuf ライブラリが使用するイベントを初期化します。
    nn::os::SystemEventType eventTypes[nn::mbuf::TypeCountMax];
    nn::mbuf::Type types[nn::mbuf::TypeCountMax];
    for (int i = 0; i < nn::mbuf::TypeCountMax; ++i)
    {
        nn::os::CreateSystemEvent(&eventTypes[i], nn::os::EventClearMode_AutoClear, false);
        types[i].pEvent = &eventTypes[i];
    }

    // 全て同じサイズの mbuf プールを構築します。
    nn::mbuf::Config configs[nn::mbuf::TypeCountMax] = { };
    for (int i = 0; i < nn::mbuf::TypeCountMax; ++i)
    {
        configs[i].unitSize = nn::mbuf::MbufUnitSizeMin;
        configs[i].count = nn::mbuf::PoolCapacityMin;
        configs[i].type = i;
    }

    // 必要なバッファサイズを計算します。
    size_t requiredSize = nn::mbuf::GetRequiredBufferSize(configs, nn::mbuf::TypeCountMax);
    ASSERT_LE(requiredSize, sizeof(g_MbufBuffer));

    // mbuf ライブラリを初期化します。
    nn::mbuf::Initialize(
        types, nn::mbuf::TypeCountMax, configs, nn::mbuf::TypeCountMax,
        g_MbufBuffer, requiredSize);

    // mbuf の確保を繰り返し実行できることを確認します。
    for (int i = 0; i < 256; ++i)
    {
        // 適当な順番のプールから mbuf クラスタを確保します。
        nn::mbuf::Mbuf* mbufs[nn::mbuf::TypeCountMax][nn::mbuf::PoolCapacityMin];
        for (int j = 0; j < nn::mbuf::TypeCountMax; ++j)
        {
            int poolIndex = (i + j) % nn::mbuf::TypeCountMax;
            int type = poolIndex;
            for (int k = 0; k < nn::mbuf::PoolCapacityMin; ++k)
            {
                mbufs[poolIndex][k] = nn::mbuf::MbufGetm(
                    nullptr, nn::mbuf::MbufDataSizeMin,
                    nn::mbuf::MbufAllocationMode_DontWait, type);
                ASSERT_NE(nullptr, mbufs[poolIndex][k]);
                ASSERT_EQ(nullptr, mbufs[poolIndex][k]->_pNext);
                ASSERT_EQ(poolIndex, mbufs[poolIndex][k]->_pool);
            }
        }

        // 空きがないため確保できません。
        for (int j = 0; j < nn::mbuf::TypeCountMax; ++j)
        {
            nn::mbuf::Mbuf* pMbuf;
            int poolIndex = (i + j) % nn::mbuf::TypeCountMax;
            int type = poolIndex;
            pMbuf = nn::mbuf::MbufGetm(
                nullptr, nn::mbuf::MbufDataSizeMin, nn::mbuf::MbufAllocationMode_DontWait, type);
            ASSERT_EQ(nullptr, pMbuf);
        }

        // mbuf を全て解放します。
        for (int j = 0; j < nn::mbuf::TypeCountMax; ++j)
        {
            for (int k = 0; k < nn::mbuf::PoolCapacityMin; ++k)
            {
                nn::mbuf::MbufFreem(mbufs[j][k]);
            }
        }
    }
    nn::mbuf::Finalize();

    // SystemEvent を解放します。
    for (int i = 0; i < nn::mbuf::TypeCountMax; ++i)
    {
        nn::os::DestroySystemEvent(&eventTypes[i]);
    }
}
