﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/os.h>
#include <nn/os/os_MemoryAttribute.h>

#include "MemoryTest.h"
#include "RandomList.h"
#include "Util.h"

namespace {
    const int VeryBigNumber = 1024 * 1024 * 1024;

    // メモリテストを実行する CPU コアの番号
    const int CoreNumber = 1; // コア 0 は描画を行うので、コア 1 を使う

    // テスト対象のメモリ領域サイズ
    const uint64_t HeapByteSize = 2048ull * 1024ull * 1024ull;

    // Write -> Verify のチェックを行うブロックのサイズ
    const uint32_t BlockBitWidth = 21;
    const uint32_t BlockByteSize = (1 << BlockBitWidth) * sizeof(uint32_t);

    // テストで用いるパターン
    const uint32_t PatternList[] = {
        0x00000000,
        0xa5a5a5a5,
        0x0f0f0f0f,
        0x69696969,
    };

    // テスト回数
    const int PatternTestCount = 4;
    const int RandomTestCount = 16;

    // エラー発生時に表示する前後のワード数（目安）
    const int ErrorPrintSize = 32;

    // スレッド
    uint8_t g_ThreadStack[4096] NN_ALIGNAS(4096);
    nn::os::ThreadType g_Thread;

    // 0 - 2^N をランダムな順番で保持するリスト
    RandomList g_RandomList;

    void ErrorLoop(volatile uint32_t* pErrorData)
    {
        // HW 的に調査がしやすいように、エラーが起きたアドレスをリードし続ける
        NN_LOG("Read virtual address %p repeatedly\n", pErrorData);

        size_t pageSize = nn::os::MemoryPageSize;
        uintptr_t pageAddress = (reinterpret_cast<uintptr_t>(pErrorData) / pageSize) * pageSize;
        nn::os::SetMemoryAttribute(pageAddress, pageSize, nn::os::MemoryAttribute_Uncached);

        uint32_t sum = 0;
        while (true)
        {
            for (int i = 0; i < VeryBigNumber; i++)
            {
                for (int j = 0; j < VeryBigNumber; j++)
                {
                    for (int k = 0; k < VeryBigNumber; k++)
                    {
                        // ダミーの適当な計算
                        sum += *pErrorData;
                    }
                }
            }
            NN_LOG("Dummy Print (sum: %d)\n", sum);
        }
    }

    void Print(volatile uint32_t* buffer, uint32_t minIndex, uint32_t maxIndex, uint32_t bufferSize)
    {
        // 表示するインデックスの先頭と末尾
        uint32_t start = std::max<uint32_t>(0, (minIndex / 4) * 4);
        uint32_t end = std::min<uint32_t>(bufferSize, (maxIndex / 4 + 1) * 4);

        for (int i = start; i < end; i += 4)
        {
            NN_LOG("Virtual %p: %08x %08x %08x %08x\n", &buffer[i], buffer[i], buffer[i + 1], buffer[i + 2], buffer[i + 3]);
        }
    }

    void FillRandom(volatile uint32_t* buffer, uint32_t value, uint32_t size)
    {
        NN_LOG("%s(buffer=0x%p, value=0x%08x, size=0x%08x).\n", __FUNCTION__, buffer, value, size);

        for (int i = 0; i < g_RandomList.GetSize(); i++)
        {
            uint32_t index = g_RandomList.GetValue(i);
            if (index >= size)
            {
                continue;
            }
            buffer[index] = value;
        }
    }

    void VerifySequential(volatile uint32_t* buffer, uint32_t value, uint32_t size)
    {
        NN_LOG("%s(buffer=0x%p, value=0x%08x, size=0x%08x).\n", __FUNCTION__, buffer, value, size);

        for (uint32_t i = 0; i < size; i++)
        {
            if (buffer[i] != value)
            {
                NN_LOG("Error addr=0x%p value=0x%08x.\n\n", &buffer[i], buffer[i]);
                Print(buffer, i - ErrorPrintSize / 2, i + ErrorPrintSize / 2, size);
                ErrorLoop(&buffer[i]);
            }
        }
    }

    void Check(volatile uint32_t* buffer, uint32_t value, uint32_t size, uint32_t blockBitWidth)
    {
        uint32_t blockSize = 1 << blockBitWidth;

        NN_ASSERT(size % blockSize == 0);
        g_RandomList.Init(blockBitWidth);

        for (size_t offset = 0; offset < size; offset += blockSize)
        {
            FillRandom(&buffer[offset], value, blockSize);
            VerifySequential(&buffer[offset], value, blockSize);
        }

        for (size_t offset = 0; offset < size; offset += blockSize)
        {
            FillRandom(&buffer[offset], ~value, blockSize);
            VerifySequential(&buffer[offset], ~value, blockSize);
        }
    }

    void MemoryTestThread(void*)
    {
        NN_LOG("MemoryTest: Heap = %ld byte, Block = %ld byte\n", HeapByteSize, BlockByteSize);

        srand(static_cast<uint32_t>(nn::os::GetSystemTick().GetInt64Value()));

        uint32_t* buffer = reinterpret_cast<uint32_t*>(malloc(HeapByteSize));
        if (buffer == nullptr)
        {
            NN_LOG("Allocate Failed.\n");
            return;
        }
        NN_LOG("Allocate %d MB\n", HeapByteSize / (1024 * 1024));

        const uint32_t size = HeapByteSize / sizeof(uint32_t);

        while (true)
        {
            for (int i = 0; i < PatternTestCount; i++)
            {
                for (int j = 0; j < sizeof(PatternList) / sizeof(PatternList[0]); j++)
                {
                    Check(buffer, PatternList[j], size, BlockBitWidth);
                }
            }

            for (int i = 0; i < RandomTestCount; i++)
            {
                Check(buffer, Rand32(), size, BlockBitWidth);
            }
        }
    }
}

void StartMemoryTestThread()
{
    nn::os::CreateThread(&g_Thread, MemoryTestThread, nullptr, g_ThreadStack, sizeof(g_ThreadStack), nn::os::DefaultThreadPriority, CoreNumber);
    nn::os::StartThread(&g_Thread);
}
