﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <string>
#include <random>
#include <chrono>
#include <set>
#include <tuple>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_BitUtil.h>

#include <nn/lr/lr_Result.h>
#include <nn/lr/lr_Service.h>
#include <nn/lr/lr_ContentLocationResolverImpl.h>

#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentStorageImpl.h>
#include <nn/ncm/ncm_IContentStorage.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentInfoData.h>

#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaDatabaseImpl.h>
#include <nn/ncm/ncm_IContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>

#include <nn/fs/fs_Bis.h>

#include <nn/sf/sf_StdAllocationPolicy.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <new>

namespace {
    typedef std::string String;
    typedef nn::sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> ContentStorageFactory;
    typedef nn::sf::SharedPointer<nn::ncm::IContentStorage> ContentStorageShared;
    typedef nn::sf::ObjectFactory<nn::sf::StdAllocationPolicy<std::allocator>> ContentMetaDatabaseFactory;
    typedef nn::sf::SharedPointer<nn::ncm::IContentMetaDatabase> ContentMetaDatabaseShared;

    struct ContentData
    {
        nn::ncm::ContentId      id;
        size_t                  size;
        std::vector<int64_t>    data;
    };

    struct ApplicationContentData
    {
        ContentData meta;
        ContentData program;
        ContentData control;
        ContentData htmlDocument;
        ContentData legalInformation;
    };

    class LocationResolverTest : public testing::Test
    {
    protected:
#if defined(NN_BUILD_CONFIG_OS_WIN)
        virtual void SetUp()
        {
        }

        virtual void TearDown()
        {
        }

        static void SetUpTestCase()
        {
            NN_ABORT_UNLESS(nn::fs::MountHostRoot().IsSuccess());
        }

        static void TearDownTestCase()
        {
            nn::fs::UnmountHostRoot();
        }

#endif

        nn::Result RegisterRandomContent(ContentData* outValue, nn::ncm::ContentStorage* storage)
        {
            auto placeHolderId = storage->GeneratePlaceHolderId();

            auto size = GetRandomSize();
            auto data = GetRandomData(size);
            auto contentId = GetRandomContentId();

            auto result = storage->Delete(contentId);
            if (result.IsFailure())
            {
                NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentNotFound, result);
            }

            NN_RESULT_DO(storage->CreatePlaceHolder(placeHolderId, contentId, size));
            NN_RESULT_DO(storage->WritePlaceHolder(placeHolderId, 0, data.data(), size));
            NN_RESULT_DO(storage->Register(placeHolderId, contentId));

            ContentData ret = { contentId, size, data };
            *outValue = ret;

            NN_RESULT_SUCCESS;
        }

        nn::ncm::PackagedContentInfo MakePackagedContentInfo(nn::ncm::ContentId contentId, size_t size, nn::ncm::ContentType contentType)
        {
            nn::ncm::PackagedContentInfo packagedContentInfo = {};
            packagedContentInfo.info = nn::ncm::ContentInfo::Make(contentId, size, contentType, 0);
            return packagedContentInfo;
        }

        std::unique_ptr<char[]> ConvertToContentMeta(
            size_t *outSize,
            const ContentData& metaContent,
            const void* packagedContentMeta,
            size_t packagedContentMetaSize)
        {
            nn::ncm::PackagedContentMetaReader pr(packagedContentMeta, packagedContentMetaSize);
            size_t contentMetaSize = pr.CalculateConvertContentMetaSize();
            std::unique_ptr<char[]> contentMeta(new char[contentMetaSize]);

            auto contentInfo = nn::ncm::ContentInfo::Make(metaContent.id, metaContent.size, nn::ncm::ContentType::Meta, 0);
            pr.ConvertToContentMeta(contentMeta.get(), contentMetaSize, contentInfo);

            *outSize = contentMetaSize;
            return contentMeta;
        }

        nn::Result InstallRandomContentApplication(ApplicationContentData* outValue, nn::ncm::ApplicationId id, uint16_t version, nn::ncm::ContentMetaDatabase* metaDatabase, nn::ncm::ContentStorage* storage)
        {
            ContentData meta;
            ContentData program;
            ContentData control;
            ContentData htmlDocument;
            ContentData legalInformation;

            NN_RESULT_DO(RegisterRandomContent(&meta, storage));
            NN_RESULT_DO(RegisterRandomContent(&program, storage));
            NN_RESULT_DO(RegisterRandomContent(&control, storage));
            NN_RESULT_DO(RegisterRandomContent(&htmlDocument, storage));
            NN_RESULT_DO(RegisterRandomContent(&legalInformation, storage));

            auto packagedMetaSize = nn::ncm::PackagedContentMetaWriter::CalculateSize(nn::ncm::ContentMetaType::Application, 4, 0, 0);
            std::unique_ptr<char[]> packagedContentMeta(new char[packagedMetaSize]);
            nn::ncm::PackagedContentMetaWriter writer(packagedContentMeta.get(), packagedMetaSize);
            writer.WriteHeader(id.value, version, nn::ncm::ContentMetaType::Application, nn::ncm::ContentMetaAttribute_None, 0, 4, 0);
            writer.WriteContentInfo(MakePackagedContentInfo(program.id, program.size, nn::ncm::ContentType::Program), 0);
            writer.WriteContentInfo(MakePackagedContentInfo(control.id, control.size, nn::ncm::ContentType::Control), 1);
            writer.WriteContentInfo(MakePackagedContentInfo(htmlDocument.id, htmlDocument.size, nn::ncm::ContentType::HtmlDocument), 2);
            writer.WriteContentInfo(MakePackagedContentInfo(legalInformation.id, legalInformation.size, nn::ncm::ContentType::LegalInformation), 3);

            size_t contentMetaSize;
            auto contentMeta = ConvertToContentMeta(&contentMetaSize, meta, packagedContentMeta.get(), packagedMetaSize);
            NN_RESULT_DO(metaDatabase->Set(nn::ncm::ContentMetaKey::Make(id, version), contentMeta.get(), contentMetaSize));

            ApplicationContentData appData = { meta, program, control, htmlDocument, legalInformation };
            *outValue = appData;

            NN_RESULT_SUCCESS;
        }

        nn::ncm::ContentId GetRandomContentId()
        {
            nn::ncm::ContentId contentId = {};
            auto low = m_RandomEngine();
            auto high = m_RandomEngine();
            std::memcpy(&contentId.data[0], &low, sizeof(low));
            std::memcpy(&contentId.data[8], &high, sizeof(high));

            return contentId;
        }

        size_t GetRandomSize()
        {
            return std::uniform_int_distribution<>(1, 1024 * 1024)(m_RandomEngine);
        }

        void ExpectContent(const nn::lr::Path& path, const ContentData& contentData)
        {
            NN_LOG("ExpectContent path %s\n", path.string);
            NN_LOG("ExpectContent contentData.size %lld\n", contentData.size);

            nn::fs::FileHandle fileHandle;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path.string, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(fileHandle); };

            int64_t fileSize64;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize64, fileHandle));
            size_t fileSize = static_cast<size_t>(fileSize64);
            EXPECT_EQ(contentData.size, fileSize);

            std::unique_ptr<char[]> buffer(new char[fileSize]);
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, buffer.get(), fileSize));
            EXPECT_TRUE(memcmp(contentData.data.data(), buffer.get(), contentData.size) == 0);
        }

        std::vector<int64_t> GetRandomData(size_t size)
        {
            size_t count = nn::util::align_up(size, sizeof(int64_t)) / sizeof(int64_t);
            std::vector<int64_t> data;
            data.reserve(count);
            for (size_t i = 0; i < count; i++)
            {
                data.push_back(m_RandomEngine());
            }

            return data;
        }

        nn::ncm::ApplicationId GetRandomApplicationId()
        {
            nn::ncm::ApplicationId id = { m_RandomEngine() };
            return id;
        }

        void TestResolveRedirectContentLocationResolver(nn::ncm::StorageId storageId)
        {
            nn::ncm::ContentMetaDatabase metaDb;
            nn::ncm::ContentStorage storage;
            NNT_EXPECT_RESULT_SUCCESS(nn::ncm::OpenContentMetaDatabase(&metaDb, storageId));
            NNT_EXPECT_RESULT_SUCCESS(nn::ncm::OpenContentStorage(&storage, storageId));

            nn::lr::LocationResolver locationResolver;
            NNT_EXPECT_RESULT_SUCCESS(nn::lr::OpenLocationResolver(&locationResolver, storageId));

            auto appId = GetRandomApplicationId();
            nn::lr::Path path;

            auto result = metaDb.Remove(appId, 0);
            if (result.IsFailure())
            {
                NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, result);
            }
            result = metaDb.Remove(appId, 1);
            if (result.IsFailure())
            {
                NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, result);
            }
            result = metaDb.Remove(appId, 2);
            if (result.IsFailure())
            {
                NNT_EXPECT_RESULT_FAILURE(nn::ncm::ResultContentMetaNotFound, result);
            }

            ApplicationContentData contentDatav1;
            NNT_EXPECT_RESULT_SUCCESS(InstallRandomContentApplication(&contentDatav1, appId, 1, &metaDb, &storage));
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, appId));

            ApplicationContentData contentDatav0;
            NNT_EXPECT_RESULT_SUCCESS(InstallRandomContentApplication(&contentDatav0, appId, 0, &metaDb, &storage));
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, appId));

            ApplicationContentData contentDatav2;
            NNT_EXPECT_RESULT_SUCCESS(InstallRandomContentApplication(&contentDatav2, appId, 2, &metaDb, &storage));
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, appId));

            auto redirectProgramId = GetRandomApplicationId();

            {
                nn::lr::Path redirectedPath = { "redirected" };
                locationResolver.RedirectProgramPath(redirectProgramId, redirectedPath);
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, redirectProgramId));
                EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

                nn::lr::Path overwrotePath = { "overwrote" };
                locationResolver.RedirectProgramPath(redirectProgramId, overwrotePath);
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, redirectProgramId));
                EXPECT_TRUE(strncmp(overwrotePath.string, path.string, sizeof(overwrotePath.string)) == 0);
            }

            {
                nn::lr::Path redirectedPath = { "redirectedControl" };
                locationResolver.RedirectApplicationControlPath(redirectProgramId, redirectedPath);
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationControlPath(&path, redirectProgramId));
                EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

                nn::lr::Path overwrotePath = { "overwroteControl" };
                locationResolver.RedirectApplicationControlPath(redirectProgramId, overwrotePath);
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationControlPath(&path, redirectProgramId));
                EXPECT_TRUE(strncmp(overwrotePath.string, path.string, sizeof(overwrotePath.string)) == 0);
            }

            {
                nn::lr::Path redirectedPath = { "redirectedHtmlDocument" };
                locationResolver.RedirectApplicationHtmlDocumentPath(redirectProgramId, redirectedPath);
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationHtmlDocumentPath(&path, redirectProgramId));
                EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

                nn::lr::Path overwrotePath = { "overwroteHtmlDocument" };
                locationResolver.RedirectApplicationHtmlDocumentPath(redirectProgramId, overwrotePath);
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationHtmlDocumentPath(&path, redirectProgramId));
                EXPECT_TRUE(strncmp(overwrotePath.string, path.string, sizeof(overwrotePath.string)) == 0);
            }

            {
                nn::lr::Path redirectedPath = { "redirectedLegalInformation" };
                locationResolver.RedirectApplicationLegalInformationPath(redirectProgramId, redirectedPath);
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationLegalInformationPath(&path, redirectProgramId));
                EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

                nn::lr::Path overwrotePath = { "overwroteLegalInformation" };
                locationResolver.RedirectApplicationLegalInformationPath(redirectProgramId, overwrotePath);
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationLegalInformationPath(&path, redirectProgramId));
                EXPECT_TRUE(strncmp(overwrotePath.string, path.string, sizeof(overwrotePath.string)) == 0);
            }
        }

        void TestClearRedirection(nn::ncm::StorageId storageId)
        {
            nn::lr::LocationResolver locationResolver;
            NNT_EXPECT_RESULT_SUCCESS(nn::lr::OpenLocationResolver(&locationResolver, storageId));

            nn::lr::Path path;

            const int ListCount = 3;
            nn::ncm::ApplicationId appIdList[ListCount];
            for (int i = 0; i < ListCount; i++)
            {
                appIdList[i] = GetRandomApplicationId();

                {
                    nn::lr::Path redirectedPath = { "redirected" };
                    locationResolver.RedirectApplicationProgramPath(appIdList[i], redirectedPath);
                    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, appIdList[i]));
                    EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);
                }

                {
                    nn::lr::Path redirectedPath = { "redirectedControl" };
                    locationResolver.RedirectApplicationControlPath(appIdList[i], redirectedPath);
                    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationControlPath(&path, appIdList[i]));
                    EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);
                }

                {
                    nn::lr::Path redirectedPath = { "redirectedHtmlDocument" };
                    locationResolver.RedirectApplicationHtmlDocumentPath(appIdList[i], redirectedPath);
                    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationHtmlDocumentPath(&path, appIdList[i]));
                    EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);
                }

                {
                    nn::lr::Path redirectedPath = { "redirectedLegalInformation" };
                    locationResolver.RedirectApplicationLegalInformationPath(appIdList[i], redirectedPath);
                    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationLegalInformationPath(&path, appIdList[i]));
                    EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);
                }
            }

            nn::ncm::ProgramId programIdList[ListCount];
            for (int i = 0; i < ListCount; i++)
            {
                programIdList[i] = { GetRandomApplicationId().value };

                {
                    nn::lr::Path redirectedPath = { "redirected" };
                    locationResolver.RedirectProgramPath(programIdList[i], redirectedPath);
                    NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, programIdList[i]));
                    EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);
                }
            }

            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ClearApplicationRedirection());

            for (int i = 0; i < ListCount; i++)
            {
                NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultProgramNotFound, locationResolver.ResolveProgramPath(&path, appIdList[i]));
                NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultControlNotFound, locationResolver.ResolveApplicationControlPath(&path, appIdList[i]));
                NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultHtmlDocumentNotFound, locationResolver.ResolveApplicationHtmlDocumentPath(&path, appIdList[i]));
                NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultLegalInformationNotFound, locationResolver.ResolveApplicationLegalInformationPath(&path, appIdList[i]));
            }

            for (int i = 0; i < ListCount; i++)
            {
                nn::lr::Path redirectedPath = { "redirected" };
                NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, programIdList[i]));
                EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

                locationResolver.EraseProgramRedirection(programIdList[i]);
                NNT_EXPECT_RESULT_FAILURE(nn::lr::ResultProgramNotFound, locationResolver.ResolveProgramPath(&path, appIdList[i]));
            }
        }

#if defined(NN_BUILD_CONFIG_OS_WIN)
        void TestNullTerminatedPath(nn::ncm::StorageId storageId)
        {
            nn::lr::LocationResolver locationResolver;
            NNT_EXPECT_RESULT_SUCCESS(nn::lr::OpenLocationResolver(&locationResolver, storageId));

            nn::lr::Path path;
            std::fill(path.string, path.string + sizeof(path.string), 'a');
            nn::ncm::ApplicationId appId = GetRandomApplicationId();

            EXPECT_DEATH(locationResolver.RedirectApplicationProgramPath(appId, path), "");
            EXPECT_DEATH(locationResolver.RedirectApplicationControlPath(appId, path), "");
            EXPECT_DEATH(locationResolver.RedirectApplicationHtmlDocumentPath(appId, path), "");
            EXPECT_DEATH(locationResolver.RedirectApplicationLegalInformationPath(appId, path), "");

            nn::ncm::ProgramId programId = { GetRandomApplicationId().value };
            EXPECT_DEATH(locationResolver.RedirectProgramPath(programId, path), "");
        }
#endif

        std::mt19937_64 m_RandomEngine;
    };
}

TEST_F(LocationResolverTest, ResolveRedirectLocationResolver)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::System));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::System));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::User));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::User));
    };

    nn::ncm::Initialize();
    nn::lr::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::lr::Finalize();
        nn::ncm::Finalize();
    };

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NNT_EXPECT_RESULT_SUCCESS(nn::ncm::CreateContentMetaDatabase(nn::ncm::StorageId::BuiltInUser));
    NNT_EXPECT_RESULT_SUCCESS(nn::ncm::CreateContentStorage(nn::ncm::StorageId::BuiltInUser));
    NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ActivateContentMetaDatabase(nn::ncm::StorageId::BuiltInUser));
    NNT_EXPECT_RESULT_SUCCESS(nn::ncm::ActivateContentStorage(nn::ncm::StorageId::BuiltInUser));
#endif

    TestResolveRedirectContentLocationResolver(nn::ncm::StorageId::BuildInSystem);
    TestResolveRedirectContentLocationResolver(nn::ncm::StorageId::BuildInUser);

    {
        nn::lr::LocationResolver locationResolver;
        NNT_EXPECT_RESULT_SUCCESS(nn::lr::OpenLocationResolver(&locationResolver, nn::ncm::StorageId::Host));

        auto redirectProgramId = GetRandomApplicationId();
        nn::lr::Path path;

        {
            nn::lr::Path redirectedPath = { "redirected" };
            locationResolver.RedirectProgramPath(redirectProgramId, redirectedPath);
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, redirectProgramId));
            EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

            nn::lr::Path overwrotePath = { "overwrote" };
            locationResolver.RedirectProgramPath(redirectProgramId, overwrotePath);
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveProgramPath(&path, redirectProgramId));
            EXPECT_TRUE(strncmp(overwrotePath.string, path.string, sizeof(overwrotePath.string)) == 0);
        }
        {
            nn::lr::Path redirectedPath = { "redirectedControl" };
            locationResolver.RedirectApplicationControlPath(redirectProgramId, redirectedPath);
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationControlPath(&path, redirectProgramId));
            EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

            nn::lr::Path overwrotePath = { "overwroteControl" };
            locationResolver.RedirectApplicationControlPath(redirectProgramId, overwrotePath);
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationControlPath(&path, redirectProgramId));
            EXPECT_TRUE(strncmp(overwrotePath.string, path.string, sizeof(overwrotePath.string)) == 0);
        }

        {
            nn::lr::Path redirectedPath = { "redirectedHtmlDocument" };
            locationResolver.RedirectApplicationHtmlDocumentPath(redirectProgramId, redirectedPath);
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationHtmlDocumentPath(&path, redirectProgramId));
            EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

            nn::lr::Path overwrotePath = { "overwroteHtmlDocument" };
            locationResolver.RedirectApplicationHtmlDocumentPath(redirectProgramId, overwrotePath);
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationHtmlDocumentPath(&path, redirectProgramId));
            EXPECT_TRUE(strncmp(overwrotePath.string, path.string, sizeof(overwrotePath.string)) == 0);
        }

        {
            nn::lr::Path redirectedPath = { "redirectedLegalInformation" };
            locationResolver.RedirectApplicationLegalInformationPath(redirectProgramId, redirectedPath);
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationLegalInformationPath(&path, redirectProgramId));
            EXPECT_TRUE(strncmp(redirectedPath.string, path.string, sizeof(redirectedPath.string)) == 0);

            nn::lr::Path overwrotePath = { "overwroteLegalInformation" };
            locationResolver.RedirectApplicationLegalInformationPath(redirectProgramId, overwrotePath);
            NNT_EXPECT_RESULT_SUCCESS(locationResolver.ResolveApplicationLegalInformationPath(&path, redirectProgramId));
            EXPECT_TRUE(strncmp(overwrotePath.string, path.string, sizeof(overwrotePath.string)) == 0);
        }
    }
}

TEST_F(LocationResolverTest, ClearRedirection)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::System));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::System));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::User));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::User));
    };

    nn::ncm::Initialize();
    nn::lr::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::lr::Finalize();
        nn::ncm::Finalize();
    };


#if defined(NN_BUILD_CONFIG_OS_WIN)
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::CreateContentMetaDatabase(nn::ncm::StorageId::BuiltInUser));
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::CreateContentStorage(nn::ncm::StorageId::BuiltInUser));
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::ActivateContentMetaDatabase(nn::ncm::StorageId::BuiltInUser));
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::ActivateContentStorage(nn::ncm::StorageId::BuiltInUser));
#endif

    TestClearRedirection(nn::ncm::StorageId::BuiltInSystem);
    TestClearRedirection(nn::ncm::StorageId::BuiltInUser);
    TestClearRedirection(nn::ncm::StorageId::Host);
}

#if defined(NN_BUILD_CONFIG_OS_WIN)
TEST_F(LocationResolverTest, NullTerminatedPath)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::System));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::System));
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountContentStorage(nn::fs::ContentStorageId::User));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(nn::fs::GetContentStorageMountName(nn::fs::ContentStorageId::User));
    };

    nn::ncm::Initialize();
    nn::lr::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::lr::Finalize();
        nn::ncm::Finalize();
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::CreateContentMetaDatabase(nn::ncm::StorageId::BuiltInUser));
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::CreateContentStorage(nn::ncm::StorageId::BuiltInUser));
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::ActivateContentMetaDatabase(nn::ncm::StorageId::BuiltInUser));
    NNT_ASSERT_RESULT_SUCCESS(nn::ncm::ActivateContentStorage(nn::ncm::StorageId::BuiltInUser));

    TestNullTerminatedPath(nn::ncm::StorageId::BuiltInSystem);
    TestNullTerminatedPath(nn::ncm::StorageId::BuiltInUser);
    TestNullTerminatedPath(nn::ncm::StorageId::Host);
}
#endif
