﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LdnNifm.cpp,PageSampleLdnNifm}
 *
 * @brief LDN ライブラリのサンプルプログラム（アクセスポイント）
 */

/**
 * @page PageSampleLdnNifm LDN ライブラリのサンプルプログラム（アクセスポイント）
 *
 * @tableofcontents
 *
 * @brief LDN ライブラリのサンプルプログラム（アクセスポイント）の解説です。
 *
 * @section PageSampleLdnNifm_SectionBrief 概要
 * LDN ライブラリのアクセスポイントのサンプルです。ステーション側のサンプルについては
 * @link PageSampleLdnStation LdnStation @endlink を参照してください。
 *
 * @section PageSampleLdnNifm_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/LdnNifm
 * Samples/Sources/Applications/LdnBasic/LdnNifm @endlink 以下にあります。
 *
 * @section PageSampleLdnNifm_SectionHowToOperate 操作方法
 * サンプルプログラムを実行するとログメッセージで操作方法が表示されます。
 * この操作方法に従い、デバッグパッドで操作してください。
 *
 * @section PageSampleLdnNifm_SectionPrecaution 注意事項
 * このサンプルプログラムは 2.4GHz 帯を使用して無線通信を行いますので、
 * 同じ周波数帯を使用する無線通信機器が多数存在する環境は避けてください。
 * 電波状況が悪いと以下のような問題が多発します。
 *
 * - 接続対象の LDN ネットワークが見つからない
 * - LDN ネットワークへの接続に失敗する
 * - UDP 通信のパケットロス率が高くなる
 *
 * 事情があって電波状況を改善できない場合、サンプルプログラムを修正することで 5GHz 帯を使用できます。
 * nn::ldn::CreateNetwork() の引数として渡す nn::ldn::NetworkConfig 構造体の
 * channel メンバを nn::ldn::AutoChannel から 36, 40, 44, 48 のいずれかに変更してください。
 * 併せて @link PageSampleLdnStation LdnStation @endlink サンプルも変更が必要になります。
 * 詳細は @link PageSampleLdnStation LdnStation @endlink サンプルの解説を参照してください。
 *
 * @section PageSampleLdnNifm_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleLdnNifm_SectionDetail 解説
 * LDN ライブラリは、中継器やインターネットを介さずに複数のターゲットを接続し、
 * それぞれのターゲット上で動作するアプリケーション間で通信する機能を提供します。
 *
 * LDN ライブラリを使用すると最大 8 台のターゲットを接続して通信できます。
 * 最初に 1 台のターゲットが「アクセスポイント」となり、LDN ネットワークを構築し、
 * 他のターゲットは「ステーション」として LDN ネットワークを探索して接続します。
 *
 * このサンプルプログラムではアクセスポイントとして LDN ネットワークを構築します。
 * ネットワークを構築した後、@link PageSampleLdnStation LdnStation @endlink
 * サンプルを使用してステーションを LDN ネットワークに接続してください。
 * LDN ネットワークを構築した後は SOCKET ライブラリを用いて UDP 通信を行います。
 * 全てのターゲットは独自にカウンタをもっており、そのカウンタの値を全員で共有します。
 * 具体的には、ステーションは自身のカウンタの値をユニキャストでアクセスポイントに送信し、
 * アクセスポイントは自身を含む全員のカウンタの値をブロードキャストでステーションに配信します。
 */

#include <memory>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include <nn/socket/socket_TypesPrivate.h>
#include "../Common/Definition.h"
#include "../Common/Util.h"

#include <nn/settings/fwdbg/settings_SettingsCommon.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForMenu.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>

#include <curl/curl.h>

#include <nn/wlan/wlan_Types.h>
#include <nn/wlan/wlan_Ssid.h>
#include <nn/wlan/wlan_LocalApi.h>

nn::ldn::NetworkInfo g_ScanResult[nn::ldn::ScanResultCountMax];

namespace
{
    NN_ALIGNAS(4096) uint8_t g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize];

    const nn::hid::NpadIdType NpadIds[] = {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::No4,
        nn::hid::NpadId::No5,
        nn::hid::NpadId::No6,
        nn::hid::NpadId::No7,
        nn::hid::NpadId::No8,
        nn::hid::NpadId::Handheld,
    };
}

//#define SIGLO_42479_RANDOM_NONE // 5GHz/40CH固定

#define NN_LDNNIFM_RESULT_SUCCESS(ret)                                  \
    if (ret.IsSuccess() != true)                                        \
        NN_LOG("%s --> L.%d Result Module(%d) Description(%d)\n\n",     \
                __FUNCTION__, __LINE__,                                 \
                ret.GetModule(), ret.GetDescription());                 \
    NN_ABORT_UNLESS_RESULT_SUCCESS(ret);


NN_ALIGNAS(4096) char g_SendThreadStack[4096 * 3];
NN_ALIGNAS(4096) char g_ReceiveThreadStack[4096 * 3];
NN_ALIGNAS(4096) char g_LdnThreadStack[4096 * 4];
NN_ALIGNAS(4096) char g_MonitorThreadStack[4096 * 3];
NN_ALIGNAS(4096) char g_LdnAgingThreadStack[4096 * 5];

int SendThreadPriority      = nn::os::DefaultThreadPriority + 4;
int ReceiveThreadPriority   = nn::os::DefaultThreadPriority + 3;
int LdnThreadPriority       = nn::os::DefaultThreadPriority + 2;
int MonitorThreadPriority   = nn::os::DefaultThreadPriority + 1;

const int ModeSettingServerWaitTime     = 30;
const int RandamModeCount               = 80;
const int ModeAgingMinTime              = 43200;
const int ModeAgingMaxTime              = 43200;
const int ModeScanWaitTime              = 3;

const int ModeLocalAgingMinTime         = 86400;
const int ModeLocalAgingMaxTime         = 86400;
const int ModeInfraEtherAgingMinTime    = 86400;
const int ModeInfraEtherAgingMaxTime    = 86400;
const int ModeInfraWiFiAgingMinTime     = 86400;
const int ModeInfraWiFiAgingMaxTime     = 86400;

const int ModeLocalInfraAgingMinTime    = 259200;
const int ModeLocalInfraAgingMaxTime    = 259200;

const int ModeChangeMinTime             = 15;
const int ModeChangeMaxTime             = 30;
const int ClientConRetry                = 100;
const int ConWaitTime                   = 3;
const int ModeWaitTime                  = 180;
const int ModeLocalConnectRetryCount    = 10;

const bool NifmCtrlEnable   = true;


// 一時的な接続設定（テンポラリ接続設定）の構築
// 周波数帯別のネットワーク設定に変更
const nn::nifm::WirelessSettingData WirelessSettingData2_4Ghz = {
    {               // ssidConfig
        {           // ssid
            15,      // length
            { 0x77, 0x69, 0x72, 0x65, 0x6c, 0x65, 0x73, 0x73, 0x5f, 0x74, 0x65, 0x73, 0x74, 0x5f, 0x39 }     // wireless_test_9
        },
    false       // nonBroadcast
    },
    {               // security
        {           // authEncryption
            nn::nifm::Authentication_Open,  // authentication
            nn::nifm::Encryption_None       // encryption
        },
        {           // sharedKey
            0,      // length
            ""      // keyMaterial
        }
    }
};

const nn::nifm::WirelessSettingData WirelessSettingData5Ghz = {
    {               // ssidConfig
        {           // ssid
            16,      // length
            { 0x77, 0x69, 0x72, 0x65, 0x6c, 0x65, 0x73, 0x73, 0x5f, 0x74, 0x65, 0x73, 0x74, 0x5f, 0x31, 0x30 }     // wireless_test_10
        },
    false       // nonBroadcast
    },
    {               // security
        {           // authEncryption
            nn::nifm::Authentication_Open,  // authentication
            nn::nifm::Encryption_None       // encryption
        },
        {           // sharedKey
            0,      // length
            ""      // keyMaterial
        }
    }
};

const nn::nifm::IpSettingData ipServerSettingData2_4Ghz = {
    {               // ip
        false,       // isAuto *固定IPで対応しているためDHCPは無効のみです
        { {192, 168, 10, 10} },      // ipAddress
        { {255, 255, 255, 0} },       // subnetMask
        { {192, 168, 10, 1} }        // defaultGateway
    },
    {               // dns
        false,       // isAuto
        { {192, 168, 10, 1} },       // preferredDns
        { {192, 168, 10, 1} }        // alternateDns
    },
    {               // proxy
        false,      // isEnabled
        0,          // port
        "",         // proxy
        {           // authentication
            false,  // isEnabled
            "",     // username
            ""      // password
        }
    },
    1400            // mtu
};

const nn::nifm::IpSettingData ipServerWiFiSettingData2_4Ghz = {
    {               // ip
        false,       // isAuto *固定IPで対応しているためDHCPは無効のみです
        { { 192, 168, 10, 12 } },      // ipAddress
        { { 255, 255, 255, 0 } },       // subnetMask
        { { 192, 168, 10, 1 } }        // defaultGateway
    },
    {               // dns
        false,       // isAuto
        { { 192, 168, 10, 1 } },       // preferredDns
        { { 192, 168, 10, 1 } }        // alternateDns
    },
    {               // proxy
        false,      // isEnabled
        0,          // port
        "",         // proxy
        {           // authentication
            false,  // isEnabled
            "",     // username
            ""      // password
        }
    },
    1400            // mtu
};

const nn::nifm::IpSettingData ipServerSettingData5Ghz = {
    {               // ip
        false,       // isAuto *固定IPで対応しているためDHCPは無効のみです
        { {192, 168, 10, 10} },      // ipAddress
        { {255, 255, 255, 0} },       // subnetMask
        { {192, 168, 10, 1} }        // defaultGateway
    },
    {               // dns
        false,       // isAuto
        { {192, 168, 10, 1} },       // preferredDns
        { {192, 168, 10, 1} }        // alternateDns
    },
    {               // proxy
        false,      // isEnabled
        0,          // port
        "",         // proxy
        {           // authentication
            false,  // isEnabled
            "",     // username
            ""      // password
        }
    },
    1400            // mtu
};

const nn::nifm::IpSettingData ipServerWiFiSettingData5Ghz = {
    {               // ip
        false,       // isAuto *固定IPで対応しているためDHCPは無効のみです
        { { 192, 168, 10, 12 } },      // ipAddress
        { { 255, 255, 255, 0 } },       // subnetMask
        { { 192, 168, 10, 1 } }        // defaultGateway
    },
    {               // dns
        false,       // isAuto
        { { 192, 168, 10, 1 } },       // preferredDns
        { { 192, 168, 10, 1 } }        // alternateDns
    },
    {               // proxy
        false,      // isEnabled
        0,          // port
        "",         // proxy
        {           // authentication
            false,  // isEnabled
            "",     // username
            ""      // password
        }
    },
    1400            // mtu
};

const nn::nifm::IpSettingData ipClientSettingData2_4Ghz = {
    {               // ip
        false,       // isAuto *固定IPで対応しているためDHCPは無効のみです
        { {192, 168, 10, 11} },      // ipAddress
        { {255, 255, 255, 0} },       // subnetMask
        { {192, 168, 10, 1} }        // defaultGateway
    },
    {               // dns
        false,       // isAuto
        { {192, 168, 10, 1} },       // preferredDns
        { {192, 168, 10, 1} }        // alternateDns
    },
    {               // proxy
        false,      // isEnabled
        0,          // port
        "",         // proxy
        {           // authentication
            false,  // isEnabled
            "",     // username
            ""      // password
        }
    },
    1400            // mtu
};

const nn::nifm::IpSettingData ipClientWiFiSettingData2_4Ghz = {
    {               // ip
        false,       // isAuto *固定IPで対応しているためDHCPは無効のみです
        { { 192, 168, 10, 13 } },      // ipAddress
        { { 255, 255, 255, 0 } },       // subnetMask
        { { 192, 168, 10, 1 } }        // defaultGateway
    },
    {               // dns
        false,       // isAuto
        { { 192, 168, 10, 1 } },       // preferredDns
        { { 192, 168, 10, 1 } }        // alternateDns
    },
    {               // proxy
        false,      // isEnabled
        0,          // port
        "",         // proxy
        {           // authentication
            false,  // isEnabled
            "",     // username
            ""      // password
        }
    },
    1400            // mtu
};

const nn::nifm::IpSettingData ipClientSettingData5Ghz = {
    {               // ip
        false,       // isAuto *固定IPで対応しているためDHCPは無効のみです
        { {192, 168, 10, 11} },      // ipAddress
        { {255, 255, 255, 0} },       // subnetMask
        { {192, 168, 10, 1} }        // defaultGateway
    },
    {               // dns
        false,       // isAuto
        { {192, 168, 10, 1} },       // preferredDns
        { {192, 168, 10, 1} }        // alternateDns
    },
    {               // proxy
        false,      // isEnabled
        0,          // port
        "",         // proxy
        {           // authentication
            false,  // isEnabled
            "",     // username
            ""      // password
        }
    },
    1400            // mtu
};

const nn::nifm::IpSettingData ipClientWiFiSettingData5Ghz = {
    {               // ip
        false,       // isAuto *固定IPで対応しているためDHCPは無効のみです
        { { 192, 168, 10, 13 } },      // ipAddress
        { { 255, 255, 255, 0 } },       // subnetMask
        { { 192, 168, 10, 1 } }        // defaultGateway
    },
    {               // dns
        false,       // isAuto
        { { 192, 168, 10, 1 } },       // preferredDns
        { { 192, 168, 10, 1 } }        // alternateDns
    },
    {               // proxy
        false,      // isEnabled
        0,          // port
        "",         // proxy
        {           // authentication
            false,  // isEnabled
            "",     // username
            ""      // password
        }
    },
    1400            // mtu
};

// WiFi AP 2.4GHz設定
const nn::nifm::NetworkProfileData WifiServerProfileData2_4Ghz = {
    nn::util::InvalidUuid,  // id
    "",                     // profileName
    nn::nifm::NetworkProfileType_User,  // networkProfileType
    nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, // networkInterfaceType
    true,  // isAutoConnect
    true,  // isLargeCapacity
    {
        WirelessSettingData2_4Ghz
    },
    ipServerWiFiSettingData2_4Ghz
};

const nn::nifm::NetworkProfileData WifiClientProfileData2_4Ghz = {
    nn::util::InvalidUuid,  // id
    "",                     // profileName
    nn::nifm::NetworkProfileType_User,  // networkProfileType
    nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, // networkInterfaceType
    true,  // isAutoConnect
    true,  // isLargeCapacity
    {
        WirelessSettingData2_4Ghz
    },
    ipClientWiFiSettingData2_4Ghz
};

const nn::nifm::NetworkProfileData EthernetServerProfileData2_4Ghz = {
    nn::util::InvalidUuid,  // id
    {},                     // profileName
    nn::nifm::NetworkProfileType_User,  // networkProfileType
    nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, // networkInterfaceType
    true,  // isAutoConnect
    true,  // isLargeCapacity
    {
        WirelessSettingData2_4Ghz
    },
    ipServerSettingData2_4Ghz
};

const nn::nifm::NetworkProfileData EthernetClientProfileData2_4Ghz = {
    nn::util::InvalidUuid,  // id
    {},                     // profileName
    nn::nifm::NetworkProfileType_User,  // networkProfileType
    nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, // networkInterfaceType
    true,  // isAutoConnect
    true,  // isLargeCapacity
    {
        WirelessSettingData2_4Ghz
    },
    ipClientSettingData2_4Ghz
};

// WiFi AP 5GHz設定
const nn::nifm::NetworkProfileData WifiServerProfileData5Ghz = {
    nn::util::InvalidUuid,  // id
    "",                     // profileName
    nn::nifm::NetworkProfileType_User,  // networkProfileType
    nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, // networkInterfaceType
    true,  // isAutoConnect
    true,  // isLargeCapacity
    {
        WirelessSettingData5Ghz
    },
    ipServerWiFiSettingData5Ghz
};

const nn::nifm::NetworkProfileData WifiClientProfileData5Ghz = {
    nn::util::InvalidUuid,  // id
    "",                     // profileName
    nn::nifm::NetworkProfileType_User,  // networkProfileType
    nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ieee80211, // networkInterfaceType
    true,  // isAutoConnect
    true,  // isLargeCapacity
    {
        WirelessSettingData5Ghz
    },
    ipClientWiFiSettingData5Ghz
};

const nn::nifm::NetworkProfileData EthernetServerProfileData5Ghz = {
    nn::util::InvalidUuid,  // id
    "",                     // profileName
    nn::nifm::NetworkProfileType_User,  // networkProfileType
    nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, // networkInterfaceType
    true,  // isAutoConnect
    true,  // isLargeCapacity
    {
        WirelessSettingData5Ghz
    },
    ipServerSettingData5Ghz
};

const nn::nifm::NetworkProfileData EthernetClientProfileData5Ghz = {
    nn::util::InvalidUuid,  // id
    "",                     // profileName
    nn::nifm::NetworkProfileType_User,  // networkProfileType
    nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, // networkInterfaceType
    true,  // isAutoConnect
    true,  // isLargeCapacity
    {
        WirelessSettingData5Ghz
    },
    ipClientSettingData5Ghz
};

const int16_t Channels_2_4Ghz[] = {
#ifdef SIGLO_42479_RANDOM_NONE
    1
#else
    1,  6,  11
#endif
};

const int16_t Channels_5Ghz[] = {
#ifdef SIGLO_42479_RANDOM_NONE
    40
#else
    36,  40,  44,   48
#endif
};

enum ApplicationState
{
    ApplicationState_Initialized,
    ApplicationState_NetworkCreating,
    ApplicationState_NetworkCreated,
    ApplicationState_NetworkDestroying,
    ApplicationState_NetworkDestroyed,
    ApplicationState_Connecting,
    ApplicationState_Connected,
    ApplicationState_Disconnecting,
    ApplicationState_Disconnected,
    ApplicationState_ChangeMode
};


enum ChangeMode
{
    Mode_Init = 0,
    Mode_InfraEtherServer,
    Mode_InfraEtherClient,
    Mode_InfraWiFiServer,
    Mode_InfraWiFiClient,
    Mode_LdnLocalMaster,
    Mode_LdnLocalClient,

    // 以降はテスト起動モードで使用
    Mode_TestMaster,
    Mode_TestClient
};

enum TestDataType
{
    Type_Non = 0,
    Type_TestPattern,
};

enum TestNetPattern
{
    NetPttern_2_4Ghz,
    NetPttern_5Ghz,
};

const char* ChangeModeString[] =
{
    "Mode Init",
    "Mode Infra Ether Server Test",
    "Mode Infra Ether Client Test",
    "Mode Infra WiFi Server Test",
    "Mode Infra WiFi Client Test",
    "Mode Local Master Test",
    "Mode Local Client Test",

    // 以降はテスト起動モードで使用
    "Mode Test Master Execute",
    "Mode Test Client Execute"
};


#ifdef SIGLO_42479_RANDOM_NONE
struct Siglo42479_Trace
{
    ChangeMode mode;
    uint32_t changeTime;
};

// SIGLO-42479問題発生時試験パターン
const Siglo42479_Trace TraceTestData[] =
{
    { Mode_InfraEtherClient, 25 },
    { Mode_InfraEtherClient, 25 },
    { Mode_InfraWiFiServer, 25 },
    { Mode_InfraWiFiServer, 25 },
    { Mode_InfraEtherServer, 25 },
    { Mode_InfraEtherServer, 25 },
    { Mode_InfraWiFiServer, 23 },
    { Mode_InfraWiFiServer, 23 },
    { Mode_InfraEtherClient, 16 },
    { Mode_InfraEtherClient, 16 },
    { Mode_InfraWiFiClient, 26 },
    { Mode_InfraWiFiClient, 26 },
    { Mode_InfraWiFiServer, 25 },
    { Mode_InfraWiFiServer, 25 },
    { Mode_LdnLocalMaster, 25 },
    { Mode_LdnLocalMaster, 25 },
    { Mode_LdnLocalMaster, 19 },
    { Mode_LdnLocalMaster, 19 },
    { Mode_LdnLocalMaster, 15 },
    { Mode_LdnLocalMaster, 15 },
    { Mode_InfraEtherServer, 16 },
    { Mode_InfraEtherServer, 16 },
    { Mode_LdnLocalClient, 17 },
    { Mode_LdnLocalClient, 17 },
    { Mode_LdnLocalClient, 26 },
    { Mode_LdnLocalClient, 26 },
    { Mode_InfraWiFiClient, 20 },
    { Mode_InfraWiFiClient, 20 },
    { Mode_InfraEtherClient, 15 },
    { Mode_InfraEtherClient, 15 },
    { Mode_InfraEtherClient, 26 },
    { Mode_InfraEtherClient, 26 },
    { Mode_LdnLocalMaster, 30 },
    { Mode_LdnLocalMaster, 30 },
    { Mode_LdnLocalMaster, 23 },
    { Mode_LdnLocalMaster, 23 },
    { Mode_InfraEtherServer, 20 },
    { Mode_InfraEtherServer, 20 },
    { Mode_InfraEtherClient, 24 },
    { Mode_InfraEtherClient, 24 },
    { Mode_InfraEtherServer, 26 },
    { Mode_InfraEtherServer, 26 },
    { Mode_InfraWiFiClient, 22 },
    { Mode_InfraWiFiClient, 22 },
    { Mode_LdnLocalMaster, 21 },
    { Mode_LdnLocalMaster, 21 },
    { Mode_InfraEtherServer, 21 },
    { Mode_InfraEtherServer, 21 },
    { Mode_InfraEtherClient, 21 },
    { Mode_InfraEtherClient, 21 },
    { Mode_InfraWiFiClient, 29 },
    { Mode_InfraWiFiClient, 29 },
    { Mode_InfraEtherServer, 25 },
    { Mode_InfraEtherServer, 25 },
    { Mode_InfraWiFiClient, 25 },
    { Mode_InfraWiFiClient, 25 },
    { Mode_LdnLocalClient, 28 },
    { Mode_LdnLocalClient, 28 },
    { Mode_InfraWiFiClient, 27 },
    { Mode_InfraWiFiClient, 27 },
    { Mode_LdnLocalMaster, 19 },
    { Mode_LdnLocalMaster, 19 },
    { Mode_LdnLocalClient, 20 },
    { Mode_LdnLocalClient, 20 },
    { Mode_LdnLocalMaster, 27 },
    { Mode_LdnLocalMaster, 27 },
    { Mode_InfraEtherServer, 28 },
    { Mode_InfraEtherServer, 28 },
    { Mode_InfraEtherServer, 29 },
    { Mode_InfraEtherServer, 29 },
    { Mode_InfraEtherServer, 27 },
    { Mode_InfraEtherServer, 27 },
    { Mode_InfraWiFiClient, 20 },
    { Mode_InfraWiFiClient, 20 },
    { Mode_LdnLocalMaster, 15 },
    { Mode_LdnLocalMaster, 15 },
    { Mode_LdnLocalClient, 28 },
    { Mode_LdnLocalClient, 28 },
    { Mode_InfraWiFiServer, 16 },
    { Mode_InfraWiFiServer, 16 }
};

#endif

// ローカル通信／インフラ通信周波数帯試験パターン
// | 2.4GHz → 5GHz | 5GHz → 2.4GHz | 2.4GHz → 2.4 GHz | 5GHz → 5GHz |
const TestNetPattern NetProfilePattern[] =
{
#ifdef SIGLO_42479_RANDOM_NONE
    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_5Ghz,
#else
    TestNetPattern::NetPttern_2_4Ghz,
    TestNetPattern::NetPttern_2_4Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_2_4Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_2_4Ghz,
    TestNetPattern::NetPttern_2_4Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_2_4Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_2_4Ghz,

    TestNetPattern::NetPttern_2_4Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_2_4Ghz,
    TestNetPattern::NetPttern_2_4Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_2_4Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_2_4Ghz,
    TestNetPattern::NetPttern_2_4Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_2_4Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_2_4Ghz,

    TestNetPattern::NetPttern_2_4Ghz,
    TestNetPattern::NetPttern_5Ghz,

    TestNetPattern::NetPttern_5Ghz,
    TestNetPattern::NetPttern_5Ghz,
#endif
};

struct TestDataPayload
{
    uint32_t dataType;
    uint8_t data[1];
};

struct ChangeModePayload
{
    uint32_t executeTime;
    uint32_t length;
    ChangeMode mode[RandamModeCount];
    uint32_t changeTimes[RandamModeCount];
};

struct ChangeModeInfo
{
    uint32_t length;
    ChangeMode mode[RandamModeCount];
    uint32_t changeTimes[RandamModeCount];
};

struct ApplicationResource
{
    nn::os::ThreadType          monitorThread;
    nn::os::ThreadType          sendThread;
    nn::os::ThreadType          receiveThread;
    nn::os::SystemEventType     stateChangeEvent;
    nn::os::EventType           cancelEvent;
    nn::os::EventType           initEvent;
    nn::os::MutexType           mutex;
    nn::os::MutexType           apiMutex;
    nn::ldn::NetworkInfo        networkInfo;
    nn::ldn::Ipv4Address        ipv4Address;
    nn::ldn::SubnetMask         subnetMask;
    uint16_t                    port;
    uint32_t                    myCounter;
    uint32_t                    counter[nn::ldn::NodeCountMax];
    int                         socket;
    int                         clientSocket;
    ApplicationState            state;
    uint32_t                    executeTime;
    ChangeMode                  mode;
    ChangeMode                  currentMode;
    ChangeModeInfo              TestPattern;
    bool                        isAgingEnable;
    nn::nifm::NetworkProfileData* pNetProfile;
    int16_t                     setChannel;
};

void CreateSocket(ApplicationResource* pApp) NN_NOEXCEPT;
void CreateTcpServerSocket(ApplicationResource* pApp) NN_NOEXCEPT;
void CreateTcpClientSocket(ApplicationResource* pApp) NN_NOEXCEPT;
void CreateNifmBroadCastSocket(ApplicationResource* pApp) NN_NOEXCEPT;
void DestroySocket(ApplicationResource* pApp, bool isClientClose) NN_NOEXCEPT;


void TraceCurrentnetwork() NN_NOEXCEPT
{
    nn::Result result;
    nn::socket::InAddr ipAddr, subAddr, gateAddr, dns1Addr, dns2Addr;
    char strAddress[INET_ADDRSTRLEN];

    result = nn::nifm::GetCurrentIpConfigInfo(&ipAddr, &subAddr, &gateAddr, &dns1Addr, &dns2Addr);
    if (result.IsSuccess())
    {
        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &ipAddr.S_addr, strAddress, sizeof(strAddress));

        NN_LOG("現在のWlan接続ネットワーク設定情報です。\n");
        NN_LOG("IP Address     : %s\n",
            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &ipAddr.S_addr, strAddress, sizeof(strAddress)));
        NN_LOG("Subnet Mask    : %s\n",
            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &subAddr.S_addr, strAddress, sizeof(strAddress)));
        NN_LOG("Default GateWay: %s\n",
            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &gateAddr.S_addr, strAddress, sizeof(strAddress)));
        NN_LOG("Dns1 Address   : %s\n",
            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &dns1Addr.S_addr, strAddress, sizeof(strAddress)));
        NN_LOG("Dns2 Address   : %s\n\n",
            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &dns2Addr.S_addr, strAddress, sizeof(strAddress)));
    }
    else
    {
        NN_LOG("nn::nifm::GetCurrentIpConfigInfo Error Module(%d) Description(%d)\n\n", result.GetModule(), result.GetDescription());
    }
}

int GetRandom(int min, int max) NN_NOEXCEPT
{
    static time_t check;
    int rand_ret;
    if (check != time(nullptr))
    {
        check = time(nullptr);
        srand(static_cast<unsigned int>(time(nullptr)));
    }
    rand_ret = min + static_cast<int>(rand() * (max - min + 1.0) / (1.0 + RAND_MAX));
    return rand_ret;
}

nn::Bit32 ConvertIpArrayToBit32(const nn::nifm::IpV4Address& ipAddress)
{
    nn::Bit32* pIpAddress;
    nn::Bit8 ipArray[nn::nifm::IpV4Address::Size];
    int offSet = nn::nifm::IpV4Address::Size;

    for (int i = 0; i < nn::nifm::IpV4Address::Size; i++)
    {
        offSet--;
        ipArray[i] = ipAddress.data[offSet];
    }

    pIpAddress = reinterpret_cast<nn::Bit32*>(&ipArray[0]);

    return *pIpAddress;
}

void TestSocketClose(int &sock, nn::os::MutexType* pMutex, bool isShutdown = false)
{
    if (pMutex != nullptr)
    {
        nn::os::LockMutex(pMutex);
    }

    int nResult;
    if (sock != -1)
    {
        if (isShutdown == true)
        {
            nResult = nn::socket::Shutdown(sock, nn::socket::ShutdownMethod::Shut_RdWr);
            NN_LOG("Socket Shutdown Result : %d\n", nResult);
            if (nResult != 0)
            {
                NN_LOG("Socket Shutdown Last Error : %d\n", nn::socket::GetLastError());
            }
        }

        // ソケットを破棄します。
        nResult = nn::socket::Close(sock);
        NN_LOG("Socket Close Result : %d\n", nResult);
        sock = -1;
    }

    if (pMutex != nullptr)
    {
        nn::os::UnlockMutex(pMutex);
    }
}

bool ExecuteTcpIpCommunication() NN_NOEXCEPT
{
    nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::DefaultSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);

    curl_global_init(CURL_GLOBAL_DEFAULT);

    CURL* curl = curl_easy_init();
    if( curl == nullptr )
    {
        nn::socket::Finalize();

        NN_LOG("curl_easy_init failed.\n");
        return false;
    }

    curl_easy_setopt(curl, CURLOPT_URL, "http://example.com");

    // Debug/Develop 版では、サーバからの受信内容が表示されます
    CURLcode curlCode = curl_easy_perform(curl);
    NN_LOG("curl_easy_perform returned %d.\n", curlCode);

    curl_easy_cleanup(curl);

    nn::socket::Finalize();

    return CURLE_OK;
}


void EthernetTcpServer(ApplicationResource* pApp, uint32_t waitTime) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(EthernetServerProfileData2_4Ghz);
    nn::nifm::NetworkConnection networkConnection;
    nn::nifm::RequestHandle requestHandle;
    nn::util::Uuid profileId;
    nn::Result result;

    if (pApp->pNetProfile == nullptr)
    {
        result = temporaryNetworkProfile.Update(EthernetServerProfileData2_4Ghz);
    }
    else
    {
        result = temporaryNetworkProfile.Update(*pApp->pNetProfile);
    }
    if (result.IsSuccess() != true)
    {
        pApp->isAgingEnable = false;
        NN_LOG("temporaryNetworkProfile Update Error(%d:%d) %d\n\n",
            result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }

    requestHandle = networkConnection.GetRequestHandle();
    profileId = temporaryNetworkProfile.GetId();
    result = nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    while (!nn::os::TryWaitEvent(&pApp->cancelEvent))
    {
        NN_LOG("%s ---> SubmitRequestAndWait\n", __FUNCTION__);
        networkConnection.SubmitRequestAndWait();
        if (networkConnection.IsAvailable() == true)
        {
            break;
        }
    }

    // 外部アクセスできる環境が必須です。外部アクセスに接続しないとsocket接続が失敗します。
    if (networkConnection.IsAvailable() == true)
    {
        TraceCurrentnetwork();

        nn::os::ClearEvent(&pApp->initEvent);
        nn::os::ClearEvent(&pApp->cancelEvent);

        // socket ライブラリを初期化します。
        result = nn::socket::Initialize(
            g_SocketMemoryPoolBuffer,
            nn::socket::DefaultSocketMemoryPoolSize,
            nn::socket::DefaultSocketAllocatorSize,
            nn::socket::DefaultConcurrencyLimit);
        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

        pApp->state = ApplicationState_Connected;

        // TCP ソケット生成、サーバー送受信を別スレッドで開始します
        CreateTcpServerSocket(pApp);

        // サーバーにクライアントが接続されたか待機します。
        if (nn::os::TimedWaitEvent(&pApp->initEvent, nn::TimeSpan::FromSeconds(ModeWaitTime)) == true)
        {
            // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
            nn::os::TimedWaitEvent(&pApp->cancelEvent, nn::TimeSpan::FromSeconds(waitTime));
        }

        // ネットワークを破棄し、データの送受信を終了します。
        DestroySocket(pApp, true);

        networkConnection.CancelRequest();
        if (networkConnection.IsAvailable() != true)
        {
            NN_LOG("NIFM ネットワークが切断されました。\n\n");
        }
        else
        {
            result = networkConnection.GetResult();
            NN_LOG("NIFM ネットワークが接続状態で終了しました。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                result.GetDescription(), result.GetInnerValueForDebug());
        }

        // socket ライブラリの使用を終了します。
        result = nn::socket::Finalize();
        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    else
    {
        pApp->isAgingEnable = false;
        result = networkConnection.GetResult();
        NN_LOG("NIFM ネットワーク接続されていません。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
            result.GetDescription(), result.GetInnerValueForDebug());
    }
}

void EthernetTcpClient(ApplicationResource* pApp, uint32_t waitTime) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(EthernetClientProfileData2_4Ghz);
    nn::nifm::NetworkConnection networkConnection;
    nn::nifm::RequestHandle requestHandle;
    nn::util::Uuid profileId;
    nn::Result result;

    if (pApp->pNetProfile == nullptr)
    {
        result = temporaryNetworkProfile.Update(EthernetClientProfileData2_4Ghz);
    }
    else
    {
        result = temporaryNetworkProfile.Update(*pApp->pNetProfile);
    }
    if (result.IsSuccess() != true)
    {
        pApp->isAgingEnable = false;
        NN_LOG("temporaryNetworkProfile Update Error(%d:%d) %d\n\n",
            result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }

    requestHandle = networkConnection.GetRequestHandle();
    profileId = temporaryNetworkProfile.GetId();
    result = nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);

    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    while (!nn::os::TryWaitEvent(&pApp->cancelEvent))
    {
        NN_LOG("%s ---> SubmitRequestAndWait\n", __FUNCTION__);
        networkConnection.SubmitRequestAndWait();
        if (networkConnection.IsAvailable() == true)
        {
            break;
        }
    }

    // 外部アクセスできる環境が必須です。外部アクセスに接続しないとsocket接続が失敗します。
    if (networkConnection.IsAvailable() == true)
    {
        TraceCurrentnetwork();

        nn::os::ClearEvent(&pApp->initEvent);
        nn::os::ClearEvent(&pApp->cancelEvent);

        // socket ライブラリを初期化します。
        result = nn::socket::Initialize(
            g_SocketMemoryPoolBuffer,
            nn::socket::DefaultSocketMemoryPoolSize,
            nn::socket::DefaultSocketAllocatorSize,
            nn::socket::DefaultConcurrencyLimit);

        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

        pApp->state = ApplicationState_Connected;

        // TCP ソケット生成、クライアント送受信を別スレッドで開始します
        CreateTcpClientSocket(pApp);

        // メインスレッドから停止を指示されるか、試験実行時間が経過するか、ネットワークが破棄されるまで待機します。
        nn::os::TimedWaitEvent(&pApp->cancelEvent, nn::TimeSpan::FromSeconds(waitTime));

        // ネットワークを破棄し、データの送受信を終了します。
        DestroySocket(pApp, false);
        pApp->clientSocket = -1;

        networkConnection.CancelRequest();
        if (networkConnection.IsAvailable() != true)
        {
            NN_LOG("NIFM ネットワークが切断されました。\n\n");
        }
        else
        {
            result = networkConnection.GetResult();
            NN_LOG("NIFM ネットワークが接続状態で終了しました。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                result.GetDescription(), result.GetInnerValueForDebug());
        }

        // socket ライブラリの使用を終了します。
        result = nn::socket::Finalize();
        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    else
    {
        pApp->isAgingEnable = false;
        result = networkConnection.GetResult();
        NN_LOG("NIFM ネットワーク接続されていません。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
            result.GetDescription(), result.GetInnerValueForDebug());
    }
}

void WifiServer(ApplicationResource* pApp, uint32_t waitTime) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);
    nn::Result result;

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(WifiServerProfileData2_4Ghz);
    if (pApp->pNetProfile == nullptr)
    {
        result = temporaryNetworkProfile.Update(WifiServerProfileData2_4Ghz);
    }
    else
    {
        result = temporaryNetworkProfile.Update(*pApp->pNetProfile);
    }
    if (result.IsSuccess() != true)
    {
        pApp->isAgingEnable = false;
        NN_LOG("temporaryNetworkProfile Update Error(%d:%d) %d\n\n",
            result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }

    nn::nifm::NetworkConnection networkConnection;
    nn::nifm::RequestHandle requestHandle = networkConnection.GetRequestHandle();
    nn::util::Uuid profileId = temporaryNetworkProfile.GetId();
    nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);
    networkConnection.SubmitRequestAndWait();

    // 外部アクセスできる環境が必須です。外部アクセスに接続しないとsocket接続が失敗します。
    if (networkConnection.IsAvailable() == true)
    {
        TraceCurrentnetwork();

        nn::os::ClearEvent(&pApp->initEvent);
        nn::os::ClearEvent(&pApp->cancelEvent);

        // socket ライブラリを初期化します。
        result = nn::socket::Initialize(
            g_SocketMemoryPoolBuffer,
            nn::socket::DefaultSocketMemoryPoolSize,
            nn::socket::DefaultSocketAllocatorSize,
            nn::socket::DefaultConcurrencyLimit);

        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

        pApp->state = ApplicationState_Connected;

        // TCP ソケット生成、サーバー送受信を別スレッドで開始します
        CreateTcpServerSocket(pApp);

        // Clientが接続されるか、試験時間が経過するまで待機します
        if (nn::os::TimedWaitEvent(&pApp->initEvent, nn::TimeSpan::FromSeconds(ModeWaitTime)) == true)
        {
            // メインスレッドから停止を指示されるか、試験実行時間が経過するか、ネットワークが破棄されるまで待機します。
            nn::os::TimedWaitEvent(&pApp->cancelEvent, nn::TimeSpan::FromSeconds(waitTime));
        }

        // ネットワークを破棄し、データの送受信を終了します。
        DestroySocket(pApp, true);

        networkConnection.CancelRequest();
        if (networkConnection.IsAvailable() != true)
        {
            NN_LOG("NIFM ネットワークが切断されました。\n\n");
        }
        else
        {
            result = networkConnection.GetResult();
            NN_LOG("NIFM ネットワークが接続状態で終了しました。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                result.GetDescription(), result.GetInnerValueForDebug());
        }

        // socket ライブラリの使用を終了します。
        result = nn::socket::Finalize();
        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    else{
        pApp->isAgingEnable = false;
        result = networkConnection.GetResult();
        NN_LOG("NIFM ネットワーク接続されていません。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
            result.GetDescription(), result.GetInnerValueForDebug());
    }

}

void WifiClient(ApplicationResource* pApp, uint32_t waitTime) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);
    nn::Result result;

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(WifiClientProfileData2_4Ghz);
    if (pApp->pNetProfile == nullptr)
    {
        result = temporaryNetworkProfile.Update(WifiClientProfileData2_4Ghz);
    }
    else
    {
        result = temporaryNetworkProfile.Update(*pApp->pNetProfile);
    }
    if (result.IsSuccess() != true)
    {
        pApp->isAgingEnable = false;
        NN_LOG("temporaryNetworkProfile Update Error(%d:%d) %d\n\n",
            result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        return;
    }

    nn::nifm::NetworkConnection networkConnection;
    nn::nifm::RequestHandle requestHandle = networkConnection.GetRequestHandle();
    nn::util::Uuid profileId = temporaryNetworkProfile.GetId();
    nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);
    NN_LOG("%s ---> SubmitRequestAndWait\n", __FUNCTION__);
    networkConnection.SubmitRequestAndWait();

    // 外部アクセスできる環境が必須です。外部アクセスに接続しないとsocket接続が失敗します。
    if (networkConnection.IsAvailable() == true)
    {
        TraceCurrentnetwork();

        nn::os::ClearEvent(&pApp->initEvent);
        nn::os::ClearEvent(&pApp->cancelEvent);

        // socket ライブラリを初期化します。
        result = nn::socket::Initialize(
            g_SocketMemoryPoolBuffer,
            nn::socket::DefaultSocketMemoryPoolSize,
            nn::socket::DefaultSocketAllocatorSize,
            nn::socket::DefaultConcurrencyLimit);

        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

        pApp->state = ApplicationState_Connected;

        // TCP ソケット生成、クライアント送受信を別スレッドで開始します
        CreateTcpClientSocket(pApp);

        // メインスレッドから停止を指示されるか、試験実行時間が経過するか、ネットワークが破棄されるまで待機します。
        nn::os::TimedWaitEvent(&pApp->cancelEvent, nn::TimeSpan::FromSeconds(waitTime));

        // ネットワークを破棄し、データの送受信を終了します。
        DestroySocket(pApp, false);
        pApp->clientSocket = -1;

        networkConnection.CancelRequest();
        if (networkConnection.IsAvailable() != true)
        {
            NN_LOG("NIFM ネットワークが切断されました。\n\n");
        }
        else
        {
            result = networkConnection.GetResult();
            NN_LOG("NIFM ネットワークが接続状態で終了しました。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                result.GetDescription(), result.GetInnerValueForDebug());
        }

        // socket ライブラリの使用を終了します。
        result = nn::socket::Finalize();
        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }
    else{
        pApp->isAgingEnable = false;
        result = networkConnection.GetResult();
        NN_LOG("NIFM ネットワーク接続されていません。Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
            result.GetDescription(), result.GetInnerValueForDebug());
    }
}

void MonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::Result result;
    nn::ldn::NetworkInfo networkInfo;
    uint32_t connectCnt, disConnectCnt;

    NN_LOG("Monitorスレッドが開始されました。\n");

    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

    if (app.state == ApplicationState_NetworkCreating)
    {
        nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];

        // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
        while (nn::os::WaitAny(&multiWait) != &cancelHolder)
        {
            // WaitAny では必ず手動でシグナルをクリアしなければなりません。
            nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

            // ネットワークが破棄された場合には監視を終了します。
            nn::ldn::State ldnState = nn::ldn::GetState();
            if (ldnState != nn::ldn::State_AccessPointCreated)
            {
                nn::os::SignalEvent(&app.cancelEvent);
                break;
            }

            // 接続中のステーションを取得します。
            result = nn::ldn::GetNetworkInfo(&networkInfo, updates, nn::ldn::NodeCountMax);
            if (nn::ldn::ResultInvalidState::Includes(result))
            {
                nn::os::SignalEvent(&app.cancelEvent);
                break;
            }
            else if (nn::ldn::ResultDeviceNotAvailable::Includes(result))
            {
                nn::os::SignalEvent(&app.cancelEvent);
                break;
            }
            NN_ABORT_UNLESS(result.IsSuccess());

            connectCnt = disConnectCnt = 0;
            // 接続状態が変化したステーションを検索します。ノード番号 0 は AP なので無視します。
            for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
            {
                const auto& oldNode = app.networkInfo.ldn.nodes[i];
                const auto& newNode = networkInfo.ldn.nodes[i];
                const auto& update = updates[i];
                if (update.stateChange == nn::ldn::NodeStateChange_Disconnect ||
                    update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect)
                {
                    NN_LOG("ステーションが切断しました。\n");
                    NN_LOG("Node ID    : %d\n", oldNode.nodeId);
                    NN_LOG("IP Address : %s\n", ConvertToString(oldNode.ipv4Address));
                    NN_LOG("User Name  : %s\n\n", oldNode.userName);
                    disConnectCnt++;
                }
                if (update.stateChange == nn::ldn::NodeStateChange_Connect ||
                    update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect)
                {
                    NN_LOG("新しいステーションの接続です。\n");
                    NN_LOG("Node ID    : %d\n", newNode.nodeId);
                    NN_LOG("IP Address : %s\n", ConvertToString(newNode.ipv4Address));
                    NN_LOG("User Name  : %s\n\n", newNode.userName);
                    connectCnt++;
                }
            }

            // 接続状態が１台もなく切断が発生していたら監視終了追加
            if (connectCnt == 0 && disConnectCnt > 0)
            {
                nn::os::SignalEvent(&app.cancelEvent);
                break;
            }
            nn::os::LockMutex(&app.mutex);
            std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
            nn::os::UnlockMutex(&app.mutex);
        }
    }
    else if (app.state == ApplicationState_Connecting)
    {
        // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
        while (nn::os::WaitAny(&multiWait) != &cancelHolder)
        {
            // WaitAny では必ず手動でシグナルをクリアしなければなりません。
            nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

            // ネットワークが破棄された場合には監視を終了します。
            nn::ldn::State ldnState = nn::ldn::GetState();
            if (ldnState != nn::ldn::State_StationConnected)
            {
                nn::os::SignalEvent(&app.cancelEvent);
                break;
            }

            // 現在のネットワークの状態を取得します。
            result = nn::ldn::GetNetworkInfo(&networkInfo);
            if (result <= nn::ldn::ResultInvalidState())
            {
                nn::os::SignalEvent(&app.cancelEvent);
                break;
            }
            nn::os::LockMutex(&app.mutex);
            std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
            nn::os::UnlockMutex(&app.mutex);
        }
    }
    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);

    NN_LOG("Monitorスレッドが終了されました。\n");
} // NOLINT(impl/function_size)

void SendThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    char strIpAddress[INET_ADDRSTRLEN];

    if (app.state == ApplicationState_NetworkCreated)
    {
        // ブロードキャスト送信用のアドレスを生成します。
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(UdpPort);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::ldn::BroadcastIpv4Address.raw);
        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
        NN_LOG("Server Send Broadcast IP ADDRESS (%s)\n\n", strIpAddress);

        // キャンセルされるまで一定時間毎にパケットを送信します。
        const auto timeout = nn::TimeSpan::FromMilliSeconds(PacketInterval);
        while (!nn::os::TimedWaitEvent(&app.cancelEvent, timeout))
        {
            // 自身のカウンタをインクリメントします。
            nn::os::LockMutex(&app.mutex);
            app.counter[0] += 1;

            // 送信データを生成します。
            AccesspointData data;
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                data.counter[i] = nn::socket::InetHtonl(app.counter[i]);
            }

            // 現在の状態をログ出力します。
            NN_LOG("[Counter]\n");
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (app.networkInfo.ldn.nodes[i].isConnected)
                {
                    NN_LOG("Node %d : %u\n", i, app.counter[i]);
                }
            }
            NN_LOG("\n");
            nn::os::UnlockMutex(&app.mutex);

            NN_LOG("%s ---> Send Start\n", __FUNCTION__);
            // パケットを送信します。
            auto size = nn::socket::SendTo(
                app.socket,
                &data,
                sizeof(data),
                nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr),
                sizeof(addr));
            NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, size);
        }
    }
    else if (app.state == ApplicationState_Connected)
    {
        // アクセスポイントへの送信用のアドレスを生成します。
        const auto& ap = app.networkInfo.ldn.nodes[0];
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(UdpPort);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(ap.ipv4Address.raw);

        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
        NN_LOG("Client Send Broadcast IP ADDRESS (%s)\n\n", strIpAddress);

        // キャンセルされるまで一定時間毎にパケットを送信します。
        const auto timeout = nn::TimeSpan::FromMilliSeconds(PacketInterval);
        while (!nn::os::TimedWaitEvent(&app.cancelEvent, timeout))
        {
            // 自身のカウンタをインクリメントします。
            nn::os::LockMutex(&app.mutex);
            ++app.myCounter;

            // 送信データを生成します。
            StationData data;
            data.counter = nn::socket::InetHtonl(app.myCounter);
            nn::os::UnlockMutex(&app.mutex);

            NN_LOG("%s ---> Send Start\n", __FUNCTION__);
            // パケットを送信します。
            auto size = nn::socket::SendTo(
                app.socket,
                &data,
                sizeof(data),
                nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr),
                sizeof(addr));
            NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, size);
        }
    }
}

void ReceiveThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    char strIpAddress[INET_ADDRSTRLEN];
    bool isBroadConnect = false;

    if (app.state == ApplicationState_NetworkCreated)
    {
        // 受信用のバッファです。
        StationData data;

        // 任意のステーションからパケットを受信するためのアドレスを生成します。
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(UdpPort);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;

        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
        NN_LOG("Server Receive Broadcast IP ADDRESS (%s)\n\n", strIpAddress);

        // キャンセルされるまでパケットを受信し続けます。
        while (!nn::os::TryWaitEvent(&app.cancelEvent))
        {
            // パケットの受信まで待機します。失敗したらその時点でスレッドを終了します。
            nn::socket::SockLenT length = sizeof(addr);
            NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
            auto size = nn::socket::RecvFrom(
                app.socket,
                &data,
                sizeof(data),
                nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr),
                &length);

            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
            NN_LOG("%s ---> Server Recv End From Address : %s Result : %dByte\n\n", __FUNCTION__, strIpAddress , size);
            if (size == 0 || size == -1)
            {
                break;
            }

            nn::ldn::Ipv4Address from = { nn::socket::InetNtohl(addr.sin_addr.S_addr) };

            // 初回且つ、自身のIPではない場合ブロードキャスト通信開始イベント
            if (isBroadConnect != true && app.ipv4Address != from)
            {
                NN_LOG("%s ---> Server BroadCast Event Sygnal\n\n", __FUNCTION__);
                nn::os::SignalEvent(&app.initEvent);
                isBroadConnect = true;
            }

            // 受信したカウンタを保存します。
            nn::os::LockMutex(&app.mutex);
            for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
            {
                const auto& node = app.networkInfo.ldn.nodes[i];
                if (node.isConnected && node.ipv4Address == from)
                {
                    app.counter[i] = nn::socket::InetNtohl(data.counter);
                }
            }
            nn::os::UnlockMutex(&app.mutex);
        }
    }
    else if (app.state == ApplicationState_Connected)
    {
        // 受信用のバッファです。
        AccesspointData data;

        // アクセスポイントからパケットを受信するためのアドレスを生成します。
        const auto& ap = app.networkInfo.ldn.nodes[0];
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(UdpPort);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(ap.ipv4Address.raw);;

        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
        NN_LOG("Client Receive Broadcast IP ADDRESS (%s)\n\n", strIpAddress);

        // キャンセルされるまでパケットを受信し続けます。
        while (!nn::os::TryWaitEvent(&app.cancelEvent))
        {
            // パケットの受信まで待機します。失敗したらその時点でスレッドを終了します。
            nn::socket::SockLenT length = sizeof(addr);
            NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
            auto size = nn::socket::RecvFrom(
                app.socket,
                &data,
                sizeof(data),
                nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr),
                &length);

            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
            NN_LOG("%s ---> Client Recv End From Address : %s Result : %dByte\n\n", __FUNCTION__, strIpAddress, size);
            if (size == 0 || size == -1)
            {
                break;
            }

            nn::ldn::Ipv4Address from = { nn::socket::InetNtohl(addr.sin_addr.S_addr) };

            // 初回且つ、自身のIPではない場合ブロードキャスト通信開始イベント
            if (isBroadConnect != true && app.ipv4Address != from)
            {
                NN_LOG("%s ---> Client BroadCast Event Sygnal\n\n", __FUNCTION__);
                nn::os::SignalEvent(&app.initEvent);
                isBroadConnect = true;
            }

            // 受信したカウンタを保存します。
            nn::os::LockMutex(&app.mutex);
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                app.counter[i] = nn::socket::InetNtohl(data.counter[i]);
            }

            // 現在の状態をログ出力します。
            NN_LOG("[Counter]\n");
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (app.networkInfo.ldn.nodes[i].isConnected)
                {
                    NN_LOG("Node %d : %u\n", i, app.counter[i]);
                }
            }
            NN_LOG("\n");
            nn::os::UnlockMutex(&app.mutex);
        }
    }
}

void TcpSendThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    // ブロードキャスト送信用のアドレスを生成します。

    if (app.currentMode == ChangeMode::Mode_TestMaster)
    {
        nn::os::WaitEvent(&app.initEvent);
        NN_LOG("Tcp Server Accept Event Sygnal\n\n");
    }
    else
    {
        NN_LOG("Tcp Mode Check %s\n\n", ChangeModeString[app.currentMode]);
    }

    // キャンセルされるまで一定時間毎にパケットを送信します。
    const auto timeout = nn::TimeSpan::FromMilliSeconds(PacketInterval);
    while (!nn::os::TimedWaitEvent(&app.cancelEvent, timeout))
    {
        // 自身のカウンタをインクリメントします。
        nn::os::LockMutex(&app.mutex);
        app.counter[0] += 1;

        // 送信データを生成します。
        AccesspointData data;
        for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
        {
            data.counter[i] = nn::socket::InetHtonl(app.counter[i]);
        }
        nn::os::UnlockMutex(&app.mutex);

        NN_LOG("%s ---> Send Start\n", __FUNCTION__);
        auto sendSize = nn::socket::Send(
            app.clientSocket,
            &data,
            sizeof(data),
            nn::socket::MsgFlag::Msg_None);
        NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, sendSize);

        if (sendSize < 0)
        {
            NN_LOG("Tcp Send Error(%d)\n\n", nn::socket::GetLastError());
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        else
        {
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (app.counter[i] != 0)
                {
                    NN_LOG("Tcp Send Data %d : %lu\n\n", i, app.counter[i]);
                }
            }
        }
    }
}

void TcpServerReceiveThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    // 受信用のバッファです。
    AccesspointData data;

    // パケットを受信するためのアドレスを生成します。
    nn::socket::SockAddrIn addr;

    // クライアント接続時に取得する情報です。
    nn::socket::SockAddrIn clientAddr;
    nn::socket::SockLenT nsaddrLen;
    char strIpAddress[INET_ADDRSTRLEN];

    std::memset(&addr, 0x00, sizeof(nn::socket::SockAddrIn));
    std::memset(&clientAddr, 0x00, sizeof(nn::socket::SockAddrIn));
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(TcpServerPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;

    NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
        app.socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)), 0);

    nn::socket::Listen(app.socket, 0);

    nsaddrLen = sizeof(nn::socket::SockAddrIn);
    NN_LOG("%s ---> Accept Start\n", __FUNCTION__);
    app.clientSocket = nn::socket::Accept(app.socket, reinterpret_cast<nn::socket::SockAddr *>(&clientAddr), &nsaddrLen);
    if (app.clientSocket == -1)
    {
        NN_LOG("TCP Server nn::socket::Accept Error(%d)\n\n", nn::socket::GetLastError());
        return;
    }
    NN_LOG("%s ---> Accept End Socket(%d) \n\n", __FUNCTION__, app.clientSocket);

    nn::socket::InetNtop(nn::socket::Family::Af_Inet, &clientAddr.sin_addr,
        strIpAddress, INET_ADDRSTRLEN);
    NN_LOG("TCP Server Client Connect : %s(%d)\n", strIpAddress, nn::socket::InetNtohs(clientAddr.sin_port));

    nn::os::SignalEvent(&app.initEvent);

    // キャンセルされるまでパケットを受信し続けます。
    while (!nn::os::TryWaitEvent(&app.cancelEvent))
    {
        NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
        auto size = nn::socket::Recv(
            app.clientSocket,
            &data,
            sizeof(data),
            nn::socket::MsgFlag::Msg_None);
        NN_LOG("%s ---> Recv End Result : %dByte\n\n", __FUNCTION__, size);
        if (size == 0 || size == -1)
        {
            NN_LOG("Tcp Server Receive Error(%d)\n\n", nn::socket::GetLastError());
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        else
        {
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (nn::socket::InetHtonl(data.counter[i]) != 0)
                {
                    NN_LOG("Tcp Server Receive Data %d : %lu\n\n", i, nn::socket::InetHtonl(data.counter[i]));
                }
            }
        }
    }
}

void TcpClientReceiveThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    // 受信用のバッファです。
    AccesspointData data;

    // キャンセルされるまでパケットを受信し続けます。
    while (!nn::os::TryWaitEvent(&app.cancelEvent))
    {
        NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
        auto size = nn::socket::Recv(
            app.socket,
            &data,
            sizeof(data),
            nn::socket::MsgFlag::Msg_None);
        NN_LOG("%s ---> Recv End Result : %dByte\n\n", __FUNCTION__, size);
        if (size == 0 || size == -1)
        {
            NN_LOG("Tcp Client Receive Error(%d)\n\n", nn::socket::GetLastError());
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        else
        {
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (nn::socket::InetHtonl(data.counter[i]) != 0)
                {
                    NN_LOG("Tcp Client Receive Data %d : %lu\n\n", i, nn::socket::InetHtonl(data.counter[i]));
                }
            }
        }
    }
}

void NifmSendThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    char strIpAddress[INET_ADDRSTRLEN];
    nn::socket::SockAddrIn addr;
    nn::Result result;

    std::memset(&addr, 0x00, sizeof(nn::socket::SockAddrIn));

    // ブロードキャスト送信用のアドレスを生成します。
    if (app.mode == ChangeMode::Mode_TestMaster)
    {
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(NifmUdpPort);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::ldn::BroadcastIpv4Address.raw);

        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
        NN_LOG("Nifm Server Send Broadcast IP ADDRESS (%s)\n\n", strIpAddress);
    }
    else if (app.mode == Mode_TestClient)
    {
        nn::socket::InAddr ipAddr, subAddr, gateAddr, dns1Addr, dns2Addr;

        result = nn::nifm::GetCurrentIpConfigInfo(&ipAddr, &subAddr, &gateAddr, &dns1Addr, &dns2Addr);
        if (result.IsSuccess() == true)
        {
            addr.sin_family = nn::socket::Family::Af_Inet;
            addr.sin_port = nn::socket::InetHtons(NifmUdpPort);
            addr.sin_addr.S_addr = gateAddr.S_addr;
            //addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::ldn::BroadcastIpv4Address.raw);

            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
            NN_LOG("Nifm Client Send Broadcast IP ADDRESS (%s)\n\n", strIpAddress);
        }
        else
        {
            NN_LOG("Nifm Client GetCurrentIpConfigInfo Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                result.GetDescription() , result.GetInnerValueForDebug());
            nn::os::SignalEvent(&app.cancelEvent);
        }
    }
    else
    {
        NN_LOG("Nifm Send Mode Failler\n\n");
        nn::os::SignalEvent(&app.cancelEvent);
    }

    // キャンセルされるまで一定時間毎にパケットを送信します。
    const auto timeout = nn::TimeSpan::FromMilliSeconds(PacketInterval);
    while (!nn::os::TimedWaitEvent(&app.cancelEvent, timeout))
    {
        // 自身のカウンタをインクリメントします。
        nn::os::LockMutex(&app.mutex);
        app.counter[0] += 1;

        // 送信データを生成します。
        AccesspointData data;
        for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
        {
            data.counter[i] = nn::socket::InetHtonl(app.counter[i]);
        }
        nn::os::UnlockMutex(&app.mutex);

        NN_LOG("%s ---> Send Start\n", __FUNCTION__);
        auto sendSize = nn::socket::SendTo(
            app.socket,
            &data,
            sizeof(data),
            nn::socket::MsgFlag::Msg_None,
            reinterpret_cast<nn::socket::SockAddr*>(&addr),
            sizeof(addr));
        NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, sendSize);

        if (sendSize < 0)
        {
            if (app.mode == ChangeMode::Mode_TestMaster)
            {
                NN_LOG("Nifm Server BraodCast Send Error(%d)\n\n",
                    nn::socket::GetLastError());
            }
            else
            {
                NN_LOG("Nifm Client BraodCast Send Error(%d)\n\n",
                    nn::socket::GetLastError());
            }
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        else
        {
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (app.counter[i] != 0)
                {
                    if (app.mode == ChangeMode::Mode_TestMaster)
                    {
                        NN_LOG("Nifm Server BroadCast Send Data %d : %lu\n\n", i, app.counter[i]);
                    }
                    else
                    {
                        NN_LOG("Nifm Client BroadCast Send Data %d : %lu\n\n", i, app.counter[i]);
                    }
                }
            }
        }
    }
}

void NifmReceiveThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    // 受信用のバッファです。
    AccesspointData data;
    // パケットを受信するためのアドレスを生成します。
    nn::socket::SockAddrIn addr;
    char strIpAddress[INET_ADDRSTRLEN];
    nn::Result result;
    // Client受信設定
    nn::socket::InAddr ipAddr, subAddr, gateAddr, dns1Addr, dns2Addr;

    std::memset(&addr, 0x00, sizeof(nn::socket::SockAddrIn));

    if (app.mode == ChangeMode::Mode_TestMaster)
    {
        // Server受信設定
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(NifmUdpPort);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;

        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
        NN_LOG("Nifm Server Receive Tcp Server IP ADDRESS (%s)\n\n", strIpAddress);
    }
    else if (app.mode == ChangeMode::Mode_TestClient)
    {
        result = nn::nifm::GetCurrentIpConfigInfo(&ipAddr, &subAddr, &gateAddr, &dns1Addr, &dns2Addr);
        if (result.IsSuccess() == true)
        {
            // Client受信設定
            addr.sin_family = nn::socket::Family::Af_Inet;
            addr.sin_port = nn::socket::InetHtons(NifmUdpPort);
            //addr.sin_addr.S_addr = gateAddr.S_addr;
            addr.sin_addr.S_addr = nn::socket::InAddr_Any;

            nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
            NN_LOG("Nifm Client Receive Broadcast IP ADDRESS (%s)\n\n", strIpAddress);
        }
        else
        {
            NN_LOG("Nifm Client GetCurrentIpConfigInfo Error(%d:%d) : 0x%08x\n\n", result.GetModule(),
                result.GetDescription(), result.GetInnerValueForDebug());
            nn::os::SignalEvent(&app.cancelEvent);
        }
    }
    else
    {
        NN_LOG("Nifm Receive Mode Failler\n\n");
        nn::os::SignalEvent(&app.cancelEvent);
    }

    nn::os::SignalEvent(&app.initEvent);

    // キャンセルされるまでパケットを受信し続けます。
    while (!nn::os::TryWaitEvent(&app.cancelEvent))
    {
        nn::socket::SockLenT length = sizeof(addr);
        NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
        auto size = nn::socket::RecvFrom(
            app.socket,
            &data,
            sizeof(data),
            nn::socket::MsgFlag::Msg_None,
            reinterpret_cast<nn::socket::SockAddr*>(&addr),
            &length);

        nn::socket::InetNtop(nn::socket::Family::Af_Inet, &addr.sin_addr, strIpAddress, sizeof(strIpAddress));
        NN_LOG("%s ---> Recv End From Address : %s Result : %dByte\n\n", __FUNCTION__, strIpAddress, size);
        if (size == 0 || size == -1)
        {
            if (app.mode == ChangeMode::Mode_TestMaster)
            {
                NN_LOG("Nifm BroadCast Server Receive Error(%d)\n\n",
                    nn::socket::GetLastError());
            }
            else
            {
                NN_LOG("Nifm BroadCast Client Receive Error(%d)\n\n",
                    nn::socket::GetLastError());
            }
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        else
        {
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (nn::socket::InetHtonl(data.counter[i]) != 0)
                {
                    if (app.mode == ChangeMode::Mode_TestMaster)
                    {
                        NN_LOG("Nifm BroadCast Server Receive Data %d : %lu\n\n", i, nn::socket::InetHtonl(data.counter[i]));
                    }
                    else
                    {
                        NN_LOG("Nifm BroadCast Client Receive Data %d : %lu\n\n", i, nn::socket::InetHtonl(data.counter[i]));
                    }
                }
            }
        }
    }
}

nn::Result Scan(ApplicationResource* pApp, int* pOutScanResultCount) NN_NOEXCEPT
{
    // IntentId が一致する LDN ネットワークのみを探索対象とします。
    int scanChannel[] = { nn::ldn::AutoChannel, 36, 40, 44, 48 };
    nn::ldn::ScanFilter filter = {};
    filter.networkType = nn::ldn::NetworkType_Ldn;
    filter.networkId.intentId = BasicSampleIntentId;
    filter.flag = nn::ldn::ScanFilterFlag_IntentId | nn::ldn::ScanFilterFlag_NetworkType;

    // LDN ネットワークを探索します。
    for (auto channel : scanChannel)
    {
        NNS_LDN_RETURN_IF_FAILED(nn::ldn::Scan(
            g_ScanResult, pOutScanResultCount, nn::ldn::ScanResultCountMax,
            filter, channel));
        if (*pOutScanResultCount > 0)
        {
            break;
        }
    }

    return nn::ResultSuccess();
}

nn::Result Connect(ApplicationResource* pApp, const nn::ldn::NetworkInfo& network) NN_NOEXCEPT
{
    // 接続に必要なセキュリティパラメータを設定します。
    nn::ldn::SecurityConfig security = {};
    nn::ldn::UserConfig     user = {};
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Any);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    std::strncpy(user.userName, "Station", nn::ldn::UserNameBytesMax);
    nn::Result result;


    // LDN ネットワークに接続します。
    for (int i = 0; i < ModeLocalConnectRetryCount; i++)
    {
        result = nn::ldn::Connect(network, security, user, 0, nn::ldn::ConnectOption_None);
        if (result.IsSuccess() == true)
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(ConWaitTime));
        NN_LOG("LDN ネットワーク接続をリトライしています。Result(%d:%d) : 0x%08x\n",
            result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
    }

    NNS_LDN_RETURN_IF_FAILED(result);

    // ネットワークの情報を取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&pApp->networkInfo));

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask));

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, MonitorThread, pApp,
        g_MonitorThreadStack, sizeof(g_MonitorThreadStack),
        MonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void Disconnect(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークから切断します。
    nn::ldn::Disconnect();
}

nn::Result CreateAccessPoint(ApplicationResource* pApp) NN_NOEXCEPT
{
    nn::Result result;

    // アクセスポイントモードとして起動します。
    NN_LOG("%s ---> nn::ldn::OpenAccessPoint Start\n", __FUNCTION__);
    result = nn::ldn::OpenAccessPoint();
    NN_LOG("%s ---> nn::ldn::OpenAccessPoint End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_RETURN_IF_FAILED(result);

    // Advertise で配信するデータを設定します。
    const char message[] = "Welcome to LDN network!";
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::SetAdvertiseData(message, sizeof(message)));

    // 新規に構築する LDN ネットワークのパラメータを設定します。
    nn::ldn::NetworkConfig  network  = {};
    nn::ldn::SecurityConfig security = {};
    nn::ldn::UserConfig     user     = {};
    network.intentId        = BasicSampleIntentId;
    network.channel         = pApp->setChannel;
    network.nodeCountMax    = nn::ldn::NodeCountMax;
    security.securityMode   = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    std::strncpy(user.userName, "AP", nn::ldn::UserNameBytesMax);

    // アクセスポイントとして新規にネットワークを構築します。
    NN_LOG("%s ---> nn::ldn::CreateNetwork Start\n", __FUNCTION__);
    result = nn::ldn::CreateNetwork(network, security, user);
    NN_LOG("%s ---> nn::ldn::CreateNetwork End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_RETURN_IF_FAILED(result);

    // ネットワークの情報を取得します。
    NN_LOG("%s ---> nn::ldn::GetNetworkInfo Start\n", __FUNCTION__);
    result = nn::ldn::GetNetworkInfo(&pApp->networkInfo);
    NN_LOG("%s ---> nn::ldn::GetNetworkInfo End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_RETURN_IF_FAILED(result);

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NN_LOG("%s ---> nn::ldn::GetIpv4Address Start\n", __FUNCTION__);
    result = nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask);
    NN_LOG("%s ---> nn::ldn::GetIpv4Address End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_RETURN_IF_FAILED(result);

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, MonitorThread, pApp,
        g_MonitorThreadStack, sizeof(g_MonitorThreadStack),
        MonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void DestroyAccessPoint(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークを破棄し、 LDN ライブラリの使用を終了します。
    nn::ldn::DestroyNetwork();
    nn::ldn::CloseAccessPoint();
    NN_LOG("LDN MoniterThreadを解放しました\n\n");
}

uint32_t GetIpAddress(const char* interfaceName) NN_NOEXCEPT
{
    int fd_ip;
    struct ifreq ifr;

    /* Create Socket */
    fd_ip = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);

    /* IP v4 */
    ifr.ifr_addr.sa_family = static_cast<sa_family_t>(nn::socket::Family::Af_Inet);

    /* network adapter */
    strncpy(ifr.ifr_name, interfaceName, nn::socket::IfNamSiz - 1);

    nn::socket::Ioctl(fd_ip, static_cast<nn::socket::IoctlCommand>(nn::socket::IoctlCommandPrivate::SiocGIfAddr), &ifr, sizeof(ifr));
    fd_ip = nn::socket::Close(fd_ip);
    NN_LOG("GetIpAddress Socket Close Result : %d\n", fd_ip);

    return ((nn::socket::SockAddrIn *)&ifr.ifr_addr)->sin_addr.S_addr;
}

void CreateSocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    // データの送受信に使用する socket を生成します。
    pApp->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    NN_ABORT_UNLESS_NOT_EQUAL(pApp->socket, -1);

    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;
    NN_LOG("%s ---> nn::socket::Bind Start\n", __FUNCTION__);
    NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
        pApp->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)), 0);
    NN_LOG("%s ---> nn::socket::Bind End\n\n", __FUNCTION__);

    // ブロードキャスト通信を有効化します。
    int isEnabled = 1;
    NN_LOG("%s ---> nn::socket::SetSockOpt SO_BROADCAST Start\n", __FUNCTION__);
    int nResult = nn::socket::SetSockOpt(
        pApp->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));
    if (nResult != 0)
    {
        NN_LOG("nn::socket::SetSockOpt nn::socket::Option::So_Broadcast Error : %d\n\n", nResult);
    }
    NN_LOG("%s ---> nn::socket::SetSockOpt SO_BROADCAST End\n\n", __FUNCTION__);

    // データ送信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->sendThread, SendThread, pApp,
        g_SendThreadStack, sizeof(g_SendThreadStack),
        SendThreadPriority));
    nn::os::StartThread(&pApp->sendThread);

    // データ受信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->receiveThread, ReceiveThread, pApp,
        g_ReceiveThreadStack, sizeof(g_ReceiveThreadStack),
        ReceiveThreadPriority));
    nn::os::StartThread(&pApp->receiveThread);
}

void CreateTcpServerSocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    // データの送受信に使用する socket を生成します。
    pApp->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
    NN_ABORT_UNLESS_NOT_EQUAL(pApp->socket, -1);

    int isEnabled = 1;
    int nResult = nn::socket::SetSockOpt(
        pApp->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_ReuseAddr, &isEnabled, sizeof(isEnabled));
    if (nResult != 0)
    {
        NN_LOG("nn::socket::SetSockOpt nn::socket::Option::So_ReuseAddr Error : %d\n\n", nResult);
    }

    nn::os::ClearEvent(&pApp->initEvent);

    // データ送信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->sendThread, TcpSendThread, pApp,
        g_SendThreadStack, sizeof(g_SendThreadStack),
        SendThreadPriority));
    nn::os::StartThread(&pApp->sendThread);

    // データ受信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->receiveThread, TcpServerReceiveThread, pApp,
        g_ReceiveThreadStack, sizeof(g_ReceiveThreadStack),
        ReceiveThreadPriority));
    nn::os::StartThread(&pApp->receiveThread);
}

void CreateTcpClientSocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    nn::socket::SockAddrIn addr;
    char strIpAddress[INET_ADDRSTRLEN];
    int nResult;

    nn::os::ClearEvent(&pApp->initEvent);

    std::memset(&addr, 0x00, sizeof(nn::socket::SockAddrIn));
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(TcpServerPort);
    addr.sin_addr.S_addr = pApp->ipv4Address.raw;

    sprintf(strIpAddress, "%s", ConvertToString(pApp->ipv4Address));

    NN_LOG("TCP Client 接続先 : %s:%d\n\n", strIpAddress, TcpServerPort);
    nn::socket::InetPton(nn::socket::Family::Af_Inet, strIpAddress, &addr.sin_addr);

    for (int i = 0; i < ClientConRetry; i++)
    {
        // データの送受信に使用する socket を生成します。
        pApp->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        NN_ABORT_UNLESS_NOT_EQUAL(pApp->socket, -1);

        NN_LOG("%s ---> Connect Start %d/%d\n", __FUNCTION__, i + 1, ClientConRetry);
        nResult = nn::socket::Connect(pApp->socket, reinterpret_cast<nn::socket::SockAddr *>(&addr), sizeof(nn::socket::SockAddr));
        NN_LOG("%s ---> Connect End %d/%d Result(%d)\n\n", __FUNCTION__, i + 1, ClientConRetry, nResult);
        if (nResult == 0)
        {
            pApp->clientSocket = pApp->socket;
            break;
        }
        else
        {
            NN_LOG("nn::socket::Connect Error Retry Count : %d\n\n", i + 1);
            TestSocketClose(pApp->socket, &pApp->mutex, false);
            pApp->clientSocket = -1;
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(ConWaitTime));
        }
    }

    if (nResult == 0)
    {
        // データ送信用のスレッドを生成します。
        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &pApp->sendThread, TcpSendThread, pApp,
            g_SendThreadStack, sizeof(g_SendThreadStack),
            SendThreadPriority));
        nn::os::StartThread(&pApp->sendThread);

        // データ受信用のスレッドを生成します。
        NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &pApp->receiveThread, TcpClientReceiveThread, pApp,
            g_ReceiveThreadStack, sizeof(g_ReceiveThreadStack),
            ReceiveThreadPriority));
        nn::os::StartThread(&pApp->receiveThread);
    }
    else
    {
        nn::os::SignalEvent(&pApp->cancelEvent);
        NN_LOG("nn::socket::Connect Error : %d\n\n", nResult);
    }
}

void CreateNifmBroadCastSocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    nn::os::ClearEvent(&pApp->initEvent);

    // データの送受信に使用する socket を生成します。
    pApp->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    NN_ABORT_UNLESS_NOT_EQUAL(pApp->socket, -1);
    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(NifmUdpPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;
    NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
        pApp->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)), 0);

    // ブロードキャスト通信を有効化します。
    int isEnabled = 1;
    int nResult = nn::socket::SetSockOpt(
        pApp->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));
    if (nResult != 0)
    {
        NN_LOG("nn::socket::SetSockOpt nn::socket::Option::So_Broadcast Error : %d\n\n", nResult);
    }

    // データ送信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->sendThread, NifmSendThread, pApp,
        g_SendThreadStack, sizeof(g_SendThreadStack),
        SendThreadPriority));
    nn::os::StartThread(&pApp->sendThread);

    // データ受信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->receiveThread, NifmReceiveThread, pApp,
        g_ReceiveThreadStack, sizeof(g_ReceiveThreadStack),
        ReceiveThreadPriority));
    nn::os::StartThread(&pApp->receiveThread);
}

void DestroySocket(ApplicationResource* pApp, bool isClientClose) NN_NOEXCEPT
{
    nn::os::SignalEvent(&pApp->cancelEvent);

    if (isClientClose)
    {
        TestSocketClose(pApp->clientSocket, &pApp->mutex, true);
    }

    TestSocketClose(pApp->socket, &pApp->mutex, true);

    // スレッドの終了まで待機します。
    NN_LOG("%s ---> Send Thread Wait Start\n", __FUNCTION__);
    nn::os::WaitThread(&pApp->sendThread);
    NN_LOG("%s ---> Send Thread Wait End\n\n", __FUNCTION__);
    NN_LOG("%s ---> Send Thread Destroy Start\n", __FUNCTION__);
    nn::os::DestroyThread(&pApp->sendThread);
    NN_LOG("%s ---> Send Thread Destroy End\n\n", __FUNCTION__);

    NN_LOG("%s ---> Recv Thread Wait Start\n", __FUNCTION__);
    nn::os::WaitThread(&pApp->receiveThread);
    NN_LOG("%s ---> Recv Thread Wait End\n\n", __FUNCTION__);
    NN_LOG("%s ---> Recv Thread Destroy Start\n", __FUNCTION__);
    nn::os::DestroyThread(&pApp->receiveThread);
    NN_LOG("%s ---> Recv Thread Destroy End\n\n", __FUNCTION__);
}

void LdnAccessPoint(void* arg) NN_NOEXCEPT
{
    nn::Result result;

    // LDN ライブラリを初期化します。
    NN_LOG("%s ---> nn::ldn::Initialize Start\n", __FUNCTION__);
    result = nn::ldn::Initialize();
    NN_LOG("%s ---> nn::ldn::Initialize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    if (result.IsSuccess() != true)
    {
        NN_LOG("nn::ldn::Initialize Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
    }

    // LDN ネットワークを構築します。
    auto& app = *static_cast<ApplicationResource*>(arg);

    // LDN ネットワーク構築前にイベント初期化
    nn::os::ClearEvent(&app.initEvent);

    if (result <= nn::ldn::ResultDeviceOccupied())
    {
        // 他の通信機能を使用中のためローカル通信機能を利用できません。
        // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    app.currentMode = Mode_TestMaster;

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    NN_LOG("%s ---> nn::ldn::AttachStateChangeEvent Start\n", __FUNCTION__);
    nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);
    NN_LOG("%s ---> nn::ldn::AttachStateChangeEvent End\n\n", __FUNCTION__);

    NN_LOG("%s ---> CreateAccessPoint Start\n", __FUNCTION__);
    result = CreateAccessPoint(&app);
    if (result.IsSuccess() != true)
    {
        NN_LOG("CreateAccessPoint Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
        return;
    }
    NN_LOG("%s ---> CreateAccessPoint End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("LDN ネットワークの構築に成功しました。\n");
    NN_LOG("Channel    : %d\n", app.networkInfo.common.channel);
    NN_LOG("IP Address : %s/%s\n",
           ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));
    NN_LOG("User Name  : %s\n\n", app.networkInfo.ldn.nodes[0].userName);
    app.state = ApplicationState_NetworkCreated;

    // socket ライブラリを初期化します。
    NN_LOG("%s ---> nn::socket::Initialize Start\n", __FUNCTION__);
    result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::DefaultSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);
    NN_LOG("%s ---> nn::socket::Initialize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // データの送受信を開始します.
    NN_LOG("%s ---> CreateSocket Start\n", __FUNCTION__);
    CreateSocket(&app);
    NN_LOG("%s ---> CreateSocket End\n\n", __FUNCTION__);

    // ブロードキャスト通信は自身のパケット送信を受けてしまうため別IPの初回受信するまで待機
    NN_LOG("%s ---> WaitEvent BroadCast Connect Event Start\n", __FUNCTION__);
    nn::os::WaitEvent(&app.initEvent);
    NN_LOG("%s ---> WaitEvent BroadCast Connect Event End\n\n", __FUNCTION__);

    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
    NN_LOG("%s ---> WaitEvent CancelEvent Start\n", __FUNCTION__);
    nn::os::WaitEvent(&app.cancelEvent);
    NN_LOG("%s ---> WaitEvent CancelEvent End\n\n", __FUNCTION__);

    // LDN ネットワークを破棄し、データの送受信を終了します。
    NN_LOG("%s ---> DestroySocket Start\n", __FUNCTION__);
    DestroySocket(&app, true);
    NN_LOG("%s ---> DestroySocket End\n\n", __FUNCTION__);

    // socket ライブラリの使用を終了します。
    NN_LOG("%s ---> nn::socket::Finalize Start\n", __FUNCTION__);
    result = nn::socket::Finalize();
    NN_LOG("%s ---> nn::socket::Finalize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("%s ---> DestroyAccessPoint Start\n", __FUNCTION__);
    DestroyAccessPoint(&app);
    NN_LOG("%s ---> DestroyAccessPoint End\n\n", __FUNCTION__);
    app.state = ApplicationState_NetworkDestroyed;
    NN_LOG("LDN ネットワークが破棄されました。\n\n");

    nn::os::DestroySystemEvent(&app.stateChangeEvent);
}

void LdnStation(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::Result result;

    // LDN ライブラリを初期化します。
    NN_LOG("%s ---> nn::ldn::Initialize Start\n", __FUNCTION__);
    result = nn::ldn::Initialize();
    NN_LOG("%s ---> nn::ldn::Initialize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    if (result.IsSuccess() != true)
    {
        NN_LOG("nn::ldn::Initialize Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
        return;
    }

    // LDN ステーション起動前にイベント初期化
    nn::os::ClearEvent(&app.initEvent);

    if (result <= nn::ldn::ResultDeviceOccupied())
    {
        // 他の通信機能を使用中のためローカル通信機能を利用できません。
        // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    app.currentMode = Mode_TestClient;

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    NN_LOG("%s ---> nn::ldn::AttachStateChangeEvent Start\n", __FUNCTION__);
    nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);
    NN_LOG("%s ---> nn::ldn::AttachStateChangeEvent End\n\n", __FUNCTION__);

    // ステーションとして起動します。
    NN_LOG("%s ---> nn::ldn::OpenStation Start\n", __FUNCTION__);
    result = nn::ldn::OpenStation();
    NN_LOG("%s ---> nn::ldn::OpenStation End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークを探索します。
    NN_LOG("LDN ネットワークを探索中です・・・。\n\n");
    int scanResultCount;

    // LDN ネットワークを探索します。
    NN_LOG("%s ---> nn::ldn::Scan Start\n", __FUNCTION__);
    for (int i = 0; i < (ModeWaitTime / ModeScanWaitTime); i++)
    {
        result = Scan(&app, &scanResultCount);
        if (result.IsSuccess() != true)
        {
            // 無線スイッチオフなど通信機能を利用できない状態です。
            // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
            NN_LOG("Scan Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
            nn::ldn::CloseStation();
            app.state = ApplicationState_Disconnected;
            return;
        }
        else if (scanResultCount == 0)
        {
            NN_LOG("接続対象のネットワークが見つかりませんでした。\n\n");
        }
        else
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(ModeScanWaitTime));
    }
    NN_LOG("%s ---> nn::ldn::Scan End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークへの接続を試行します。
    NN_LOG("%s ---> nn::ldn::Connect Start\n", __FUNCTION__);
    result = Connect(&app, g_ScanResult[0]);
    NN_LOG("%s ---> nn::ldn::Connect End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    if (result <= nn::ldn::ResultInvalidState() ||
        result <= nn::ldn::ResultConnectionFailed())
    {
        NN_LOG("LDN ネットワークへの接続に失敗しました。\n\n");
        nn::ldn::CloseStation();
        app.state = ApplicationState_Disconnected;
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_LOG("LDN ネットワークへの接続に成功しました。\n");
    NN_LOG("Name (AP)        : %s\n", app.networkInfo.ldn.nodes[0].userName);
    NN_LOG("IP Address (AP)  : %s/%s\n",
        ConvertToString(app.networkInfo.ldn.nodes[0].ipv4Address),
        ConvertToString(app.subnetMask));
    NN_LOG("IP Address (STA) : %s/%s\n\n",
        ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));
    app.state = ApplicationState_Connected;

    // socket ライブラリを初期化します。
    NN_LOG("%s ---> nn::socket::Initialize Start\n", __FUNCTION__);
    result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::DefaultSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);
    NN_LOG("%s ---> nn::socket::Initialize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // データの送受信を開始します.
    NN_LOG("%s ---> CreateSocket Start\n", __FUNCTION__);
    CreateSocket(&app);
    NN_LOG("%s ---> CreateSocket End\n\n", __FUNCTION__);

    // ブロードキャスト通信は自身のパケット送信を受けてしまうため別IPの初回受信するまで待機
    NN_LOG("%s ---> WaitEvent BroadCast Connect Event Start\n", __FUNCTION__);
    nn::os::WaitEvent(&app.initEvent);
    NN_LOG("%s ---> WaitEvent BroadCast Connect Event End\n\n", __FUNCTION__);

    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
    NN_LOG("%s ---> WaitEvent CancelEvent Start\n", __FUNCTION__);
    nn::os::WaitEvent(&app.cancelEvent);
    NN_LOG("%s ---> WaitEvent CancelEvent End\n\n", __FUNCTION__);

    // LDN ネットワークから切断し、データの送受信を終了します。
    NN_LOG("%s ---> DestroySocket Start\n", __FUNCTION__);
    DestroySocket(&app, false);
    NN_LOG("%s ---> DestroySocket End\n\n", __FUNCTION__);

    // socket ライブラリの使用を終了します。
    NN_LOG("%s ---> nn::socket::Finalize Start\n", __FUNCTION__);
    result = nn::socket::Finalize();
    NN_LOG("%s ---> nn::socket::Finalize End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_LOG("%s ---> Disconnect Start\n", __FUNCTION__);
    Disconnect(&app);
    NN_LOG("%s ---> Disconnect End\n\n", __FUNCTION__);

    NN_LOG("%s ---> nn::ldn::CloseStation Start\n", __FUNCTION__);
    result = nn::ldn::CloseStation();
    NN_LOG("%s ---> nn::ldn::CloseStation End Result : 0x%08x\n\n", __FUNCTION__, result.GetInnerValueForDebug());
    app.state = ApplicationState_Disconnected;
    NN_LOG("LDN ネットワークから切断されました。\n\n");

    nn::os::DestroySystemEvent(&app.stateChangeEvent);
} // NOLINT(impl/function_size)

void LdnNetWorkCreaate(ApplicationResource* pApp, nn::os::ThreadType* pldnThread)
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);
    NN_ABORT_UNLESS_NOT_NULL(pldnThread);

    pApp->state = ApplicationState_NetworkCreating;
    NN_LOG("LDN ネットワークの構築を開始します。\n\n");
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        pldnThread, LdnAccessPoint, pApp,
        g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
    nn::os::StartThread(pldnThread);
}

void NetWorkDestroy(nn::os::ThreadType* pldnThread)
{
    NN_ABORT_UNLESS_NOT_NULL(pldnThread);

    nn::os::WaitThread(pldnThread);
    nn::os::DestroyThread(pldnThread);
}

void CreateRandomTestData(ApplicationResource* pApp, ChangeMode testPatternMin, ChangeMode testPatternMax, int changeTimeMin, int changeTimeMax, int agingTimeMin, int agingTimeMax)
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);

    int minMode, maxMode, pos;

    minMode = testPatternMin;
    maxMode = testPatternMax;

    pApp->TestPattern.length = 0;
    pos = 0;
    while (pos < RandamModeCount)
    {
#ifdef SIGLO_42479_RANDOM_NONE
        for (auto testPtn : TraceTestData)
        {
            if (pos < RandamModeCount)
            {
                pApp->TestPattern.mode[pos]         = testPtn.mode;
                pApp->TestPattern.changeTimes[pos]  = testPtn.changeTime;
                pApp->TestPattern.length++;
                pos++;
            }
        }
#else
        pApp->TestPattern.mode[pos] = static_cast<ChangeMode>(GetRandom(minMode, maxMode));
        pApp->TestPattern.changeTimes[pos] = static_cast<ChangeMode>(GetRandom(changeTimeMin, changeTimeMax));
        pApp->TestPattern.length++;

        // 2.4GHz／5GHzのランダムの2セットにするため同じモードを２個１のランダムに変更
        if (pos + 1 < RandamModeCount)
        {
            pApp->TestPattern.mode[pos + 1] = pApp->TestPattern.mode[pos];
            pApp->TestPattern.changeTimes[pos + 1] = pApp->TestPattern.changeTimes[pos];
            pApp->TestPattern.length++;
            pos++;
        }
        pos++;
#endif
    }

    pApp->executeTime = GetRandom(agingTimeMin, agingTimeMax);
}

void CreateTestSocket(ApplicationResource* pApp, nn::socket::SockAddrIn& addr)
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);

    nn::Result result;

    // socket ライブラリを初期化します。
    result = nn::socket::Initialize(
        g_SocketMemoryPoolBuffer,
        nn::socket::DefaultSocketMemoryPoolSize,
        nn::socket::DefaultSocketAllocatorSize,
        nn::socket::DefaultConcurrencyLimit);

    NN_LDNNIFM_RESULT_SUCCESS(result);

    pApp->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    NN_ABORT_UNLESS_NOT_EQUAL(pApp->socket, -1);

    NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
        pApp->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)), 0);

    // ブロードキャスト通信を有効化します。
    int isEnabled = 1;
    nn::socket::SetSockOpt(
        pApp->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));
}

void DestroyTestSocket(ApplicationResource* pApp)
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);

    nn::Result result;

    TestSocketClose(pApp->socket, &pApp->mutex);
    result = nn::socket::Finalize();

    NN_LDNNIFM_RESULT_SUCCESS(result);
}

bool TestPatternSettingUdpServer(ApplicationResource* pApp)
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);

    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;
    size_t bufSize = 1024;
    std::unique_ptr<uint8_t[]> randomBuffer(new uint8_t[bufSize]);
    nn::Result result;
    bool isSetting = false;

    // LDN ライブラリを初期化します。
    result = nn::ldn::Initialize();
    if (result.IsSuccess() != true)
    {
        NN_LOG("nn::ldn::Initialize Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
        return false;
    }

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    nn::ldn::AttachStateChangeEvent(&pApp->stateChangeEvent);

    result = CreateAccessPoint(pApp);
    if (result.IsSuccess() != true)
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        nn::ldn::Finalize();
        nn::os::DestroySystemEvent(&pApp->stateChangeEvent);
        return false;
    }
    NN_LOG("LDN ネットワークの構築に成功しました。\n");
    NN_LOG("Channel    : %d\n", pApp->networkInfo.common.channel);
    NN_LOG("IP Address : %s/%s\n",
        ConvertToString(pApp->ipv4Address), ConvertToString(pApp->subnetMask));
    NN_LOG("User Name  : %s\n\n", pApp->networkInfo.ldn.nodes[0].userName);
    pApp->state = ApplicationState_NetworkCreated;

    // socket ライブラリの初期処理。
    CreateTestSocket(pApp, addr);

    // 任意のステーションからパケットを受信するためのアドレスを生成します。
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;

    // パケットの受信まで待機します。失敗したらその時点でスレッドを終了します。
    nn::socket::SockLenT length = sizeof(addr);
    NN_LOG("%s ---> Recv Start\n", __FUNCTION__);
    auto size = nn::socket::RecvFrom(
        pApp->socket,
        randomBuffer.get(),
        bufSize,
        nn::socket::MsgFlag::Msg_None,
        reinterpret_cast<nn::socket::SockAddr*>(&addr),
        &length);
    NN_LOG("%s ---> Recv End Result : %dByte\n\n", __FUNCTION__, size);

    nn::os::ClearSystemEvent(&pApp->stateChangeEvent);
    nn::os::TimedWaitSystemEvent(&pApp->stateChangeEvent, nn::TimeSpan::FromSeconds(ModeSettingServerWaitTime));

    nn::os::LockMutex(&pApp->mutex);
    if (size <= static_cast<size_t>(sizeof(ChangeModePayload)))
    {
        // ランダムな試験パターンデータ解析
        ChangeModePayload* pPayload = reinterpret_cast<ChangeModePayload*>(randomBuffer.get());

        pApp->executeTime = pPayload->executeTime;
        pApp->TestPattern.length = pPayload->length;
        for (int i = 0; i < pPayload->length; i++)
        {
            pApp->TestPattern.mode[i] = pPayload->mode[i];
            pApp->TestPattern.changeTimes[i] = pPayload->changeTimes[i];
        }
        isSetting = true;
    }
    else
    {
        isSetting = false;
    }
    nn::os::UnlockMutex(&pApp->mutex);

    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // socket ライブラリの終了処理。
    DestroyTestSocket(pApp);

    nn::ldn::Finalize();

    nn::os::DestroySystemEvent(&pApp->stateChangeEvent);
    return isSetting;
}

void TestPatternSettingUdpClient(ApplicationResource* pApp, ChangeMode testPatternMin, ChangeMode testPatternMax, int changeTimeMin, int changeTimeMax, int agingTimeMin, int agingTimeMax)
{
    NN_ABORT_UNLESS_NOT_NULL(pApp);

    std::unique_ptr<ChangeModePayload[]> sendBuffer(new ChangeModePayload);
    nn::Result result;
    NN_ABORT_UNLESS_NOT_NULL(sendBuffer);

    // ランダムな試験パターンデータ作成
    CreateRandomTestData(pApp, testPatternMin, testPatternMax, changeTimeMin, changeTimeMax, agingTimeMin, agingTimeMax);

    sendBuffer.get()->length = pApp->TestPattern.length;
    sendBuffer.get()->executeTime = pApp->executeTime;
    std::memcpy(sendBuffer.get()->mode, pApp->TestPattern.mode, sizeof(pApp->TestPattern.mode));
    std::memcpy(sendBuffer.get()->changeTimes, pApp->TestPattern.changeTimes, sizeof(pApp->TestPattern.changeTimes));

    // ここからLDN・UDP通信処理
    // LDN ライブラリを初期化します。
    result = nn::ldn::Initialize();
    if (result.IsSuccess() != true)
    {
        NN_LOG("nn::ldn::Initialize Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
        return;
    }

    // socket ライブラリを初期化します。

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    nn::ldn::AttachStateChangeEvent(&pApp->stateChangeEvent);

    // ステーションとして起動します。
    result = nn::ldn::OpenStation();
    if (result.IsSuccess() != true)
    {
        NN_LOG("nn::ldn::OpenStation Error Module(%d) Description(%d)\n", result.GetModule(), result.GetDescription());
        nn::ldn::Finalize();
        nn::os::DestroySystemEvent(&pApp->stateChangeEvent);
        return;
    }

    // LDN ネットワークを探索します。
    for (int i = 0; i < (ModeWaitTime / ModeScanWaitTime); i++)
    {
        NN_LOG("LDN ネットワークを探索中です・・・。\n\n");
        int scanResultCount;
        result = Scan(pApp, &scanResultCount);
        if (result.IsSuccess() != true)
        {
            // 無線スイッチオフなど通信機能を利用できない状態です。
            // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
            nn::ldn::CloseStation();
            nn::ldn::Finalize();
            nn::os::DestroySystemEvent(&pApp->stateChangeEvent);
            return;
        }
        else if (scanResultCount == 0)
        {
            NN_LOG("接続対象のネットワークが見つかりませんでした。\n\n");
        }
        else
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(ModeScanWaitTime));
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ネットワークへの接続を試行します。
    result = Connect(pApp, g_ScanResult[0]);
    if (result <= nn::ldn::ResultInvalidState() ||
        result <= nn::ldn::ResultConnectionFailed())
    {
        NN_LOG("LDN ネットワークへの接続に失敗しました。\n\n");
        nn::ldn::CloseStation();
        pApp->state = ApplicationState_Disconnected;
        nn::ldn::Finalize();
        nn::os::DestroySystemEvent(&pApp->stateChangeEvent);
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    pApp->state = ApplicationState_Connected;

    // データの送受信に使用する socket を生成します。
    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;

    // socket ライブラリの初期処理。
    CreateTestSocket(pApp, addr);

    const auto& ap = pApp->networkInfo.ldn.nodes[0];
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InetHtonl(ap.ipv4Address.raw);

    // ランダムな試験パターンデータ送信
    NN_LOG("%s ---> Send Start\n", __FUNCTION__);
    auto size = nn::socket::SendTo(
        pApp->socket,
        sendBuffer.get(),
        sizeof(ChangeModePayload),
        nn::socket::MsgFlag::Msg_None,
        reinterpret_cast<nn::socket::SockAddr*>(&addr),
        sizeof(addr));
    NN_LOG("%s ---> Send End Result(%d)\n\n", __FUNCTION__, size);

    // LDN ネットワークから切断し、データの送受信を終了します。
    // socket ライブラリの使用を終了します。
    DestroyTestSocket(pApp);

    Disconnect(pApp);
    nn::ldn::CloseStation();
    pApp->state = ApplicationState_Disconnected;
    NN_LOG("LDN ネットワークから切断されました。\n\n");


    nn::ldn::Finalize();
    nn::os::DestroySystemEvent(&pApp->stateChangeEvent);
}

void LdnNifmAlternately(void* arg)
{
    NN_ABORT_UNLESS_NOT_NULL(arg);

    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::ThreadType ldnThread;
    nn::os::Tick sysTick;
    uint32_t count = 0;
    int pattern = 0;
    int channelPattern = 0;
    int netLdnPattern = 0;
    int netNifmEtherPattern = 0;
    int netNifmWifiPattern = 0;
    bool isStart = true;

    uint32_t secTime = app.executeTime % 60;
    uint32_t minuteTime = app.executeTime % (60 * 60) / 60;
    uint32_t hourTime = app.executeTime / (60 * 60);

    NN_LOG("LDN AND NIFM試験を開始します。実行予定時間 : %lu:%02lu:%02lu\n\n", hourTime, minuteTime, secTime);

    sysTick = nn::os::GetSystemTick();
    while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetSeconds() <= app.executeTime && app.isAgingEnable == true)
    {
        nn::os::ClearEvent(&app.cancelEvent);

        // ローカル通信のデフォルトチャンネル設定(自動取得)
        app.setChannel = nn::ldn::AutoChannel;
        // ローカル通信／インフラ通信切り替え試験のデフォルトネットワークプロファイル設定(2.4GHz設定)
        app.pNetProfile = nullptr;

        NN_LOG("ローカル通信とインフラ通信が交互切り替えを開始します。%s[%d回目]パターン 時間：%d 秒\n\n", ChangeModeString[app.TestPattern.mode[pattern]], ++count, app.TestPattern.changeTimes[pattern]);
        if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalClient) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalMaster))
        {
            if (NetProfilePattern[netLdnPattern] == TestNetPattern::NetPttern_2_4Ghz)
            {
                channelPattern = GetRandom(0, (sizeof(Channels_2_4Ghz) / sizeof(Channels_2_4Ghz[0]) - 1));
                app.setChannel = Channels_2_4Ghz[channelPattern];
                NN_LOG("ローカル通信 AccessPoint(2.4GHz) Channel(%d) 生成を開始します。\n\n", app.setChannel);
            }
            else
            {
                channelPattern = GetRandom(0, (sizeof(Channels_5Ghz) / sizeof(Channels_5Ghz[0]) - 1));
                app.setChannel = Channels_5Ghz[channelPattern];
                NN_LOG("ローカル通信 AccessPoint(5GHz) Channel(%d) 生成を開始します。\n\n", app.setChannel);
            }
            app.currentMode = Mode_TestMaster;
            ++netLdnPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));

            LdnNetWorkCreaate(&app, &ldnThread);

            nn::os::WaitEvent(&app.initEvent);
            nn::os::ClearEvent(&app.initEvent);

            if (isStart == true)
            {
                isStart = false;
                sysTick = nn::os::GetSystemTick();
            }

            nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(app.TestPattern.changeTimes[pattern]));

            app.state = ApplicationState_NetworkDestroying;
            NN_LOG("LDN ネットワークを破棄します。\n\n");
            nn::os::SignalEvent(&app.cancelEvent);

            NetWorkDestroy(&ldnThread);
            nn::ldn::Finalize();

        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalMaster) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalClient))
        {
            app.state = ApplicationState_Connecting;
            NN_LOG("LDN ネットワークをスキャンして接続します。\n\n");

            app.currentMode = Mode_TestClient;
            NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                &ldnThread, LdnStation, &app,
                g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
            nn::os::StartThread(&ldnThread);

            nn::os::WaitEvent(&app.initEvent);
            nn::os::ClearEvent(&app.initEvent);

            if (isStart == true)
            {
                isStart = false;
                sysTick = nn::os::GetSystemTick();
            }

            nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(app.TestPattern.changeTimes[pattern]));
            nn::os::SignalEvent(&app.cancelEvent);
            NetWorkDestroy(&ldnThread);
            nn::ldn::Finalize();
        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherClient) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherServer))
        {
            nn::os::Tick loopTick;
            loopTick = nn::os::GetSystemTick();

            if (NetProfilePattern[netNifmEtherPattern] == TestNetPattern::NetPttern_2_4Ghz)
            {
                app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&EthernetServerProfileData2_4Ghz);
                NN_LOG("インフラ通信 Ethernetサーバー(2.4GHz)を開始します。\n\n");
            }
            else
            {
                app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&EthernetServerProfileData5Ghz);
                NN_LOG("インフラ通信 Ethernetサーバー(5GHz)を開始します。\n\n");
            }
            app.currentMode = Mode_TestMaster;
            ++netNifmEtherPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));
            EthernetTcpServer(&app, app.TestPattern.changeTimes[pattern]);
        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherServer) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherClient))
        {
            nn::os::Tick loopTick;
            loopTick = nn::os::GetSystemTick();

            if (NetProfilePattern[netNifmEtherPattern] == TestNetPattern::NetPttern_2_4Ghz)
            {
                app.ipv4Address.raw = ConvertIpArrayToBit32(EthernetServerProfileData2_4Ghz.ipSetting.ip.ipAddress);
                app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&EthernetClientProfileData2_4Ghz);
                NN_LOG("インフラ通信 Ethernetクライアント(2.4GHz)を開始します。\n\n");
            }
            else
            {
                app.ipv4Address.raw = ConvertIpArrayToBit32(EthernetServerProfileData5Ghz.ipSetting.ip.ipAddress);
                app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&EthernetClientProfileData5Ghz);
                NN_LOG("インフラ通信 Ethernetクライアント(5GHz)を開始します。\n\n");
            }
            app.currentMode = Mode_TestClient;
            ++netNifmEtherPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));
            EthernetTcpClient(&app, app.TestPattern.changeTimes[pattern]);
        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiClient) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiServer))
        {
            nn::os::Tick loopTick;
            loopTick = nn::os::GetSystemTick();

            if (NetProfilePattern[netNifmWifiPattern] == TestNetPattern::NetPttern_2_4Ghz)
            {
                app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&WifiServerProfileData2_4Ghz);
                NN_LOG("インフラ通信 WiFiサーバー(2.4GHz)を開始します。\n\n");
            }
            else
            {
                app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&WifiServerProfileData5Ghz);
                NN_LOG("インフラ通信 WiFiサーバー(5GHz)を開始します。\n\n");
            }
            app.currentMode = Mode_TestMaster;
            ++netNifmWifiPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));

            WifiServer(&app, app.TestPattern.changeTimes[pattern]);
        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiServer) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiClient))
        {
            nn::os::Tick loopTick;
            loopTick = nn::os::GetSystemTick();

            if (NetProfilePattern[netNifmWifiPattern] == TestNetPattern::NetPttern_2_4Ghz)
            {
                app.ipv4Address.raw = ConvertIpArrayToBit32(WifiServerProfileData2_4Ghz.ipSetting.ip.ipAddress);
                app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&WifiClientProfileData2_4Ghz);
                NN_LOG("インフラ通信 WiFiクライアント(2.4GHz)を開始します。\n\n");
            }
            else
            {
                app.ipv4Address.raw = ConvertIpArrayToBit32(WifiServerProfileData5Ghz.ipSetting.ip.ipAddress);
                app.pNetProfile = const_cast<nn::nifm::NetworkProfileData*>(&WifiClientProfileData5Ghz);
                NN_LOG("インフラ通信 WiFiクライアント(5GHz)を開始します。\n\n");
            }
            app.currentMode = Mode_TestClient;
            ++netNifmWifiPattern %= (sizeof(NetProfilePattern) / sizeof(NetProfilePattern[0]));
            WifiClient(&app, app.TestPattern.changeTimes[pattern]);
        }
        ++pattern %= app.TestPattern.length;
    }

    NN_LOG("LDN AND NIFM試験を終了します。\n\n", hourTime, minuteTime, secTime);
} // NOLINT(impl/function_size)

void LdnAlternately(void* arg)
{
    NN_ABORT_UNLESS_NOT_NULL(arg);

    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::ThreadType ldnThread;
    nn::os::Tick sysTick;
    uint32_t count = 0;
    int pattern = 0;
    bool isStart = true;

    uint32_t secTime = app.executeTime % 60;
    uint32_t minuteTime = app.executeTime % (60 * 60) / 60;
    uint32_t hourTime = app.executeTime / (60 * 60);

    NN_LOG("LDN試験を開始します。実行予定時間 : %lu:%02lu:%02lu\n\n", hourTime, minuteTime, secTime);

    sysTick = nn::os::GetSystemTick();
    while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetSeconds() <= app.executeTime && app.isAgingEnable == true)
    {
        nn::os::ClearEvent(&app.cancelEvent);

        // ローカル通信のデフォルトチャンネル設定(自動取得)
        app.setChannel = nn::ldn::AutoChannel;
        // ローカル通信／インフラ通信切り替え試験のデフォルトネットワークプロファイル設定(2.4GHz設定)
        app.pNetProfile = nullptr;

        NN_LOG("%s[%d回目]パターン 切り替え予定時間：%d 秒\n\n", ChangeModeString[app.TestPattern.mode[pattern]], ++count, app.TestPattern.changeTimes[pattern]);
        if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalClient) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalMaster))
        {
            NN_LOG("ローカル通信 AccessPoint 生成を開始します。\n\n");

            app.currentMode = Mode_TestMaster;

            LdnNetWorkCreaate(&app, &ldnThread);

            nn::os::WaitEvent(&app.initEvent);
            nn::os::ClearEvent(&app.initEvent);

            if (isStart == true)
            {
                isStart = false;
                sysTick = nn::os::GetSystemTick();
            }

            nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(app.TestPattern.changeTimes[pattern]));

            app.state = ApplicationState_NetworkDestroying;
            NN_LOG("LDN ネットワークを破棄します。\n\n");
            nn::os::SignalEvent(&app.cancelEvent);

            NetWorkDestroy(&ldnThread);
            nn::ldn::Finalize();
        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalMaster) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_LdnLocalClient))
        {
            app.state = ApplicationState_Connecting;
            NN_LOG("LDN ネットワークをスキャンして接続します。\n\n");

            app.currentMode = Mode_TestClient;

            NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                &ldnThread, LdnStation, &app,
                g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
            nn::os::StartThread(&ldnThread);

            nn::os::WaitEvent(&app.initEvent);
            nn::os::ClearEvent(&app.initEvent);

            if (isStart == true)
            {
                isStart = false;
                sysTick = nn::os::GetSystemTick();
            }

            nn::os::TimedWaitEvent(&app.cancelEvent, nn::TimeSpan::FromSeconds(app.TestPattern.changeTimes[pattern]));

            nn::os::SignalEvent(&app.cancelEvent);
            NetWorkDestroy(&ldnThread);
            nn::ldn::Finalize();
        }
        else
        {
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(app.TestPattern.changeTimes[pattern] + 2));
        }
        ++pattern %= app.TestPattern.length;
    }
    NN_LOG("LDN試験を終了します。\n\n");
}

void NifmAlternately(void* arg)
{
    NN_ABORT_UNLESS_NOT_NULL(arg);

    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::Tick sysTick;
    uint32_t count = 0;
    int pattern = 0;

    uint32_t secTime = app.executeTime % 60;
    uint32_t minuteTime = app.executeTime % (60 * 60) / 60;
    uint32_t hourTime = app.executeTime / (60 * 60);

    NN_LOG("NIFM試験を開始します。実行予定時間 : %lu:%02lu:%02lu\n\n", hourTime, minuteTime, secTime);

    sysTick = nn::os::GetSystemTick();
    while ((nn::os::GetSystemTick() - sysTick).ToTimeSpan().GetSeconds() <= app.executeTime && app.isAgingEnable == true)
    {
        // ローカル通信／インフラ通信切り替え試験のデフォルトネットワークプロファイル設定(2.4GHz設定)
        app.pNetProfile = nullptr;

        NN_LOG("%s[%d回目]パターン 切り替え予定時間：%d 秒\n\n", ChangeModeString[app.TestPattern.mode[pattern]], ++count, app.TestPattern.changeTimes[pattern]);
        if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherClient) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherServer))
        {
            nn::os::Tick loopTick;
            loopTick = nn::os::GetSystemTick();

            app.currentMode = Mode_TestMaster;

            NN_LOG("インフラ通信 Ethernetサーバーを開始します。\n\n");
            EthernetTcpServer(&app, app.TestPattern.changeTimes[pattern]);
        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherServer) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraEtherClient))
        {
            nn::os::Tick loopTick;
            loopTick = nn::os::GetSystemTick();
            app.ipv4Address.raw = ConvertIpArrayToBit32(EthernetServerProfileData2_4Ghz.ipSetting.ip.ipAddress);

            app.currentMode = Mode_TestClient;

            NN_LOG("インフラ通信 Ethernetクライアントを開始します。\n\n");
            EthernetTcpClient(&app, app.TestPattern.changeTimes[pattern]);
        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiClient) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiServer))
        {
            nn::os::Tick loopTick;
            loopTick = nn::os::GetSystemTick();

            NN_LOG("インフラ通信 WiFiサーバーを開始します。\n\n");
            app.currentMode = Mode_TestMaster;

            WifiServer(&app, app.TestPattern.changeTimes[pattern]);
        }
        else if ((app.mode == ChangeMode::Mode_TestClient && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiServer) ||
            (app.mode == ChangeMode::Mode_TestMaster && app.TestPattern.mode[pattern] == ChangeMode::Mode_InfraWiFiClient))
        {
            nn::os::Tick loopTick;
            loopTick = nn::os::GetSystemTick();
            app.ipv4Address.raw = ConvertIpArrayToBit32(WifiServerProfileData2_4Ghz.ipSetting.ip.ipAddress);

            app.currentMode = Mode_TestClient;

            NN_LOG("インフラ通信 WiFiクライアントを開始します。\n\n");
            WifiClient(&app, app.TestPattern.changeTimes[pattern]);
        }
        ++pattern %= app.TestPattern.length;
    }

    NN_LOG("NIFM試験を終了します。\n\n");
}

void LdnNifmTestInitialize(ApplicationResource* pApp)
{
    auto& app = *static_cast<ApplicationResource*>(pApp);
    nn::Result result;

    // スレッド毎にOS APIのInitialize／Finalizeを呼んでいた箇所をmainに変更(スレッドタイミング調整不要に対応)
    nn::os::InitializeEvent(&app.cancelEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&app.initEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeMutex(&app.mutex, false, 0);
    nn::os::InitializeMutex(&app.apiMutex, false, 0);

    // Socket初期値設定
    app.socket = -1;
    app.clientSocket = -1;

    // ローカル通信のデフォルトチャンネル設定(自動取得)
    app.setChannel = nn::ldn::AutoChannel;
    // ローカル通信／インフラ通信切り替え試験のデフォルトネットワークプロファイル設定(2.4GHz設定)
    app.pNetProfile = nullptr;

    result = nn::nifm::Initialize();
    NN_LDNNIFM_RESULT_SUCCESS(result);

    nn::settings::fwdbg::SetSettingsItemValue("nifm", "is_communication_control_enabled_for_test", &NifmCtrlEnable, sizeof(NifmCtrlEnable));
    nn::nifm::SetWirelessCommunicationEnabledForTest(true);
}

void LdnNifmTestKeyLog()
{
    // サンプルの操作方法を出力します。
    NN_LOG("操作方法 (LDN ネットワークを構築していない場合)\n");
    NN_LOG("    ↑ + ZR ボタンで ローカル通信の１２時間エージング(MASTER)を実施します。\n");
    NN_LOG("    ↓ + ZR ボタンで ローカル通信の１２時間エージング(CLIENT)を実施します。\n");
    NN_LOG("    ↑ +  R ボタンで Ethernetインフラ通信の２４時間エージング(MASTER)を実施します。\n");
    NN_LOG("    ↓ +  R ボタンで Ethernetインフラ通信の２４時間エージング(CLIENT)を実施します。\n");
    NN_LOG("    ↑ +  L ボタンで Wifiインフラの２４時間エージング(MASTER)を実施します。\n");
    NN_LOG("    ↓ +  L ボタンで Wifiインフラの２４時間エージング(CLIENT)を実施します。\n");
    NN_LOG("    ← + ZL ボタンで ローカル通信インフラ通信交互切り替え(MASTER)を15秒～30秒ランダムに切り替え（実行時間７２時間）を実施します。\n");
    NN_LOG("    → + ZL ボタンで ローカル通信インフラ通信交互切り替え(CLIENT)を15秒～30秒ランダムに切り替え（実行時間７２時間）を実施します。\n");
    NN_LOG("    ← + ZR ボタンで ローカル通信でマスターとクライアントの交互切り替え(MASTER)を15秒～30秒ランダムに切り替え（実行時間２４時間）を実施します。\n");
    NN_LOG("    → + ZR ボタンで ローカル通信でマスターとクライアントの交互切り替え(CLIENT)を15秒～30秒ランダムに切り替え（実行時間２４時間）を実施します。\n");
    NN_LOG("    A ボタンで LDN ネットワークを構築します。\n");
    NN_LOG("    X ボタンで LDN ネットワークをスキャンして接続します。\n");
    NN_LOG("    Start ボタンでサンプルを終了します。\n");
    NN_LOG("操作方法 (LDN ネットワークを構築している場合)\n");
    NN_LOG("    B ボタンで LDN ネットワークを破棄／切断します。\n");
    NN_LOG("    Y ボタンで全ステーションを強制的に切断します。\n");
    NN_LOG("\n");
}

void LdnNifmTestFinalize(ApplicationResource* pApp, nn::os::ThreadType* pldnThread)
{
    auto& app = *static_cast<ApplicationResource*>(pApp);
    auto& ldnThread = *static_cast<nn::os::ThreadType*>(pldnThread);
    nn::Result result;

    if (app.mode != ChangeMode::Mode_Init)
    {
        app.isAgingEnable = false;
        app.state = ApplicationState_NetworkDestroying;
        nn::os::SignalEvent(&app.cancelEvent);
        nn::os::WaitThread(&ldnThread);
        nn::os::DestroyThread(&ldnThread);
        app.mode = ChangeMode::Mode_Init;
        app.state = ApplicationState_Initialized;
    }

    nn::os::FinalizeMutex(&app.mutex);
    nn::os::FinalizeMutex(&app.apiMutex);
    nn::os::FinalizeEvent(&app.cancelEvent);
    nn::os::FinalizeEvent(&app.initEvent);
}

bool SetPushKeyDebugPadState(nn::hid::DebugPadButtonSet& setButtons, nn::hid::DebugPadState& pushState)
{
    bool isSetRet = false;
    nn::hid::DebugPadState oldDebugState = pushState;

    pushState.buttons.Reset();
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::Left>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::Left>();
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::Right>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::Right>(true);
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::Up>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::Up>(true);
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::Down>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::Down>(true);
    }

    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::A>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::A>(true);
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::B>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::B>(true);
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::X>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::X>(true);
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::Y>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::Y>(true);
    }

    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::L>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::L>(true);
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::R>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::R>(true);
    }

    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::ZL>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::ZL>(true);
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::ZR>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::ZR>(true);
    }

    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::Start>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::Start>(true);
    }
    if (oldDebugState.buttons.Test<nn::hid::DebugPadButton::Select>() == true)
    {
        isSetRet = true;
        setButtons.Set<nn::hid::DebugPadButton::Select>(true);
    }

    return isSetRet;
}

bool GetConvertNpadKeyToDebugPadState(nn::hid::DebugPadState& retState, nn::hid::NpadButtonSet getButtons)
{
    bool isSetRet = false;

    if (getButtons.Test<nn::hid::NpadButton::Left>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::Left>();
    }
    if (getButtons.Test<nn::hid::NpadButton::Right>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::Right>(true);
    }
    if (getButtons.Test<nn::hid::NpadButton::Up>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::Up>(true);
    }
    if (getButtons.Test<nn::hid::NpadButton::Down>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::Down>(true);
    }

    if (getButtons.Test<nn::hid::NpadButton::A>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::A>(true);
    }
    if (getButtons.Test<nn::hid::NpadButton::B>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::B>(true);
    }
    if (getButtons.Test<nn::hid::NpadButton::X>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::X>(true);
    }
    if (getButtons.Test<nn::hid::NpadButton::Y>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::Y>(true);
    }

    if (getButtons.Test<nn::hid::NpadButton::L>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::L>(true);
    }
    if (getButtons.Test<nn::hid::NpadButton::R>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::R>(true);
    }

    if (getButtons.Test<nn::hid::NpadButton::ZL>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::ZL>(true);
    }
    if (getButtons.Test<nn::hid::NpadButton::ZR>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::ZR>(true);
    }

    if (getButtons.Test<nn::hid::NpadButton::Plus>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::Start>(true);
    }
    if (getButtons.Test<nn::hid::NpadButton::Minus>() == true)
    {
        isSetRet = true;
        retState.buttons.Set<nn::hid::DebugPadButton::Select>(true);
    }

    return isSetRet;
}

bool GetNpadKey(nn::hid::DebugPadState& debugState)
{
    nn::hid::NpadHandheldState handlendState;
    nn::hid::NpadFullKeyState fullKey;
    nn::hid::NpadJoyDualState jyoiDual;

    for (int i = 0; i < NN_ARRAY_SIZE(NpadIds); i++)
    {
        // 現在有効な操作形態(NpadStyleSet)を取得
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(NpadIds[i]);

        // フルキー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleFullKey>())
        {
            //最新のNpadのステートを取得
            nn::hid::GetNpadState(&fullKey, NpadIds[i]);
            if (GetConvertNpadKeyToDebugPadState(debugState, jyoiDual.buttons) == true)
            {
                return true;
            }
        }
        // Joy-Con 操作が有効な場合
        if (style.Test<nn::hid::NpadStyleJoyDual>())
        {
            // 最新のNpadのステートを取得
            nn::hid::GetNpadState(&jyoiDual, NpadIds[i]);

            if (GetConvertNpadKeyToDebugPadState(debugState, jyoiDual.buttons) == true)
            {
                return true;
            }
        }
        // 携帯機コントローラー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleHandheld>())
        {
            // 最新のNpadのステートを取得
            nn::hid::GetNpadState(&handlendState, NpadIds[i]);

            if (GetConvertNpadKeyToDebugPadState(debugState, handlendState.buttons) == true)
            {
                return true;
            }
        }
    }

    return false;
}

extern "C" void nnMain()
{
    nn::Result result;

    // デバッグパッドから入力を取得するために HID ライブラリを取得します。
    nn::hid::InitializeDebugPad();

    // ジョイコンからの入力取得に対応（EDEV対応）
    nn::hid::InitializeNpad();

    // 使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask |
        nn::hid::NpadStyleJoyDual::Mask |
        nn::hid::NpadStyleHandheld::Mask);

    // 使用する Npad を設定
    nn::hid::SetSupportedNpadIdType(NpadIds, NN_ARRAY_SIZE(NpadIds));
    // アプリケーションが使用するリソースを初期化します。
    ApplicationResource app;
    std::memset(&app, 0, sizeof(app));

    // メインループです。
    nn::hid::DebugPadState previousPadState = { };
    nn::hid::DebugPadState padState = { };
    nn::hid::DebugPadButtonSet trigger = { };
    nn::os::ThreadType ldnThread;

    LdnNifmTestKeyLog();
    LdnNifmTestInitialize(&app);

    for (;;)
    {
        // Nパッドの入力を取得します。
        if (GetNpadKey(padState) != true)
        {
            // デバッグパッドの入力を取得します。
            nn::hid::GetDebugPadState(&padState);
        }

        // トリガー入力を取得します。
        previousPadState = padState;
        SetPushKeyDebugPadState(padState.buttons, previousPadState);
        trigger = (~previousPadState.buttons) & padState.buttons;
        previousPadState = padState;

        // 各状態とボタン入力に応じた処理です。
        if (app.state == ApplicationState_Initialized || app.mode != ChangeMode::Mode_Init)
        {
            // ボタン組み合わせするためSTARTを先に変更
            if (trigger.Test<nn::hid::DebugPadButton::Start>())
            {
                NN_LOG("入力を受け付けました。サンプルを終了します。\n\n");
                break;
            }

            if (app.mode != ChangeMode::Mode_Init)
            {
                if (trigger.Test<nn::hid::DebugPadButton::B>())
                {
                    app.isAgingEnable = false;
                    app.state = ApplicationState_NetworkDestroying;
                    NN_LOG("入力を受け付けました。 LDN ネットワークを破棄します。\n\n");
                    nn::os::SignalEvent(&app.cancelEvent);
                    TestSocketClose(app.socket, &app.mutex, true);
                    nn::os::WaitThread(&ldnThread);
                    nn::os::DestroyThread(&ldnThread);
                    app.mode = ChangeMode::Mode_Init;
                    app.currentMode = ChangeMode::Mode_Init;
                    app.state = ApplicationState_Initialized;
                }
            }
            else
            {
                // ボタン組み合わせするため十字キー条件を先頭
                if (trigger.Test<nn::hid::DebugPadButton::Up>() == true && trigger.Test<nn::hid::DebugPadButton::ZR>() == true)
                {
                    NN_LOG("入力を受け付けました。ローカル通信の１２時間エージング(MASTER)を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestMaster;

                    if (TestPatternSettingUdpServer(&app) != true)
                    {
                        NN_LOG("LDN 初期試験データ取得に失敗しました。\n\n");
                        app.isAgingEnable = false;
                        app.mode = ChangeMode::Mode_Init;
                        app.currentMode = ChangeMode::Mode_Init;
                        app.state = ApplicationState_Initialized;
                        continue;
                    }

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Down>() == true && trigger.Test<nn::hid::DebugPadButton::ZR>() == true)
                {
                    NN_LOG("入力を受け付けました。ローカル通信の１２時間エージング(CLIENT)を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestClient;
                    TestPatternSettingUdpClient(&app, Mode_LdnLocalMaster, Mode_LdnLocalMaster, ModeAgingMinTime, ModeAgingMaxTime, ModeAgingMinTime, ModeAgingMaxTime);

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Left>() == true && trigger.Test<nn::hid::DebugPadButton::ZR>() == true)
                {
                    NN_LOG("入力を受け付けました。ローカル通信でマスターとクライアントの交互切り替え(MASTER)を15秒～30秒ランダムに切り替え（実行時間２４時間）を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestMaster;
                    if (TestPatternSettingUdpServer(&app) != true)
                    {
                        NN_LOG("LDN 初期試験データ取得に失敗しました。\n\n");
                        app.isAgingEnable = false;
                        app.mode = ChangeMode::Mode_Init;
                        app.state = ApplicationState_Initialized;
                        continue;
                    }

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Right>() == true && trigger.Test<nn::hid::DebugPadButton::ZR>() == true)
                {
                    NN_LOG("入力を受け付けました。 ローカル通信でマスターとクライアントの交互切り替え(CLIENT)を15秒～30秒ランダムに切り替え（実行時間２４時間）を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestClient;
                    TestPatternSettingUdpClient(&app, Mode_LdnLocalMaster, Mode_LdnLocalClient, ModeChangeMinTime, ModeChangeMaxTime, ModeLocalAgingMinTime, ModeLocalAgingMaxTime);

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Left>() == true && trigger.Test<nn::hid::DebugPadButton::ZL>() == true)
                {
                    NN_LOG("入力を受け付けました。 ローカル通信インフラ通信交互切り替え(MASTER)を15秒～30秒ランダムに切り替え（実行時間７２時間）を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestMaster;
                    if (TestPatternSettingUdpServer(&app) != true)
                    {
                        NN_LOG("LDN 初期試験データ取得に失敗しました。\n\n");
                        app.isAgingEnable = false;
                        app.mode = ChangeMode::Mode_Init;
                        app.state = ApplicationState_Initialized;
                        continue;
                    }

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnNifmAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Right>() == true && trigger.Test<nn::hid::DebugPadButton::ZL>() == true)
                {
                    NN_LOG("入力を受け付けました。 ローカル通信インフラ通信交互切り替え(CLIENT)を15秒～30秒ランダムに切り替え（実行時間７２時間）を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestClient;

                    TestPatternSettingUdpClient(&app, Mode_InfraEtherServer, Mode_LdnLocalClient, ModeChangeMinTime, ModeChangeMaxTime, ModeLocalInfraAgingMinTime, ModeLocalInfraAgingMaxTime);

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnNifmAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Up>() == true && trigger.Test<nn::hid::DebugPadButton::R>())
                {
                    NN_LOG("入力を受け付けました。 Ethernet インフラ通信の２４時間エージング(Master)を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestMaster;
                    if (TestPatternSettingUdpServer(&app) != true)
                    {
                        NN_LOG("LDN 初期試験データ取得に失敗しました。\n\n");
                        app.isAgingEnable = false;
                        app.mode = ChangeMode::Mode_Init;
                        app.state = ApplicationState_Initialized;
                        continue;
                    }

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, NifmAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Down>() == true && trigger.Test<nn::hid::DebugPadButton::R>())
                {
                    NN_LOG("入力を受け付けました。 Ethernet インフラ通信の２４時間エージング(CLIENT)を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestClient;
                    TestPatternSettingUdpClient(&app, Mode_InfraEtherServer, Mode_InfraEtherServer, ModeInfraEtherAgingMinTime, ModeInfraEtherAgingMaxTime, ModeAgingMinTime, ModeInfraEtherAgingMaxTime);

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, NifmAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Up>() == true && trigger.Test<nn::hid::DebugPadButton::L>())
                {
                    NN_LOG("入力を受け付けました。 Wifi インフラ通信の２４時間エージング(Master)を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestMaster;
                    if (TestPatternSettingUdpServer(&app) != true)
                    {
                        NN_LOG("LDN 初期試験データ取得に失敗しました。\n\n");
                        app.isAgingEnable = false;
                        app.mode = ChangeMode::Mode_Init;
                        app.state = ApplicationState_Initialized;
                        continue;
                    }

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, NifmAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::Down>() == true && trigger.Test<nn::hid::DebugPadButton::L>())
                {
                    NN_LOG("入力を受け付けました。 Wifi インフラ通信の２４時間エージング(CLIENT)を実施します。\n\n");
                    app.isAgingEnable = true;
                    app.mode = ChangeMode::Mode_TestClient;
                    TestPatternSettingUdpClient(&app, Mode_InfraWiFiServer, Mode_InfraEtherServer, ModeInfraWiFiAgingMinTime, ModeInfraWiFiAgingMaxTime, ModeInfraWiFiAgingMinTime, ModeInfraWiFiAgingMaxTime);

                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, NifmAlternately, &app,
                        g_LdnAgingThreadStack, sizeof(g_LdnAgingThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::A>())
                {
                    app.state = ApplicationState_NetworkCreating;
                    NN_LOG("入力を受け付けました。 LDN ネットワークの構築を開始します。\n\n");
                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnAccessPoint, &app,
                        g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
                else if (trigger.Test<nn::hid::DebugPadButton::X>())
                {
                    app.state = ApplicationState_Connecting;
                    NN_LOG("入力を受け付けました。 LDN ネットワークをスキャンして接続します。\n\n");
                    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                        &ldnThread, LdnStation, &app,
                        g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
                    nn::os::StartThread(&ldnThread);
                }
            }
        }
        else if (app.state == ApplicationState_NetworkCreated)
        {
            if (trigger.Test<nn::hid::DebugPadButton::B>())
            {
                app.state = ApplicationState_NetworkDestroying;
                NN_LOG("入力を受け付けました。 LDN ネットワークを破棄します。\n\n");
                nn::os::SignalEvent(&app.cancelEvent);
            }
            else if (trigger.Test<nn::hid::DebugPadButton::Y>())
            {
                nn::os::LockMutex(&app.mutex);
                if (0 < app.networkInfo.ldn.nodeCount)
                {
                    NN_LOG("入力を受け付けました。全ステーションを切断します。\n\n");
                    for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
                    {
                        const auto& node = app.networkInfo.ldn.nodes[i];
                        if (node.isConnected)
                        {
                            nn::ldn::Reject(node.ipv4Address);
                        }
                    }
                    nn::ldn::GetNetworkInfo(&app.networkInfo);
                }
                nn::os::UnlockMutex(&app.mutex);
            }
        }
        else if (app.state == ApplicationState_Connected)
        {
            if (trigger.Test<nn::hid::DebugPadButton::B>())
            {
                app.state = ApplicationState_Disconnecting;
                NN_LOG("入力を受け付けました。 LDN ネットワークから切断します。\n\n");
                nn::os::SignalEvent(&app.cancelEvent);
            }
        }
        else if (app.state == ApplicationState_NetworkDestroyed || app.state == ApplicationState_Disconnected)
        {
            app.state = ApplicationState_Initialized;
            nn::os::WaitThread(&ldnThread);
            nn::os::DestroyThread(&ldnThread);

            // LDN ライブラリの使用を終了します。
            nn::ldn::Finalize();
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    LdnNifmTestFinalize(&app, &ldnThread);
} // NOLINT(impl/function_size)
