﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ldn/ldn_Ipv4Address.h>
#include <nn/ldn/ldn_MacAddress.h>
#include <nn/ldn/ldn_NetworkId.h>
#include <nn/ldn/ldn_Ssid.h>
#include <nn/ldn/detail/Utility/ldn_Stringize.h>
#include <nnt.h>

namespace
{

} // namespace <unnamed>

TEST(Stringize, IntentId)
{
    nn::ldn::IntentId intentId = nn::ldn::MakeIntentId(0x0123456789ABCDEF, 0x09AF);
    NN_LDN_STRINGIZE_HELPER(helper, nn::ldn::detail::StringizedIntentIdLength);
    ASSERT_STREQ("0123456789abcdef-0000-09af-00000000", helper.ToString(intentId));
}

TEST(Stringize, SessionId)
{
    nn::ldn::SessionId sessionId = {{
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
    }};
    NN_LDN_STRINGIZE_HELPER(helper, nn::ldn::detail::StringizedSessionIdLength);
    ASSERT_STREQ("0001020304050607f8f9fafbfcfdfeff", helper.ToString(sessionId));
}

TEST(Stringize, NetworkId)
{
    nn::ldn::IntentId intentId = nn::ldn::MakeIntentId(0x0123456789ABCDEF, 0x09AF);
    nn::ldn::SessionId sessionId = {{
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF
    }};
    nn::ldn::NetworkId networkId = { intentId, sessionId };
    NN_LDN_STRINGIZE_HELPER(helper, nn::ldn::detail::StringizedNetworkIdLength);
    ASSERT_STREQ("0123456789abcdef-0000-09af-00000000 0001020304050607f8f9fafbfcfdfeff",
                 helper.ToString(networkId));
}

TEST(Stringize, Ipv4Address)
{
    NN_LDN_STRINGIZE_HELPER(helper, nn::ldn::detail::StringizedIpv4AddressLengthMax);
    {
        nn::ldn::Ipv4Address ipv4Address = nn::ldn::MakeIpv4Address(0, 0, 0, 0);
        ASSERT_STREQ("0.0.0.0", helper.ToString(ipv4Address));
    }
    {
        nn::ldn::Ipv4Address ipv4Address = nn::ldn::MakeIpv4Address(255, 255, 255, 255);
        ASSERT_STREQ("255.255.255.255", helper.ToString(ipv4Address));
    }
    {
        nn::ldn::Ipv4Address ipv4Address = nn::ldn::MakeIpv4Address(192, 168, 10, 2);
        ASSERT_STREQ("192.168.10.2", helper.ToString(ipv4Address));
    }
}

TEST(Stringize, MacAddress)
{
    NN_LDN_STRINGIZE_HELPER(helper, nn::ldn::detail::StringizedMacAddressLength);
    {
        nn::ldn::MacAddress macAddress = nn::ldn::MakeMacAddress(0x01, 0x23, 0x45, 0x67, 0x89, 0xAB);
        ASSERT_STREQ("01-23-45-67-89-ab", helper.ToString(macAddress));
    }
    {
        nn::ldn::MacAddress macAddress = nn::ldn::MakeMacAddress(0xCD, 0xEF, 0x00, 0x55, 0xAA, 0xFF);
        ASSERT_STREQ("cd-ef-00-55-aa-ff", helper.ToString(macAddress));
    }
}

TEST(Stringize, Ssid)
{
    NN_LDN_STRINGIZE_HELPER(helper, nn::ldn::detail::StringizedSsidLengthMax);
    {
        nn::ldn::Ssid ssid = nn::ldn::MakeSsid("");
        ASSERT_STREQ("", helper.ToString(ssid));
    }
    {
        nn::ldn::Ssid ssid = nn::ldn::MakeSsid(" !\"#$%&'()*+,-./0123456789:;<=>");
        ASSERT_STREQ(" !\"#$%&'()*+,-./0123456789:;<=>", helper.ToString(ssid));
    }
    {
        nn::ldn::Ssid ssid = nn::ldn::MakeSsid("?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^");
        ASSERT_STREQ("?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]^", helper.ToString(ssid));
    }
    {
        nn::ldn::Ssid ssid = nn::ldn::MakeSsid("_`abcdefghijklmnopqrstuvwxyz{|}~");
        ASSERT_STREQ("_`abcdefghijklmnopqrstuvwxyz{|}~", helper.ToString(ssid));
    }
    {
        nn::ldn::Ssid ssid;
        ssid.raw[0] = 0x1F;
        ssid.length = 1;
        ASSERT_STREQ("", helper.ToString(ssid));
    }
    {
        nn::ldn::Ssid ssid;
        ssid.raw[0] = 0x7F;
        ssid.length = 1;
        ASSERT_STREQ("", helper.ToString(ssid));
    }
}
