﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <limits>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ldn/detail/Utility/ldn_Random.h>
#include <nn/ldn/detail/Utility/ldn_SecureRandom.h>
#include <nnt.h>

namespace
{
    // テストに使用する乱数列の長さです。
    const size_t RandomBits = 20000;
    const size_t RandomBytes = RandomBits / 8;
    uint8_t g_Random[RandomBytes + sizeof(uint64_t)];

    // 検定の繰り返し回数です。
    const int RepeatCount = 3;

    // Monobit Test のパラメータです。
    const int MonobitTestThresholdMin = 9725;
    const int MonobitTestThresholdMax = 10275;

    // Poker Test のパラメータです。
    const float PokerTestThresholdMin = 2.16f;
    const float PokerTestThresholdMax = 46.17f;

    // Run Test のパラメータです。
    const int RunLengthThreshold = 26;

    // 乱数列を生成します。
    template <typename T>
    void GenerateRandom(void* buffer, size_t size) NN_NOEXCEPT
    {
        for (size_t i = 0; i < size; i += sizeof(T))
        {
            void* p = static_cast<char*>(buffer) + i;
            *static_cast<T*>(p) = nn::ldn::detail::Random<T>();
        }
    }

    // 1 にセットされているビットの数をカウントします。
    int CalculateSetBitCount(uint32_t word) NN_NOEXCEPT
    {
        word = (word & UINT32_C(0x55555555)) + (word >> 1 & UINT32_C(0x55555555));
        word = (word & UINT32_C(0x33333333)) + (word >> 2 & UINT32_C(0x33333333));
        word = (word & UINT32_C(0x0f0f0f0f)) + (word >> 4 & UINT32_C(0x0f0f0f0f));
        word = (word & UINT32_C(0x00ff00ff)) + (word >> 8 & UINT32_C(0x00ff00ff));
        word = (word & UINT32_C(0x0000ffff)) + (word >>16 & UINT32_C(0x0000ffff));
        return static_cast<int>(word);
    }

    // モノビットテストです。1 になっているビットを数えます。
    bool MonobitTest(const void* random, size_t randomSize) NN_NOEXCEPT
    {
        int monobit = 0;
        for (size_t i = 0; i < randomSize / sizeof(uint32_t); ++i)
        {
            auto word = static_cast<const uint32_t*>(random)[i];
            monobit += CalculateSetBitCount(word);
        }
        return MonobitTestThresholdMin < monobit && monobit < MonobitTestThresholdMax;
    }

    // ポーカーテストです。
    bool PokerTest(const void* random, size_t randomSize) NN_NOEXCEPT
    {
        int f[16] = { };
        const int blockCount = RandomBits / 4;
        for (size_t i = 0; i < randomSize; ++i)
        {
            auto byte = reinterpret_cast<const uint8_t*>(random)[i];
            int high = (byte >> 4) & UINT8_C(0x0F);
            ++f[high];
            int low  = byte & UINT8_C(0x0F);
            ++f[low];
        }
        int64_t sum = 0;
        for (int i = 0; i < 16; ++i)
        {
            sum += f[i] * f[i];
        }
        float value = 16.0f * sum / blockCount - blockCount;
        return PokerTestThresholdMin < value && value < PokerTestThresholdMax;
    }

    // ランデストです。1 になっているビットがどのくらい連続して存在するかを確認します。
    bool RunTest(const void* random, size_t randomSize) NN_NOEXCEPT
    {
        int length = 0;
        int f[RunLengthThreshold] = { };
        for (size_t pos = 0; pos < randomSize * 8; ++pos)
        {
            auto byte = reinterpret_cast<const uint8_t*>(random)[pos / 8];
            bool isOne = (byte & (1 << (7 - pos % 8))) != 0;
            if (isOne)
            {
                ++length;
            }
            else if (RunLengthThreshold <= length)
            {
                return false;
            }
            else if (0 < length)
            {
                ++f[length];
                length = 0;
            }
        }
        int rest = 0;
        for (int i = 6; i < RunLengthThreshold; ++i)
        {
            rest += f[i];
        }
        return 2315 <= f[1] && f[1] <= 2685 &&
               1114 <= f[2] && f[2] <= 1386 &&
               527 <= f[3] && f[3] <= 723 &&
               240 <= f[4] && f[4] <= 384 &&
               103 <= f[5] && f[5] <= 209 &&
               103 <= rest && rest <= 209;
    }
};

//
// ランダム性のテストです。
//
TEST(Random, RandomnessBit8)
{
    // 乱数列を生成します。
    GenerateRandom<uint8_t>(g_Random, sizeof(g_Random));

    // 検定の結果です。
    bool isMonobitTestPassed = false;
    bool isPokerTestPassed = false;
    bool isRunTestPassed =false;

    // 検定の性質上、稀に失敗することがあるため、複数回テストを実行します。
    for (int i = 0; i < RepeatCount; ++i)
    {
        // 検定を実施します。
        isMonobitTestPassed |= MonobitTest(g_Random, RandomBytes);
        isPokerTestPassed |= PokerTest(g_Random, RandomBytes);
        isRunTestPassed |= RunTest(g_Random, RandomBytes);
    }

    // 検定の結果を確認します。
    ASSERT_TRUE(isMonobitTestPassed);
    ASSERT_TRUE(isPokerTestPassed);
    ASSERT_TRUE(isRunTestPassed);
}

//
// 16 ビット単位の乱数生成です。
//
TEST(Random, RandomnessBit16)
{
    // 乱数列を生成します。
    GenerateRandom<uint16_t>(g_Random, sizeof(g_Random));

    // 検定の結果です。
    bool isMonobitTestPassed = false;
    bool isPokerTestPassed = false;
    bool isRunTestPassed =false;

    // 検定の性質上、稀に失敗することがあるため、複数回テストを実行します。
    for (int i = 0; i < RepeatCount; ++i)
    {
        // 検定を実施します。
        isMonobitTestPassed |= MonobitTest(g_Random, RandomBytes);
        isPokerTestPassed |= PokerTest(g_Random, RandomBytes);
        isRunTestPassed |= RunTest(g_Random, RandomBytes);
    }

    // 検定の結果を確認します。
    ASSERT_TRUE(isMonobitTestPassed);
    ASSERT_TRUE(isPokerTestPassed);
    ASSERT_TRUE(isRunTestPassed);
}

//
// 32 ビット単位の乱数生成です。
//
TEST(Random, RandomnessBit32)
{
    // 乱数列を生成します。
    GenerateRandom<uint32_t>(g_Random, sizeof(g_Random));

    // 検定の結果です。
    bool isMonobitTestPassed = false;
    bool isPokerTestPassed = false;
    bool isRunTestPassed =false;

    // 検定の性質上、稀に失敗することがあるため、複数回テストを実行します。
    for (int i = 0; i < RepeatCount; ++i)
    {
        // 検定を実施します。
        isMonobitTestPassed |= MonobitTest(g_Random, RandomBytes);
        isPokerTestPassed |= PokerTest(g_Random, RandomBytes);
        isRunTestPassed |= RunTest(g_Random, RandomBytes);
    }

    // 検定の結果を確認します。
    ASSERT_TRUE(isMonobitTestPassed);
    ASSERT_TRUE(isPokerTestPassed);
    ASSERT_TRUE(isRunTestPassed);
}

//
// 64 ビット単位の乱数生成です。
//
TEST(Random, RandomnessBit64)
{
    // 乱数列を生成します。
    GenerateRandom<uint64_t>(g_Random, sizeof(g_Random));

    // 検定の結果です。
    bool isMonobitTestPassed = false;
    bool isPokerTestPassed = false;
    bool isRunTestPassed =false;

    // 検定の性質上、稀に失敗することがあるため、複数回テストを実行します。
    for (int i = 0; i < RepeatCount; ++i)
    {
        // 検定を実施します。
        isMonobitTestPassed |= MonobitTest(g_Random, RandomBytes);
        isPokerTestPassed |= PokerTest(g_Random, RandomBytes);
        isRunTestPassed |= RunTest(g_Random, RandomBytes);
    }

    // 検定の結果を確認します。
    ASSERT_TRUE(isMonobitTestPassed);
    ASSERT_TRUE(isPokerTestPassed);
    ASSERT_TRUE(isRunTestPassed);
}

//
// 乱数生成です。
//
TEST(FillRandom, Randomness)
{
    // 乱数列を生成します。
    nn::ldn::detail::FillRandom(g_Random, RandomBytes);

    // 検定の結果です。
    bool isMonobitTestPassed = false;
    bool isPokerTestPassed = false;
    bool isRunTestPassed =false;

    // 検定の性質上、稀に失敗することがあるため、複数回テストを実行します。
    for (int i = 0; i < RepeatCount; ++i)
    {
        // 検定を実施します。
        isMonobitTestPassed |= MonobitTest(g_Random, RandomBytes);
        isPokerTestPassed |= PokerTest(g_Random, RandomBytes);
        isRunTestPassed |= RunTest(g_Random, RandomBytes);
    }

    // 検定の結果を確認します。
    ASSERT_TRUE(isMonobitTestPassed);
    ASSERT_TRUE(isPokerTestPassed);
    ASSERT_TRUE(isRunTestPassed);
}

//
// 先頭がアライメントされていなくても正しく乱数を生成できることを検証します。
//
TEST(FillRandom, Preceding)
{
    const size_t dataSize = 16;
    const int dataCount = 8;

    // オフセットを変更しながらテストを行います。
    for (size_t offset = 0; offset < dataSize; ++offset)
    {
        // 複数回乱数を生成します。
        uint8_t data[dataCount][dataSize] = { };
        for (int i = 0; i < dataCount; ++i)
        {
            nn::ldn::detail::FillRandom(data[i] + offset, sizeof(data[i]) - offset);
        }

        // 全バイトについて、3 種類以上のデータが含まれていれば成功とします。
        for (size_t i = offset; i < dataSize; ++i)
        {
            int8_t counter[256] = { };
            for (int j = 0; j < dataCount; ++j)
            {
                ++counter[data[j][i]];
            }

            int count = 0;
            for (int j = 0; j < 256; ++j)
            {
                if (counter[j] != 0)
                {
                    ++count;
                }
            }
            ASSERT_GE(count, 3);
        }
    }
}

//
// 末尾がアライメントされていなくても正しく乱数を生成できることを検証します。
//
TEST(FillRandom, Following)
{
    const size_t dataSize = 16;
    const int dataCount = 8;

    // サイズを変更しながらテストを行います。
    for (size_t size = 1; size < dataSize; ++size)
    {
        // 複数回乱数を生成します。
        uint8_t data[dataCount][dataSize] = { };
        for (int i = 0; i < dataCount; ++i)
        {
            nn::ldn::detail::FillRandom(data[i], size);
        }

        // 全バイトについて、3 種類以上のデータが含まれていれば成功とします。
        for (size_t i = 0; i < size; ++i)
        {
            int8_t counter[256] = { };
            for (int j = 0; j < dataCount; ++j)
            {
                ++counter[data[j][i]];
            }

            int count = 0;
            for (int j = 0; j < 256; ++j)
            {
                if (counter[j] != 0)
                {
                    ++count;
                }
            }
            ASSERT_GE(count, 3);
        }
    }
}

//
// 乱数生成です。
//
TEST(FillSecureRandom, Randomness)
{
    // 乱数列を生成します。
    nn::ldn::detail::FillSecureRandom(g_Random, RandomBytes);

    // 検定の結果です。
    bool isMonobitTestPassed = false;
    bool isPokerTestPassed = false;
    bool isRunTestPassed =false;

    // 検定の性質上、稀に失敗することがあるため、複数回テストを実行します。
    for (int i = 0; i < RepeatCount; ++i)
    {
        // 検定を実施します。
        isMonobitTestPassed |= MonobitTest(g_Random, RandomBytes);
        isPokerTestPassed |= PokerTest(g_Random, RandomBytes);
        isRunTestPassed |= RunTest(g_Random, RandomBytes);
    }

    // 検定の結果を確認します。
    ASSERT_TRUE(isMonobitTestPassed);
    ASSERT_TRUE(isPokerTestPassed);
    ASSERT_TRUE(isRunTestPassed);
}

//
// 先頭がアライメントされていなくても正しく乱数を生成できることを検証します。
//
TEST(FillSecureRandom, Preceding)
{
    const size_t dataSize = 16;
    const int dataCount = 8;

    // オフセットを変更しながらテストを行います。
    for (size_t offset = 0; offset < dataSize; ++offset)
    {
        // 複数回乱数を生成します。
        uint8_t data[dataCount][dataSize] = { };
        for (int i = 0; i < dataCount; ++i)
        {
            nn::ldn::detail::FillSecureRandom(data[i] + offset, sizeof(data[i]) - offset);
        }

        // 全バイトについて、3 種類以上のデータが含まれていれば成功とします。
        for (size_t i = offset; i < dataSize; ++i)
        {
            int8_t counter[256] = { };
            for (int j = 0; j < dataCount; ++j)
            {
                ++counter[data[j][i]];
            }

            int count = 0;
            for (int j = 0; j < 256; ++j)
            {
                if (counter[j] != 0)
                {
                    ++count;
                }
            }
            ASSERT_GE(count, 3);
        }
    }
}

//
// 末尾がアライメントされていなくても正しく乱数を生成できることを検証します。
//
TEST(FillSecureRandom, Following)
{
    const size_t dataSize = 16;
    const int dataCount = 8;

    // サイズを変更しながらテストを行います。
    for (size_t size = 1; size < dataSize; ++size)
    {
        // 複数回乱数を生成します。
        uint8_t data[dataCount][dataSize] = { };
        for (int i = 0; i < dataCount; ++i)
        {
            nn::ldn::detail::FillSecureRandom(data[i], size);
        }

        // 全バイトについて、3 種類以上のデータが含まれていれば成功とします。
        for (size_t i = 0; i < size; ++i)
        {
            int8_t counter[256] = { };
            for (int j = 0; j < dataCount; ++j)
            {
                ++counter[data[j][i]];
            }

            int count = 0;
            for (int j = 0; j < 256; ++j)
            {
                if (counter[j] != 0)
                {
                    ++count;
                }
            }
            ASSERT_GE(count, 3);
        }
    }
}
