﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/ldn/detail/Utility/ldn_Digest.h>
#include <nnt.h>

namespace
{
    char ConvertToHexChar(int n) NN_NOEXCEPT
    {
        if (0 <= n && n <= 9)
        {
            return static_cast<char>('0' + n);
        }
        else
        {
            return static_cast<char>('a' + (n - 10));
        }
    }

    void ConvertToHexString(void* outBuffer, size_t bufferSize,
                            const void* input, size_t inputSize) NN_NOEXCEPT
    {
        ASSERT_TRUE(inputSize * 2 < bufferSize);
        auto out = static_cast<char*>(outBuffer);
        auto in  = static_cast<const uint8_t*>(input);
        for (size_t i = 0; i < inputSize; ++i)
        {
            auto high = (in[i] >> 4) & 0x0F;
            auto low  = in[i] & 0x0F;
            out[i * 2 + 0] = ConvertToHexChar(high);
            out[i * 2 + 1] = ConvertToHexChar(low);
        }
        out[inputSize * 2] = 0;
    }

    template <int N>
    void Verify(const char* hash, size_t hashSize, const char* expected) NN_NOEXCEPT
    {
        ASSERT_EQ(hashSize, N);
        char buffer[N * 2 + 1];
        ConvertToHexString(buffer, sizeof(buffer), hash, hashSize);
        ASSERT_STREQ(expected, buffer);
    }

    char g_Buffer[1024 * 1024];
};

//
// 空の入力に対する署名が仕様と一致することを確認します。
//
TEST(CalculateSha256, Empty)
{
    char   hash[32];
    size_t hashSize;
    nn::ldn::detail::CalculateSha256(hash, &hashSize, sizeof(hash), nullptr, 0);
    Verify<32>(hash, hashSize, "e3b0c44298fc1c149afbf4c8996fb92427ae41e4649b934ca495991b7852b855");
}

//
// 小さな入力に対する署名が仕様と一致することを確認します。
//
TEST(CalculateSha256, SmallData)
{
    char   hash[32];
    size_t hashSize;

    nn::ldn::detail::CalculateSha256(hash, &hashSize, sizeof(hash), "abc", 3);
    Verify<32>(hash, hashSize, "ba7816bf8f01cfea414140de5dae2223b00361a396177a9cb410ff61f20015ad");

    nn::ldn::detail::CalculateSha256(hash, &hashSize, sizeof(hash),
        "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq", 56);
    Verify<32>(hash, hashSize, "248d6a61d20638b8e5c026930c3e6039a33ce45964ff2167f6ecedd419db06c1");
}

//
// 大きな入力に対する署名が仕様と一致することを確認します。
//
TEST(CalculateSha256, LargeData)
{
    char   hash[32];
    size_t hashSize;
    std::memset(g_Buffer, 'a', sizeof(g_Buffer));
    nn::ldn::detail::CalculateSha256(hash, &hashSize, sizeof(hash), g_Buffer, 1000000);
    Verify<32>(hash, hashSize, "cdc76e5c9914fb9281a1c7e284d73e67f1809a48a497200e046d39ccc7112cd0");
}
