﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/ldn/ldn_Types.h>
#include <nn/ldn/detail/NetworkInterface/ldn_Wlan.h>
#include <nnt.h>

namespace
{
    // このテストでは Wlan を使用します。
    typedef nn::ldn::detail::Wlan NetworkInterface;
    const size_t NetworkInterfaceBufferSize = sizeof(nn::ldn::detail::impl::WlanBuffer);

    // ネットワーク・インタフェース用のバッファです。
    NN_ALIGNAS(4096) char g_NetworkInterfaceBuffer[NetworkInterfaceBufferSize];

} // namespace <unnamed>

//
// ネットワーク・インタフェースの初期化と解放を繰り返し実行します。
//
TEST(Initialize, MultipleTimes)
{
    const int repeatCountMax = 256;
    for (int i = 0; i < repeatCountMax; ++i)
    {
        auto* pNetworkInterface = new NetworkInterface(
            g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize);
        ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());
        delete pNetworkInterface;
    }
}

//
// ネットワーク・インタフェースから取得できる情報に問題が無いか確認します。
//
TEST(GetNetworkInterfaceProfile, Verification)
{
    std::unique_ptr<nn::ldn::detail::INetworkInterface> pNetworkInterface(
        new NetworkInterface(g_NetworkInterfaceBuffer, NetworkInterfaceBufferSize));
    ASSERT_EQ(nn::ldn::detail::L2State_None, pNetworkInterface->GetState());

    // ネットワーク・インタフェースの詳細を取得します。
    nn::ldn::detail::NetworkInterfaceProfile profile;
    pNetworkInterface->GetNetworkInterfaceProfile(&profile);

    // MTU は 576 Byte 以上であることが必須です。
    ASSERT_GE(profile.mtu, nn::ldn::detail::MtuMin);

    // 未知のケイパビリティが設定されていないことを確認します。
    ASSERT_EQ(0, profile.capability & (~nn::ldn::detail::NetworkInterfaceCapability_All));

    // ネットワークに接続可能なノードの最大数は 2 以上です。
    ASSERT_GE(profile.nodeCountMax, 2);

    // ネットワークインタフェース種別が正しく設定されていることを確認します。
    ASSERT_NE(nn::ldn::detail::NetworkInterfaceType_Invalid, profile.type);

    // 名前は文字数のみ確認します。
    const size_t nameLength = std::strlen(profile.name);
    ASSERT_TRUE(0U < nameLength && nameLength <= nn::ldn::detail::NetworkInterfaceNameLengthMax);

    // 物理アドレスが正しく設定されていることを確認します。
    ASSERT_NE(nn::ldn::ZeroMacAddress, profile.physicalAddress);
    ASSERT_NE(nn::ldn::BroadcastMacAddress, profile.physicalAddress);

    // ビーコンデータの最大サイズが正しく設定されていることを確認します。
    ASSERT_GT(profile.beaconDataSizeMax, 0);

    // 1 つ以上の無線チャンネルがサポートされています。
    ASSERT_GT(profile.supportedChannelCount, 0);
    for (int i = 0; i < profile.supportedChannelCount; ++i)
    {
        ASSERT_GT(profile.supportedChannels[i], 0);
    }

    // 取得した無線チャンネルに重複が無いことを確認します。
    bool channels[256] = { };
    for (int i = 0; i < profile.supportedChannelCount; ++i)
    {
        ASSERT_FALSE(channels[profile.supportedChannels[i]]);
        channels[profile.supportedChannels[i]] = true;
    }

    // IsSupportedChannels() と対応していることを確認します。
    for (int i = 0; i < 256; ++i)
    {
        ASSERT_EQ(channels[i], pNetworkInterface->IsSupportedChannel(i));
    }
}
