﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <limits>
#include <nn/ldn.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/os.h>
#include <nnt.h>
#include <nnt/ldn/testLdn_CommandLineParser.h>
#include <nnt/ldn/testLdn_Log.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>

namespace
{
    nnt::ldn::TestConfig g_TestConfig;

    const nn::Bit64 LocalId = UINT64_C(0x0005000c10000000);
    const nn::Bit16 SceneId = UINT16_C(0x1200);
    const nn::ldn::IntentId IntentId = nn::ldn::MakeIntentId(LocalId, SceneId);
    const char Passphrase[] = "LdnTestPassphrase";
    const size_t PassphraseSize = std::strlen(Passphrase);
    const char UserName[] = "PerfTest";

    nn::ldn::NetworkInfo g_Networks[nn::ldn::ScanResultCountMax];

    void SetLdnSettings() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetWirelessControllerRestriction(
            static_cast<nn::ldn::WirelessControllerRestriction>(
                g_TestConfig.wirelessControllerRestriction)));
    }

} // namespace <unnamed>

//
// Initialize の実行速度を計測します。
//
TEST(Performance, Initialize)
{
    // テストのパラメータです。
    const int allowedTimeMin = 0;
    const int allowedTimeMax = 8000;
    const int repeatCount = 256;

    // 所要時間のバラツキを考慮して十分な回数繰り返します。
    int timeMin = std::numeric_limits<int>::max();
    int timeMax = std::numeric_limits<int>::min();
    int timeSum = 0;
    for (int i = 0; i < repeatCount; ++i)
    {
        // 初期化処理の実行時間を計測します。
        auto begin = nn::os::GetSystemTick();
        nnt::ldn::SystemInitializer initializer;
        SetLdnSettings();
        auto end = nn::os::GetSystemTick();
        auto time = static_cast<int>((end - begin).ToTimeSpan().GetMilliSeconds());

        // 初期化時間の統計値を更新します。
        timeMin  = std::min(timeMin, time);
        timeMax  = std::max(timeMax, time);
        timeSum += time;
    }

    // テスト結果を出力します。
    NNT_LDN_LOG_INFO("MIN: %d ms\n", timeMin);
    NNT_LDN_LOG_INFO("MAX: %d ms\n", timeMax);
    NNT_LDN_LOG_INFO("AVG: %d ms\n", timeSum / repeatCount);
    ASSERT_GE(timeMin, allowedTimeMin);
    ASSERT_LE(timeMax, allowedTimeMax);
}

//
// Scan の実行速度を計測します。
//
TEST(Performance, Scan)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::StationStarter starter;
    ASSERT_EQ(nn::ldn::State_Station, nn::ldn::GetState());

    nn::ldn::ScanFilter filter;
    filter.flag = nn::ldn::ScanFilterFlag_IntentId;
    filter.networkId.intentId = IntentId;

    // テストのパラメータです。
    const int allowedTimeMin = 330; // 110 ms * 3 ch
    const int allowedTimeMax = 430; // allowedTimeMin + 100 ms (overhead)
    const int repeatCount = 64;

    // スキャン時間のバラツキを考慮して十分な回数繰り返します。
    int timeMin = std::numeric_limits<int>::max();
    int timeMax = std::numeric_limits<int>::min();
    int timeSum = 0;
    for (int i = 0; i < repeatCount; ++i)
    {
        // スキャンの実行時間を計測します。
        auto begin = nn::os::GetSystemTick();
        int count;
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
            g_Networks, &count, nn::ldn::ScanResultCountMax,
            filter, nn::ldn::AutoChannel));
        auto end = nn::os::GetSystemTick();
        auto time = static_cast<int>((end - begin).ToTimeSpan().GetMilliSeconds());

        // スキャン時間の統計値を更新します。
        timeMin  = std::min(timeMin, time);
        timeMax  = std::max(timeMax, time);
        timeSum += time;
    }

    // テスト結果を出力します。
    NNT_LDN_LOG_INFO("MIN: %d ms\n", timeMin);
    NNT_LDN_LOG_INFO("MAX: %d ms\n", timeMax);
    NNT_LDN_LOG_INFO("AVG: %d ms\n", timeSum / repeatCount);
    ASSERT_GE(timeMin, allowedTimeMin);
    ASSERT_LE(timeMax, allowedTimeMax);
}

//
// CreateNetwork の実行速度を計測します。
//
TEST(Performance, CreateNetwork)
{
    nnt::ldn::SystemInitializer initializer;
    SetLdnSettings();
    nnt::ldn::AccessPointStarter starter;
    ASSERT_EQ(nn::ldn::State_AccessPoint, nn::ldn::GetState());

    nn::ldn::NetworkConfig network;
    network.channel = nn::ldn::AutoChannel;
    network.intentId = IntentId;
    network.nodeCountMax = 2;
    nn::ldn::SecurityConfig security;
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = static_cast<uint16_t>(PassphraseSize);
    nn::ldn::UserConfig user;
    std::strncpy(user.userName, UserName, nn::ldn::UserNameBytesMax);

    // テストのパラメータです。
    const int allowedTimeMin = 300; // AutoChannel が作用していることを確認するため下限を設ける
    const int allowedTimeMax = 600;
    const int repeatCount = 64;

    // 所要時間のバラツキを考慮して十分な回数繰り返します。
    int timeMin = std::numeric_limits<int>::max();
    int timeMax = std::numeric_limits<int>::min();
    int timeSum = 0;
    for (int i = 0; i < repeatCount; ++i)
    {
        // ネットワーク構築の実行時間を計測します。
        auto begin = nn::os::GetSystemTick();
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        auto end = nn::os::GetSystemTick();
        auto time = static_cast<int>((end - begin).ToTimeSpan().GetMilliSeconds());
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::DestroyNetwork());

        // ネットワーク構築に要した時間の統計値を更新します。
        timeMin  = std::min(timeMin, time);
        timeMax  = std::max(timeMax, time);
        timeSum += time;
    }

    // テスト結果を出力します。
    NNT_LDN_LOG_INFO("MIN: %d ms\n", timeMin);
    NNT_LDN_LOG_INFO("MAX: %d ms\n", timeMax);
    NNT_LDN_LOG_INFO("AVG: %d ms\n", timeSum / repeatCount);
    ASSERT_GE(timeMin, allowedTimeMin);
    ASSERT_LE(timeMax, allowedTimeMax);
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::ldn::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::ldn::CommandLineOptionFlag_OperationMode |
        nnt::ldn::CommandLineOptionFlag_WirelessControllerRestriction);

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::ldn::Parse(&g_TestConfig, setting, argc, argv);

    // 無線スイッチをオンにします。
    {
        nnt::ldn::NifmInitializer nifmInitializer;
        nnt::ldn::SetWirelessCommunicationSwitch(true);
    }

    // テストを実行します。
    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}
