﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * 1 台の開発機だけで実行できるテストです。
 */

#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/lcs/lcs_PrivateDebugApi.h>
#include <nn/ldn.h>
#include <nn/nn_Log.h>
#include <nn/ns/ns_ApplicationRecordApi.h>
#include <nn/ns/ns_ApplicationViewApi.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/socket.h>
#include <nnt.h>
#include <nnt/lcs/testLcs_Utility.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>
#include "Precondition.h"

namespace
{
    // LCS/Socket の初期化用バッファです。
    NN_ALIGNAS(nn::lcs::RequiredBufferAlignment) char g_Buffer[4 * 1024 * 1024];
    NN_STATIC_ASSERT(nn::lcs::RequiredBufferSize < sizeof(g_Buffer));
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // EULA 取得用のバッファです。
    char g_EulaBuffer[1 * 1024 * 1024];

    void ValidateSessionInfo(
        const nn::lcs::SessionInfo& session, const nn::lcs::Config& config,
        const nn::lcs::SessionSettings& settings) NN_NOEXCEPT
    {
        // ユーザー名の検証です。
        char name[nn::lcs::UserNameBytesMax + 1];
        size_t size;
        config.GetName(name, &size, sizeof(name));
        ASSERT_STREQ(name, session.hostUserName);

        // 配信されるコンテンツ情報の検証です。
        ASSERT_EQ(settings.applicationCount, session.contentsCount);
        for (int i = 0; i < settings.applicationCount; ++i)
        {
            const auto& cond = GetPrecondition(nullptr, nullptr, settings.applications[i]);
            const auto& app = cond.application;
            const auto& content = session.contents[i];
            ASSERT_EQ(false, content.contentsFlag.Test<nn::lcs::ContentsType::System>());
            ASSERT_EQ(false, content.contentsFlag.Test<nn::lcs::ContentsType::Application>());
            ASSERT_EQ(true, content.contentsFlag.Test<nn::lcs::ContentsType::Patch>());
            ASSERT_STREQ(app.latestDisplayVersion, content.displayVersion);
            ASSERT_EQ(0U, content.attributeFlag);
            ASSERT_EQ(settings.applications[i], content.applicationId);
        }

        // 端末接続数の検証です。この時点では接続数 1 を想定しています。
        ASSERT_EQ(settings.nodeCountMax, session.nodeCountMax);
        ASSERT_EQ(1, session.nodeCount);

        // クライアントが存在しない状況ではリンクレベル最大です。
        ASSERT_EQ(nn::lcs::LinkLevelMax, session.linkLevel);
    }

} // namespace <unnamed>

//
// パッチをもっていなくてもローカルパッチ配信を開始できることを確認します。
//
TEST(OpenSession, NoPatch)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone01);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // セッション情報を取得して正しくセッションを構築できていることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
}

//
// 最小のバッファサイズでセッションを構築できることを確認します。
//
TEST(OpenSession, MinimumBufferSize)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, nn::lcs::RequiredBufferSize, config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone02);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // セッション情報を取得して正しくセッションを構築できていることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
}

//
// 最大長のユーザー名でローカルパッチ配信を開始できることを確認します。
//
TEST(OpenSession, UserNameBytesMax)
{
    nn::lcs::Config config = nnt::lcs::CreateConfig(nnt::lcs::LongestUserName);

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone02);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // セッション情報を取得して正しくセッションを構築できていることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::LongestUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
}

//
// 最小の最大端末数でローカルパッチ配信を開始できることを確認します。
//
TEST(OpenSession, MinimumNodeCountMax)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        1, nnt::lcs::Application::Standalone02);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // セッション情報を取得して正しくセッションを構築できていることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
}

//
// OpenSession() の後、LeaveSession() ではなく Finalize() でセッションを抜けます。
//
TEST(OpenSession, LeaveByFinalize)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        1, nnt::lcs::Application::Standalone02);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // セッション情報を取得して正しくセッションを構築できていることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);
}

//
// OpenSession() に存在しないアプリケーション識別子を指定すると失敗します。
//
TEST(OpenSession, ApplicationNotFound)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 無効な ApplicationID を設定します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Invalid);

    // セッションを構築できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultApplicationNotFound, nn::lcs::OpenSession(settings));
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
}

//
// パッチをもっていなくてもローカルパッチ配信を開始できることを確認します。
//
TEST(ContentsShare, NoPatch)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone01);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // 配信を開始します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_NE(nn::lcs::State_Opened, nn::lcs::GetState());

    // 配信の完了まで待機します。
    while (nn::lcs::GetState() != nn::lcs::State_Completed)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, nn::TimeSpan::FromSeconds(5)));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 正しくセッション情報を取得できることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    ASSERT_EQ(myIndex, nn::lcs::GetMyIndex());

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // 何も配信されないため、進捗はありません。
    nn::lcs::Progress progress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetProgress(&progress));
    ASSERT_EQ(0U, progress.size);
    ASSERT_EQ(0U, progress.downloadedSize);
    ASSERT_EQ(nn::lcs::TransferRole_None, progress.transferRole);

    // 何も配信されないため、進捗はありません。
    nn::lcs::NodeProgress nodeProgress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodeProgress(&nodeProgress, myIndex));
    ASSERT_EQ(0, nodeProgress.contentCount);
    ASSERT_EQ(0, nodeProgress.downloadedContentCount);

    // 何もダウンロードしていません。
    nn::lcs::ContentsInfo content;
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(&content, &count, 1));
    ASSERT_EQ(0, count);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
}

//
// 最小限のバッファでローカルコンテンツ配信を開始できることを検証します。
//
TEST(ContentsShare, MiminumBufferSize)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, nn::lcs::RequiredBufferSize, config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone02);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // 配信を開始します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_NE(nn::lcs::State_Opened, nn::lcs::GetState());

    // 配信の完了まで待機します。
    while (nn::lcs::GetState() != nn::lcs::State_Completed)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, nn::TimeSpan::FromSeconds(5)));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 正しくセッション情報を取得できることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    ASSERT_EQ(myIndex, nn::lcs::GetMyIndex());

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // 何も配信されないため、進捗はありません。
    nn::lcs::Progress progress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetProgress(&progress));
    ASSERT_EQ(0U, progress.size);
    ASSERT_EQ(0U, progress.downloadedSize);
    ASSERT_EQ(nn::lcs::TransferRole_None, progress.transferRole);

    // 何も配信されないため、進捗はありません。
    nn::lcs::NodeProgress nodeProgress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodeProgress(&nodeProgress, myIndex));
    ASSERT_EQ(0, nodeProgress.contentCount);
    ASSERT_EQ(0, nodeProgress.downloadedContentCount);

    // 何もダウンロードしていません。
    nn::lcs::ContentsInfo content;
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(&content, &count, 1));
    ASSERT_EQ(0, count);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
}

//
// 最大長のバッファでローカルコンテンツ配信を開始できることを検証します。
//
TEST(ContentsShare, UserNameBytesMax)
{
    nn::lcs::Config config = nnt::lcs::CreateConfig(nnt::lcs::LongestUserName);

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone02);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // 配信を開始します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_NE(nn::lcs::State_Opened, nn::lcs::GetState());

    // 配信の完了まで待機します。
    while (nn::lcs::GetState() != nn::lcs::State_Completed)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, nn::TimeSpan::FromSeconds(5)));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 正しくセッション情報を取得できることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    ASSERT_EQ(myIndex, nn::lcs::GetMyIndex());

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::LongestUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // 何も配信されないため、進捗はありません。
    nn::lcs::Progress progress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetProgress(&progress));
    ASSERT_EQ(0U, progress.size);
    ASSERT_EQ(0U, progress.downloadedSize);
    ASSERT_EQ(nn::lcs::TransferRole_None, progress.transferRole);

    // 何も配信されないため、進捗はありません。
    nn::lcs::NodeProgress nodeProgress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodeProgress(&nodeProgress, myIndex));
    ASSERT_EQ(0, nodeProgress.contentCount);
    ASSERT_EQ(0, nodeProgress.downloadedContentCount);

    // 何もダウンロードしていません。
    nn::lcs::ContentsInfo content;
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(&content, &count, 1));
    ASSERT_EQ(0, count);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
}

//
// 最小の最大接続端末数でローカルコンテンツ配信を開始できることを検証します。
//
TEST(ContentsShare, MiminumNodeCountMax)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        1, nnt::lcs::Application::Standalone02);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // 配信を開始します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_NE(nn::lcs::State_Opened, nn::lcs::GetState());

    // 配信の完了まで待機します。
    while (nn::lcs::GetState() != nn::lcs::State_Completed)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, nn::TimeSpan::FromSeconds(5)));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 正しくセッション情報を取得できることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    ASSERT_EQ(myIndex, nn::lcs::GetMyIndex());

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // 何も配信されないため、進捗はありません。
    nn::lcs::Progress progress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetProgress(&progress));
    ASSERT_EQ(0U, progress.size);
    ASSERT_EQ(0U, progress.downloadedSize);
    ASSERT_EQ(nn::lcs::TransferRole_None, progress.transferRole);

    // 何も配信されないため、進捗はありません。
    nn::lcs::NodeProgress nodeProgress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodeProgress(&nodeProgress, myIndex));
    ASSERT_EQ(0, nodeProgress.contentCount);
    ASSERT_EQ(0, nodeProgress.downloadedContentCount);

    // 何もダウンロードしていません。
    nn::lcs::ContentsInfo content;
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(&content, &count, 1));
    ASSERT_EQ(0, count);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
}

//
// ローカルコンテンツの開始後、LeaveSession ではなく Finalize でセッションを破棄します。
//
TEST(ContentsShare, LeaveByFinalize)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // 対象のアプリケーションを取得します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone02);

    // セッションを構築します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());

    // インデックスが正しく設定されていることを確認します。
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);

    // 配信を開始します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent)); // Manual Clear
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_NE(nn::lcs::State_Opened, nn::lcs::GetState());

    // 配信の完了まで待機します。
    while (nn::lcs::GetState() != nn::lcs::State_Completed)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, nn::TimeSpan::FromSeconds(5)));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 正しくセッション情報を取得できることを確認します。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionInfo(&session));
    ValidateSessionInfo(session, config, settings);

    // インデックスが正しく設定されていることを確認します。
    ASSERT_EQ(myIndex, nn::lcs::GetMyIndex());

    // クライアントは接続されていません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));
    ASSERT_EQ(1, nodeCount);
    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
    ASSERT_EQ(myIndex, nodes[0].index);

    // 何も配信されないため、インストールサイズは 0 になるはずです。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetRequiredStorageSize(&requiredStorageSize));
    ASSERT_EQ(0U, requiredStorageSize);

    // 何も配信されないため、進捗はありません。
    nn::lcs::Progress progress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetProgress(&progress));
    ASSERT_EQ(0U, progress.size);
    ASSERT_EQ(0U, progress.downloadedSize);
    ASSERT_EQ(nn::lcs::TransferRole_None, progress.transferRole);

    // 何も配信されないため、進捗はありません。
    nn::lcs::NodeProgress nodeProgress;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodeProgress(&nodeProgress, myIndex));
    ASSERT_EQ(0, nodeProgress.contentCount);
    ASSERT_EQ(0, nodeProgress.downloadedContentCount);

    // 何もダウンロードしていません。
    nn::lcs::ContentsInfo content;
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetDownloadedContents(&content, &count, 1));
    ASSERT_EQ(0, count);

    // セッションを破棄します。
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
}

//
// 最小のバッファサイズで Scan() を実行します。
//
TEST(Scan, BufferCountMin)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, nn::lcs::RequiredBufferSize, config);

    // 最小のバッファサイズで Scan() を実行します。
    nn::lcs::SessionInfo sessions[1];
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::Scan(sessions, &count, 1));
    ASSERT_GE(count, 0);
    ASSERT_LE(count, 1);
}

//
// 最大のバッファサイズで Scan() を実行します。
//
TEST(Scan, BufferCountMax)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, nn::lcs::RequiredBufferSize, config);

    // 最小のバッファサイズで Scan() を実行します。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::Scan(sessions, &count, nn::lcs::ScanResultCountMax));
    ASSERT_GE(count, 0);
    ASSERT_LE(count, nn::lcs::ScanResultCountMax);
}

//
// 不正な引数を指定して GetNodeProgress() を使用します。
//
TEST(InvalidArgument, GetNodeProgress)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションを構築します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone02);
    nnt::lcs::HostStarter starter(settings);

    // 配信を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());

    // 配信の完了まで待機します。
    while (nn::lcs::GetState() != nn::lcs::State_Completed)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, nn::TimeSpan::FromSeconds(5)));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // 不正なインデックスを指定すると ResultNodeNotFound を返します。
    nn::lcs::NodeProgress nodeProgress;
    auto myIndex = nn::lcs::GetMyIndex();
    ASSERT_NE(0U, myIndex);
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultNodeNotFound, nn::lcs::GetNodeProgress(&nodeProgress, myIndex + 1U));
}

//
// 不正な引数を指定して ResumeSession() を使用します。
//
TEST(InvalidArgument, ResumeSession)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // 不正なコンテキストを指定すると ResultInvalidContext を返します。
    nn::lcs::SessionContext context = {};
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidContext, nn::lcs::ResumeSession(context));
}

//
// Initialized 状態で実行できない API を実行して ResultInvalidState が返ることを確認します。
//
TEST(InvalidState, Initialized)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // State_Initialized 状態では GetSessionInfo() は使用できません。
    nn::lcs::SessionInfo session;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetSessionInfo(&session));

    // State_Initialized 状態では GetMyIndex() は使用できません。
    ASSERT_EQ(0U, nn::lcs::GetMyIndex());

    // State_Initialized では GetNodes() は使用できません。
    nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
    int nodeCount;
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState,
        nn::lcs::GetNodes(nodes, &nodeCount, nn::lcs::NodeCountMax));

    // State_Initialized 状態では GetRequiredStorageSize() は使用できません。
    size_t requiredStorageSize;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetRequiredStorageSize(&requiredStorageSize));

    // State_Initialized 状態では GetProgress() は使用できません。
    nn::lcs::Progress progress;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetProgress(&progress));

    // State_Initialized 状態では GetNodeProgress() は使用できません。
    nn::lcs::NodeProgress noreProgress;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetNodeProgress(&noreProgress, 0U));

    // State_Initialized 状態では GetDownloadedContents() は使用できません。
    nn::lcs::ContentsInfo content;
    int contentCount;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedContents(&content, &contentCount, 1));

    // State_Initialized 状態では GetDownloadedEulaDataSize() は使用できません。
    size_t eulaSize;
    const char* path = "JPja/Eula.msbt.szs";
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedEulaDataSize(&eulaSize, path));

    // State_Initialized 状態では GetDownloadedEulaData() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedEulaData(&eulaSize, g_EulaBuffer, sizeof(g_EulaBuffer), path));

    // State_Initialized 状態では GetSessionContext() は使用できません。
    nn::lcs::SessionContext context;
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetSessionContext(&context));

    // State_Initialized 状態では ResumeContentsShare() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeContentsShare());

    // State_Initialized 状態では LeaveSession() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::LeaveSession());

    // State_Initialized 状態では GetContentsShareFailureReason() は使用できません。
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_None, nn::lcs::GetContentsShareFailureReason());

    // State_Initialized 状態では GetSuspendedReason() は使用できません。
    ASSERT_EQ(nn::lcs::SuspendedReason_None, nn::lcs::GetSuspendedReason());

    // State_Initialized では SetClientAcceptPolicy() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState,
        nn::lcs::SetClientAcceptPolicy(nn::lcs::AcceptPolicy_AlwaysReject));

    // State_Initialized では StartContentsShare() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::StartContentsShare());
}

//
// Opened 状態で実行できない API を実行して ResultInvalidState が返ることを確認します。
//
TEST(InvalidState, Opened)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // セッションを構築します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone02);
    nnt::lcs::HostStarter starter(settings);

    // State_Opened 状態では GetDownloadedContents() は使用できません。
    nn::lcs::ContentsInfo content;
    int contentCount;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedContents(&content, &contentCount, 1));

    // State_Opened 状態では GetDownloadedEulaDataSize() は使用できません。
    size_t eulaSize;
    const char* path = "JPja/Eula.msbt.szs";
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedEulaDataSize(&eulaSize, path));

    // State_Opened 状態では GetDownloadedEulaData() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedEulaData(&eulaSize, g_EulaBuffer, sizeof(g_EulaBuffer), path));

    // State_Opened 状態では GetSessionContext() は使用できません。
    nn::lcs::SessionContext context = {};
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetSessionContext(&context));

    // State_Opened 状態では ResumeContentsShare() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeContentsShare());

    // State_Opened 状態では ResumeSession() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeSession(context));

    // State_Opened 状態では GetContentsShareFailureReason() は使用できません。
     ASSERT_EQ(nn::lcs::ContentsShareFailureReason_None, nn::lcs::GetContentsShareFailureReason());

    // State_Opened 状態では GetSuspendedReason() は使用できません。
    ASSERT_EQ(nn::lcs::SuspendedReason_None, nn::lcs::GetSuspendedReason());

    // State_Opened 状態では OpenSession() は実行できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::OpenSession(settings));

    // State_Opened では SetClientAcceptPolicy() が成功します。
    NNT_ASSERT_RESULT_SUCCESS(
        nn::lcs::SetClientAcceptPolicy(nn::lcs::AcceptPolicy_AlwaysReject));
    NNT_ASSERT_RESULT_SUCCESS(
        nn::lcs::SetClientAcceptPolicy(nn::lcs::AcceptPolicy_AlwaysAccept));

    // State_Opened 状態では Scan() は使用できません。
    nn::lcs::SessionInfo session = {};
    int scanResultCount;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::Scan(&session, &scanResultCount, 1));

    // State_Opened 状態では JoinSession() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::JoinSession(session));
}

//
// Completed 状態で実行できない API を実行して ResultInvalidState が返ることを確認します。
//
TEST(InvalidState, Completed)
{
    nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // セッションを構築します。
    nn::lcs::SessionSettings settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, nnt::lcs::Application::Standalone02);
    nnt::lcs::HostStarter starter(settings);

    // 配信を開始します。
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());

    // 配信の完了まで待機します。
    while (nn::lcs::GetState() != nn::lcs::State_Completed)
    {
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, nn::TimeSpan::FromSeconds(5)));
        nn::os::ClearSystemEvent(&stateChangeEvent);
    }

    // State_Completed 状態では GetDownloadedEulaDataSize() は使用できません。
    size_t eulaSize;
    const char* path = "JPja/Eula.msbt.szs";
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::GetDownloadedEulaDataSize(&eulaSize, path));

    // State_Completed 状態では GetDownloadedEulaData() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState,
        nn::lcs::GetDownloadedEulaData(&eulaSize, g_EulaBuffer, sizeof(g_EulaBuffer), path));

    // State_Completed 状態では GetSessionContext() は使用できません。
    nn::lcs::SessionContext context = {};
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::GetSessionContext(&context));

    // State_Completed 状態では ResumeContentsShare() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeContentsShare());

    // State_Completed 状態では ResumeSession() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::ResumeSession(context));

    // State_Completed 状態では GetContentsShareFailureReason() は使用できません。
    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_None, nn::lcs::GetContentsShareFailureReason());

    // State_Completed 状態では GetSuspendedReason() は使用できません。
    ASSERT_EQ(nn::lcs::SuspendedReason_None, nn::lcs::GetSuspendedReason());

    // State_Completed 状態では OpenSession() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::OpenSession(settings));

    // State_Completed 状態では SetClientAcceptPolicy() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState,
        nn::lcs::SetClientAcceptPolicy(nn::lcs::AcceptPolicy_AlwaysReject));

    // State_Completed では StartContentsShare() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::StartContentsShare());

    // State_Completed 状態では Scan() は使用できません。
    nn::lcs::SessionInfo session = {};
    int scanResultCount;
    NNT_ASSERT_RESULT_FAILURE(
        nn::lcs::ResultInvalidState, nn::lcs::Scan(&session, &scanResultCount, 1));

    // State_Completed 状態では JoinSession() は使用できません。
    NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, nn::lcs::JoinSession(session));
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::lcs::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::lcs::CommandLineOptionFlag_NodeCount |
        nnt::lcs::CommandLineOptionFlag_NodeIndex |
        nnt::lcs::CommandLineOptionFlag_SceneId);
    setting.nodeCountMin = 1;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax;

    // コマンドライン引数を解析します。
    nnt::lcs::TestConfig config;
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::lcs::Parse(&config, setting, argc, argv);

    // Socket ライブラリを初期化します。
    nn::socket::Initialize(g_SocketConfig);

    // テストの準備です。
    nnt::lcs::Setup(config.sceneId);

    // テストを実行します。
    int result = RUN_ALL_TESTS();
    nnt::Exit(result);
}
