﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lcs.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnt/nntest.h>

namespace nnt { namespace lcs { namespace
{
}}} // namespace nnt::lcs::<unnamed>

namespace nnt { namespace lcs
{
    nn::Result TimedScan(
        nn::lcs::SessionInfo* pOutSession, int* pOutCount, int* pOutIndex, int bufferCount,
        const char* hostName, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // スキャン開始時刻を記憶します。
        nn::os::Tick start = nn::os::GetSystemTick();
        nn::os::Tick now = nn::os::GetSystemTick();
        nn::os::Tick diff = now - start;

        // 目的のセッションを発見するか、タイムアウトするまで検索します。
        do
        {
            // セッションをスキャンします。
            NN_RESULT_DO(nn::lcs::Scan(pOutSession, pOutCount, bufferCount));

            // 目的のセッションがスキャン結果に含まれるかどうかを確認します。
            for (int i = 0; i < *pOutCount; ++i)
            {
                const auto& session = pOutSession[i];
                if (std::strncmp(hostName, session.hostUserName, nn::lcs::UserNameBytesMax) == 0)
                {
                    *pOutIndex = i;
                    NN_RESULT_SUCCESS;
                }
            }

            // 目的のセッションが含まれなかった場合はリトライします。
            now = nn::os::GetSystemTick();
            diff = now - start;
        } while (diff.ToTimeSpan() < timeout);

        return nn::lcs::ResultSessionNotFound();
    }

    nn::Result TimedJoin(
        const nn::lcs::SessionInfo& session, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // 接続試行の開始時刻を記憶します。
        nn::os::Tick start = nn::os::GetSystemTick();
        nn::os::Tick now = nn::os::GetSystemTick();
        nn::os::Tick diff = now - start;

        // セッションへの接続に成功するかタイムアウトするまでリトライします。
        nn::Result result;
        do
        {
            result = nn::lcs::JoinSession(session);

            // 成功した場合、意図しない Result の場合は return します。
            if (nn::lcs::ResultCommunicationError::Includes(result))
            {
                now = nn::os::GetSystemTick();
                diff = now - start;
            }
            else
            {
                break;
            }
        } while (diff.ToTimeSpan() < timeout);
        return result;
    }

    nn::Result TimedResumeSession(
        const nn::lcs::SessionContext& context, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // 接続試行の開始時刻を記憶します。
        nn::os::Tick start = nn::os::GetSystemTick();
        nn::os::Tick now = nn::os::GetSystemTick();
        nn::os::Tick diff = now - start;

        // セッションへの接続に成功するかタイムアウトするまでリトライします。
        nn::Result result;
        do
        {
            result = nn::lcs::ResumeSession(context);

            // 成功した場合、意図しない Result の場合は return します。
            if (nn::lcs::ResultCommunicationError::Includes(result))
            {
                now = nn::os::GetSystemTick();
                diff = now - start;
            }
            else
            {
                break;
            }
        } while (diff.ToTimeSpan() < timeout);
        return result;
    }

    void TimedWaitTransition(
        nn::lcs::State state,
        nn::os::SystemEventType* pEvent,
        nn::TimeSpan timeout) NN_NOEXCEPT
    {
        // 接続試行の開始時刻を記憶します。
        nn::os::Tick start = nn::os::GetSystemTick();
        nn::os::Tick now = nn::os::GetSystemTick();
        auto diff = (now - start).ToTimeSpan();

        // 状態遷移の完了まで待機します。
        nn::lcs::State currentState;
        while ((currentState = nn::lcs::GetState()) != state && diff < timeout)
        {
            auto rest = timeout - diff;
            ASSERT_EQ(nn::lcs::State_Transferring, currentState);
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(pEvent, rest));
            nn::os::ClearSystemEvent(pEvent);
        }
        nn::os::ClearSystemEvent(pEvent);
        ASSERT_EQ(state, currentState);
    }

}} // namespace nnt::lcs
